// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
#include "vtkTriangularTexture.h"

#include "vtkImageData.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkMath.h"
#include "vtkObjectFactory.h"
#include "vtkPointData.h"
#include "vtkStreamingDemandDrivenPipeline.h"
#include "vtkUnsignedCharArray.h"

VTK_ABI_NAMESPACE_BEGIN
vtkStandardNewMacro(vtkTriangularTexture);

// Instantiate object with XSize and YSize = 64; the texture pattern =1
// (opaque at centroid); and the scale factor set to 1.0.

vtkTriangularTexture::vtkTriangularTexture()
{
  this->XSize = this->YSize = 64;
  this->TexturePattern = 1;
  this->ScaleFactor = 1.0;
  this->SetNumberOfInputPorts(0);
}

static void vtkOpaqueAtElementCentroid(
  int XSize, int YSize, double ScaleFactor, vtkUnsignedCharArray* newScalars)
{
  int i, j;
  double opacity;
  double point[3];
  double XScale = XSize + 1.0;
  double YScale = YSize + 1.0;
  unsigned char AGrayValue[2];
  double dist, distToV2, distToV3;
  double v1[3] = { 0.0, 0.0, 0.0 };
  double v2[3] = { 1.0, 0.0, 0.0 };
  double v3[3] = { 0.5, sqrt(3.0) / 2.0, 0.0 };

  point[2] = 0.0;
  AGrayValue[0] = AGrayValue[1] = 255;

  for (j = 0; j < YSize; j++)
  {
    for (i = 0; i < XSize; i++)
    {
      point[0] = i / XScale;
      point[1] = j / YScale;
      dist = vtkMath::Distance2BetweenPoints(point, v1);
      distToV2 = vtkMath::Distance2BetweenPoints(point, v2);
      dist = std::min(distToV2, dist);
      distToV3 = vtkMath::Distance2BetweenPoints(point, v3);
      dist = std::min(distToV3, dist);

      opacity = sqrt(dist) * ScaleFactor;
      if (opacity < .5)
      {
        opacity = 0.0;
      }
      if (opacity > .5)
      {
        opacity = 1.0;
      }
      AGrayValue[1] = static_cast<unsigned char>(opacity * 255);
      newScalars->SetValue((XSize * j + i) * 2, AGrayValue[0]);
      newScalars->SetValue((XSize * j + i) * 2 + 1, AGrayValue[1]);
    }
  }
}

static void vtkOpaqueAtVertices(
  int XSize, int YSize, double ScaleFactor, vtkUnsignedCharArray* newScalars)
{
  int i, j;
  double opacity;
  double point[3];
  double XScale = XSize + 1.0;
  double YScale = YSize + 1.0;
  unsigned char AGrayValue[2];
  double dist, distToV2, distToV3;
  double v1[3] = { 0.0, 0.0, 0.0 };
  double v2[3] = { 1.0, 0.0, 0.0 };
  double v3[3] = { 0.5, sqrt(3.0) / 2.0, 0.0 };

  point[2] = 0.0;
  AGrayValue[0] = AGrayValue[1] = 255;

  for (j = 0; j < YSize; j++)
  {
    for (i = 0; i < XSize; i++)
    {
      point[0] = i / XScale;
      point[1] = j / YScale;
      dist = vtkMath::Distance2BetweenPoints(point, v1);
      distToV2 = vtkMath::Distance2BetweenPoints(point, v2);
      dist = std::min(distToV2, dist);
      distToV3 = vtkMath::Distance2BetweenPoints(point, v3);
      dist = std::min(distToV3, dist);

      opacity = sqrt(dist) * ScaleFactor;
      if (opacity < .5)
      {
        opacity = 0.0;
      }
      if (opacity > .5)
      {
        opacity = 1.0;
      }
      opacity = 1.0 - opacity;
      AGrayValue[1] = static_cast<unsigned char>(opacity * 255);
      newScalars->SetValue((XSize * j + i) * 2, AGrayValue[0]);
      newScalars->SetValue((XSize * j + i) * 2 + 1, AGrayValue[1]);
    }
  }
}

//------------------------------------------------------------------------------
int vtkTriangularTexture::RequestInformation(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** vtkNotUsed(inputVector), vtkInformationVector* outputVector)
{
  // get the info objects
  vtkInformation* outInfo = outputVector->GetInformationObject(0);

  int wExt[6] = { 0, this->XSize - 1, 0, this->YSize - 1, 0, 0 };

  outInfo->Set(vtkStreamingDemandDrivenPipeline::WHOLE_EXTENT(), wExt, 6);
  vtkDataObject::SetPointDataActiveScalarInfo(outInfo, VTK_UNSIGNED_CHAR, 2);
  return 1;
}

void vtkTriangularTexture::ExecuteDataWithInformation(vtkDataObject* outp, vtkInformation* outInfo)
{
  vtkImageData* output = this->AllocateOutputData(outp, outInfo);
  vtkUnsignedCharArray* newScalars =
    vtkArrayDownCast<vtkUnsignedCharArray>(output->GetPointData()->GetScalars());

  if (this->XSize * this->YSize < 1)
  {
    vtkErrorMacro(<< "Bad texture (xsize,ysize) specification!");
    return;
  }

  switch (this->TexturePattern)
  {
    case 1: // opaque at element vertices
      vtkOpaqueAtVertices(this->XSize, this->YSize, this->ScaleFactor, newScalars);
      break;

    case 2: // opaque at element centroid
      vtkOpaqueAtElementCentroid(this->XSize, this->YSize, this->ScaleFactor, newScalars);
      break;

    case 3: // opaque in rings around vertices
      vtkErrorMacro(<< "Opaque vertex rings not implemented");
      break;
  }
}

void vtkTriangularTexture::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);

  os << indent << "XSize:" << this->XSize << "\n";
  os << indent << "YSize:" << this->YSize << "\n";

  os << indent << "Texture Pattern:" << this->TexturePattern << "\n";

  os << indent << "Scale Factor:" << this->ScaleFactor << "\n";
}
VTK_ABI_NAMESPACE_END
