/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkMutableGraphHelper.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/*----------------------------------------------------------------------------
 Copyright (c) Sandia Corporation
 See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.
----------------------------------------------------------------------------*/

#include "vtkMutableGraphHelper.h"

#include "vtkGraphEdge.h"
#include "vtkMolecule.h"
#include "vtkMutableDirectedGraph.h"
#include "vtkMutableUndirectedGraph.h"
#include "vtkObjectFactory.h"

vtkCxxSetObjectMacro(vtkMutableGraphHelper, InternalGraph, vtkGraph);
vtkStandardNewMacro(vtkMutableGraphHelper);
//----------------------------------------------------------------------------
vtkMutableGraphHelper::vtkMutableGraphHelper()
{
  this->InternalGraph = nullptr;
  this->Molecule = nullptr;
  this->DirectedGraph = nullptr;
  this->UndirectedGraph = nullptr;
  this->GraphEdge = vtkGraphEdge::New();
  this->GraphEdge->SetId(-1);
  this->GraphEdge->SetSource(-1);
  this->GraphEdge->SetTarget(-1);
}

//----------------------------------------------------------------------------
vtkMutableGraphHelper::~vtkMutableGraphHelper()
{
  if (this->InternalGraph)
  {
    this->InternalGraph->Delete();
  }
  this->GraphEdge->Delete();
}

//----------------------------------------------------------------------------
void vtkMutableGraphHelper::SetGraph(vtkGraph* g)
{
  this->SetInternalGraph(g);
  this->DirectedGraph = vtkMutableDirectedGraph::SafeDownCast(this->InternalGraph);
  this->UndirectedGraph = vtkMutableUndirectedGraph::SafeDownCast(this->InternalGraph);
  this->Molecule = vtkMolecule::SafeDownCast(this->InternalGraph);
  if (!this->DirectedGraph && !this->UndirectedGraph && !this->Molecule)
  {
    vtkErrorMacro("The graph must be mutable.");
  }
}

//----------------------------------------------------------------------------
vtkGraph* vtkMutableGraphHelper::GetGraph()
{
  return this->GetInternalGraph();
}

//----------------------------------------------------------------------------
vtkIdType vtkMutableGraphHelper::AddVertex()
{
  if (!this->InternalGraph)
  {
    return -1;
  }
  if (this->DirectedGraph)
  {
    return this->DirectedGraph->AddVertex();
  }
  else if (this->UndirectedGraph)
  {
    return this->UndirectedGraph->AddVertex();
  }
  else if (this->Molecule)
  {
    vtkAtom atom = this->Molecule->AppendAtom();
    return atom.GetId();
  }

  return vtkIdType();
}

//----------------------------------------------------------------------------
vtkEdgeType vtkMutableGraphHelper::AddEdge(vtkIdType u, vtkIdType v)
{
  if (!this->InternalGraph)
  {
    return vtkEdgeType();
  }
  if (this->DirectedGraph)
  {
    return this->DirectedGraph->AddEdge(u, v);
  }
  else if (this->UndirectedGraph)
  {
    return this->UndirectedGraph->AddEdge(u, v);
  }
  else if (this->Molecule)
  {
    vtkBond bond = this->Molecule->AppendBond(u, v);
    return vtkEdgeType(u, v, bond.GetId());
  }

  return vtkEdgeType();
}

//----------------------------------------------------------------------------
vtkGraphEdge* vtkMutableGraphHelper::AddGraphEdge(vtkIdType u, vtkIdType v)
{
  if (!this->InternalGraph || this->Molecule)
  {
    return this->GraphEdge;
  }
  if (this->DirectedGraph)
  {
    return this->DirectedGraph->AddGraphEdge(u, v);
  }
  else
  {
    return this->UndirectedGraph->AddGraphEdge(u, v);
  }
}

//----------------------------------------------------------------------------
void vtkMutableGraphHelper::RemoveVertex(vtkIdType v)
{
  if (!this->InternalGraph)
  {
    return;
  }
  if (this->DirectedGraph)
  {
    this->DirectedGraph->RemoveVertex(v);
  }
  else if (this->UndirectedGraph)
  {
    this->UndirectedGraph->RemoveVertex(v);
  }
  else
  {
    // todo in vtkMolecule !!
    // return this->Molecule->RemoveAtom(v);
  }
}

//----------------------------------------------------------------------------
void vtkMutableGraphHelper::RemoveVertices(vtkIdTypeArray* verts)
{
  if (!this->InternalGraph)
  {
    return;
  }
  if (this->DirectedGraph)
  {
    return this->DirectedGraph->RemoveVertices(verts);
  }
  else
  {
    return this->UndirectedGraph->RemoveVertices(verts);
  }
}

//----------------------------------------------------------------------------
void vtkMutableGraphHelper::RemoveEdge(vtkIdType e)
{
  if (!this->InternalGraph)
  {
    return;
  }
  if (this->DirectedGraph)
  {
    return this->DirectedGraph->RemoveEdge(e);
  }
  else
  {
    return this->UndirectedGraph->RemoveEdge(e);
  }
}

//----------------------------------------------------------------------------
void vtkMutableGraphHelper::RemoveEdges(vtkIdTypeArray* edges)
{
  if (!this->InternalGraph)
  {
    return;
  }
  if (this->DirectedGraph)
  {
    return this->DirectedGraph->RemoveEdges(edges);
  }
  else
  {
    return this->UndirectedGraph->RemoveEdges(edges);
  }
}

//----------------------------------------------------------------------------
void vtkMutableGraphHelper::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
  os << indent << "InternalGraph: " << (this->InternalGraph ? "" : "(null)") << endl;
  if (this->InternalGraph)
  {
    this->InternalGraph->PrintSelf(os, indent.GetNextIndent());
  }
}
