/*=========================================================================

   Library: iMSTK

   Copyright (c) Kitware, Inc. & Center for Modeling, Simulation,
   & Imaging in Medicine, Rensselaer Polytechnic Institute.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0.txt

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

=========================================================================*/

#include "imstkSPHSolver.h"
#include "imstkSPHObject.h"
#include "imstkSPHModel.h"
#include "imstkLogUtility.h"

#include <tbb/tbb.h>
#include <climits>
#undef min
#undef max

namespace imstk
{
void
SPHSolver::addSPHObject(const std::shared_ptr<SPHObject>& sphObject)
{
    LOG_IF(FATAL, (!sphObject)) << "Adding nullptr SPH object";
    LOG_IF(FATAL, (m_mSPHModelToIndex.find(sphObject->getSPHModel().get()) != m_mSPHModelToIndex.end()))
        << "SPH object has previously been added";

    const auto sphModel = sphObject->getSPHModel();
    LOG_IF(FATAL, (!sphModel)) << "SPH model has not been initialized for the SPH object";
    LOG_IF(FATAL, (sphModel->m_connectedToSolver)) << "SPH model has been added to some other SPH solver";

    // Update solver object group data
    sphModel->m_connectedToSolver = true;
    m_mSPHModelToIndex.emplace(sphObject->getSPHModel().get(), m_vSPHModels.size());
    m_vSPHModels.push_back(sphObject->getSPHModel());

    // Update SPH model group for all sph models in this solver
    SPHModel::setupModelGroup(m_vSPHModels);
}

void
SPHSolver::removeSPHObject(const std::shared_ptr<SPHObject>& sphObject)
{
    LOG_IF(FATAL, (!sphObject)) << "Removing nullptr SPH object";
    const auto sphModel = sphObject->getSPHModel();
    LOG_IF(FATAL, (m_mSPHModelToIndex.find(sphModel.get()) == m_mSPHModelToIndex.end()))
        << "Cannot remove SPH object: object not found in the solver";

    // Update solver object group data
    const auto objIdx = m_mSPHModelToIndex[sphModel.get()];
    std::swap(m_vSPHModels.back(), m_vSPHModels[objIdx]);
    m_vSPHModels.resize(m_vSPHModels.size() - 1);
    m_mSPHModelToIndex.erase(sphModel.get());
    sphModel->m_connectedToSolver = false;

    if (objIdx < m_vSPHModels.size())
    {
        m_mSPHModelToIndex[m_vSPHModels[objIdx].get()] = objIdx;
    }

    // Update SPH model group for all sph models in this solver
    SPHModel::setupModelGroup(m_vSPHModels);
}

void
SPHSolver::solve()
{
    tbb::task_group tg;

    // For each SPH object, before finding neighbors for particles across multiple SPH models,
    // firstly we need to compute bounding boxes for particles in each model,
    // then collect particle indices into the internal neighbor search data structure
    for (size_t objIdx = 0; objIdx < m_vSPHModels.size(); ++objIdx)
    {
        tg.run([&, objIdx] { m_vSPHModels[objIdx]->updateNeighborSearchData(); });
    }
    tg.wait();

    // After sync, find neighbors for all particles in all models
    for (size_t objIdx = 0; objIdx < m_vSPHModels.size(); ++objIdx)
    {
        tg.run([&, objIdx] { m_vSPHModels[objIdx]->findParticleNeighbors(); });
    }
    tg.wait();

    // Compute particle densities
    for (size_t objIdx = 0; objIdx < m_vSPHModels.size(); ++objIdx)
    {
        tg.run([&, objIdx] {
                const auto& sphModel = m_vSPHModels[objIdx];
                sphModel->computeNeighborRelativePositions();
                sphModel->computeDensity();
                sphModel->normalizeDensity();
        });
    }
    tg.wait();

    // Compute particle pressure accelerations and surface normals
    for (size_t objIdx = 0; objIdx < m_vSPHModels.size(); ++objIdx)
    {
        tg.run([&, objIdx] {
                const auto& sphModel = m_vSPHModels[objIdx];
                sphModel->collectNeighborDensity();
                sphModel->computePressureAcceleration();
                sphModel->computeSurfaceNormal();
        });
    }
    tg.wait();

    // Compute surface tensions from surface normals
    for (size_t objIdx = 0; objIdx < m_vSPHModels.size(); ++objIdx)
    {
        tg.run([&, objIdx] { m_vSPHModels[objIdx]->computeSurfaceTensionAcceleration(); });
    }

    // Compute time step size for each model
    for (size_t objIdx = 0; objIdx < m_vSPHModels.size(); ++objIdx)
    {
        tg.run([&, objIdx] { m_vSPHModels[objIdx]->computeTimeStepSize(); });
    }
    tg.wait();

    // Also compute the minimum time step of all models
    Real timestepCFL = MAX_REAL;
    for (size_t objIdx = 0; objIdx < m_vSPHModels.size(); ++objIdx)
    {
        const auto timestep = m_vSPHModels[objIdx]->getTimeStep();
        timestepCFL = (timestepCFL < timestep) ? timestepCFL : timestep;
    }

    // Update velocities and move particles
    for (size_t objIdx = 0; objIdx < m_vSPHModels.size(); ++objIdx)
    {
        tg.run([&, objIdx, timestepCFL] {
                const auto& sphModel = m_vSPHModels[objIdx];
                sphModel->updateVelocity(timestepCFL);
                sphModel->computeViscosity();
                sphModel->moveParticles(timestepCFL);
        });
    }
    tg.wait();
}
} // end namespace imstk
