/*=========================================================================

   Library: iMSTK

   Copyright (c) Kitware, Inc. & Center for Modeling, Simulation,
   & Imaging in Medicine, Rensselaer Polytechnic Institute.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0.txt

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

=========================================================================*/

#pragma once

#include "imstkPointSet.h"
#include "imstkDynamicalModel.h"
#include "imstkSPHState.h"
#include "imstkSPHKernels.h"
#include "imstkNeighborSearch.h"

namespace imstk
{
///
/// \class SPHModelConfig
/// \brief Class that holds the SPH model parameters
/// Upon instantiating an SPH model, user must specify a particle radius
/// Other parameters are optional
///
class SPHModelConfig
{
public:
    SPHModelConfig(const Real particleRadius) : m_particleRadius(particleRadius) {}

    ///
    /// \brief Compute dependent parameters, such as particle mass, kernel radius etc.
    ///
    void initialize();

    ///
    /// \brief Min/Max time step sizes, used to limits the time step size during simulation
    /// This is necessary to avoid integration with too large time step size which can lead to explosion,
    /// or integration with too small time step size, which stalls the entire system
    /// \todo Move this to solver or time integrator in the future
    ///
    Real m_minTimestep = Real(1e-6);
    Real m_maxTimestep = Real(1e-3);

    ///
    /// \brief CFL factor, used to scale the time step size
    /// (The time step size is computed based on CFL condition: dt = 2 * particle_radius / max{ |v| } *CFLFactor)
    ///
    Real m_CFLFactor = Real(1.0);

    ///
    /// \brief Particle radius, MUST be specified by user, which is half distance between two sampled points
    ///
    Real m_particleRadius = Real(0);

    ///
    /// \brief Rest density of the fluid (1000 kg/m^3 for water, 1.225 kg/m^3 for air)
    ///
    Real m_restDensity = Real(1000.0);

    ///
    /// \brief Scale value for particle mass, making the mass of each fluid particle slightly smaller/bigger than the ideal value
    /// This is necessary if the particles were not sampled by a regular grid pattern
    ///
    Real m_particleMassScale = Real(0.95);

    ///
    /// \brief Normalize the particle densities, prodcing more accurate results
    ///
    bool m_normalizeDensity = false;

    ///
    /// \brief This flag allows to compute fluid particle densties condidering the boundary particles (if they exist)
    ///
    bool m_densityWithBoundary = false;

    ///
    /// \brief Stiffness value K for computing particle pressure
    ///
    Real m_pressureStiffnessConstant = Real(50000.0);

    ///
    /// \brief Viscosity coefficient for computing viscosity between fluid particles
    ///
    Real m_fluidViscosityConstant = Real(1e-2);

    ///
    /// \brief Viscosity coefficient for computing viscosity between fluid particle and boundary particles
    ///
    Real m_boundaryViscosityConstant = Real(1e-3);

    ///
    /// \brief Surface tension coefficient for computing surface tension
    ///
    Real m_surfaceTensionConstant = Real(1);

    ///
    /// \brief Friction coefficient for computing friction between fluid particles and solid boundaries
    ///
    Real m_frictionBoundaryConstant = Real(0.1);

    ///
    /// \brief The ratio between SPH kernel and particles radius
    /// During initialization, the SPH kernel will be computed as kernel_radius = kernel_ratio * particle_radius
    ///
    Real m_kernelOverParticleRadiusRatio = Real(4.0);

    ///
    /// \brief Set gravity acceleration for the particles
    ///
    Vec3r m_gravity = Vec3r(0, -9.81, 0);

    ///
    /// \brief Method to perform neighbor search
    ///
    NeighborSearch::Method m_neighborSearchMethod = NeighborSearch::Method::UniformGridBasedSearch;

private:
    // Allow SPHModel to access private members
    friend class SPHModel;

    /// Squared particle radius, computed from particle radius during initialization
    Real m_particleRadiusSqr = Real(0);

    /// Particle mass, computed from rest density during initialization
    Real m_particleMass = Real(0);

    /// Squared Rest density, computed from rest density during initialization
    Real m_restDensitySqr = Real(1000000.0);

    /// Inversed rest density, computed from rest density during initialization
    Real m_restDensityInv = Real(1.0 / 1000.0);

    /// Kernel radius, computed from particle radius during initialization
    Real m_kernelRadius;

    /// Squared of kernel radius, computed from kernel radius during initialization
    Real m_kernelRadiusSqr;
};

///
/// \class SPHModel
/// \brief SPH fluid model
///
class SPHModel : public DynamicalModel<SPHKinematicState>, public std::enable_shared_from_this<SPHModel>
{
public:
    ///
    /// \brief Constructor
    ///
    SPHModel() : DynamicalModel<SPHKinematicState>(DynamicalModelType::SPH) {}

    ///
    /// \brief Set simulation parameters, MUST be called upon instantiating SPH model
    ///
    void configure(const std::shared_ptr<SPHModelConfig>& params) { m_modelParameters = params; }

    ///
    /// \brief Set the geometry (particle positions)
    ///
    void setModelGeometry(const std::shared_ptr<PointSet>& geo) { m_geometry = geo; }

    ///
    /// \brief Initialize the dynamical model
    ///
    virtual bool initialize() override;

    ///
    /// \brief Update states
    ///
    virtual void updateBodyStates(const Vectord&, const stateUpdateType) override {}

    ///
    /// \brief Update positions of point set geometry
    ///
    virtual void updatePhysicsGeometry() override
    { assert(m_geometry); m_geometry->setVertexPositions(this->m_currentState->getPositions()); }

    ///
    /// \brief Reset the current state to the initial state
    ///
    virtual void resetToInitialState() override
    { this->m_currentState->setState(this->m_initialState); }

    ///
    /// \brief Get the simulation parameters
    ///
    const std::shared_ptr<SPHModelConfig>& getParameters() const
    { assert(m_modelParameters); return m_modelParameters; }

    ///
    /// \brief Get the kinematics particle data (positions + velocities)
    ///
    SPHKinematicState& getKinematicsState()
    {
        assert(this->m_currentState);
        return *this->m_currentState;
    }

    const SPHKinematicState& getKinematicsState() const
    {
        assert(this->m_currentState);
        return *this->m_currentState;
    }

    ///
    /// \brief Get particle simulation data
    ///
    SPHSimulationState& getState() { return m_simulationState; }
    const SPHSimulationState& getState() const { return m_simulationState; }

    ///
    /// \brief Set the default time step size,
    /// valid only if using a fixed time step for integration
    ///
    virtual void setTimeStep(const double timeStep) override
    { setDefaultTimeStep(static_cast<Real>(timeStep)); }

    ///
    /// \brief Set the default time step size,
    /// valid only if using a fixed time step for integration
    ///
    void setDefaultTimeStep(const Real timeStep)
    { m_defaultDt = static_cast<Real>(timeStep); }

    ///
    /// \brief Returns the time step size
    ///
    virtual double getTimeStep() const override
    { return static_cast<double>(m_dt); }

    ///
    /// \brief Do one time step simulation (SPH solver should execute all steps in this function)
    ///
    void advanceTimeStep();

private:
    friend class SPHSolver;

    ///
    /// \brief Update the bounding box of particles in all SPH models of the same SPH solver, then collect particles indices
    /// This is necessary to synchronize neighbor search between SPH objects
    ///
    void updateNeighborSearchData();

    ///
    /// \brief Compute time step size, do nothing if using a fixed time step size for integration
    ///
    void computeTimeStepSize();

    ///
    /// \brief Compute time step size based on CFL condition
    ///
    Real computeCFLTimeStepSize();

    ///
    /// \brief Find the neighbors for each particle
    ///
    void findParticleNeighbors();

    ///
    /// \brief Pre-compute relative positions x_{ij} = x_i - x_j with all neighbor particles j
    /// x_{ij} are used multiples times, thus caching them can improve performance significantly
    ///
    void computeNeighborRelativePositions();

    ///
    /// \brief Collect the densities of neighbor particles,
    /// called after all density computation (after density normalization, if applicable)
    ///
    void collectNeighborDensity();

    ///
    /// \brief Compute particle densities
    ///
    void computeDensity();

    ///
    /// \brief Normalize densities, producing smoother density field
    ///
    void normalizeDensity();

    ///
    /// \brief Compute particle accelerations due to pressure
    ///
    void computePressureAcceleration();

    ///
    /// \brief Compute surface normals
    ///
    void computeSurfaceNormal();

    ///
    /// \brief Compute surface tension using Akinci et at. 2013 model
    /// (Versatile Surface Tension and Adhesion for SPH Fluids)
    ///
    void computeSurfaceTensionAcceleration();

    ///
    /// \brief Update particle velocities from acceleration
    ///
    void updateVelocity(const Real timestep);

    ///
    /// \brief Compute viscosity (in the XSPH viscosity model, visocisty is computed after velocity update)
    ///
    void computeViscosity();

    ///
    /// \brief Move particles
    ///
    void moveParticles(const Real timestep);

    /// A set containing all SPH models belonging to the same SPH solver
    using SPHModelGroup = std::vector<std::shared_ptr<SPHModel>>;

    ///
    /// \brief Initialize group of SPH models for a newly created SPH model
    /// Each SPH model must associate with a model group, which consists of at least itself
    /// Thus, this function MUST be called during SPH model initialization
    ///
    static void initializeModelGroup(const std::shared_ptr<SPHModel>& sphModel);

    ///
    /// \brief Return the set of SPH models which are in the same SPH solver
    ///
    static const SPHModelGroup& getModelGroup(SPHModel* const sphModel);

    ///
    /// \brief Setup model group for all SPH models in the same SPH solver
    ///
    static void setupModelGroup(const std::vector<std::shared_ptr<SPHModel>>& models);

    /// Map from a pointer of an SPH model to the set of its mode group
    static std::unordered_map<SPHModel*, SPHModelGroup> s_mSPHModelGroups;

    /// Turn this flag on/off to know whether an SPH model has been added to an SPH solver or not
    /// This is used to check to avoid adding the same SPH model to multiple SPH solver
    /// Thus, it should be access only in SPH solver during adding/removing model
    bool m_connectedToSolver = false;

    std::shared_ptr<PointSet> m_geometry;               ///> Original pointset geometry
    SPHSimulationState        m_simulationState;        ///> The simulation state of the SPH model

    /// Time step size, which is computed from particle velocities at every time step, if using CFL condition
    Real m_dt;

    /// If using a fixed time step for simulation, the time step size will be fixed by this value
    Real m_defaultDt = Real(1e-4);

    /// The SPH kernels, which are initialized during model initialization
    SPHSimulationKernels m_kernels;

    /// Simulation parameters, which MUST be set upon instantiating an SPH model
    std::shared_ptr<SPHModelConfig> m_modelParameters;

    /// Neighbor search data structure, which is initialized during model initialization
    std::shared_ptr<NeighborSearch> m_neighborSearcher;
};
} // end namespace imstk
