/*=========================================================================

   Library: iMSTK

   Copyright (c) Kitware, Inc. & Center for Modeling, Simulation,
   & Imaging in Medicine, Rensselaer Polytechnic Institute.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0.txt

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

=========================================================================*/

#include "imstkSimulationManager.h"
#include "imstkSPHObject.h"
#include "imstkSPHSolver.h"
#include "imstkSPHModel.h"

using namespace imstk;

///
/// \brief Generate a sphere-shape fluid object
///
StdVectorOfVec3d
generateSphereShapeFluid(const double particleRadius)
{
    const double sphereRadius = 2.0;
    const Vec3d  sphereCenter(0, 1, 0);

    const auto  sphereRadiusSqr = sphereRadius * sphereRadius;
    const auto  spacing         = 2.0 * particleRadius;
    const auto  N               = static_cast<size_t>(2.0 * sphereRadius / spacing);              // Maximum number of particles in each dimension
    const Vec3d lcorner         = sphereCenter - Vec3d(sphereRadius, sphereRadius, sphereRadius); // Cannot use auto here, due to Eigen bug

    StdVectorOfVec3d particles;
    particles.reserve(N * N * N);

    for (size_t i = 0; i < N; ++i)
    {
        for (size_t j = 0; j < N; ++j)
        {
            for (size_t k = 0; k < N; ++k)
            {
                Vec3d ppos = lcorner + Vec3d(spacing * double(i), spacing * double(j), spacing * double(k));
                Vec3d cx   = ppos - sphereCenter;
                if (cx.squaredNorm() < sphereRadiusSqr)
                {
                    particles.push_back(ppos);
                }
            }
        }
    }

    return particles;
}

///
/// \brief Generate a box-shape fluid object
///
StdVectorOfVec3d
generateBoxShapeFluid(const double particleRadius)
{
    const double boxWidth = 4.0;
    const Vec3d  boxLowerCorner(-2, -3, -2);

    const auto spacing = 2.0 * particleRadius;
    const auto N       = static_cast<size_t>(boxWidth / spacing);

    StdVectorOfVec3d particles;
    particles.reserve(N * N * N);

    for (size_t i = 0; i < N; ++i)
    {
        for (size_t j = 0; j < N; ++j)
        {
            for (size_t k = 0; k < N; ++k)
            {
                Vec3d ppos = boxLowerCorner + Vec3d(spacing * double(i), spacing * double(j), spacing * double(k));
                particles.push_back(ppos);
            }
        }
    }

    return particles;
}

#if (SCENE_ID == 3) || (SCENE_ID == 4)
StdVectorOfVec3d getBunny(); // Defined in Bunny.cpp
#endif
///
/// \brief Generate a bunny-shape fluid object
///
StdVectorOfVec3d
generateBunnyShapeFluid(const double particleRadius)
{
    LOG_IF(FATAL, (std::abs(particleRadius - 0.08) > 1e-6)) << "Particle radius for this scene must be 0.08";
    StdVectorOfVec3d particles;
#if (SCENE_ID == 3) || (SCENE_ID == 4)
    particles = getBunny();
#endif
    return particles;
}

std::vector<std::shared_ptr<SPHObject>>
generateFluid(const std::shared_ptr<Scene>& scene, const double particleRadius)
{
    StdVectorOfVec3d particles;
    switch (SCENE_ID)
    {
    case 1:
        particles = generateSphereShapeFluid(particleRadius);
        break;
    case 2:
        particles = generateBoxShapeFluid(particleRadius);
        break;
    case 3:
    case 4:
        particles = generateBunnyShapeFluid(particleRadius);
        break;
    default:
        LOG(FATAL) << "Invalid scene index";
    }
    if (SCENE_ID == 4)
    {
        const double translation = 3.0;
        for (size_t i = 0; i < particles.size(); ++i)
        {
            particles[i] -= Vec3d(translation, 0, 0);
        }
    }

    LOG(INFO) << "Number of particles: " << particles.size();

    // Create a geometry object
    auto fluidGeometry = std::make_shared<PointSet>();
    fluidGeometry->initialize(particles);

    // Create a fluids object
    auto fluidObj1 = std::make_shared<SPHObject>("FluidObj1");

    // Create a visual model
    auto fluidVisualModel = std::make_shared<VisualModel>(fluidGeometry);
    auto fluidMaterial    = std::make_shared<RenderMaterial>();
    fluidMaterial->setColor(Color::Blue);
    fluidMaterial->setSphereGlyphSize(particleRadius);
    fluidVisualModel->setRenderMaterial(fluidMaterial);

    // Create a physics model
    auto sphModel = std::make_shared<SPHModel>();
    sphModel->setModelGeometry(fluidGeometry);

    // configure model
    auto sphParams = std::make_shared<SPHModelConfig>(particleRadius);
    sphParams->m_normalizeDensity = true;
    if (SCENE_ID == 2)   // highly viscous fluid
    {
        sphParams->m_kernelOverParticleRadiusRatio = 6.0;
        sphParams->m_fluidViscosityConstant        = 0.5;
        sphParams->m_surfaceTensionConstant        = 5.0;
    }
    else if (SCENE_ID == 3 || SCENE_ID == 4) // bunny-shaped fluid
    {
        sphParams->m_frictionBoundaryConstant = 0.3;
    }

    if (SCENE_ID == 4)   // multiple fluid
    {
        sphParams->m_restDensity = 5000;
        sphParams->m_surfaceTensionConstant = 0.5;
    }

    sphModel->configure(sphParams);
    sphModel->setTimeStepSizeType(TimeSteppingType::realTime);

    // Add the component models
    fluidObj1->addVisualModel(fluidVisualModel);
    fluidObj1->setCollidingGeometry(fluidGeometry);
    fluidObj1->setDynamicalModel(sphModel);
    fluidObj1->setPhysicsGeometry(fluidGeometry); // TODO: Look into API duplication and resulting conflicts
    scene->addSceneObject(fluidObj1);

    // Configure the solver
    auto sphSolver = std::make_shared<SPHSolver>();
    sphSolver->addSPHObject(fluidObj1);
    scene->addNonlinearSolver(sphSolver);

    std::vector<std::shared_ptr<SPHObject>> fluids = { fluidObj1 };

    if (SCENE_ID == 4) // multiple fluid
    {
        auto         particlesDup = particles;
        const double translation  = 3.0;
        for (size_t i = 0; i < particlesDup.size(); ++i)
        {
            particlesDup[i] += Vec3d(translation * 2.0, 0, 0);
        }

        // Create a geometry object
        auto fluidGeometry2 = std::make_shared<PointSet>();
        fluidGeometry2->initialize(particlesDup);

        // Create a fluids object
        auto fluidObj2 = std::make_shared<SPHObject>("FluidObj2");

        // Create a visiual model
        auto fluidVisualModel2 = std::make_shared<VisualModel>(fluidGeometry2);
        auto fluidMaterial2    = std::make_shared<RenderMaterial>();
        fluidMaterial2->setColor(Color::Red);
        fluidMaterial2->setSphereGlyphSize(particleRadius);
        fluidVisualModel2->setRenderMaterial(fluidMaterial2);

        // Create a physics model
        auto sphModel2 = std::make_shared<SPHModel>();
        sphModel2->setModelGeometry(fluidGeometry2);

        // configure model
        auto sphParams2 = std::make_shared<SPHModelConfig>(particleRadius);
        sphParams2->m_normalizeDensity         = true;
        sphParams2->m_frictionBoundaryConstant = 0.3;
        sphParams2->m_restDensity              = 1000;

        sphModel2->configure(sphParams2);
        sphModel2->setTimeStepSizeType(TimeSteppingType::realTime);

        // Add the component models
        fluidObj2->addVisualModel(fluidVisualModel2);
        fluidObj2->setCollidingGeometry(fluidGeometry2);
        fluidObj2->setDynamicalModel(sphModel2);
        fluidObj2->setPhysicsGeometry(fluidGeometry2); // TODO: Look into API duplication and resulting conflicts
        scene->addSceneObject(fluidObj2);
        fluids.push_back(fluidObj2);

        // Configure the solver
        sphSolver->addSPHObject(fluidObj2);
    }

    return fluids;
}
