// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
#include "vtkOverlappingAMR.h"
#include "vtkAMRInformation.h"
#include "vtkCellData.h"
#include "vtkDataSetAttributes.h"
#include "vtkInformationIdTypeKey.h"
#include "vtkObjectFactory.h"
#include "vtkUniformGrid.h"
#include "vtkUniformGridAMRDataIterator.h"
#include "vtkUnsignedCharArray.h"
#include <vector>

VTK_ABI_NAMESPACE_BEGIN
vtkStandardNewMacro(vtkOverlappingAMR);

vtkInformationKeyMacro(vtkOverlappingAMR, NUMBER_OF_BLANKED_POINTS, IdType);

//------------------------------------------------------------------------------
vtkOverlappingAMR::vtkOverlappingAMR() = default;

//------------------------------------------------------------------------------
vtkOverlappingAMR::~vtkOverlappingAMR() = default;

//------------------------------------------------------------------------------
void vtkOverlappingAMR::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//------------------------------------------------------------------------------
vtkCompositeDataIterator* vtkOverlappingAMR::NewIterator()
{
  vtkUniformGridAMRDataIterator* iter = vtkUniformGridAMRDataIterator::New();
  iter->SetDataSet(this);
  return iter;
}

//------------------------------------------------------------------------------
void vtkOverlappingAMR::SetRefinementRatio(unsigned int level, int ratio)
{
  this->GetAMRInfo()->SetRefinementRatio(level, ratio);
}

//------------------------------------------------------------------------------
int vtkOverlappingAMR::GetRefinementRatio(unsigned int level)
{
  vtkAMRInformation* amrInfo = this->GetAMRInfo();
  if (!amrInfo->HasRefinementRatio())
  {
    amrInfo->GenerateRefinementRatio();
  }
  return amrInfo->GetRefinementRatio(level);
}

//------------------------------------------------------------------------------
int vtkOverlappingAMR::GetRefinementRatio(vtkCompositeDataIterator* iter)
{
  vtkUniformGridAMRDataIterator* amrIter = vtkUniformGridAMRDataIterator::SafeDownCast(iter);

  unsigned int level = amrIter->GetCurrentLevel();
  return this->GetAMRInfo()->GetRefinementRatio(level);
}

//------------------------------------------------------------------------------
void vtkOverlappingAMR::GenerateParentChildInformation()
{
  this->GetAMRInfo()->GenerateParentChildInformation();
}

//------------------------------------------------------------------------------
bool vtkOverlappingAMR::HasChildrenInformation()
{
  return this->GetAMRInfo()->HasChildrenInformation();
}

//------------------------------------------------------------------------------
unsigned int* vtkOverlappingAMR::GetParents(
  unsigned int level, unsigned int index, unsigned int& num)
{
  return this->GetAMRInfo()->GetParents(level, index, num);
}

//------------------------------------------------------------------------------
unsigned int* vtkOverlappingAMR::GetChildren(
  unsigned int level, unsigned int index, unsigned int& num)
{
  return this->GetAMRInfo()->GetChildren(level, index, num);
}

//------------------------------------------------------------------------------
void vtkOverlappingAMR::PrintParentChildInfo(unsigned int level, unsigned int index)
{
  this->GetAMRInfo()->PrintParentChildInfo(level, index);
}

//------------------------------------------------------------------------------
void vtkOverlappingAMR::SetAMRBox(unsigned int level, unsigned int id, const vtkAMRBox& box)
{
  this->GetAMRInfo()->SetAMRBox(level, id, box);
}

//------------------------------------------------------------------------------
const vtkAMRBox& vtkOverlappingAMR::GetAMRBox(unsigned int level, unsigned int id)
{
  const vtkAMRBox& box = this->GetAMRInfo()->GetAMRBox(level, id);
  if (box.IsInvalid())
  {
    vtkErrorMacro("Invalid AMR box");
  }
  return box;
}

//------------------------------------------------------------------------------
void vtkOverlappingAMR::SetSpacing(unsigned int level, const double spacing[3])
{
  this->GetAMRInfo()->SetSpacing(level, spacing);
}

//------------------------------------------------------------------------------
void vtkOverlappingAMR::GetSpacing(unsigned int level, double spacing[3])
{
  this->GetAMRInfo()->GetSpacing(level, spacing);
}

//------------------------------------------------------------------------------
void vtkOverlappingAMR::GetBounds(unsigned int level, unsigned int id, double bb[6])
{
  this->GetAMRInfo()->GetBounds(level, id, bb);
}

//------------------------------------------------------------------------------
void vtkOverlappingAMR::GetOrigin(unsigned int level, unsigned int id, double origin[3])
{
  double bb[6];
  this->GetBounds(level, id, bb);
  origin[0] = bb[0];
  origin[1] = bb[2];
  origin[2] = bb[4];
}

//------------------------------------------------------------------------------
void vtkOverlappingAMR::SetOrigin(const double origin[3])
{
  this->GetAMRInfo()->SetOrigin(origin);
}

//------------------------------------------------------------------------------
double* vtkOverlappingAMR::GetOrigin()
{
  return this->GetAMRInfo()->GetOrigin();
}

//------------------------------------------------------------------------------
void vtkOverlappingAMR::SetAMRBlockSourceIndex(unsigned int level, unsigned int id, int sourceId)
{
  vtkAMRInformation* amrInfo = this->GetAMRInfo();
  unsigned int index = amrInfo->GetIndex(level, id);
  amrInfo->SetAMRBlockSourceIndex(index, sourceId);
}

//------------------------------------------------------------------------------
int vtkOverlappingAMR::GetAMRBlockSourceIndex(unsigned int level, unsigned int id)
{
  vtkAMRInformation* amrInfo = this->GetAMRInfo();
  unsigned int index = amrInfo->GetIndex(level, id);
  return amrInfo->GetAMRBlockSourceIndex(index);
}

//------------------------------------------------------------------------------
void vtkOverlappingAMR::Audit()
{
  vtkAMRInformation* amrInfo = this->GetAMRInfo();
  amrInfo->Audit();

  int emptyDimension(-1);
  switch (this->GetGridDescription())
  {
    case VTK_YZ_PLANE:
      emptyDimension = 0;
      break;
    case VTK_XZ_PLANE:
      emptyDimension = 1;
      break;
    case VTK_XY_PLANE:
      emptyDimension = 2;
      break;
  }

  vtkSmartPointer<vtkUniformGridAMRDataIterator> iter;
  iter.TakeReference(vtkUniformGridAMRDataIterator::SafeDownCast(this->NewIterator()));
  iter->SetSkipEmptyNodes(1);
  for (iter->GoToFirstItem(); !iter->IsDoneWithTraversal(); iter->GoToNextItem())
  {
    vtkUniformGrid* grid = vtkUniformGrid::SafeDownCast(iter->GetCurrentDataObject());
    int hasGhost = grid->HasAnyGhostCells();

    unsigned int level = iter->GetCurrentLevel();
    unsigned int id = iter->GetCurrentIndex();
    const vtkAMRBox& box = amrInfo->GetAMRBox(level, id);
    int dims[3];
    box.GetNumberOfNodes(dims);

    double spacing[3];
    this->GetSpacing(level, spacing);

    double origin[3];
    this->GetOrigin(level, id, origin);

    for (int d = 0; d < 3; d++)
    {
      if (d == emptyDimension)
      {
        if (grid->GetSpacing()[d] != spacing[d])
        {
          vtkErrorMacro(
            "The grid spacing does not match AMRInfo at (" << level << ", " << id << ")");
        }
        if (!hasGhost && grid->GetOrigin()[d] != origin[d])
        {
          vtkErrorMacro(
            "The grid origin does not match AMRInfo at (" << level << ", " << id << ")");
        }
        if (!hasGhost && grid->GetDimensions()[d] != dims[d])
        {
          vtkErrorMacro(
            "The grid dimensions does not match AMRInfo at (" << level << ", " << id << ")");
        }
      }
    }
  }
}

bool vtkOverlappingAMR::FindGrid(double q[3], unsigned int& level, unsigned int& gridId)
{
  return this->GetAMRInfo()->FindGrid(q, level, gridId);
}
VTK_ABI_NAMESPACE_END
