#ifndef DIY_EXAMPLES_POINT_H
#define DIY_EXAMPLES_POINT_H

#include <vector>
#include <cassert>
#include <diy/types.hpp>
#include <diy/log.hpp>

// D-dimensional point
template<unsigned D>
using SimplePoint = diy::Point<float, D>;

// block structure
// the contents of a block are completely user-defined
// however, a block must have functions defined to create, destroy, save, and load it
// create and destroy allocate and free the block, while
// save and load serialize and deserialize the block
// these four functions are called when blocks are cycled in- and out-of-core
// they can be member functions of the block, as below, or separate standalone functions
template<unsigned DIM>
struct PointBlock
{
    typedef         SimplePoint<DIM>                            Point;
    typedef         diy::ContinuousBounds                       Bounds;

PointBlock(const Bounds& bounds_):
    bounds(bounds_)                         {}

    // allocate a new block
    static void*    create()                { return new PointBlock; }
    // free a block
    static void     destroy(void* b)        { delete static_cast<PointBlock*>(b); }
    // serialize the block and write it
    static void     save(const void* b_, diy::BinaryBuffer& bb)
    {
        const PointBlock* b = static_cast<const PointBlock*>(b_);
        diy::save(bb, b->bounds);
        diy::save(bb, b->box);
        diy::save(bb, b->points);
    }
    // read the block and deserialize it
    static void     load(void* b_, diy::BinaryBuffer& bb)
    {
        PointBlock* b = static_cast<PointBlock*>(b_);
        diy::load(bb, b->bounds);
        diy::load(bb, b->box);
        diy::load(bb, b->points);
    }
    // initialize block values
    void            generate_points(const Bounds& domain, // overall data bounds
                                    size_t n)             // number of points
    {
        box = domain;
        points.resize(n);
        for (size_t i = 0; i < n; ++i)
            for (unsigned j = 0; j < DIM; ++j)
                points[i][j] = domain.min[j] + float(rand() % 1024)/1024 *
                    (domain.max[j] - domain.min[j]);
    }

    // --- foreach callback functions ---//

    // callbacks can be member functions of the block
    // when they are, there is no need for the block to be passed as the first argument

    // check that block values are in the block bounds (debug)
    void          verify_block(const diy::Master::ProxyWithLink& cp) // communication proxy
    {
        for (size_t i = 0; i < points.size(); ++i)
            for (unsigned j = 0; j < DIM; ++j)
                if (points[i][j] < box.min[j] || points[i][j] > box.max[j])
                {
                    fmt::print(stderr, "!!! Point outside the box !!!\n");
                    fmt::print(stderr, "    {} {} {}\n", points[i][0], points[i][1], points[i][2]);
                    fmt::print(stderr, "    {} {} {} - {} {} {}\n",
                               box.min[0], box.min[1], box.min[2],
                               box.max[0], box.max[1], box.max[2]);
                }
    }
    // print block values
    void          print_block(const diy::Master::ProxyWithLink& cp,  // communication proxy
                              bool verbose)                          // amount of output
    {
        fmt::print("[{}] Box:    {} {} {} -- {} {} {}\n",
                cp.gid(),
                box.min[0], box.min[1], box.min[2],
                box.max[0], box.max[1], box.max[2]);
        fmt::print("[{}] Bounds: {} {} {} -- {} {} {}\n",
                cp.gid(),
                bounds.min[0], bounds.min[1], bounds.min[2],
                bounds.max[0], bounds.max[1], bounds.max[2]);

        if (verbose)
        {
            for (size_t i = 0; i < points.size(); ++i)
                fmt::print("  {}\n", points[i]);
        } else
            fmt::print("[{}] Points: {}\n", cp.gid(), points.size());
    }

    // block data
    Bounds                bounds;
    Bounds                box;
    std::vector<Point>    points;

private:
    PointBlock()                                  {}
};

// diy::decompose needs to have a function defined to create a block
// here, it is wrapped in an object to add blocks with an overloaded () operator
// it could have also been written as a standalone function
template<unsigned DIM>
struct AddPointBlock
{
    typedef   PointBlock<DIM>                                     Block;
    typedef   diy::ContinuousBounds                               Bounds;
    typedef   diy::RegularContinuousLink                          RCLink;

    AddPointBlock(diy::Master& master_, size_t num_points_):
        master(master_),
        num_points(num_points_)
        {}

    // this is the function that is needed for diy::decompose
    void  operator()(int gid,                // block global id
                     const Bounds& core,     // block bounds without any ghost added
                     const Bounds& bounds,   // block bounds including any ghost region added
                     const Bounds& domain,   // global data bounds
                     const RCLink& link)     // neighborhood
        const
        {
            Block*          b   = new Block(core);
            RCLink*         l   = new RCLink(link);
            diy::Master&    m   = const_cast<diy::Master&>(master);

            m.add(gid, b, l); // add block to the master (mandatory)

            b->generate_points(domain, num_points); // initialize block data (typical)
        }

    diy::Master&  master;
    size_t        num_points;
};

#endif
