#include "File.h"

#include <vtkm/io/FileUtils.h>
#include <vtkm/io/VTKDataSetWriter.h>

#include <fstream>
#include <glob.h>
#include <string.h>
#include <sys/stat.h>

namespace beams
{
namespace io
{

bool File::FileExists(const std::string& filePath)
{
  std::ifstream file(filePath);
  return file.is_open();
}

bool File::FolderExists(const std::string& folderPath)
{
  struct stat sb;
  return stat(folderPath.c_str(), &sb) == 0 && S_ISDIR(sb.st_mode);
}

std::vector<std::string> File::Glob(const std::string& pattern)
{
  glob_t globResult;
  memset(&globResult, 0, sizeof(globResult));
  int globRetCode = glob(pattern.c_str(), GLOB_TILDE, nullptr, &globResult);
  if (globRetCode != 0)
  {
    globfree(&globResult);
    return std::vector<std::string>();
  }
  std::vector<std::string> filesNames;
  for (size_t i = 0; i < globResult.gl_pathc; ++i)
  {
    filesNames.push_back(globResult.gl_pathv[i]);
  }
  globfree(&globResult);
  return filesNames;
}

void File::SaveDataSet(const vtkm::cont::DataSet& dataSet, const std::string& name)
{
  vtkm::io::VTKDataSetWriter writer(name);
  writer.WriteDataSet(dataSet);
}

std::string FilenameExt(const std::string& filePath)
{
  auto lastSlashPos = filePath.rfind('.');
  if (lastSlashPos != std::string::npos)
  {
    return filePath.substr(lastSlashPos);
  }
  return filePath;
}

std::string FilenameWithoutExt(const std::string& filePath)
{
  auto lastSlashPos = filePath.rfind('.');
  if (lastSlashPos != std::string::npos)
  {
    return filePath.substr(0, lastSlashPos);
  }
  return filePath;
}

std::string SuffixStringToFilename(const std::string& filePath, const std::string& suffix)
{
  auto parentPath = vtkm::io::ParentPath(filePath);
  auto filename = vtkm::io::Filename(filePath);
  auto filenameWithoutExt = FilenameWithoutExt(filename);
  auto ext = FilenameExt(filename);
  filename = filenameWithoutExt + suffix + ext;
  return vtkm::io::MergePaths(parentPath, filename);
}

}
} // namespace beams::io