#pragma once

#include "CoordinatesType.h"

#include <vtkm/source/Source.h>

namespace beams
{
namespace source
{
/**
 * @brief The PerlinNoise source creates a uniform dataset.
 *
 * This class generates a uniform grid dataset with a tileable perlin
 * noise scalar point field.
 *
 * The Execute method creates a complete structured dataset that have a
 * scalar point field named 'perlinnoise'.
**/
class PerlinNoise : public vtkm::source::Source
{
public:
  VTKM_CONT PerlinNoise(
    beams::source::CoordinatesType coordinatesType = beams::source::CoordinatesType::Uniform);
  VTKM_CONT ~PerlinNoise() = default;

  VTKM_CONT PerlinNoise(const PerlinNoise&) = default;
  VTKM_CONT PerlinNoise(PerlinNoise&&) = default;
  VTKM_CONT PerlinNoise& operator=(const PerlinNoise&) = default;
  VTKM_CONT PerlinNoise& operator=(PerlinNoise&&) = default;

  VTKM_CONT vtkm::Id3 GetPointDimensions() const { return this->PointDimensions; }
  VTKM_CONT void SetPointDimensions(vtkm::Id3 dims) { this->PointDimensions = dims; }

  VTKM_CONT vtkm::Id3 GetCellDimensions() const { return this->PointDimensions - vtkm::Id3(1); }
  VTKM_CONT void SetCellDimensions(vtkm::Id3 dims) { this->PointDimensions = dims + vtkm::Id3(1); }

  VTKM_CONT vtkm::Vec3f GetMinExtent() const { return this->MinExtent; }
  VTKM_CONT void SetMinExtent(vtkm::Vec3f minExtent) { this->MinExtent = minExtent; }

  VTKM_CONT vtkm::Vec3f GetMaxExtent() const { return this->MaxExtent; }
  VTKM_CONT void SetMaxExtent(vtkm::Vec3f maxExtent) { this->MaxExtent = maxExtent; }

  /// \brief The seed used for the pseudorandom number generation of the noise.
  ///
  /// If the seed is not set, then a new, unique seed is picked each time `Execute` is run.
  VTKM_CONT vtkm::IdComponent GetSeed() const { return this->Seed; }
  VTKM_CONT void SetSeed(vtkm::IdComponent seed)
  {
    this->Seed = seed;
    this->SeedSet = true;
  }

private:
  vtkm::cont::DataSet DoExecute() const override;

  beams::source::CoordinatesType CoordinatesType;
  vtkm::Id3 PointDimensions = { 16, 16, 16 };
  vtkm::Vec3f MinExtent{ 0.0f, 0.0f, 0.0f };
  vtkm::Vec3f MaxExtent{ 1.0f, 1.0f, 1.0f };
  vtkm::IdComponent Seed = 0;
  bool SeedSet = false;
};
} //namespace source
} //namespace beams