//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//============================================================================

#ifndef vtk_m_rendering_compositing_ImageCompositor_h
#define vtk_m_rendering_compositing_ImageCompositor_h

#include <vtkm/rendering/vtkm_rendering_export.h>

#include "../../mpi/MpiEnv.h"
#include "Image.h"
#include "fmt/core.h"
#include <algorithm>

namespace vtkm
{
namespace rendering
{
namespace compositing
{

class VTKM_RENDERING_EXPORT ImagefCompositor
{
public:
  void Blend(vtkm::rendering::compositing::Imagef& front,
             vtkm::rendering::compositing::Imagef& back)
  {
    assert(front.Bounds.X.Min == back.Bounds.X.Min);
    assert(front.Bounds.Y.Min == back.Bounds.Y.Min);
    assert(front.Bounds.X.Max == back.Bounds.X.Max);
    assert(front.Bounds.Y.Max == back.Bounds.Y.Max);
    const int size = static_cast<int>(front.Pixels.size() / 4);

    auto rank = beams::mpi::MpiEnv::GetInstance()->Rank;
    auto debug_rank = 0;
    static int count = 0;
    if (rank == debug_rank)
    {
      front.Save(fmt::format("{}_a.png", count), {});
      back.Save(fmt::format("{}_b.png", count), {});
    }

#ifdef VTKH_OPENMP_ENABLED
#pragma omp parallel for
#endif
    for (int i = 0; i < size; ++i)
    {
      const int offset = i * 4;
      float alpha = front.Pixels[offset + 3];
      const float opacity = 1.0f - alpha;

      front.Pixels[offset + 0] += opacity * back.Pixels[offset + 0];
      front.Pixels[offset + 1] += opacity * back.Pixels[offset + 1];
      front.Pixels[offset + 2] += opacity * back.Pixels[offset + 2];
      front.Pixels[offset + 3] += opacity * back.Pixels[offset + 3];

      float d1 = std::min(front.Depths[i], 1.001f);
      float d2 = std::min(back.Depths[i], 1.001f);
      float depth = std::min(d1, d2);
      front.Depths[i] = depth;
    }
    if (rank == debug_rank)
    {
      front.Save(fmt::format("{}_c.png", count), {});
      ++count;
    }
  }

  void OrderedComposite(std::vector<vtkm::rendering::compositing::Imagef>& images)
  {
    const int total_images = images.size();
    std::sort(images.begin(), images.end(), CompositeOrderSortf());
    for (int i = 1; i < total_images; ++i)
    {
      Blend(images[0], images[i]);
    }
  }
};
}
}
} //namespace vtkm::rendering::compositing

#endif //vtk_m_rendering_compositing_ImageComposititing_h
