//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//  Copyright 2015 National Technology & Engineering Solutions of Sandia, LLC (NTESS).
//  Copyright 2015 UT-Battelle, LLC.
//  Copyright 2015 Los Alamos National Security.
//
//  Under the terms of Contract DE-NA0003525 with NTESS,
//  the U.S. Government retains certain rights in this software.
//
//  Under the terms of Contract DE-AC52-06NA25396 with Los Alamos National
//  Laboratory (LANL), the U.S. Government retains certain rights in
//  this software.
//============================================================================
#ifndef vtk_m_rendering_raytracing_ToneMapping_h
#define vtk_m_rendering_raytracing_ToneMapping_h

#include <vtkm/Math.h>
#include <vtkm/cont/Algorithm.h>
#include <vtkm/cont/ArrayHandleCounting.h>
#include <vtkm/rendering/raytracing/Ray.h>
#include <vtkm/worklet/DispatcherMapField.h>
#include <vtkm/worklet/WorkletMapField.h>

namespace vtkm
{
namespace rendering
{
namespace raytracing
{
namespace detail
{
class Luminance : public vtkm::worklet::WorkletMapField
{
public:
  VTKM_CONT
  Luminance() {}
  typedef void ControlSignature(FieldInOut, WholeArrayIn);
  typedef void ExecutionSignature(_1, _2, WorkIndex);
  template <typename Precision, typename ColorPortalType>
  VTKM_EXEC void operator()(Precision& luminance,
                            ColorPortalType& colors,
                            const vtkm::Id& idx) const
  {
    vtkm::Id offset = idx * 4;
    vtkm::Vec<Precision, 3> rgb;

    rgb[0] = colors.Get(offset + 0);
    rgb[1] = colors.Get(offset + 1);
    rgb[2] = colors.Get(offset + 2);

    luminance = 0.2126f * rgb[0] + 0.7152f * rgb[1] + 0.0722f * rgb[2];
  }
}; //class Luminance

class ToneMapExp : public vtkm::worklet::WorkletMapField
{
private:
  const vtkm::Float32 Exposure;

public:
  VTKM_CONT
  ToneMapExp(vtkm::Float32 exposure = 0.9f)
    : Exposure(exposure)
  {
  }
  typedef void ControlSignature(FieldIn, WholeArrayInOut);
  typedef void ExecutionSignature(_1, _2);
  template <typename ColorPortalType>
  VTKM_EXEC void operator()(const vtkm::Id& idx, ColorPortalType& colors) const
  {
    vtkm::Id offset = idx * 4;
    vtkm::Vec<vtkm::Float32, 3> rgb;

    rgb[0] = static_cast<vtkm::Float32>(colors.Get(offset + 0));
    rgb[1] = static_cast<vtkm::Float32>(colors.Get(offset + 1));
    rgb[2] = static_cast<vtkm::Float32>(colors.Get(offset + 2));
    bool bg = rgb[0] == 1.f;

    if (!bg || true)
    {
      rgb[0] = 1.f - vtkm::Exp(-(rgb[0] / Exposure));
      rgb[1] = 1.f - vtkm::Exp(-(rgb[1] / Exposure));
      rgb[2] = 1.f - vtkm::Exp(-(rgb[2] / Exposure));
    }
    rgb[0] = vtkm::Max(0.f, vtkm::Min(1.f, rgb[0]));
    rgb[1] = vtkm::Max(0.f, vtkm::Min(1.f, rgb[1]));
    rgb[2] = vtkm::Max(0.f, vtkm::Min(1.f, rgb[2]));

    colors.Set(offset + 0, rgb[0]);
    colors.Set(offset + 1, rgb[1]);
    colors.Set(offset + 2, rgb[2]);
    colors.Set(offset + 3, 1.f);
  }
}; //class tone map

class ToneMap : public vtkm::worklet::WorkletMapField
{
private:
  const vtkm::Float32 LogAverage; // luminance
  const vtkm::Float32 Alpha;      // simulates exposure
public:
  VTKM_CONT
  ToneMap(vtkm::Float32 logAverage)
    : LogAverage(logAverage)
    , Alpha(.0056f)
  {
  }
  typedef void ControlSignature(FieldIn, WholeArrayInOut);
  typedef void ExecutionSignature(_1, _2, WorkIndex);
  template <typename Precision, typename ColorPortalType>
  VTKM_EXEC void operator()(const Precision& luminance,
                            ColorPortalType& colors,
                            const vtkm::Id& idx) const
  {
    vtkm::Id offset = idx * 4;
    vtkm::Vec<Precision, 3> rgb;

    rgb[0] = static_cast<Precision>(colors.Get(offset + 0));
    rgb[1] = static_cast<Precision>(colors.Get(offset + 1));
    rgb[2] = static_cast<Precision>(colors.Get(offset + 2));
    bool bg = rgb[0] == 1.f;
    // TODO: precalculate
    if (!bg)
    {
      vtkm::Float32 scaledAverage = (Alpha / LogAverage) * luminance;
      vtkm::Float32 scaleDown = scaledAverage / (1.f + scaledAverage);
      rgb[0] *= scaleDown;
      rgb[1] *= scaleDown;
      rgb[2] *= scaleDown;
    }
    rgb[0] = vtkm::Max(0.f, vtkm::Min(1.f, rgb[0]));
    rgb[1] = vtkm::Max(0.f, vtkm::Min(1.f, rgb[1]));
    rgb[2] = vtkm::Max(0.f, vtkm::Min(1.f, rgb[2]));

    colors.Set(offset + 0, rgb[0]);
    colors.Set(offset + 1, rgb[1]);
    colors.Set(offset + 2, rgb[2]);
    colors.Set(offset + 3, 1.f);
  }
}; //class tone map

} // namespace detail

class ToneMapper
{
public:
  template <typename FloatType>
  static void ToneMap(Ray<FloatType>& rays)
  {
    vtkm::cont::ArrayHandle<vtkm::Float32> lum;
    lum.Allocate(rays.NumRays);

    vtkm::worklet::DispatcherMapField<detail::Luminance> lumDispatcher;
    lumDispatcher.Invoke(lum, rays.Buffers[0].Buffer);

    vtkm::Float32 initValue = 0.f;
    vtkm::Float32 sum = vtkm::cont::Algorithm::Reduce(lum, initValue);
    const vtkm::Float32 delta = .000001f;
    sum += delta; // avoid zero;
    vtkm::Float32 geomMean = vtkm::Pow(sum, 1.f / vtkm::Float32(rays.NumRays));

    vtkm::worklet::DispatcherMapField<detail::ToneMap> toneDispatcher(geomMean);
    toneDispatcher.Invoke(lum, rays.Buffers[0].Buffer);
  }

  template <typename FloatType>
  static void ToneMapExp(Ray<FloatType>& rays, vtkm::Float32 exposure)
  {
    vtkm::cont::ArrayHandleCounting<vtkm::Id> index(0, 1, rays.NumRays);

    vtkm::worklet::DispatcherMapField<detail::ToneMapExp> toneDispatcher(exposure);
    toneDispatcher.Invoke(index, rays.Buffers[0].Buffer);
  }
};

}
}
} // namespace vtkm::rendering::raytracing
#endif
