#include "SpheresScene.h"
#include "../source/Spheres.h"
#include "mpi/MpiEnv.h"

namespace beams
{
namespace rendering
{
void SpheresScene::LoadDataSet(const beams::Config& config,
                               const beams::Preset& preset,
                               beams::mpi::MpiEnv& mpi)
{
  const beams::DataSetOptions& dataSetOptions = config.DataSets.at(preset.DataSetId);
  const vtkm::Id SIZE = std::stoi(dataSetOptions.Params.at("size"));
  const vtkm::Id3 dims{ SIZE };
  this->Dims = dims;

  const vtkm::Float32 DEFAULT_FIELD_VAL = 0.1f;
  const vtkm::Vec3f_32 origin{ static_cast<vtkm::Float32>(mpi.XRank),
                               static_cast<vtkm::Float32>(mpi.YRank),
                               static_cast<vtkm::Float32>(mpi.ZRank) };
  const vtkm::Vec3f_32 spacing{ 1.0f / static_cast<vtkm::Float32>(dims[0]),
                                1.0f / static_cast<vtkm::Float32>(dims[1]),
                                1.0f / static_cast<vtkm::Float32>(dims[2]) };

  beams::source::Spheres source{
    origin, dims, spacing, DEFAULT_FIELD_VAL, beams::source::CoordinatesType::Rectilinear
  };

  bool smallSpheres = true;
  if (dataSetOptions.Params.find("smallSpheres") != dataSetOptions.Params.end())
  {
    smallSpheres = dataSetOptions.Params.at("smallSpheres") == "true";
  }

  const vtkm::Float32 CENTER_SPHERE_RAD = 0.1f;
  const vtkm::Float32 CENTER_SPHERE_VAL = 1.0f;
  const vtkm::Vec3f_32 CENTER_SPHERE_OFFSET{ 0.5f, 0.5f, 0.5f };
  source.AddSphere(
    origin + CENTER_SPHERE_OFFSET, CENTER_SPHERE_RAD, CENTER_SPHERE_VAL, CENTER_SPHERE_VAL);

  if (smallSpheres)
  {
    const vtkm::Float32 SMALL_FIELD_VALS[] = { 1.0f, 1.0f, 1.0f, 1.0f, 1.0f };
    const vtkm::Float32 SMALL_SPHERE_RAD = 0.04f;
    vtkm::Vec3f partialSphere{ 0.5f * static_cast<vtkm::Float32>(mpi.XLength),
                               0.5f * static_cast<vtkm::Float32>(mpi.YLength),
                               0.5f * static_cast<vtkm::Float32>(mpi.ZLength) };
    for (int i = 0; i < 8; ++i)
    {
      vtkm::Vec3f_32 centerPos = origin + CENTER_SPHERE_OFFSET;
      vtkm::Float32 multiplier = (i < 4) ? 1.5f : 5.0f;
      multiplier *= 0.5f + 0.8f * 0.5f;
      int j = i % 4;
      int x = ((j & 1) == 1) ? 1 : -1;
      int z = ((j & 2) == 2) ? 1 : -1;
      int y = (i < 4) ? 1 : -1;

      centerPos[0] += (CENTER_SPHERE_RAD + SMALL_SPHERE_RAD * multiplier) * x;
      centerPos[1] += (CENTER_SPHERE_RAD + SMALL_SPHERE_RAD * multiplier) * y;
      centerPos[2] += (CENTER_SPHERE_RAD + SMALL_SPHERE_RAD * multiplier) * z;
      source.AddSphere(centerPos, SMALL_SPHERE_RAD, SMALL_FIELD_VALS[j], SMALL_FIELD_VALS[j]);
    }
  }

  // Add floor if self rank is bottom layer of ranks
  if (mpi.YRank == 0)
  {
    const vtkm::Float32 FLOOR_FIELD_VAL = 1.0f;
    const vtkm::Float32 FLOOR_HEIGHT = 0.1f;
    source.SetFloor(FLOOR_HEIGHT, FLOOR_FIELD_VAL); // Floor
  }

  this->DataSet = source.Execute();
  this->FieldName = "spheres";
  this->DataSet.GetField(this->FieldName).GetRange(&this->LocalRange);
}
}
} //namespace beams::rendering
