#ifndef beams_rendering_scene_h
#define beams_rendering_scene_h

#include "../Config.h"
#include "../Result.h"
#include "BoundsMap.h"
#include "MapperDirectVolume.h"
#include "MapperLitVolume.h"
#include "MapperPhongVolume.h"
#include "MapperVolumeBase.h"
#include "RangeMap.h"

#include <vtkm/Types.h>
#include <vtkm/cont/DataSet.h>
#include <vtkm/rendering/Camera.h>
#include <vtkm/rendering/CanvasRayTracer.h>
#include <vtkm/rendering/Mapper.h>
#include <vtkm/rendering/MapperVolume.h>

#include <string>

namespace beams
{
namespace rendering
{
struct Scene
{
  Scene() = default;
  virtual ~Scene() = default;

  virtual void LoadFromPreset(const beams::Config& config,
                              const beams::Preset& preset,
                              beams::mpi::MpiEnv& mpi);
  virtual void LoadMpiTopology(const beams::Config& config,
                               const beams::Preset& preset,
                               beams::mpi::MpiEnv& mpi);
  virtual void LoadDataSet(const beams::Config& config,
                           const beams::Preset& preset,
                           beams::mpi::MpiEnv& mpi) = 0;
  virtual void LoadHistogram(const beams::Config& config,
                             const beams::Preset& preset,
                             beams::mpi::MpiEnv& mpi);
  virtual void LoadBoundsMap(const beams::Config& config,
                             const beams::Preset& preset,
                             beams::mpi::MpiEnv& mpi);
  virtual void LoadRangeMap(const beams::Config& config,
                            const beams::Preset& preset,
                            beams::mpi::MpiEnv& mpi);
  virtual void LoadCamera(const beams::Config& config,
                          const beams::Preset& preset,
                          beams::mpi::MpiEnv& mpi);
  virtual void LoadLights(const beams::Config& config,
                          const beams::Preset& preset,
                          beams::mpi::MpiEnv& mpi);
  virtual void LoadOpacityMap(const beams::Config& config,
                              const beams::Preset& preset,
                              beams::mpi::MpiEnv& mpi);
  virtual void LoadCanvas(const beams::Config& config,
                          const beams::Preset& preset,
                          beams::mpi::MpiEnv& mpi);
  virtual void LoadMapper(const beams::Config& config,
                          const beams::Preset& preset,
                          beams::mpi::MpiEnv& mpi);
  virtual void LoadColorTable(const beams::Config& config,
                              const beams::Preset& preset,
                              beams::mpi::MpiEnv& mpi);

  virtual void Ready();
  virtual void ReadyCanvas();
  virtual void ReadyCamera();
  virtual void ReadyColorTable();
  virtual void ReadyMapper();
  virtual void ReadyOpacityMap();
  virtual void ReadyLights();

  vtkm::Vec3f_32 GetTrueLightPosition() const;

  void SetTrueLightPosition(const vtkm::Vec3f_32& trueLightPosition);

  beams::rendering::MapperDirectVolume* GetMapperDirectVolume();

  beams::rendering::MapperLitVolume* GetMapperLitVolume();

  beams::rendering::MapperPhongVolume* GetMapperPhongVolume();

  virtual void PrintSummary(bool dataSet = true, bool opacityMap = true, bool render = true) const;

  std::string Id;
  vtkm::cont::DataSet DataSet;
  vtkm::Id3 Dims;
  vtkm::Id3 GlobalDims;
  std::string FieldName;
  vtkm::Range LocalRange;
  vtkm::Range OriginalLocalRange;
  vtkm::cont::ArrayHandle<vtkm::Id> DataSetHistogram;

  std::shared_ptr<beams::rendering::RangeMap> RangeMap;

  std::shared_ptr<beams::rendering::BoundsMap> BoundsMap;

  vtkm::rendering::Camera Camera;
  beams::Optional<vtkm::Vec3f_32> CameraPosition;
  beams::Optional<vtkm::Vec3f_32> CameraLookAt;
  beams::Optional<vtkm::Vec3f_32> CameraUp;
  beams::Optional<vtkm::Float32> CameraFov;

  vtkm::Vec3f_32 LightPosition;
  vtkm::Vec3f_32 LightColor;
  vtkm::Float32 LightIntensity = 1.0f;
  bool ShowLight = false;
  vtkm::Float32 ShowLightLimits = 3.0f;

  RendererType Renderer;
  vtkm::Id3 ShadowMapSize;
  vtkm::Id ShadowMapNumSteps = 128;
  bool UseClamp;
  bool UseReinhard;
  bool UsePhongDiffuse;
  bool UsePhongSpecular;

  //beams::rendering::MapperLitVolume Mapper;
  std::shared_ptr<beams::rendering::MapperVolumeBase> Mapper;
  vtkm::rendering::CanvasRayTracer* Canvas;

  vtkm::Float32 SampleDistance;
  vtkm::Id S0 = 128;
  vtkm::Id NumShadowSamples;

  vtkm::cont::ColorTable ColorTable;
  std::string ColorTableName;
  std::vector<beams::PointAlpha> PointAlphas;
}; // struct Scene
}
} // namespace beams::rendering

#endif // beams_rendering_scene_h
