#include "Scene.h"

#include "../utils/Fmt.h"
#include "MapperDirectVolume.h"
#include "PointLight.h"
#include "Utils.h"

#include <vtkm/filter/density_estimate/Histogram.h>

namespace beams
{
namespace rendering
{
void Scene::LoadFromPreset(const beams::Config& config,
                           const beams::Preset& preset,
                           beams::mpi::MpiEnv& mpi)
{
  this->Id = preset.Id;

  this->LoadMpiTopology(config, preset, mpi);
  this->LoadDataSet(config, preset, mpi);
  this->LoadBoundsMap(config, preset, mpi);
  this->LoadRangeMap(config, preset, mpi);
  this->LoadHistogram(config, preset, mpi);
  this->LoadCamera(config, preset, mpi);
  this->LoadLights(config, preset, mpi);
  this->LoadColorTable(config, preset, mpi);
  this->LoadOpacityMap(config, preset, mpi);
  this->LoadCanvas(config, preset, mpi);
  this->LoadMapper(config, preset, mpi);
}

void Scene::LoadMpiTopology(const beams::Config& vtkmNotUsed(config),
                            const beams::Preset& vtkmNotUsed(preset),
                            beams::mpi::MpiEnv& mpi)
{
  //if (mpi.Size < 8)
  {
    mpi.ReshapeAsRectangle();
  }
  //else
  {
    //mpi.ReshapeAsCuboid();
  }
}

void Scene::LoadHistogram(const beams::Config& vtkmNotUsed(config),
                          const beams::Preset& vtkmNotUsed(preset),
                          beams::mpi::MpiEnv& vtkmNotUsed(mpi))
{
  const vtkm::Id numBins = 256;
  vtkm::filter::density_estimate::Histogram histogramFilter;
  histogramFilter.SetRange(this->LocalRange);
  histogramFilter.SetNumberOfBins(numBins);
  histogramFilter.SetActiveField(this->FieldName);
  auto histogramResult = histogramFilter.Execute(this->DataSet);
  histogramResult.GetField("histogram").GetData().AsArrayHandle(this->DataSetHistogram);
}

void Scene::LoadBoundsMap(const beams::Config& vtkmNotUsed(config),
                          const beams::Preset& vtkmNotUsed(preset),
                          beams::mpi::MpiEnv& vtkmNotUsed(mpi))
{
  this->BoundsMap = std::make_shared<beams::rendering::BoundsMap>(this->DataSet);
}

void Scene::LoadRangeMap(const beams::Config& vtkmNotUsed(config),
                         const beams::Preset& vtkmNotUsed(preset),
                         beams::mpi::MpiEnv& mpi)
{
  this->RangeMap = std::make_shared<beams::rendering::RangeMap>(mpi, this->LocalRange);
}

void Scene::LoadCamera(const beams::Config& config,
                       const beams::Preset& preset,
                       beams::mpi::MpiEnv& vtkmNotUsed(mpi))
{
  this->Camera = vtkm::rendering::Camera();
  const auto& cameraOptions = config.Cameras.at(preset.CameraId);
  this->CameraPosition = cameraOptions.Position;
  this->CameraLookAt = cameraOptions.LookAt;
  this->CameraUp = cameraOptions.Up;
  this->CameraFov = cameraOptions.Fov;
}

void Scene::LoadLights(const beams::Config& config,
                       const beams::Preset& preset,
                       beams::mpi::MpiEnv& vtkmNotUsed(mpi))
{
  const auto& lightCollection = config.LightCollections.at(preset.LightCollectionId);
  this->LightColor = lightCollection.Lights[0].Color;
  this->LightPosition = lightCollection.Lights[0].Position;
  this->LightIntensity = lightCollection.Lights[0].Intensity;
}

void Scene::LoadOpacityMap(const beams::Config& vtkmNotUsed(config),
                           const beams::Preset& preset,
                           beams::mpi::MpiEnv& vtkmNotUsed(mpi))
{
  this->ShadowMapSize = preset.OpacityMapOptions.Size;
  this->ShadowMapNumSteps = preset.OpacityMapOptions.NumSteps;
}

void Scene::LoadCanvas(const beams::Config& config,
                       const beams::Preset& preset,
                       beams::mpi::MpiEnv& vtkmNotUsed(mpi))
{
  const beams::CanvasOptions& canvasOptions = config.Canvases.at(preset.RenderOptions.CanvasId);
  this->Canvas = new vtkm::rendering::CanvasRayTracer(canvasOptions.Width, canvasOptions.Height);
}

void Scene::LoadMapper(const beams::Config& vtkmNotUsed(config),
                       const beams::Preset& preset,
                       beams::mpi::MpiEnv& vtkmNotUsed(mpi))
{
  this->Renderer = preset.RenderOptions.Renderer;
  vtkm::Float32 sampleDistance = preset.RenderOptions.SampleDistance;
  if (preset.RenderOptions.NumSamples > 0)
  {
    sampleDistance = Utils::SampleDistanceFromBounds(
      this->DataSet.GetCoordinateSystem().GetBounds(), preset.RenderOptions.NumSamples);
  }
  this->SampleDistance = sampleDistance;
  this->S0 = preset.RenderOptions.S0;
  this->NumShadowSamples = preset.RenderOptions.NumShadowSamples;
  this->UseClamp = preset.RenderOptions.UseClamp;
  this->UseReinhard = preset.RenderOptions.UseReinhard;
  this->UsePhongDiffuse = preset.RenderOptions.UsePhongDiffuse;
  this->UsePhongSpecular = preset.RenderOptions.UsePhongSpecular;
}

void Scene::LoadColorTable(const beams::Config& config,
                           const beams::Preset& preset,
                           beams::mpi::MpiEnv& vtkmNotUsed(mpi))
{
  const auto& colorTableOptions = config.ColorTables.at(preset.ColorTableId);
  this->ColorTableName = colorTableOptions.Name;
  this->PointAlphas = colorTableOptions.PointAlphas;
  this->ColorTable = vtkm::cont::ColorTable(this->ColorTableName);
}

void Scene::Ready()
{
  this->ReadyCanvas();
  this->ReadyCamera();
  this->ReadyColorTable();
  this->ReadyMapper();
  this->ReadyOpacityMap();
  this->ReadyLights();
}

void Scene::ReadyCanvas()
{
  vtkm::rendering::Color background(0.8f, 0.8f, 0.8f, 1.0f);
  vtkm::rendering::Color foreground(0.0f, 0.0f, 0.0f, 1.0f);
  this->Canvas->SetBackgroundColor(background);
  this->Canvas->SetForegroundColor(foreground);
  this->Canvas->Clear();
}

void Scene::ReadyCamera()
{
  if (this->CameraPosition)
  {
    this->Camera.SetPosition(this->CameraPosition.Value);
  }
  if (this->CameraLookAt)
  {
    this->Camera.SetLookAt(this->CameraLookAt.Value);
  }
  if (this->CameraUp)
  {
    this->Camera.SetViewUp(this->CameraUp.Value);
  }

  auto bounds = this->BoundsMap->GlobalBounds;
  if (this->ShowLight)
  {
    vtkm::Float32 multiplier = this->ShowLightLimits * 0.5f;
    bounds.X.Min = bounds.X.Center() - multiplier * bounds.X.Length() / 2.0f;
    bounds.Y.Min = bounds.Y.Center() - multiplier * bounds.Y.Length() / 2.0f;
    bounds.Z.Min = bounds.Z.Center() - multiplier * bounds.Z.Length() / 2.0f;
    bounds.X.Max = bounds.X.Center() + multiplier * bounds.X.Length() / 2.0f;
    bounds.Y.Max = bounds.Y.Center() + multiplier * bounds.Y.Length() / 2.0f;
    bounds.Z.Max = bounds.Z.Center() + multiplier * bounds.Z.Length() / 2.0f;
  }
  this->Camera.ResetToBounds(bounds);

  if (this->CameraFov)
  {
    this->Camera.SetFieldOfView(this->CameraFov.Value);
  }

  this->CameraPosition = this->Camera.GetPosition();
  this->CameraLookAt = this->Camera.GetLookAt();
  this->CameraUp = this->Camera.GetViewUp();
  this->CameraFov = this->Camera.GetFieldOfView();
}

void Scene::ReadyColorTable()
{
  this->ColorTable = vtkm::cont::ColorTable(this->ColorTableName);
  for (const auto& pointAlpha : this->PointAlphas)
  {
    this->ColorTable.AddPointAlpha(pointAlpha.Point, pointAlpha.Alpha);
  }
}

void Scene::ReadyMapper()
{
  vtkm::Float32 densityCorrectionRatio =
    static_cast<vtkm::Float32>(this->S0) / static_cast<vtkm::Float32>(this->ShadowMapNumSteps);

  if (this->Renderer == RendererType::DirectVolume)
  {
    this->Mapper = std::make_shared<beams::rendering::MapperDirectVolume>();
  }
  else if (this->Renderer == RendererType::ShadowVolume)
  {
    this->Mapper = std::make_shared<beams::rendering::MapperLitVolume>();
  }
  else if (this->Renderer == RendererType::PhongVolume)
  {
    this->Mapper = std::make_shared<beams::rendering::MapperPhongVolume>();
  }
  this->Mapper->SetCompositeBackground(true);
  this->Mapper->SetCanvas(this->Canvas);
  this->Mapper->SetActiveColorTable(this->ColorTable);
  this->Mapper->SetSampleDistance(this->SampleDistance);
  this->Mapper->SetDensityCorrectionRatio(densityCorrectionRatio);
  if (this->Renderer == RendererType::ShadowVolume)
  {
    auto shadowMapper = this->GetMapperLitVolume();
    shadowMapper->SetDensityScale(1.0f);
    shadowMapper->SetBoundsMap(this->BoundsMap.get());
    shadowMapper->SetNumShadowSamples(this->NumShadowSamples);
    shadowMapper->SetUseClamp(this->UseClamp);
    shadowMapper->SetUseReinhard(this->UseReinhard);
  }
  else if (this->Renderer == RendererType::PhongVolume)
  {
    auto phongMapper = this->GetMapperPhongVolume();
    phongMapper->SetUsePhongDiffuse(this->UsePhongDiffuse);
    phongMapper->SetUsePhongSpecular(this->UsePhongSpecular);
  }
}

void Scene::ReadyOpacityMap()
{
  if (this->Renderer == RendererType::ShadowVolume)
  {
    auto mapper = this->GetMapperLitVolume();
    mapper->SetShadowMapSize(this->ShadowMapSize);
    mapper->SetShadowMapNumSteps(this->ShadowMapNumSteps);
  }
}

template <typename Precision1, typename Precision2 = Precision1>
VTKM_EXEC_CONT Precision2 InvLerp(const Precision1& start,
                                  const Precision1& end,
                                  const Precision2& value)
{
  return (value - start) / (end - start);
}

vtkm::Vec3f_32 Scene::GetTrueLightPosition() const
{
  vtkm::Vec3f_32 scaledLightPosition;
  scaledLightPosition[0] = vtkm::Lerp(this->BoundsMap->GlobalBounds.X.Min,
                                      this->BoundsMap->GlobalBounds.X.Max,
                                      this->LightPosition[0]);
  scaledLightPosition[1] = vtkm::Lerp(this->BoundsMap->GlobalBounds.Y.Min,
                                      this->BoundsMap->GlobalBounds.Y.Max,
                                      this->LightPosition[1]);
  scaledLightPosition[2] = vtkm::Lerp(this->BoundsMap->GlobalBounds.Z.Min,
                                      this->BoundsMap->GlobalBounds.Z.Max,
                                      this->LightPosition[2]);
  return scaledLightPosition;
}

void Scene::SetTrueLightPosition(const vtkm::Vec3f_32& trueLightPosition)
{
  vtkm::Vec3f_32 relLightPosition;
  relLightPosition[0] = InvLerp(
    this->BoundsMap->GlobalBounds.X.Min, this->BoundsMap->GlobalBounds.X.Max, trueLightPosition[0]);
  relLightPosition[1] = InvLerp(
    this->BoundsMap->GlobalBounds.Y.Min, this->BoundsMap->GlobalBounds.Y.Max, trueLightPosition[1]);
  relLightPosition[2] = InvLerp(
    this->BoundsMap->GlobalBounds.Z.Min, this->BoundsMap->GlobalBounds.Z.Max, trueLightPosition[2]);
  this->LightPosition = relLightPosition;
}

void Scene::ReadyLights()
{
  if (this->Renderer == RendererType::ShadowVolume)
  {
    auto mapper = this->GetMapperLitVolume();
    mapper->ClearLights();
    std::shared_ptr<beams::rendering::Light> light =
      std::make_shared<beams::rendering::PointLight<vtkm::Float32>>(
        this->GetTrueLightPosition(), this->LightColor, this->LightIntensity);
    mapper->AddLight(light);
  }
  else if (this->Renderer == RendererType::PhongVolume)
  {
    auto mapper = this->GetMapperPhongVolume();
    mapper->ClearLights();
    std::shared_ptr<beams::rendering::Light> light =
      std::make_shared<beams::rendering::PointLight<vtkm::Float32>>(
        this->GetTrueLightPosition(), this->LightColor, this->LightIntensity);
    mapper->AddLight(light);
  }
}

void Scene::PrintSummary(bool dataSet, bool opacityMap, bool render) const
{
  if (dataSet)
  {
    Fmt::RawPrint0("\n**** DataSet summary ****\n");
    Fmt::RawPrint0("Dims            = {}\n", this->Dims);
    Fmt::RawPrint0("Root  Bounds    = {}\n", this->BoundsMap->BlockBounds[0]);
    Fmt::RawPrint0("Total Bounds    = {}\n", this->BoundsMap->GlobalBounds);
    Fmt::RawPrint0("Field           = {}\n", this->FieldName);
    Fmt::RawPrint0("Local Range     = {}\n", this->LocalRange);
    Fmt::RawPrint0("Global Range    = {}\n", this->RangeMap->GetGlobalRange());
    Fmt::RawPrint0("\n");
  }

  if (opacityMap)
  {
    Fmt::RawPrint0("\n**** Opacity Map summary ****\n");
    Fmt::RawPrint0("Dims            = {}\n", this->ShadowMapSize);
    Fmt::RawPrint0("Light Position  = {}\n", this->LightPosition);
    Fmt::RawPrint0("Light Color     = {}\n", this->LightColor);
    Fmt::RawPrint0("Light Intensity = {}\n", this->LightIntensity);
    Fmt::RawPrint0("NumSteps        = {}\n", this->ShadowMapNumSteps);
    Fmt::RawPrint0("\n");
  }

  if (render)
  {
    Fmt::RawPrint0("\n**** Render summary ****\n");
    Fmt::RawPrint0("Canvas Width     = {}\n", this->Canvas->GetWidth());
    Fmt::RawPrint0("Canvas Height    = {}\n", this->Canvas->GetHeight());
    Fmt::RawPrint0("SampleDistance   = {}\n", this->SampleDistance);
    Fmt::RawPrint0("S0               = {}\n", this->S0);
    Fmt::RawPrint0("NumShadowSamples = {}\n", this->NumShadowSamples);
    Fmt::RawPrint0("\n");
  }
}

beams::rendering::MapperDirectVolume* Scene::GetMapperDirectVolume()
{
  return reinterpret_cast<beams::rendering::MapperDirectVolume*>(this->Mapper.get());
}

beams::rendering::MapperLitVolume* Scene::GetMapperLitVolume()
{
  return reinterpret_cast<beams::rendering::MapperLitVolume*>(this->Mapper.get());
}

beams::rendering::MapperPhongVolume* Scene::GetMapperPhongVolume()
{
  return reinterpret_cast<beams::rendering::MapperPhongVolume*>(this->Mapper.get());
}
}
}
