#include "PerlinScene.h"
#include "utils/Fmt.h"

#include "../source/PerlinNoise.h"

namespace beams
{
namespace rendering
{
void PerlinScene::LoadDataSet(const beams::Config& config,
                              const beams::Preset& preset,
                              beams::mpi::MpiEnv& mpi)
{
  const beams::DataSetOptions& dataSetOptions = config.DataSets.at(preset.DataSetId);

  const vtkm::Id SIZE = std::stoi(dataSetOptions.Params.at("size"));
  const vtkm::Id3 globalDims{ SIZE };

  vtkm::Float32 scale = 5.0f;
  if (dataSetOptions.Params.find("scale") != dataSetOptions.Params.end())
  {
    scale = std::stof(dataSetOptions.Params.at("scale"));
  }

  const vtkm::Id seed = std::stoi(dataSetOptions.Params.at("seed"));

  int worldSize = mpi.Size;
  vtkm::Id3 localDims = globalDims;
  vtkm::IdComponent dimIdx = 0;
  this->MpiSizes = { 1, 1, 1 };

  while (worldSize > 1)
  {
    localDims[dimIdx] /= 2;
    this->MpiSizes[dimIdx] *= 2;
    worldSize /= 2;
    dimIdx = (dimIdx + 1) % 3;
  }

  mpi.ReshapeCustom(this->MpiSizes[0], this->MpiSizes[1], this->MpiSizes[2]);

  const vtkm::Vec3f_32 blockSize{ scale / static_cast<vtkm::Float32>(this->MpiSizes[0]),
                                  scale / static_cast<vtkm::Float32>(this->MpiSizes[1]),
                                  scale / static_cast<vtkm::Float32>(this->MpiSizes[2]) };
  const vtkm::Vec3f_32 minExtent =
    vtkm::Vec3f_32{ blockSize[0] * static_cast<vtkm::Float32>(mpi.XRank),
                    blockSize[1] * static_cast<vtkm::Float32>(mpi.YRank),
                    blockSize[2] * static_cast<vtkm::Float32>(mpi.ZRank) };
  const vtkm::Vec3f_32 maxExtent = minExtent + blockSize;
  beams::source::PerlinNoise source(beams::source::CoordinatesType::Uniform);
  source.SetCellDimensions(localDims);
  source.SetSeed(seed);
  source.SetMinExtent(minExtent);
  source.SetMaxExtent(maxExtent);
  this->DataSet = source.Execute();
  this->Dims = localDims;
  this->GlobalDims = globalDims;
  this->FieldName = "perlinnoise";
  this->DataSet.GetField(this->FieldName).GetRange(&this->LocalRange);
  this->OriginalLocalRange = this->LocalRange;
}

void PerlinScene::LoadMpiTopology(const beams::Config& vtkmNotUsed(config),
                                  const beams::Preset& vtkmNotUsed(preset),
                                  beams::mpi::MpiEnv& vtkmNotUsed(mpi))
{
}

void PerlinScene::LoadOpacityMap(const beams::Config& config,
                                 const beams::Preset& preset,
                                 beams::mpi::MpiEnv& mpi)
{
  Scene::LoadOpacityMap(config, preset, mpi);

  this->ShadowMapSize[0] /= mpi.XLength;
  this->ShadowMapSize[1] /= mpi.YLength;
  this->ShadowMapSize[2] /= mpi.ZLength;
}
}
}
