#include "MapperDirectVolume.h"
#include "DirectVolumeRendererStructured.h"

#include <vtkm/cont/Timer.h>
#include <vtkm/cont/TryExecute.h>

#include <vtkm/rendering/CanvasRayTracer.h>
#include <vtkm/rendering/raytracing/Camera.h>
#include <vtkm/rendering/raytracing/Logger.h>
#include <vtkm/rendering/raytracing/RayOperations.h>

#include <sstream>


namespace beams
{
namespace rendering
{
struct MapperDirectVolume::InternalsType
{
  VTKM_CONT
  InternalsType() {}
};

MapperDirectVolume::MapperDirectVolume()
  : Internals(new InternalsType)
{
}

MapperDirectVolume::~MapperDirectVolume() {}

void MapperDirectVolume::RenderCells(const vtkm::cont::UnknownCellSet& cellset,
                                     const vtkm::cont::CoordinateSystem& coords,
                                     const vtkm::cont::Field& scalarField,
                                     const vtkm::cont::ColorTable& vtkmNotUsed(colorTable),
                                     const vtkm::rendering::Camera& camera,
                                     const vtkm::Range& scalarRange)
{
  if (!cellset.CanConvert<vtkm::cont::CellSetStructured<3>>())
  {
    std::stringstream msg;
    std::string theType = typeid(cellset).name();
    msg << "Mapper volume: cell set type not currently supported\n";
    msg << "Type : " << theType << std::endl;
    throw vtkm::cont::ErrorBadValue(msg.str());
  }
  vtkm::cont::Timer tot_timer;
  tot_timer.Start();
  vtkm::cont::Timer timer;

  beams::rendering::DirectVolumeRendererStructured tracer;

  vtkm::rendering::raytracing::Camera rayCamera;
  vtkm::Int32 width = (vtkm::Int32)this->Canvas->GetWidth();
  vtkm::Int32 height = (vtkm::Int32)this->Canvas->GetHeight();
  rayCamera.SetParameters(camera, width, height);

  vtkm::rendering::raytracing::Ray<vtkm::Float32> rays;
  rayCamera.CreateRays(rays, coords.GetBounds());
  rays.Buffers.at(0).InitConst(0.f);
  vtkm::rendering::raytracing::RayOperations::MapCanvasToRays(rays, camera, *this->Canvas);

  if (this->SampleDistance <= 0.0f)
  {
    auto spatialExtent = coords.GetBounds();
    vtkm::Vec3f_32 extent;
    extent[0] = static_cast<vtkm::Float32>(spatialExtent.X.Length());
    extent[1] = static_cast<vtkm::Float32>(spatialExtent.Y.Length());
    extent[2] = static_cast<vtkm::Float32>(spatialExtent.Z.Length());
    vtkm::Float32 magExtent = vtkm::Magnitude(extent);
    const vtkm::Float32 defaultNumberOfSamples = 200.f;
    this->SampleDistance = magExtent / defaultNumberOfSamples;
  }
  tracer.SetSampleDistance(this->SampleDistance);
  tracer.SetDensityCorrectionRatio(this->DensityCorrectionRatio);

  tracer.SetData(
    coords, scalarField, cellset.AsCellSet<vtkm::cont::CellSetStructured<3>>(), scalarRange);
  tracer.SetColorMap(this->ColorMap);

  tracer.Render(rays);

  timer.Start();
  this->Canvas->WriteToCanvas(rays, rays.Buffers.at(0).Buffer, camera);

  if (this->CompositeBackground)
  {
    this->Canvas->BlendBackground();
  }
}

vtkm::rendering::Mapper* MapperDirectVolume::NewCopy() const
{
  return new beams::rendering::MapperDirectVolume(*this);
}

std::vector<beams::profiling::Record> MapperDirectVolume::GetProfilerTimes()
{
  std::vector<beams::profiling::Record> results;
  return results;
}
}
}
