#include "FileSceneBase.h"
#include "utils/Fmt.h"

#include <vtkm/Types.h>
#include <vtkm/io/FileUtils.h>
#include <vtkm/io/VTKDataSetReader.h>
#include <vtkm/rendering/CanvasRayTracer.h>

namespace beams
{
namespace rendering
{
namespace
{
bool string_replace(std::string& str, const std::string& from, const std::string& to)
{
  std::size_t pos = str.find(from);
  if (pos == std::string::npos)
  {
    return false;
  }
  str.replace(pos, from.length(), to);
  return true;
}

void string_replace_all(std::string& str, const std::string& from, const std::string& to)
{
  while (string_replace(str, from, to))
    ;
}
}

std::string FileSceneBase::GetFileName(const beams::Config& config,
                                       const beams::Preset& preset,
                                       const beams::mpi::MpiEnv& mpi) const
{
  const beams::DataSetOptions& dataSetOptions = config.DataSets.at(preset.DataSetId);
  std::string fileNamePattern = dataSetOptions.Params.at("fileNamePattern");
  std::string basePath = vtkm::io::ParentPath(config.FilePath);
  fileNamePattern = vtkm::io::MergePaths(basePath, fileNamePattern);
  string_replace_all(fileNamePattern, "%r", std::to_string(mpi.Rank));
  return fileNamePattern;
}

void FileSceneBase::LoadDataSet(const beams::Config& config,
                                const beams::Preset& preset,
                                beams::mpi::MpiEnv& mpi)
{
  std::string fileName = this->GetFileName(config, preset, mpi);
  Fmt::Println("Trying to load {}", fileName);
  vtkm::io::VTKDataSetReader source(fileName);
  this->DataSet = source.ReadDataSet();

  const beams::DataSetOptions& dataSetOptions = config.DataSets.at(preset.DataSetId);
  if (dataSetOptions.Params.count("fieldName") > 0)
  {
    this->FieldName = dataSetOptions.Params.at("fieldName");
  }
  else
  {
    Fmt::RawPrint0("No field name specified in config file. Specify one of the following fields\n");
    for (vtkm::Id i = 0; i < this->DataSet.GetNumberOfFields(); ++i)
    {
      Fmt::RawPrint0("Field {}: {}\n", i, this->DataSet.GetField(i).GetName());
    }
    exit(1);
  }
  this->DataSet.GetField(this->FieldName).GetRange(&this->LocalRange);
  if (dataSetOptions.Params.find("fieldRangeMin") != dataSetOptions.Params.end())
  {
    auto t = this->LocalRange.Min;
    this->LocalRange.Min = std::stod(dataSetOptions.Params.at("fieldRangeMin"));
    Fmt::Println0("Field range min {} to {}", t, this->LocalRange.Min);
  }
  if (dataSetOptions.Params.find("fieldRangeMax") != dataSetOptions.Params.end())
  {
    auto t = this->LocalRange.Max;
    this->LocalRange.Max = std::stod(dataSetOptions.Params.at("fieldRangeMax"));
    Fmt::Println0("Field range max {} to {}", t, this->LocalRange.Max);
  }
  this->OriginalLocalRange = this->LocalRange;

  using DefaultHandle = vtkm::cont::ArrayHandle<vtkm::FloatDefault>;
  using CartesianArrayHandle =
    vtkm::cont::ArrayHandleCartesianProduct<DefaultHandle, DefaultHandle, DefaultHandle>;
  vtkm::cont::CoordinateSystem coords = this->DataSet.GetCoordinateSystem();
  if (coords.GetData().IsType<vtkm::cont::ArrayHandleUniformPointCoordinates>())
  {
    this->Dims = coords.GetData()
                   .AsArrayHandle<vtkm::cont::ArrayHandleUniformPointCoordinates>()
                   .GetDimensions();
  }
  else if (coords.GetData().IsType<CartesianArrayHandle>())
  {
    CartesianArrayHandle data = coords.GetData().AsArrayHandle<CartesianArrayHandle>();
    this->Dims = vtkm::Id3{ data.GetFirstArray().GetNumberOfValues(),
                            data.GetSecondArray().GetNumberOfValues(),
                            data.GetThirdArray().GetNumberOfValues() };
  }
}
}
} //namespace beams::rendering