#include "Window.h"

#include "../SceneFactory.h"
#include "mpi/MpiEnv.h"
#include "source/IcoSphere.h"
#include "utils/File.h"
#include "utils/Fmt.h"
#include "utils/Image.h"
#include "utils/String.h"

#include <vtkm/io/FileUtils.h>
#include <vtkm/io/VTKDataSetWriter.h>
#include <vtkm/rendering/MapperRayTracer.h>

#include <GLFW/glfw3.h>
#include <backends/imgui_impl_glfw.h>
#include <backends/imgui_impl_opengl2.h>
#include <imgui.h>
#include <imgui_internal.h>

#include <iomanip>

namespace beams
{
namespace gui
{
static ImVec2 operator+(const ImVec2& lhs, const ImVec2& rhs)
{
  return ImVec2(lhs.x + rhs.x, lhs.y + rhs.y);
}

static ImVec2 operator-(const ImVec2& lhs, const ImVec2& rhs)
{
  return ImVec2(lhs.x - rhs.x, lhs.y - rhs.y);
}

static ImVec2 operator*(const ImVec2& lhs, const ImVec2& rhs)
{
  return ImVec2(lhs.x * rhs.x, lhs.y * rhs.y);
}

static ImVec2 operator/(const ImVec2& lhs, const ImVec2& rhs)
{
  return ImVec2(lhs.x / rhs.x, lhs.y / rhs.y);
}

static float Length(const ImVec2& v)
{
  return std::sqrt(v.x * v.x + v.y * v.y);
}

template <typename T>
static bool compareVec(const std::vector<T>& vec1, const std::vector<T>& vec2)
{
  if (vec1.size() != vec2.size())
  {
    return false;
  }

  for (size_t i = 0; i < vec1.size(); i++)
  {
    if (vec1[i] != vec2[i])
    {
      return false;
    }
  }

  return true;
}

namespace detail
{
static void GlfwErrorCallback(int error, const char* description)
{
  Fmt::Println0("Glfw Error {}: {}", error, description);
}
} // namespace detail

struct Window::WindowInternals
{
  std::string Title;
  int Width;
  int Height;
  beams::mpi::MpiEnv* Mpi = nullptr;
  std::string CurrentPresetId;
  std::shared_ptr<beams::Config> Config = nullptr;
  std::shared_ptr<beams::rendering::Scene> Scene = nullptr;

  GLFWwindow* GlfwWindow = nullptr;
  ImGuiContext* ImGuiCtx = nullptr;

  double StartMouseX = 0.0;
  double StartMouseY = 0.0;

  volatile bool _SceneDirty = true;
  volatile bool _ViewDirty = true;

  std::function<void()> OnReloadConfig;

  void LoadScene()
  {
    const beams::Preset& preset = this->Config->Presets[this->CurrentPresetId];
    this->Scene = beams::SceneFactory::CreateFromPreset(*this->Config, preset, *this->Mpi);
  }

  beams::Result InitializeGlfw(const Window* window)
  {
    glfwSetErrorCallback(detail::GlfwErrorCallback);
    if (!glfwInit())
    {
      return beams::Result::Failed("Failed to initialize GLFW");
    }
    glfwWindowHint(GLFW_RESIZABLE, GL_FALSE);
    this->GlfwWindow =
      glfwCreateWindow(this->Width, this->Height, this->Title.c_str(), nullptr, nullptr);
    if (this->GlfwWindow == nullptr)
    {
      return beams::Result::Failed("Failed to create GLFW window");
    }
    if (beams::io::File::FileExists("icon.png"))
    {
      GLFWimage icon;
      icon.pixels = stbi_load("icon.png", &icon.width, &icon.height, nullptr, 4);
      glfwSetWindowIcon(this->GlfwWindow, 1, &icon);
    }
    else
    {
      Fmt::Println0("No icon.png found");
    }
    glfwMakeContextCurrent(this->GlfwWindow);
    // Enable vsync
    glfwSwapInterval(1);
    glfwSetWindowUserPointer(this->GlfwWindow, const_cast<Window*>(window));
    return beams::Result::Succeeded();
  }

  void ShutdownGlfw()
  {
    glfwDestroyWindow(this->GlfwWindow);
    glfwTerminate();
  }

  beams::Result InitializeImGui()
  {
    IMGUI_CHECKVERSION();
    this->ImGuiCtx = ImGui::CreateContext();

    ImGuiIO& io = ImGui::GetIO();
    io.ConfigFlags |= ImGuiConfigFlags_NavEnableKeyboard;

    ImFontConfig fontConfig;
    fontConfig.SizePixels = 20.0f;
    io.Fonts->AddFontDefault(&fontConfig);

    ImGui::StyleColorsDark();

    auto result = ImGui_ImplGlfw_InitForOpenGL(this->GlfwWindow, true);
    if (!result)
    {
      return beams::Result::Failed("Failed to initialize ImGui GLFW backend");
    }
    result = ImGui_ImplOpenGL2_Init();
    if (!result)
    {
      return beams::Result::Failed("Failed to initialize ImGui OpenGL2 backend");
    }
    return beams::Result::Succeeded();
  }

  void ShutdownImGui()
  {
    ImGui_ImplOpenGL2_Shutdown();
    ImGui_ImplGlfw_Shutdown();
    ImGui::DestroyContext(this->ImGuiCtx);
  }

  beams::Result SetCallbacks()
  {
    this->SetCursorStateCallback();
    this->SetKeyCallback();
    return beams::Result::Succeeded();
  }

  beams::Result SetCursorStateCallback()
  {
    auto wrappedFunc = [](GLFWwindow* window, int button, int action, int)
    {
      ImGuiIO& io = ImGui::GetIO();
      io.AddMouseButtonEvent(button, action == GLFW_PRESS);
      if (io.WantCaptureMouse)
      {
        return;
      }

      auto* windowPtr = static_cast<Window*>(glfwGetWindowUserPointer(window));
      auto leftMouseDown = (button == GLFW_MOUSE_BUTTON_LEFT && action == GLFW_PRESS);
      double xPos, yPos;
      glfwGetCursorPos(window, &xPos, &yPos);
      if (leftMouseDown)
      {
        windowPtr->Internals->StartMouseX = xPos;
        windowPtr->Internals->StartMouseY = yPos;
      }
      windowPtr->OnCursorState(leftMouseDown, xPos, yPos);
    };
    glfwSetMouseButtonCallback(this->GlfwWindow, wrappedFunc);

    return beams::Result::Succeeded();
  }

  beams::Result SetKeyCallback()
  {
    auto wrappedFunc = [](GLFWwindow* window, int key, int scancode, int action, int mods)
    {
      ImGui_ImplGlfw_KeyCallback(window, key, scancode, action, mods);

      ImGuiIO& io = ImGui::GetIO();
      if (io.WantCaptureKeyboard)
      {
        return;
      }

      auto* windowPtr = static_cast<Window*>(glfwGetWindowUserPointer(window));
      auto leftArrowDown =
        (key == GLFW_KEY_LEFT && (action == GLFW_PRESS || action == GLFW_REPEAT));
      auto rightArrowDown =
        (key == GLFW_KEY_RIGHT && (action == GLFW_PRESS || action == GLFW_REPEAT));
      auto upArrowDown = (key == GLFW_KEY_UP && (action == GLFW_PRESS || action == GLFW_REPEAT));
      auto downArrowDown =
        (key == GLFW_KEY_DOWN && (action == GLFW_PRESS || action == GLFW_REPEAT));
      windowPtr->OnArrowKeys(leftArrowDown, rightArrowDown, upArrowDown, downArrowDown);

      if (key == GLFW_KEY_ESCAPE && action == GLFW_PRESS)
      {
        glfwSetWindowShouldClose(window, GLFW_TRUE);
      }
    };
    glfwSetKeyCallback(this->GlfwWindow, wrappedFunc);

    return beams::Result::Succeeded();
  }

  void UpdateTrackball(vtkm::Float64 startX,
                       vtkm::Float64 startY,
                       vtkm::Float64 endX,
                       vtkm::Float64 endY)
  {
    this->Scene->Camera.TrackballRotate(startX, startY, endX, endY);
    this->SetViewDirty(true);
    this->Scene->CameraPosition = this->Scene->Camera.GetPosition();
    this->Scene->CameraLookAt = this->Scene->Camera.GetLookAt();
  }

  bool IsViewDirty() const { return this->_ViewDirty; }

  void SetViewDirty(bool dirty) { this->_ViewDirty = dirty; }

  bool IsSceneDirty() const { return this->_SceneDirty; }

  void SetSceneDirty(bool dirty) { this->_SceneDirty = dirty; }
};

Window::Window()
  : Internals(new WindowInternals())
{
}

Window::~Window()
{
  this->Shutdown();
}

beams::Result Window::Initialize(const std::string& title,
                                 int width,
                                 int height,
                                 std::shared_ptr<beams::Config> config,
                                 beams::mpi::MpiEnv* mpi,
                                 std::function<void()> onReloadConfig)
{
  this->Internals->Title = title;
  this->Internals->Width = width;
  this->Internals->Height = height;
  this->Internals->Config = config;
  this->Internals->Mpi = mpi;
  this->Internals->CurrentPresetId = config->DefaultPresetId;
  this->Internals->OnReloadConfig = onReloadConfig;

  if (mpi->Rank == 0)
  {
    CHECK_RESULT(this->Internals->InitializeGlfw(this), "Error initializing GLFW");
    CHECK_RESULT(this->Internals->InitializeImGui(), "Error initializing ImGui");
    CHECK_RESULT(this->Internals->SetCallbacks(), "Error setting cursor callback");
  }
  return beams::Result::Succeeded();
}

void Window::Shutdown()
{
  if (this->Internals->Mpi->Rank == 0)
  {
    this->Internals->ShutdownImGui();
    this->Internals->ShutdownGlfw();
  }
}

void Window::Run()
{
  auto gWindow = this->Internals->GlfwWindow;
  auto& io = ImGui::GetIO();

  ImVec4 clearColor(0.45f, 0.55f, 0.60f, 1.00f);
  while (!glfwWindowShouldClose(gWindow))
  {
    glfwPollEvents();

    ImGui_ImplOpenGL2_NewFrame();
    ImGui_ImplGlfw_NewFrame();
    ImGui::NewFrame();

    auto config = this->Internals->Config;
    auto scene = this->Internals->Scene;
    ImGui::Begin("Beams");
    if (ImGui::BeginCombo("Preset", this->Internals->CurrentPresetId.c_str()))
    {
      for (auto presetId : config->PresetIds)
      {
        const bool isSelected = (this->Internals->CurrentPresetId == presetId);
        if (ImGui::Selectable(presetId.c_str(), isSelected))
        {
          this->Internals->CurrentPresetId = presetId;
          this->Internals->SetSceneDirty(true);
          this->Internals->SetViewDirty(true);
        }
        if (isSelected)
        {
          ImGui::SetItemDefaultFocus();
        }
      }
      ImGui::EndCombo();
    }
    ImGui::Separator();
    if (ImGui::CollapsingHeader("Camera"))
    {
      ImGui::LabelText("Position", "%s", Fmt::FormatFloat(scene->CameraPosition.Value).c_str());
      ImGui::LabelText("LookAt", "%s", Fmt::FormatFloat(scene->CameraLookAt.Value).c_str());
      ImGui::LabelText("Up", "%s", Fmt::FormatFloat(scene->CameraUp.Value).c_str());

      if (ImGui::SliderFloat("Camera FOV", &(scene->CameraFov.Value), 30.0f, 120.0f))
      {
        // Only allow fovs that are multiples of 5
        scene->CameraFov.Value = vtkm::Round(scene->CameraFov.Value / 5.0f) * 5.0f;
        this->Internals->SetViewDirty(true);
      }
      if (ImGui::Button("Reset Camera"))
      {
        scene->CameraPosition = vtkm::Vec3f_32{ 0.0f, 0.0f, 1.0f };
        scene->CameraLookAt = vtkm::Vec3f_32{ 0.0f, 0.0f, 0.0f };
        scene->CameraUp = vtkm::Vec3f_32{ 0.0f, 1.0f, 0.0f };
        scene->CameraFov = 60.0f;
        this->Internals->SetViewDirty(true);
      }
    }

    if (ImGui::CollapsingHeader("Light"))
    {
      if (ImGui::SliderFloat3("Light Position",
                              &(scene->LightPosition[0]),
                              -scene->ShowLightLimits,
                              scene->ShowLightLimits,
                              "%.3f"))
      {
        this->Internals->SetViewDirty(true);
      }
      if (ImGui::ColorEdit3("Light Color", &(scene->LightColor[0])))
      {
        this->Internals->SetViewDirty(true);
      }
      if (ImGui::SliderFloat("Light Intensity", &(scene->LightIntensity), 0.0f, 5.0f))
      {
        this->Internals->SetViewDirty(true);
      }
      if (ImGui::Checkbox("Show indicator", &(scene->ShowLight)))
      {
        this->Internals->SetViewDirty(true);
      }
      if (ImGui::Button("Reset Light"))
      {
        scene->LightPosition = vtkm::Vec3f(0.5010f, 1.0010f, 0.5010f);
        scene->LightColor = vtkm::Vec3f(1.0f, 1.0f, 1.0f);
        scene->LightIntensity = 1.0f;
        this->Internals->SetViewDirty(true);
      }
    }

    if (ImGui::CollapsingHeader("Rendering"))
    {
      auto rendererNames = beams::RenderOptions::GetRendererNames();
      auto rendererNamesMap = beams::RenderOptions::GetRendererNamesMap();
      static std::size_t rendererIndex = 0;
      for (std::size_t i = 0; i < rendererNames.size(); ++i)
      {
        if (beams::utils::String::CompareIgnoreCase(rendererNamesMap[scene->Renderer],
                                                    rendererNames[i]) == 0)
        {
          rendererIndex = i;
          break;
        }
      }
      const char* currentRenderer = rendererNames[rendererIndex].c_str();
      if (ImGui::BeginCombo("Renderer", currentRenderer))
      {
        for (std::size_t i = 0; i < rendererNames.size(); ++i)
        {
          const bool isSelected = (rendererIndex == i);
          if (ImGui::Selectable(rendererNames[i].c_str(), isSelected))
          {
            rendererIndex = i;
            for (auto& r : rendererNamesMap)
            {
              if (beams::utils::String::CompareIgnoreCase(r.second, rendererNames[i]) == 0)
              {
                scene->Renderer = r.first;
                break;
              }
            }
            this->Internals->SetViewDirty(true);
          }
          if (isSelected)
          {
            ImGui::SetItemDefaultFocus();
          }
        }
        ImGui::EndCombo();
      }

      if (scene->Renderer == RendererType::PhongVolume)
      {
        if (ImGui::Checkbox("Phong Diffuse", &(scene->UsePhongDiffuse)))
        {
          this->Internals->SetViewDirty(true);
        }
        if (ImGui::Checkbox("Phong Specular", &(scene->UsePhongSpecular)))
        {
          this->Internals->SetViewDirty(true);
        }
      }
    }

    if (ImGui::CollapsingHeader("Transfer Function"))
    {
      const char* colorTableNames[] = { "Default",
                                        "Cool to Warm",
                                        "Cool to Warm Extended",
                                        "Viridis",
                                        "Inferno",
                                        "Plasma",
                                        "Black-Body Radiation",
                                        "X Ray",
                                        "Green",
                                        "Black - Blue - White",
                                        "Blue to Orange",
                                        "Gray to Red",
                                        "Cold and Hot",
                                        "Blue - Green - Orange",
                                        "Yellow - Gray - Blue",
                                        "Rainbow Uniform",
                                        "Jet",
                                        "Rainbow Desaturated" };
      static int colorTableIndex = 0;
      for (int i = 0; i < IM_ARRAYSIZE(colorTableNames); ++i)
      {
        if (beams::utils::String::CompareIgnoreCase(scene->ColorTableName, colorTableNames[i]) == 0)
        {
          colorTableIndex = i;
          break;
        }
      }
      const char* currentColorTable = colorTableNames[colorTableIndex];
      if (ImGui::BeginCombo("Color table name", currentColorTable))
      {
        for (int i = 0; i < IM_ARRAYSIZE(colorTableNames); ++i)
        {
          const bool isSelected = (colorTableIndex == i);
          if (ImGui::Selectable(colorTableNames[i], isSelected))
          {
            colorTableIndex = i;
            scene->ColorTableName = colorTableNames[i];
            this->Internals->SetViewDirty(true);
          }
          if (isSelected)
          {
            ImGui::SetItemDefaultFocus();
          }
        }
        ImGui::EndCombo();
      }

      static bool showHistogram = true;
      ImGui::Checkbox("Show Histogram", &showHistogram);
      if (showHistogram)
      {
        std::vector<float> fieldHistogram;
        auto histogramPortal = scene->DataSetHistogram.ReadPortal();
        float histogramBinMax =
          *std::max_element(histogramPortal.GetIteratorBegin(), histogramPortal.GetIteratorEnd());
        for (vtkm::Id i = 0; i < histogramPortal.GetNumberOfValues(); ++i)
        {
          auto value = static_cast<float>(histogramPortal.Get(i));
          histogramBinMax = std::max(histogramBinMax, value);
          fieldHistogram.push_back(value / histogramBinMax);
        }
        ImVec2 canvasSize = ImGui::GetContentRegionAvail();
        canvasSize = ImVec2(canvasSize.x, 200);
        ImGui::PlotHistogram(scene->FieldName.c_str(),
                             fieldHistogram.data(),
                             (int)fieldHistogram.size(),
                             0,
                             nullptr,
                             0.f,
                             1.f,
                             canvasSize);
      }


      const int POINT_ALPHAS_MODE_MANUAL = 0;
      const int POINT_ALPHAS_MODE_GUI = 1;
      static int pointAlphasMode = POINT_ALPHAS_MODE_GUI;
      ImGui::RadioButton("Manual Point Alphas", &pointAlphasMode, POINT_ALPHAS_MODE_MANUAL);
      ImGui::SameLine();
      ImGui::RadioButton("GUI Point Alphas", &pointAlphasMode, POINT_ALPHAS_MODE_GUI);
      static std::vector<beams::PointAlpha> pointAlphas = scene->PointAlphas;
      static std::vector<beams::PointAlpha> manualPointAlphas = scene->PointAlphas;
      if (pointAlphasMode == POINT_ALPHAS_MODE_MANUAL)
      {
        for (std::size_t i = 0; i < manualPointAlphas.size(); ++i)
        {
          ImGui::PushID(i);
          ImGui::PushItemWidth(200);
          ImGui::InputFloat("Point", &(manualPointAlphas[i].Point), 0.f, 1.f, "%.7f");
          ImGui::SameLine();
          ImGui::InputFloat("Alpha", &(manualPointAlphas[i].Alpha), 0.f, 1.f, "%.7f");
          ImGui::PopItemWidth();
          ImGui::SameLine();
          if (ImGui::Button("+"))
          {
            if (i == manualPointAlphas.size() - 1)
            {
              manualPointAlphas.push_back(manualPointAlphas[i]);
            }
            else
            {
              vtkm::Float32 newPoint =
                (manualPointAlphas[i].Point + manualPointAlphas[i + 1].Point) / 2;
              vtkm::Float32 newAlpha =
                (manualPointAlphas[i].Alpha + manualPointAlphas[i + 1].Alpha) / 2;
              manualPointAlphas.insert(manualPointAlphas.begin() + i + 1, { newPoint, newAlpha });
            }
            --i;
          }
          ImGui::SameLine();
          if (ImGui::Button("X"))
          {
            manualPointAlphas.erase(manualPointAlphas.begin() + i);
            ++i;
          }
          ImGui::PopID();
        }
        bool manualAlphasChanged = !compareVec(manualPointAlphas, pointAlphas);
        if (!manualAlphasChanged)
        {
          ImGui::PushItemFlag(ImGuiItemFlags_Disabled, true);
          ImGui::PushStyleVar(ImGuiStyleVar_Alpha, ImGui::GetStyle().Alpha * 0.5f);
        }
        if (ImGui::Button("Apply") || scene->PointAlphas.size() != manualPointAlphas.size())
        {
          scene->PointAlphas = manualPointAlphas;
        }
        ImGui::SameLine();
        if (!manualAlphasChanged)
        {
          ImGui::PopItemFlag();
          ImGui::PopStyleVar();
        }
      }
      else if (pointAlphasMode == POINT_ALPHAS_MODE_GUI)
      {
        static std::size_t selectedPointAlpha = -1;
        ImVec2 canvasSize = ImGui::GetContentRegionAvail();
        canvasSize.y -= 20;
        const float point_radius = 10.f;
        ImVec2 canvasPos = ImGui::GetCursorScreenPos();
        ImDrawList* drawList = ImGui::GetWindowDrawList();
        ImVec2 clipRectMax = canvasPos + canvasSize;
        drawList->PushClipRect(canvasPos, clipRectMax);

        const ImVec2 viewScale(canvasSize.x, -canvasSize.y);
        const ImVec2 viewOffset(canvasPos.x, canvasPos.y + canvasSize.y);

        drawList->AddRect(canvasPos, clipRectMax, ImColor(180, 180, 180, 255));
        ImGui::InvisibleButton("tfn_canvas", canvasSize);
        static bool clickedOnItem = false;
        if (!io.MouseDown[0] && !io.MouseDown[1])
        {
          clickedOnItem = false;
        }
        if (ImGui::IsItemHovered() && (io.MouseDown[0] || io.MouseDown[1]))
        {
          clickedOnItem = true;
        }
        ImVec2 bbmin = ImGui::GetItemRectMin();
        ImVec2 bbmax = ImGui::GetItemRectMax();
        ImVec2 clippedMousePos = ImVec2(std::min(std::max(io.MousePos.x, bbmin.x), bbmax.x),
                                        std::min(std::max(io.MousePos.y, bbmin.y), bbmax.y));
        if (clickedOnItem)
        {
          ImVec2 mousePos = (clippedMousePos - viewOffset) / viewScale;
          mousePos.x = vtkm::Clamp(mousePos.x, 0.f, 1.f);
          mousePos.y = vtkm::Clamp(mousePos.y, 0.f, 1.f);

          if (io.MouseDown[0])
          {
            if (selectedPointAlpha != (std::size_t)-1)
            {
              scene->PointAlphas[selectedPointAlpha] =
                beams::PointAlpha{ .Point = mousePos.x, .Alpha = mousePos.y };

              // Keep the first and last control points at the edges
              if (selectedPointAlpha == 0)
              {
                scene->PointAlphas[selectedPointAlpha].Point = 0.f;
              }
              else if (selectedPointAlpha == scene->PointAlphas.size() - 1)
              {
                scene->PointAlphas[selectedPointAlpha].Point = 1.f;
              }
            }
            else
            {
              auto fnd = std::find_if(scene->PointAlphas.begin(),
                                      scene->PointAlphas.end(),
                                      [&](const beams::PointAlpha& p)
                                      {
                                        const ImVec2 ptPos =
                                          ImVec2(p.Point, p.Alpha) * viewScale + viewOffset;
                                        float dist = Length(ptPos - clippedMousePos);
                                        return dist <= point_radius;
                                      });
              // No nearby point, we're adding a new one
              if (fnd == scene->PointAlphas.end())
              {
                scene->PointAlphas.push_back(
                  beams::PointAlpha{ .Point = mousePos.x, .Alpha = mousePos.y });
              }
            }

            // Keep alpha control points ordered by x coordinate, update
            // selected point index to match
            std::sort(scene->PointAlphas.begin(),
                      scene->PointAlphas.end(),
                      [](const beams::PointAlpha& a, const beams::PointAlpha& b)
                      { return a.Point < b.Point; });
            if (selectedPointAlpha != 0 && selectedPointAlpha != scene->PointAlphas.size() - 1)
            {
              auto fnd = std::find_if(scene->PointAlphas.begin(),
                                      scene->PointAlphas.end(),
                                      [&](const beams::PointAlpha& p)
                                      {
                                        const ImVec2 ptPos =
                                          ImVec2(p.Point, p.Alpha) * viewScale + viewOffset;
                                        float dist = Length(ptPos - clippedMousePos);
                                        return dist <= point_radius;
                                      });
              selectedPointAlpha = std::distance(scene->PointAlphas.begin(), fnd);
            }
            this->Internals->SetViewDirty(true);
          }
          else if (ImGui::IsMouseClicked(1))
          {
            selectedPointAlpha = std::size_t(-1);
            // Find and remove the point
            auto fnd = std::find_if(scene->PointAlphas.begin(),
                                    scene->PointAlphas.end(),
                                    [&](const beams::PointAlpha& p)
                                    {
                                      const ImVec2 ptPos =
                                        ImVec2(p.Point, p.Alpha) * viewScale + viewOffset;
                                      float dist = Length(ptPos - clippedMousePos);
                                      return dist <= point_radius;
                                    });
            // We also want to prevent erasing the first and last points
            if (fnd != scene->PointAlphas.end() && fnd != scene->PointAlphas.begin() &&
                fnd != scene->PointAlphas.end() - 1)
            {
              scene->PointAlphas.erase(fnd);
            }
            this->Internals->SetViewDirty(true);
          }
          else
          {
            selectedPointAlpha = -1;
          }
        }
        else
        {
          selectedPointAlpha = -1;
        }

        // Draw the alpha control points, and build the points for the polyline
        // which connects them
        std::vector<ImVec2> polylinePts;
        for (const auto& pt : scene->PointAlphas)
        {
          const ImVec2 ptPos = ImVec2(pt.Point, pt.Alpha) * viewScale + viewOffset;
          polylinePts.push_back(ptPos);
          drawList->AddCircleFilled(ptPos, point_radius, 0xFFFFFFFF);
        }
        drawList->AddPolyline(polylinePts.data(), (int)polylinePts.size(), 0xFFFFFFFF, false, 2.f);
        drawList->PopClipRect();
      }

      struct TFInstance
      {
        std::vector<beams::PointAlpha> prevPointAlphas;
        std::vector<beams::PointAlpha> nextPointAlphas;
      };
      static std::vector<TFInstance> undoStack;
      static int undoStackTop = -1;

      if (pointAlphasMode == POINT_ALPHAS_MODE_GUI)
      {
        ImGui::PushItemFlag(ImGuiItemFlags_Disabled, true);
        ImGui::PushStyleVar(ImGuiStyleVar_Alpha, ImGui::GetStyle().Alpha * 0.5f);
        ImGui::Button("Apply");
        ImGui::PopItemFlag();
        ImGui::PopStyleVar();
        ImGui::SameLine();
      }
      bool alphasChanged = !compareVec(pointAlphas, scene->PointAlphas);
      if (alphasChanged)
      {
        undoStack.resize(undoStackTop + 2);
        undoStack[++undoStackTop].prevPointAlphas = pointAlphas;
        undoStack[undoStackTop].nextPointAlphas = scene->PointAlphas;
        pointAlphas = scene->PointAlphas;
        manualPointAlphas = scene->PointAlphas;
        this->Internals->SetViewDirty(true);
      }
      bool undoAvailable = undoStackTop >= 0;
      if (!undoAvailable)
      {
        ImGui::PushItemFlag(ImGuiItemFlags_Disabled, true);
        ImGui::PushStyleVar(ImGuiStyleVar_Alpha, ImGui::GetStyle().Alpha * 0.5f);
      }
      if (ImGui::Button("Undo"))
      {
        if (undoAvailable)
        {
          auto& tfInstance = undoStack[undoStackTop--];
          scene->PointAlphas = tfInstance.prevPointAlphas;
          pointAlphas = scene->PointAlphas;
          manualPointAlphas = scene->PointAlphas;
          this->Internals->SetViewDirty(true);
        }
      }
      if (!undoAvailable)
      {
        ImGui::PopItemFlag();
        ImGui::PopStyleVar();
      }
      ImGui::SameLine();
      bool redoAvailable = undoStackTop < (int)undoStack.size() - 1;
      if (!redoAvailable)
      {
        ImGui::PushItemFlag(ImGuiItemFlags_Disabled, true);
        ImGui::PushStyleVar(ImGuiStyleVar_Alpha, ImGui::GetStyle().Alpha * 0.5f);
      }
      if (ImGui::Button("Redo"))
      {
        if (redoAvailable)
        {
          auto& tfInstance = undoStack[++undoStackTop];
          scene->PointAlphas = tfInstance.nextPointAlphas;
          pointAlphas = scene->PointAlphas;
          manualPointAlphas = scene->PointAlphas;
          this->Internals->SetViewDirty(true);
        }
      }
      if (!redoAvailable)
      {
        ImGui::PopItemFlag();
        ImGui::PopStyleVar();
      }
    }

    if (ImGui::CollapsingHeader("Tonemap"))
    {
      if (ImGui::Checkbox("Clamp", &scene->UseClamp))
      {
        this->Internals->SetViewDirty(true);
      }
      if (ImGui::Checkbox("Reinhard", &scene->UseReinhard))
      {
        this->Internals->SetViewDirty(true);
      }
    }

    if (ImGui::Button("Save"))
    {
      int index = 0;
      bool isAvailable = false;
      std::string baseName;
      while (!isAvailable)
      {
        std::stringstream ss;
        ss << "../data/gui_presets/" << scene->Id << "_" << std::setfill('0') << std::setw(3)
           << index;
        baseName = ss.str();
        isAvailable = !beams::io::File::FileExists(baseName + "_direct.png");
        index++;
      }

      auto previousRenderer = scene->Renderer;
      scene->Renderer = RendererType::DirectVolume;
      std::string imageName = baseName + "_direct.png";
      vtkm::io::CreateDirectoriesFromFilePath(imageName);
      this->Repaint(true);
      scene->Canvas->SaveAs(imageName);
      Fmt::Println0("Saved canvas to {}", imageName);

      scene->Renderer = RendererType::ShadowVolume;
      imageName = baseName + "_shadow.png";
      this->Repaint(true);
      scene->Canvas->SaveAs(imageName);
      Fmt::Println0("Saved canvas to {}", imageName);

      scene->Renderer = RendererType::PhongVolume;
      imageName = baseName + "_phong.png";
      this->Repaint(true);
      scene->Canvas->SaveAs(imageName);
      Fmt::Println0("Saved canvas to {}", imageName);

      std::string sceneName = baseName + ".txt";
      vtkm::io::CreateDirectoriesFromFilePath(sceneName);
      std::ofstream sceneFile(sceneName);
      sceneFile << "**** Camera ****"
                << "\n";
      sceneFile << fmt::format("Camera Position: {}", Fmt::FormatFloat(scene->Camera.GetPosition()))
                << "\n";
      sceneFile << fmt::format("Camera LookAt: {}", Fmt::FormatFloat(scene->Camera.GetLookAt()))
                << "\n";
      sceneFile << fmt::format("Camera Up: {}", Fmt::FormatFloat(scene->Camera.GetViewUp()))
                << "\n";

      sceneFile << "**** Light ****"
                << "\n";
      sceneFile << fmt::format("Light Position: {}", Fmt::FormatFloat(scene->LightPosition))
                << "\n";
      sceneFile << fmt::format("Light Color: {}", Fmt::FormatFloat(scene->LightColor)) << "\n";
      sceneFile << fmt::format("Light Intensity: {}", Fmt::FormatFloat(scene->LightIntensity))
                << "\n";

      sceneFile << fmt::format("**** Transfer Function ****") << "\n";
      sceneFile << fmt::format("Color Table Name: {}", scene->ColorTableName) << "\n";
      for (const auto& pt : scene->PointAlphas)
      {
        sceneFile << fmt::format("Point: {}, Alpha: {}",
                                 Fmt::FormatFloat(pt.Point, 8),
                                 Fmt::FormatFloat(pt.Alpha, 8))
                  << "\n";
      }
      sceneFile.close();
      Fmt::Println0("Saved scene to {}", sceneName);

      scene->Renderer = previousRenderer;
      this->Internals->SetViewDirty(true);
    }

    ImGui::Separator();
    if (ImGui::Button("Reload Config"))
    {
      this->Internals->OnReloadConfig();
      this->Internals->SetSceneDirty(true);
      this->Internals->SetViewDirty(true);
    }
    ImGui::SameLine();

    ImGui::Text(
      "Application average %.3f ms/frame (%.1f FPS)", 1000.0f / io.Framerate, io.Framerate);
    ImGui::End();

    // Rendering
    ImGui::Render();
    int display_w, display_h;
    glfwGetFramebufferSize(gWindow, &display_w, &display_h);
    glViewport(0, 0, display_w, display_h);
    glClearColor(clearColor.x * clearColor.w,
                 clearColor.y * clearColor.w,
                 clearColor.z * clearColor.w,
                 clearColor.w);
    glClear(GL_COLOR_BUFFER_BIT);

    this->Repaint();

    auto& colorBuffer = this->Internals->Scene->Canvas->GetColorBuffer();
    const void* colorData =
      vtkm::cont::ArrayHandleBasic<vtkm::Vec4f_32>(colorBuffer).GetReadPointer();
    glDrawPixels((GLsizei)display_w, (GLsizei)display_h, GL_RGBA, GL_FLOAT, colorData);

    ImGui_ImplOpenGL2_RenderDrawData(ImGui::GetDrawData());
    glfwMakeContextCurrent(gWindow);
    glfwSwapBuffers(gWindow);
  }
}

void Window::Repaint(bool force)
{
  bool shouldRepaint = force || this->Internals->IsViewDirty() || this->Internals->IsSceneDirty();
  if (!shouldRepaint)
  {
    return;
  }

  if (this->Internals->IsSceneDirty())
  {
    this->Internals->LoadScene();
    this->Internals->Scene->PrintSummary();
    this->Internals->SetSceneDirty(false);
  }

  auto scene = this->Internals->Scene;
  scene->Ready();
  this->Internals->SetViewDirty(false);

  scene->Mapper->RenderCells(scene->DataSet.GetCellSet(),
                             scene->DataSet.GetCoordinateSystem(),
                             scene->DataSet.GetField(scene->FieldName),
                             scene->ColorTable,
                             scene->Camera,
                             scene->RangeMap->GetGlobalRange());

  if (scene->ShowLight)
  {
    beams::source::IcoSphere lightSource;
    vtkm::Float32 lightRadius;
    auto bounds = scene->BoundsMap->GlobalBounds;
    lightRadius =
      std::max(bounds.X.Length(), std::max(bounds.Y.Length(), bounds.Z.Length())) * 2e-2f;
    lightSource.AddSphere(scene->GetTrueLightPosition(), lightRadius, 1.0f);
    auto lightDataSet = lightSource.Execute();
    vtkm::cont::ColorTable colorTable(
      vtkm::Range(0.0f, 5.0f), scene->LightColor, scene->LightColor);
    vtkm::rendering::MapperRayTracer lightMapper;
    lightMapper.SetCanvas(scene->Canvas);
    lightMapper.SetActiveColorTable(colorTable);
    lightMapper.SetCompositeBackground(true);
    lightMapper.RenderCells(lightDataSet.GetCellSet(),
                            lightDataSet.GetCoordinateSystem(),
                            lightDataSet.GetField("ico_spheres"),
                            colorTable,
                            scene->Camera,
                            scene->RangeMap->GetGlobalRange());
  }
  scene->Canvas->RefreshColorBuffer();
}

void Window::OnCursorState(bool leftButtonDown, double xPos, double yPos)
{
  if (leftButtonDown)
  {
    return;
  }
  int displayWidth, displayHeight;
  glfwGetFramebufferSize(this->Internals->GlfwWindow, &displayWidth, &displayHeight);
  vtkm::Float64 startX = (2.0f * this->Internals->StartMouseX) / displayWidth - 1.0f;
  vtkm::Float64 startY = -((2.0f * this->Internals->StartMouseY) / displayHeight - 1.0f);
  vtkm::Float64 endX = (2.0f * xPos) / displayWidth - 1.0f;
  vtkm::Float64 endY = -((2.0f * yPos) / displayHeight - 1.0f);
  this->Internals->UpdateTrackball(startX, startY, endX, endY);
}

void Window::OnArrowKeys(bool leftArrowDown,
                         bool rightArrowDown,
                         bool upArrowDown,
                         bool downArrowDown)
{
  const vtkm::Float32 xStep = 0.01f;
  const vtkm::Float32 yStep = 0.01f;

  vtkm::Float32 startX = 0.0f;
  vtkm::Float32 startY = 0.0f;
  vtkm::Float32 endX = 0.0f;
  vtkm::Float32 endY = 0.0f;
  if (leftArrowDown)
  {
    endX += xStep;
  }
  else if (rightArrowDown)
  {
    endX -= yStep;
  }
  if (upArrowDown)
  {
    endY += yStep;
  }
  else if (downArrowDown)
  {
    endY -= yStep;
  }
  this->Internals->UpdateTrackball(startX, startY, endX, endY);
}

void Window::SetConfig(std::shared_ptr<beams::Config> config)
{
  this->Internals->Config = config;
}
} // namespace gui
} // namespace beams