#ifndef beams_config_h
#define beams_config_h

#include "Result.h"
#include <utils/Json.h>

#include <vtkm/Types.h>

#include <string>
#include <unordered_map>

namespace beams
{
template <typename T>
struct Optional
{
  bool HasValue = false;
  T Value;

  Optional() = default;

  Optional(const T& value)
    : HasValue(true)
    , Value(value)
  {
  }

  explicit operator bool() const { return this->HasValue; }

  explicit operator T() const { return this->Value; }

  static inline Optional<T> Some(const T& value) { return Optional<T>(value); }

  static inline Optional<T> None() { return Optional<T>(); }
};

struct DataSetOptions
{
  beams::Result Deserialize(const PJObj& optionsObj);

  std::string Id;
  std::string Factory;
  std::unordered_map<std::string, std::string> Params;
};

struct OpacityMapOptions
{
  beams::Result Deserialize(const PJObj& optionsObj);

  vtkm::Id3 CalculateSize(vtkm::Id3 dataSetSize) const
  {
    bool shouldCalculateSize = this->Size[0] == -1;
    if (shouldCalculateSize)
    {
      return vtkm::Id3{ static_cast<vtkm::Id>(dataSetSize[0] * this->SizeRatio),
                        static_cast<vtkm::Id>(dataSetSize[1] * this->SizeRatio),
                        static_cast<vtkm::Id>(dataSetSize[2] * this->SizeRatio) };
    }
    else
    {
      return this->Size;
    }
  }

  vtkm::Id3 Size;
  vtkm::Float32 SizeRatio;
  vtkm::IdComponent NumSteps;
};

struct CameraOptions
{
  beams::Result Deserialize(const PJObj& optionsObj);

  std::string Id;
  beams::Optional<vtkm::Vec3f_32> Position;
  beams::Optional<vtkm::Vec3f_32> LookAt;
  beams::Optional<vtkm::Vec3f_32> Up;
  beams::Optional<vtkm::Float32> Fov;
};

struct LightOption
{
  beams::Result Deserialize(const PJObj& optionsObj);

  std::string Type;
  vtkm::Vec3f_32 Position;
  vtkm::Vec3f_32 Color;
  vtkm::Float32 Intensity;
};

struct LightCollection
{
  beams::Result Deserialize(const PJObj& optionsObj);

  std::string Id;
  std::vector<LightOption> Lights;
};

enum class RendererType
{
  DirectVolume,
  ShadowVolume,
  PhongVolume
};

struct RenderOptions
{
  beams::Result Deserialize(const PJObj& optionsObj);

  static std::map<RendererType, std::string> GetRendererNamesMap();

  static std::vector<std::string> GetRendererNames();

  std::string CanvasId;
  vtkm::Id NumSamples;
  vtkm::Id NumShadowSamples = 0;
  vtkm::Float32 SampleDistance;
  vtkm::Id S0;
  RendererType Renderer;
  bool UsePhongDiffuse = true;
  bool UsePhongSpecular = true;
  bool UseClamp = true;
  bool UseReinhard = false;
};

struct Preset
{
  beams::Result Deserialize(const PJObj& presetObj);

  std::string Id;
  std::string DataSetId;
  beams::OpacityMapOptions OpacityMapOptions;
  std::string CameraId;
  std::string LightCollectionId;
  std::string ColorTableId;
  beams::RenderOptions RenderOptions;
}; // struct Preset

struct CanvasOptions
{
  beams::Result Deserialize(const PJObj& canvasObj);

  std::string Id;
  vtkm::Id Width;
  vtkm::Id Height;
}; // struct CanvasOptions

struct PointAlpha
{
  bool operator==(const PointAlpha& other) const
  {
    return this->Point == other.Point && this->Alpha == other.Alpha;
  }

  bool operator!=(const PointAlpha& other) const { return !(*this == other); }

  beams::Result Deserialize(const PJArr& pointAlphaObj);

  vtkm::Float32 Point;
  vtkm::Float32 Alpha;
};

struct ColorTableOptions
{
  beams::Result Deserialize(const PJObj& colorTableObj);

  std::string Id;
  std::string Name;
  std::vector<PointAlpha> PointAlphas;
}; // struct ColorTableOptions

struct Config
{
  Config() = default;

  beams::Result LoadFromFile(const std::string& filePath);

  std::string FilePath;
  std::string DefaultPresetId;
  std::string TimingsFileName;
  std::string SummerMoviePreset;
  std::string SummerMovieFastPreset;
  std::vector<std::string> SummerMovieStopPoints;
  int NumIters = 10;
  std::unordered_map<std::string, Preset> Presets;
  std::vector<std::string> PresetIds;
  std::unordered_map<std::string, DataSetOptions> DataSets;
  std::unordered_map<std::string, CanvasOptions> Canvases;
  std::unordered_map<std::string, CameraOptions> Cameras;
  std::unordered_map<std::string, LightCollection> LightCollections;
  std::unordered_map<std::string, ColorTableOptions> ColorTables;
}; // struct Config

} // namespace beams


#endif // beams_config_h