#include "Config.h"
#include "utils/String.h"

#include "fmt/core.h"

#include <fstream>
#include <sstream>

namespace beams
{
template <typename Type>
beams::Result DeserializeToType(const PJObj& obj, const std::string& attrName, Type& expected)
{
  if (obj.find(attrName) == obj.end())
  {
    return Result::Failed(fmt::format("No attribute {}", attrName));
  }
  const PJVal& val = obj.at(attrName);
  if (!val.is<PJObj>())
  {
    return Result::Failed(fmt::format("Invalid {} attribute structure", attrName));
  }
  const PJObj& expectedObj = val.get<PJObj>();
  auto result = expected.Deserialize(expectedObj);
  if (!result.Success)
  {
    return Result::Failed(fmt::format("{}", result.Err));
  }

  return Result::Succeeded();
}

template <typename NativeType, typename JsonType = NativeType>
beams::Result DeserializeToNativeType(const PJObj& obj,
                                      const std::string& attrName,
                                      NativeType& nativeVal)
{
  if (obj.find(attrName) == obj.end())
  {
    return Result::Failed(fmt::format("No attribute {}", attrName));
  }
  const PJVal& val = obj.at(attrName);
  if (!val.is<JsonType>())
  {
    return Result::Failed(fmt::format("Unsupported value type for {} attribute", attrName));
  }
  JsonType jsonVal = val.get<JsonType>();
  nativeVal = static_cast<NativeType>(jsonVal);
  return Result::Succeeded();
}

template <typename NativeType, typename JsonType = NativeType>
beams::Result DeserializeToOptionalNativeType(const PJObj& obj,
                                              const std::string& attrName,
                                              beams::Optional<NativeType>& nativeVal)
{
  if (obj.find(attrName) == obj.end())
  {
    nativeVal = beams::Optional<NativeType>::None();
    return Result::Succeeded();
  }
  const PJVal& val = obj.at(attrName);
  if (!val.is<JsonType>())
  {
    return Result::Failed(fmt::format("Unsupported value type for {} attribute", attrName));
  }
  JsonType jsonVal = val.get<JsonType>();
  nativeVal = static_cast<NativeType>(jsonVal);
  return Result::Succeeded();
}

template <typename MapType>
beams::Result DeserializeToMap(const PJObj& obj, const std::string& attrName, MapType& map)
{
  if (obj.find(attrName) == obj.end())
  {
    return Result::Failed(fmt::format("No attribute {}", attrName));
  }
  const PJVal& val = obj.at(attrName);
  if (!val.is<PJObj>())
  {
    return Result::Failed(fmt::format("Unsupported value type for {} attribute", attrName));
  }
  for (auto const& entry : val.get<PJObj>())
  {
    const PJVal& tmpVal = entry.second;
    if (!tmpVal.is<std::string>())
      continue;
    map[entry.first] = entry.second.get<std::string>();
  }

  return Result::Succeeded();
}

template <typename Type, typename JType = PJObj>
beams::Result DeserializeToVector(const PJObj& obj,
                                  const std::string& attrName,
                                  std::vector<Type>& expected)
{
  if (obj.find(attrName) == obj.end())
  {
    return Result::Failed(fmt::format("No attribute {}", attrName));
  }
  const PJVal& val = obj.at(attrName);
  if (!val.is<PJArr>())
  {
    return Result::Failed(fmt::format("Invalid {} attribute structure", attrName));
  }
  const PJArr& valArr = val.get<PJArr>();
  for (PJArr::const_iterator i = valArr.cbegin(); i != valArr.cend(); ++i)
  {
    const PJVal& tmpVal = *i;
    if (!tmpVal.is<JType>())
    {
      return Result::Failed(fmt::format("Invalid structure inside {}", attrName));
    }
    const JType& tmpObj = tmpVal.get<JType>();
    Type e;
    auto result = e.Deserialize(tmpObj);
    if (!result.Success)
    {
      return Result::Failed(fmt::format("{}", result.Err));
    }
    expected.push_back(e);
  }
  return Result::Succeeded();
}

template <typename NativeType, typename JsonType = NativeType>
beams::Result DeserializeToVectorNative(const PJObj& obj,
                                        const std::string& attrName,
                                        std::vector<NativeType>& expected)
{
  if (obj.find(attrName) == obj.end())
  {
    return Result::Failed(fmt::format("No attribute {}", attrName));
  }
  const PJVal& val = obj.at(attrName);
  if (!val.is<PJArr>())
  {
    return Result::Failed(fmt::format("Invalid {} attribute structure", attrName));
  }
  const PJArr& valArr = val.get<PJArr>();
  for (PJArr::const_iterator i = valArr.cbegin(); i != valArr.cend(); ++i)
  {
    const PJVal& tmpVal = *i;
    if (!tmpVal.is<JsonType>())
    {
      return Result::Failed(fmt::format("Invalid structure inside {}", attrName));
    }
    expected.push_back(static_cast<NativeType>(tmpVal.get<JsonType>()));
  }
  return Result::Succeeded();
}

beams::Result DataSetOptions::Deserialize(const PJObj& optionsObj)
{
  CHECK_RESULT(DeserializeToNativeType(optionsObj, "factory", this->Factory),
               "Error reading dataSetOptions");
  CHECK_RESULT(DeserializeToMap(optionsObj, "params", this->Params),
               "Error reading dataSetOptions");
  return Result::Succeeded();
}

beams::Result OpacityMapOptions::Deserialize(const PJObj& optionsObj)
{
  const auto DeserializeToIdComponent = DeserializeToNativeType<vtkm::IdComponent, int64_t>;
  const auto DeserializeToFloat32 = DeserializeToNativeType<vtkm::Float32, double>;
  const auto DeserializeToId3 = DeserializeToVectorNative<vtkm::Id, double>;

  this->Size = { -1, -1, -1 };
  this->SizeRatio = 0.0f;
  if (optionsObj.find("size") != optionsObj.end())
  {
    std::vector<vtkm::Id> tmpSize;
    CHECK_RESULT(DeserializeToId3(optionsObj, "size", tmpSize), "Error reading opacityMapOptions");
    this->Size = { tmpSize[0], tmpSize[1], tmpSize[2] };
  }
  else if (optionsObj.find("sizeRatio") != optionsObj.end())
  {
    CHECK_RESULT(DeserializeToFloat32(optionsObj, "sizeRatio", this->SizeRatio),
                 "Error reading opacityMapOptions");
  }
  else
  {
    return Result::Failed("Error reading opacityMapOptions: No size or sizeRatio");
  }
  CHECK_RESULT(DeserializeToIdComponent(optionsObj, "numSteps", this->NumSteps),
               "Error reading opacityMapOptions");
  return Result::Succeeded();
}

beams::Result CameraOptions::Deserialize(const PJObj& optionsObj)
{
  const auto DeserializeToVecFloat32 = DeserializeToVectorNative<vtkm::Float32, double>;
  const auto DeserializeToFloat32 = DeserializeToOptionalNativeType<vtkm::Float32, double>;

  this->Position = beams::Optional<vtkm::Vec3f_32>::None();
  if (optionsObj.find("position") != optionsObj.end())
  {
    std::vector<vtkm::Float32> tmpPos;
    CHECK_RESULT(DeserializeToVecFloat32(optionsObj, "position", tmpPos),
                 "Error reading cameraOptions");
    this->Position = vtkm::Vec3f_32{ tmpPos[0], tmpPos[1], tmpPos[2] };
  }

  this->LookAt = beams::Optional<vtkm::Vec3f_32>::None();
  if (optionsObj.find("lookAt") != optionsObj.end())
  {
    std::vector<vtkm::Float32> tmpLookAt;
    CHECK_RESULT(DeserializeToVecFloat32(optionsObj, "lookAt", tmpLookAt),
                 "Error reading cameraOptions");
    this->LookAt = vtkm::Vec3f_32{ tmpLookAt[0], tmpLookAt[1], tmpLookAt[2] };
  }

  this->Up = beams::Optional<vtkm::Vec3f_32>::None();
  if (optionsObj.find("up") != optionsObj.end())
  {
    std::vector<vtkm::Float32> tmpUp;
    CHECK_RESULT(DeserializeToVecFloat32(optionsObj, "up", tmpUp), "Error reading cameraOptions");
    this->Up = vtkm::Vec3f_32{ tmpUp[0], tmpUp[1], tmpUp[2] };
  }

  this->Fov = beams::Optional<vtkm::Float32>::Some(60.0f);
  if (optionsObj.find("fov") != optionsObj.end())
  {
    CHECK_RESULT(DeserializeToFloat32(optionsObj, "fov", this->Fov), "Error reading cameraOptions");
  }

  return Result::Succeeded();
}

beams::Result LightCollection::Deserialize(const PJObj& optionsObj)
{
  CHECK_RESULT(DeserializeToVector(optionsObj, "lights", this->Lights), "Error reading lights");
  return Result::Succeeded();
}

beams::Result LightOption::Deserialize(const PJObj& optionsObj)
{
  const auto DeserializeToFloat32 = DeserializeToNativeType<vtkm::Float32, double>;
  const auto DeserializeToVecFloat32 = DeserializeToVectorNative<vtkm::Float32, double>;
  CHECK_RESULT(DeserializeToNativeType(optionsObj, "type", this->Type),
               "Error reading lightOption");
  std::vector<vtkm::Float32> tmpPos;
  std::vector<vtkm::Float32> tmpColor;
  CHECK_RESULT(DeserializeToVecFloat32(optionsObj, "position", tmpPos),
               "Error reading lightOption");
  this->Position = { tmpPos[0], tmpPos[1], tmpPos[2] };
  CHECK_RESULT(DeserializeToVecFloat32(optionsObj, "color", tmpColor), "Error reading lightOption");
  this->Color = { tmpColor[0], tmpColor[1], tmpColor[2] };
  CHECK_RESULT(DeserializeToFloat32(optionsObj, "intensity", this->Intensity),
               "Error reading lightOption");
  return Result::Succeeded();
}

std::vector<std::string> RenderOptions::GetRendererNames()
{
  return { "direct_volume", "shadow_volume", "phong_volume" };
}

std::map<RendererType, std::string> RenderOptions::GetRendererNamesMap()
{
  return { { RendererType::DirectVolume, "direct_volume" },
           { RendererType::ShadowVolume, "shadow_volume" },
           { RendererType::PhongVolume, "phong_volume" } };
}

beams::Result RenderOptions::Deserialize(const PJObj& optionsObj)
{
  const auto DeserializeToId = DeserializeToNativeType<vtkm::Id, int64_t>;
  const auto DeserializeToFloat32 = DeserializeToNativeType<vtkm::Float32, double>;

  CHECK_RESULT(DeserializeToNativeType(optionsObj, "canvas", this->CanvasId),
               "Error reading preset");

  if (optionsObj.find("numSamples") != optionsObj.end())
  {
    CHECK_RESULT(DeserializeToId(optionsObj, "numSamples", this->NumSamples),
                 "Error reading renderOptions");
  }
  else
  {
    this->NumSamples = -1;
  }
  if (optionsObj.find("sampleDistance") != optionsObj.end())
  {
    CHECK_RESULT(DeserializeToFloat32(optionsObj, "sampleDistance", this->SampleDistance),
                 "Error reading renderOptions");
  }
  else
  {
    this->SampleDistance = -1;
  }

  if (this->NumSamples < 0 && this->SampleDistance < 0)
  {
    return Result::Failed("Error reading renderOptions: No numSamples or sampleDistance");
  }

  if (optionsObj.find("numShadowSamples") != optionsObj.end())
  {
    CHECK_RESULT(DeserializeToId(optionsObj, "numShadowSamples", this->NumShadowSamples),
                 "Error reading renderOptions");
  }
  else
  {
    this->NumShadowSamples = 0;
  }

  const RendererType DEFAULT_RENDERER = RendererType::ShadowVolume;
  if (optionsObj.find("renderer") != optionsObj.end())
  {
    std::string rendererName;
    CHECK_RESULT(DeserializeToNativeType(optionsObj, "renderer", rendererName),
                 "Error reading renderOptions");
    auto rendererNames = GetRendererNames();
    if (std::find(rendererNames.begin(), rendererNames.end(), rendererName) == rendererNames.end())
    {
      return Result::Failed(
        fmt::format("Error reading renderOptions: Invalid renderer: {}", rendererName));
    }
    if (rendererName == "direct_volume")
    {
      this->Renderer = RendererType::DirectVolume;
    }
    else if (rendererName == "shadow_volume")
    {
      this->Renderer = RendererType::ShadowVolume;
    }
    else if (rendererName == "phong_volume")
    {
      this->Renderer = RendererType::PhongVolume;
    }
  }
  else
  {
    this->Renderer = DEFAULT_RENDERER;
  }

  if (optionsObj.find("tonemap") != optionsObj.end())
  {
    std::string tonemap;
    CHECK_RESULT(DeserializeToNativeType(optionsObj, "tonemap", tonemap),
                 "Error reading renderOptions");
    this->UseClamp = tonemap == "clamp";
    this->UseReinhard = tonemap == "reinhard";
  }
  else
  {
    this->UseClamp = true;
    this->UseReinhard = false;
  }

  CHECK_RESULT(DeserializeToId(optionsObj, "s0", this->S0), "Error reading renderOptions");

  if (this->Renderer == RendererType::PhongVolume)
  {
    if (optionsObj.find("phongDiffuse") != optionsObj.end())
    {
      CHECK_RESULT(DeserializeToNativeType(optionsObj, "phongDiffuse", this->UsePhongDiffuse),
                   "Error reading renderOptions");
    }
    else
    {
      this->UsePhongDiffuse = true;
    }
    if (optionsObj.find("phongSpecular") != optionsObj.end())
    {
      CHECK_RESULT(DeserializeToNativeType(optionsObj, "phongSpecular", this->UsePhongSpecular),
                   "Error reading renderOptions");
    }
    else
    {
      this->UsePhongSpecular = true;
    }
  }

  return Result::Succeeded();
}

beams::Result CanvasOptions::Deserialize(const PJObj& optionsObj)
{
  const auto DeserializeToId = DeserializeToNativeType<vtkm::Id, int64_t>;
  CHECK_RESULT(DeserializeToId(optionsObj, "width", this->Width), "Error reading canvas");
  CHECK_RESULT(DeserializeToId(optionsObj, "height", this->Height), "Error reading canvas");
  return Result::Succeeded();
}

beams::Result PointAlpha::Deserialize(const PJArr& pointAlphasArr)
{
  if (pointAlphasArr.size() == 2 && pointAlphasArr[0].is<double>() &&
      pointAlphasArr[1].is<double>())
  {
    this->Point = static_cast<vtkm::Float32>(pointAlphasArr[0].get<double>());
    this->Alpha = static_cast<vtkm::Float32>(pointAlphasArr[1].get<double>());
  }
  else
  {
    return Result::Failed(fmt::format("Error reading pointAlpha"));
  }
  return Result::Succeeded();
}

beams::Result ColorTableOptions::Deserialize(const PJObj& colorTableObj)
{
  const auto DeserializeToVectorPointAlpha = DeserializeToVector<PointAlpha, PJArr>;
  CHECK_RESULT(DeserializeToNativeType(colorTableObj, "name", this->Name),
               "Error reading colorTable");
  CHECK_RESULT(DeserializeToVectorPointAlpha(colorTableObj, "pointAlphas", this->PointAlphas),
               "Error reading colorTable");
  return Result::Succeeded();
}

beams::Result Preset::Deserialize(const PJObj& presetObj)
{
  CHECK_RESULT(DeserializeToNativeType(presetObj, "id", this->Id), "Error reading preset");
  CHECK_RESULT(DeserializeToNativeType(presetObj, "dataSetId", this->DataSetId),
               fmt::format("Error reading preset '{}'", this->Id));
  CHECK_RESULT(DeserializeToType(presetObj, "opacityMapOptions", this->OpacityMapOptions),
               fmt::format("Error reading preset '{}'", this->Id));
  CHECK_RESULT(DeserializeToNativeType(presetObj, "cameraId", this->CameraId),
               fmt::format("Error reading preset '{}'", this->Id));
  CHECK_RESULT(DeserializeToNativeType(presetObj, "lightCollectionId", this->LightCollectionId),
               fmt::format("Error reading preset '{}'", this->Id));
  CHECK_RESULT(DeserializeToNativeType(presetObj, "colorTableId", this->ColorTableId),
               fmt::format("Error reading preset '{}'", this->Id));
  CHECK_RESULT(DeserializeToType(presetObj, "renderOptions", this->RenderOptions),
               fmt::format("Error reading preset '{}'", this->Id));
  return Result::Succeeded();
}

beams::Result Config::LoadFromFile(const std::string& filePath)
{
  std::ifstream jsonFile(filePath);

  PJVal jsonVal;
  std::string err = picojson::parse(jsonVal, jsonFile);
  if (!err.empty())
  {
    return Result::Failed(fmt::format("Error parsing config: {}", err));
  }
  if (!jsonVal.is<PJObj>())
  {
    return Result::Failed(fmt::format("Error reading config: Invalid structure"));
  }

  this->FilePath = filePath;
  const PJObj& jsonRoot = jsonVal.get<PJObj>();
  {
    if (jsonRoot.find("dataSets") == jsonRoot.end())
    {
      return Result::Failed(fmt::format("Error reading config: No 'dataSets' key"));
    }
    const PJVal& dataSetsVal = jsonRoot.at("dataSets");
    const PJObj& dataSets = dataSetsVal.get<PJObj>();
    for (const auto& dataSetIter : dataSets)
    {
      std::string dataSetId = dataSetIter.first;
      const PJVal& dataSetVal = dataSetIter.second;
      if (!dataSetVal.is<PJObj>())
      {
        return Result::Failed(fmt::format("Error reading config: Invalid dataSet structure"));
      }

      const PJObj& dataSetObj = dataSetVal.get<PJObj>();
      DataSetOptions dataSet;
      dataSet.Id = dataSetId;
      auto result = dataSet.Deserialize(dataSetObj);
      if (!result.Success)
      {
        return Result::Failed(fmt::format("Error reading config: {}", result.Err));
      }
      this->DataSets[dataSet.Id] = dataSet;
    }
  }

  {
    if (jsonRoot.find("canvases") == jsonRoot.end())
    {
      return Result::Failed(fmt::format("Error reading config: No 'canvases' key"));
    }
    const PJVal& canvasesVal = jsonRoot.at("canvases");
    const PJObj& canvases = canvasesVal.get<PJObj>();
    for (const auto& canvasIter : canvases)
    {
      std::string canvasId = canvasIter.first;
      const PJVal& canvasVal = canvasIter.second;
      if (!canvasVal.is<PJObj>())
      {
        return Result::Failed(fmt::format("Error reading config: Invalid canvas structure"));
      }

      const PJObj& canvasObj = canvasVal.get<PJObj>();
      CanvasOptions canvas;
      canvas.Id = canvasId;
      auto result = canvas.Deserialize(canvasObj);
      if (!result.Success)
      {
        return Result::Failed(fmt::format("Error reading config: {}", result.Err));
      }
      this->Canvases[canvas.Id] = canvas;
    }
  }

  {
    if (jsonRoot.find("cameras") == jsonRoot.end())
    {
      return Result::Failed(fmt::format("Error reading config: No 'cameras' key"));
    }
    const PJVal& camerasVal = jsonRoot.at("cameras");
    const PJObj& cameras = camerasVal.get<PJObj>();
    for (const auto& cameraIter : cameras)
    {
      std::string cameraId = cameraIter.first;
      const PJVal& cameraVal = cameraIter.second;
      if (!cameraVal.is<PJObj>())
      {
        return Result::Failed(fmt::format("Error reading config: Invalid camera structure"));
      }

      const PJObj& cameraObj = cameraVal.get<PJObj>();
      CameraOptions camera;
      camera.Id = cameraId;
      auto result = camera.Deserialize(cameraObj);
      if (!result.Success)
      {
        return Result::Failed(fmt::format("Error reading config: {}", result.Err));
      }
      this->Cameras[camera.Id] = camera;
    }
  }

  {
    if (jsonRoot.find("lightCollections") == jsonRoot.end())
    {
      return Result::Failed(fmt::format("Error reading config: No 'lightCollections' key"));
    }
    const PJVal& lightCollectionsVal = jsonRoot.at("lightCollections");
    const PJObj& lightCollections = lightCollectionsVal.get<PJObj>();
    for (const auto& lightCollectionIter : lightCollections)
    {
      std::string lightCollectionId = lightCollectionIter.first;
      const PJVal& lightCollectionVal = lightCollectionIter.second;
      if (!lightCollectionVal.is<PJObj>())
      {
        return Result::Failed(
          fmt::format("Error reading config: Invalid lightCollection structure"));
      }

      const PJObj& lightCollectionObj = lightCollectionVal.get<PJObj>();
      LightCollection lights;
      lights.Id = lightCollectionId;
      auto result = lights.Deserialize(lightCollectionObj);
      if (!result.Success)
      {
        return Result::Failed(fmt::format("Error reading config: {}", result.Err));
      }
      this->LightCollections[lights.Id] = lights;
    }
  }

  {
    if (jsonRoot.find("colorTables") == jsonRoot.end())
    {
      return Result::Failed(fmt::format("Error reading config: No 'colorTables' key"));
    }
    const PJVal& colorTablesVal = jsonRoot.at("colorTables");
    const PJObj& colorTables = colorTablesVal.get<PJObj>();
    for (const auto& colorTableIter : colorTables)
    {
      std::string colorTableId = colorTableIter.first;
      const PJVal& colorTableVal = colorTableIter.second;
      if (!colorTableVal.is<PJObj>())
      {
        return Result::Failed(fmt::format("Error reading config: Invalid colorTable structure"));
      }

      const PJObj& colorTableObj = colorTableVal.get<PJObj>();
      ColorTableOptions colorTable;
      colorTable.Id = colorTableId;
      auto result = colorTable.Deserialize(colorTableObj);
      if (!result.Success)
      {
        return Result::Failed(fmt::format("Error reading config: {}", result.Err));
      }
      this->ColorTables[colorTable.Id] = colorTable;
    }
  }

  {
    if (jsonRoot.find("presets") == jsonRoot.end())
    {
      return Result::Failed(fmt::format("Error reading config: No 'presets' key"));
    }
    const PJVal& presetsVal = jsonRoot.at("presets");
    if (!presetsVal.is<PJArr>())
    {
      return Result::Failed(fmt::format("Error reading config: 'presets' should be array"));
    }
    const PJArr& presetsArr = presetsVal.get<PJArr>();
    if (presetsArr.size() == 0)
    {
      return Result::Failed(fmt::format("Error reading config: 'presets' is empty array"));
    }

    for (PJArr::const_iterator i = presetsArr.begin(); i != presetsArr.end(); ++i)
    {
      const PJVal& presetVal = *i;
      if (!presetVal.is<PJObj>())
      {
        return Result::Failed(fmt::format("Error reading config: Invalid preset structure"));
      }

      const PJObj& presetObj = presetVal.get<PJObj>();
      Preset preset;
      auto result = preset.Deserialize(presetObj);
      if (!result.Success)
      {
        return Result::Failed(fmt::format("Error reading config: {}", result.Err));
      }
      this->Presets[preset.Id] = preset;
      this->PresetIds.push_back(preset.Id);
    }
  }

  {
    if (jsonRoot.find("defaultPreset") == jsonRoot.end())
    {
      return Result::Failed("Error reading config: No 'defaultPreset' key");
    }
    const PJVal& defaultPresetVal = jsonRoot.at("defaultPreset");
    if (!defaultPresetVal.is<std::string>())
    {
      return Result::Failed("Error reading config: 'defaultPreset' must be string");
    }
    std::string defaultPreset = defaultPresetVal.get<std::string>();
    if (this->Presets.find(defaultPreset) == this->Presets.end())
    {
      return Result::Failed(
        fmt::format("Error reading config: No preset with id '{}'", defaultPreset));
    }
    this->DefaultPresetId = defaultPreset;
  }

  {
    if (jsonRoot.find("timingsFile") != jsonRoot.end())
    {
      const PJVal& timingsFileVal = jsonRoot.at("timingsFile");
      if (!timingsFileVal.is<std::string>())
      {
        return Result::Failed("Error reading config: 'timingsFile' must be string");
      }
      this->TimingsFileName = timingsFileVal.get<std::string>();
    }
    else
    {
      this->TimingsFileName = "";
    }
  }

  {
    if (jsonRoot.find("numIters") != jsonRoot.end())
    {
      const PJVal& numItersVal = jsonRoot.at("numIters");
      if (!numItersVal.is<double>())
      {
        return Result::Failed("Error reading config: 'numIters' must be an integer");
      }
      this->NumIters = numItersVal.get<double>();
    }
    else
    {
      this->NumIters = -1;
    }
  }

  {
    if (jsonRoot.find("summerMoviePreset") != jsonRoot.end())
    {
      const PJVal& summerMoviePresetVal = jsonRoot.at("summerMoviePreset");
      if (!summerMoviePresetVal.is<std::string>())
      {
        return Result::Failed("Error reading config: 'summerMoviePreset' must be string");
      }
      this->SummerMoviePreset = summerMoviePresetVal.get<std::string>();
    }
    else
    {
      this->SummerMoviePreset = "";
    }
  }

  {
    if (jsonRoot.find("summerMovieFastPreset") != jsonRoot.end())
    {
      const PJVal& summerMovieFastPresetVal = jsonRoot.at("summerMovieFastPreset");
      if (!summerMovieFastPresetVal.is<std::string>())
      {
        return Result::Failed("Error reading config: 'summerMovieFastPreset' must be string");
      }
      this->SummerMovieFastPreset = summerMovieFastPresetVal.get<std::string>();
    }
    else
    {
      this->SummerMovieFastPreset = "";
    }
  }

  {
    if (jsonRoot.find("summerMovieStopPoints") != jsonRoot.end())
    {
      const PJVal& summerMovieStopPointsVal = jsonRoot.at("summerMovieStopPoints");
      if (!summerMovieStopPointsVal.is<std::string>())
      {
        return Result::Failed("Error reading config: 'summerMovieStopPoints' must be string");
      }
      this->SummerMovieStopPoints =
        beams::utils::String::Split(summerMovieStopPointsVal.get<std::string>(), ",", true);
    }
    else
    {
      this->SummerMovieStopPoints = {};
    }
  }

  return Result::Succeeded();
}
} // namespace beams
