#include "Beams.h"
#include "Config.h"
#include "SceneFactory.h"
#include "profiling/Record.h"
#include "rendering/BoundsMap.h"
#include "rendering/Scene.h"
#include "rendering/SummerMovieScene.h"
#include "source/IcoSphere.h"
#include "utils/File.h"
#include "utils/Fmt.h"
#include "utils/String.h"
#include "utils/Timings.h"

#ifdef BEAMS_USE_GUI
#include "gui/Window.h"
#endif

#include <vtkm/cont/Initialize.h>
#include <vtkm/io/FileUtils.h>
#include <vtkm/rendering/CanvasRayTracer.h>
#include <vtkm/rendering/MapperRayTracer.h>
#include <vtkm/thirdparty/diy/diy.h>
#include <vtkm/thirdparty/diy/mpi-cast.h>

#include "fmt/core.h"

#include <fstream>
#include <iomanip>
#include <map>
#include <mpi.h>

std::string P1ShadowVolumeGenLabel = "Shadow Volume Gen";
std::string P2ShadowVolumeRenderLabel = "Shadow Volume MPI";
std::string P3ShadowVolumeUpdateLabel = "Shadow Volume Update";
std::string P4VolumeRenderLabel = "Volume Render";
std::string P5CompositeLabel = "Canvas Composite";

namespace beams
{
namespace internals
{
auto ToWikiAxis(const vtkm::Vec3f_32& u) -> vtkm::Vec3f_32
{
  return { u[2], u[0], u[1] };
}

auto FromWikiAxis(const vtkm::Vec3f_32& u) -> vtkm::Vec3f_32
{
  return { u[1], u[2], u[0] };
}
}
struct Beams::InternalsType
{
  InternalsType(beams::mpi::MpiEnv& mpi)
    : Mpi(mpi)
  {
  }

  Result ParseArgs(int argc, char** argv)
  {
    // argv[0] is the name of the binary, so skipping
    for (int i = 1; i < argc; ++i)
    {
      auto arg = std::string(argv[i]);
      auto seperatorPos = arg.find("=");
      if (seperatorPos == std::string::npos)
      {
        continue;
      }

      auto argKey = arg.substr(0, seperatorPos);
      auto argVal = arg.substr(seperatorPos + 1);
      if (argKey == "--config-file")
      {
        this->ConfigFilePath = argVal;
      }
      else if (argKey == "--preset")
      {
        this->StartPresetId = argVal;
      }
      else if (argKey == "--interactive-mode")
      {
        if (beams::utils::String::CompareIgnoreCase(argVal, "headless") == 0)
        {
          this->InteractiveMode = InteractiveMode::Headless;
        }
        else if (beams::utils::String::CompareIgnoreCase(argVal, "gui") == 0)
        {
          this->InteractiveMode = InteractiveMode::Gui;
        }
        else if (beams::utils::String::CompareIgnoreCase(argVal, "summer_movie") == 0)
        {
          this->InteractiveMode = InteractiveMode::SummerMovie;
        }
        else if (beams::utils::String::CompareIgnoreCase(argVal, "summer_movie_fast") == 0)
        {
          this->InteractiveMode = InteractiveMode::SummerMovieFast;
        }
        else if (beams::utils::String::CompareIgnoreCase(argVal, "movie_camera") == 0)
        {
          this->InteractiveMode = InteractiveMode::MovieCamera;
        }
        else if (beams::utils::String::CompareIgnoreCase(argVal, "movie_light_theta") == 0)
        {
          this->InteractiveMode = InteractiveMode::MovieLightTheta;
        }
        else
        {
          return Result::Failed(fmt::format("Invalid interactive mode '{}'", argVal));
        }
      }
      else if (argKey == "--renderer")
      {
        if (beams::utils::String::CompareIgnoreCase(argVal, "direct") == 0)
        {
          this->Renderer = RendererType::DirectVolume;
        }
        else if (beams::utils::String::CompareIgnoreCase(argVal, "shadow") == 0)
        {
          this->Renderer = RendererType::ShadowVolume;
        }
        else if (beams::utils::String::CompareIgnoreCase(argVal, "phong") == 0)
        {
          this->Renderer = RendererType::PhongVolume;
        }
        else
        {
          return Result::Failed(fmt::format("Invalid renderer '{}'", argVal));
        }
      }
    }

    if (this->InteractiveMode == InteractiveMode::Unknown)
    {
      Fmt::Println0("Interactive mode not specified, defaulting to headless");
      this->InteractiveMode = InteractiveMode::Headless;
    }
    if (this->ConfigFilePath.length() == 0)
    {
      return Result::Failed("--config-file not specified");
    }
#ifndef BEAMS_USE_GUI
    if (this->InteractiveMode == InteractiveMode::Gui)
    {
      return Result::Failed(
        "Beams was not compiled with GUI support but interactive mode was set to gui");
    }
#endif

    return Result::Succeeded();
  }

  Result LoadConfig()
  {
    if (!beams::io::File::FileExists(this->ConfigFilePath))
    {
      return Result::Failed(fmt::format("Config file '{}' not found", this->ConfigFilePath));
    }

    this->Config = std::make_shared<beams::Config>();
    CHECK_RESULT(this->Config->LoadFromFile(this->ConfigFilePath), "Error loading config");
    return Result::Succeeded();
  }

  Result LoadScene(const std::string& presetId)
  {
    if (this->Config->Presets.find(presetId) == this->Config->Presets.end())
    {
      return Result::Failed(fmt::format("Preset '{}' not found", presetId));
    }
    this->CurrentPresetId = presetId;
    return this->LoadFromCurrentPreset();
  }

  Result LoadFromCurrentPreset()
  {
    beams::Preset preset = this->Config->Presets[this->CurrentPresetId];
    preset.RenderOptions.Renderer = this->Renderer;
    this->Scene = SceneFactory::CreateFromPreset(*this->Config, preset, this->Mpi);

    if (this->Scene->DataSet.GetNumberOfFields() > 0)
    {
      return Result::Succeeded();
    }
    else
    {
      return Result::Failed(
        fmt::format("Unable to load data set for preset '{}'", this->CurrentPresetId));
    }
  }

  std::string ConfigFilePath;
  std::string StartPresetId;
  std::shared_ptr<beams::Config> Config;
  std::string CurrentPresetId;
  std::shared_ptr<beams::rendering::Scene> Scene;
  beams::mpi::MpiEnv& Mpi;
  beams::InteractiveMode InteractiveMode = InteractiveMode::Unknown;
  RendererType Renderer = RendererType::ShadowVolume;
};

Beams::Beams(beams::mpi::MpiEnv& mpiEnv)
  : Internals(new InternalsType(mpiEnv))
{
}

Result Beams::Initialize(int& argc, char** argv)
{
  auto opts = vtkm::cont::InitializeOptions::RequireDevice;
  vtkm::cont::Initialize(argc, argv, opts);
  Fmt::Println("Running on host '{}'", this->Internals->Mpi.Hostname);

  CHECK_RESULT(this->Internals->ParseArgs(argc, argv), "Error initializing beams");
  CHECK_RESULT(this->Internals->LoadConfig(), "Error initializing beams");

  return Result::Succeeded();
}

void Beams::LoadScene()
{
  std::string presetId;
  if (!(this->Internals->StartPresetId.empty()))
  {
    presetId = this->Internals->StartPresetId;
  }
  else
  {
    presetId = this->Internals->Config->DefaultPresetId;
  }
  auto result = this->Internals->LoadScene(presetId);
  if (!result.Success)
  {
    Fmt::RawPrint0("Error loading scene: {}\n", result.Err);
    exit(1);
  }
}

void Beams::SetupScene()
{
  this->Internals->Scene->Ready();
}

void Beams::Run()
{
  if (this->Internals->InteractiveMode == InteractiveMode::Headless)
  {
    this->RunHeadless();
  }
  else if (this->Internals->InteractiveMode == InteractiveMode::Gui)
  {
    this->RunGui();
  }
  else if (this->Internals->InteractiveMode == InteractiveMode::SummerMovie ||
           this->Internals->InteractiveMode == InteractiveMode::SummerMovieFast)
  {
    this->RunSummerMovie(this->Internals->InteractiveMode == InteractiveMode::SummerMovieFast);
  }
  else if (this->Internals->InteractiveMode == InteractiveMode::MovieCamera)
  {
    this->RunMovieCamera();
  }
  else if (this->Internals->InteractiveMode == InteractiveMode::MovieLightTheta)
  {
    this->RunMovieLightTheta();
  }
}

void Beams::RunHeadless()
{
  this->LoadScene();
  this->SetupScene();

  auto scene = this->Internals->Scene;
  scene->PrintSummary();
  auto& mpi = this->Internals->Mpi;
  auto config = this->Internals->Config;

  int numIters = config->NumIters;
  beams::utils::Timings timings(scene->Id,
                                { P1ShadowVolumeGenLabel,
                                  P2ShadowVolumeRenderLabel,
                                  P3ShadowVolumeUpdateLabel,
                                  P4VolumeRenderLabel,
                                  P5CompositeLabel });
  for (int i = 0; i < numIters; ++i)
  {
    this->SetupScene();
    scene->Mapper->RenderCells(scene->DataSet.GetCellSet(),
                               scene->DataSet.GetCoordinateSystem(),
                               scene->DataSet.GetField(scene->FieldName),
                               scene->ColorTable,
                               scene->Camera,
                               scene->RangeMap->GetGlobalRange());

    timings.AddIteration(scene->Mapper->GetProfilerTimes());
  }
  if (mpi.Rank == 0)
  {
    this->SaveCanvas();
    timings.Save(config->TimingsFileName);
  }
}

void Beams::RunGui()
{
#ifdef BEAMS_USE_GUI
  this->LoadScene();
  this->SetupScene();

  auto scene = this->Internals->Scene;

  beams::gui::Window window;
  auto result = window.Initialize("Beams",
                                  scene->Canvas->GetWidth(),
                                  scene->Canvas->GetHeight(),
                                  this->Internals->Config,
                                  &this->Internals->Mpi,
                                  [&]
                                  {
                                    this->Internals->LoadConfig();
                                    window.SetConfig(this->Internals->Config);
                                  });
  if (!result)
  {
    Fmt::Println0("Error initializing window: {}", result.Err);
    return;
  }

  window.Run();
#endif
}



void Beams::RunSummerMovie(bool fast)
{
  auto config = this->Internals->Config;
  std::string presetId = fast ? config->SummerMovieFastPreset : config->SummerMoviePreset;
  this->Internals->StartPresetId = presetId;
  this->LoadScene();

  std::shared_ptr<beams::rendering::SummerMovieScene> scene =
    std::dynamic_pointer_cast<beams::rendering::SummerMovieScene>(this->Internals->Scene);
  auto& mpi = this->Internals->Mpi;

  auto numFiles = scene->GetFilesCount();
  std::size_t fileIndex = 0;
  int frame = 0;
  /*
  for(;fileIndex < numFiles; ++fileIndex)
  {
    std::string fileName = scene->GetCurretFileName();
    fileName = vtkm::io::Filename(fileName);
    std::string fileNumber;
    auto Split = [](const std::string& s) -> std::vector<std::string>
    {
      std::vector<std::string> result;
      std::istringstream iss(s);
      std::string temp;

      while (std::getline(iss, temp, '_'))
      {
        std::istringstream innerStream(temp);
        std::string innerTemp;
        while (std::getline(innerStream, innerTemp, '.'))
        {
          result.push_back(innerTemp);
        }
      }
      return result;
    };

    for (const auto& token : Split(fileName))
    {
      try
      {
        std::stol(token);
        fileNumber = token;
      }
      catch (...)
      {
        continue;
      }
    }
    
    std::string name = this->GetSaveCanvasName(fileNumber);
    if(beams::io::File::FileExists(name))
    {
      Fmt::RawPrintln0("Skipping {}", name);
      scene->AdvanceFileIndex();
    } 
    else 
    {
        break;
    }
  }
  */

  for (; fileIndex < numFiles; ++fileIndex)
  {
    std::string fileName = vtkm::io::Filename(scene->GetCurretFileName());
    std::string fileNumber;

    for (const auto& token : beams::utils::String::Split(fileName, "_."))
    {
      try
      {
        std::stol(token);
        fileNumber = token;
      }
      catch (...)
      {
        continue;
      }
    }

    scene->LoadDataSet(*config, config->Presets[presetId], mpi);
    this->SetupScene();
    bool shouldRotateLight = std::find(config->SummerMovieStopPoints.begin(),
                                       config->SummerMovieStopPoints.end(),
                                       fileNumber) != config->SummerMovieStopPoints.end();
    if (shouldRotateLight)
    {
      int numStillFrames = 12;
      for (int rotateFrameIndex; rotateFrameIndex < numStillFrames; ++rotateFrameIndex)
      {
        scene->Mapper->RenderCells(scene->DataSet.GetCellSet(),
                                   scene->DataSet.GetCoordinateSystem(),
                                   scene->DataSet.GetField(scene->FieldName),
                                   scene->ColorTable,
                                   scene->Camera,
                                   scene->RangeMap->GetGlobalRange());
        if (mpi.Rank == 0)
        {
          this->SaveCanvas(frame++);
        }
      }

      FMT_VAR(scene->LightPosition);
      auto pDash = scene->LightPosition;
      auto relLightPos = internals::ToWikiAxis(pDash);

      auto startR = vtkm::Magnitude(relLightPos);
      auto startTheta = vtkm::ATan2(
        vtkm::Magnitude(vtkm::Vec2f_32{ relLightPos[0], relLightPos[1] }), relLightPos[2]);
      auto startPhi = 0.0f;
      if (relLightPos[0] >= 0.0f)
      {
        startPhi = vtkm::ATan2(relLightPos[1], relLightPos[0]);
      }
      else
      {
        startPhi = vtkm::ATan2(relLightPos[1], relLightPos[0]) + vtkm::Pi();
      }

      auto phi = startPhi;
      auto theta = startTheta;
      vtkm::Float32 lightSecs = 5;
      auto numFramesPerSecond = 24;
      auto lengthInFrames = lightSecs * numFramesPerSecond;
      auto delta = vtkm::TwoPif() / lengthInFrames;
      //auto numLightFrames = static_cast<int>(vtkm::Round(vtkm::TwoPif() / deltaThetha));
      auto numLightFrames = static_cast<int>(vtkm::Round(vtkm::TwoPif() / delta));
      for (int frameIndex = 0; frameIndex < numLightFrames; ++frameIndex)
      {
        //theta = startTheta + frameIndex * delta;
        phi = startPhi + frameIndex * delta;
        vtkm::Vec3f_32 newRelLightPos;
        newRelLightPos[0] = startR * vtkm::Sin(theta) * vtkm::Cos(phi);
        newRelLightPos[1] = startR * vtkm::Sin(theta) * vtkm::Sin(phi);
        newRelLightPos[2] = startR * vtkm::Cos(theta);
        vtkm::Vec3f_32 newLightPos = internals::FromWikiAxis(newRelLightPos);
        scene->LightPosition = newLightPos;
        this->SetupScene();
        scene->Mapper->RenderCells(scene->DataSet.GetCellSet(),
                                   scene->DataSet.GetCoordinateSystem(),
                                   scene->DataSet.GetField(scene->FieldName),
                                   scene->ColorTable,
                                   scene->Camera,
                                   scene->RangeMap->GetGlobalRange());
        if (mpi.Rank == 0)
        {
          this->SaveCanvas(frame++);
        }
      }
      scene->LightPosition = pDash;
      for (int rotateFrameIndex = 0; rotateFrameIndex < numStillFrames; ++rotateFrameIndex)
      {
        scene->Mapper->RenderCells(scene->DataSet.GetCellSet(),
                                   scene->DataSet.GetCoordinateSystem(),
                                   scene->DataSet.GetField(scene->FieldName),
                                   scene->ColorTable,
                                   scene->Camera,
                                   scene->RangeMap->GetGlobalRange());
        if (mpi.Rank == 0)
        {
          this->SaveCanvas(frame++);
        }
      }
    }
    else
    {
      scene->Mapper->RenderCells(scene->DataSet.GetCellSet(),
                                 scene->DataSet.GetCoordinateSystem(),
                                 scene->DataSet.GetField(scene->FieldName),
                                 scene->ColorTable,
                                 scene->Camera,
                                 scene->RangeMap->GetGlobalRange());

      if (mpi.Rank == 0)
      {
        this->SaveCanvas(frame++);
      }
    }

    scene->AdvanceFileIndex();
    /*
    if (fileIndex >= scene->GetFilesCount() - 1)
    {
      break;
    }
    scene->LoadDataSet(*config, config->Presets[presetId], mpi);
    */
  }
}

void Beams::RunMovieCamera()
{
  this->LoadScene();
  this->SetupScene();

  auto scene = this->Internals->Scene;
  auto& mpi = this->Internals->Mpi;
  auto config = this->Internals->Config;

  auto pDash = scene->CameraPosition.Value;
  auto origin = scene->CameraLookAt.Value;

  auto relCamPos = internals::ToWikiAxis(pDash - origin);

  auto startR = vtkm::Magnitude(relCamPos);
  auto startTheta =
    vtkm::ATan2(vtkm::Magnitude(vtkm::Vec2f_32{ relCamPos[0], relCamPos[1] }), relCamPos[2]);
  auto startPhi = 0.0f;
  if (relCamPos[0] >= 0.0f)
  {
    startPhi = vtkm::ATan2(relCamPos[1], relCamPos[0]);
  }
  else
  {
    startPhi = vtkm::ATan2(relCamPos[1], relCamPos[0]) + vtkm::Pi();
  }

  auto phi = startPhi;
  auto theta = startTheta;
  auto numFramesPerSecond = 30;
  vtkm::Float32 camSecs = 5;
  auto lengthInFrames = camSecs * numFramesPerSecond;
  auto deltaPhi = vtkm::TwoPif() / lengthInFrames;
  auto numCamFrames = static_cast<int>(vtkm::Round(vtkm::TwoPif() / deltaPhi));
  for (int frameIndex = 0; frameIndex < numCamFrames; ++frameIndex)
  {
    phi = startPhi + frameIndex * deltaPhi;
    vtkm::Vec3f_32 newRelCamPos;
    newRelCamPos[0] = startR * vtkm::Sin(theta) * vtkm::Cos(phi);
    newRelCamPos[1] = startR * vtkm::Sin(theta) * vtkm::Sin(phi);
    newRelCamPos[2] = startR * vtkm::Cos(theta);
    vtkm::Vec3f_32 newCamPos = internals::FromWikiAxis(newRelCamPos) + origin;
    scene->CameraPosition = newCamPos;
    this->SetupScene();
    scene->Mapper->RenderCells(scene->DataSet.GetCellSet(),
                               scene->DataSet.GetCoordinateSystem(),
                               scene->DataSet.GetField(scene->FieldName),
                               scene->ColorTable,
                               scene->Camera,
                               scene->RangeMap->GetGlobalRange());

    if (mpi.Rank == 0)
    {
      this->SaveCanvas(frameIndex, "camera");
    }
    if ((frameIndex + 1) % numFramesPerSecond == 0)
    {
      Fmt::RawPrintln0("Finished camera {} frames for second {}",
                       (frameIndex + 1),
                       (frameIndex + 1) / numFramesPerSecond);
    }
  }
}

void Beams::RunMovieLightTheta()
{
  this->LoadScene();
  this->SetupScene();

  auto scene = this->Internals->Scene;
  auto& mpi = this->Internals->Mpi;
  auto config = this->Internals->Config;

  auto pDash = scene->LightPosition;
  // auto origin = scene->CameraLookAt.Value;
  auto relLightPos = internals::ToWikiAxis(pDash);

  auto startR = vtkm::Magnitude(relLightPos);
  auto startTheta =
    vtkm::ATan2(vtkm::Magnitude(vtkm::Vec2f_32{ relLightPos[0], relLightPos[1] }), relLightPos[2]);
  auto startPhi = 0.0f;
  if (relLightPos[0] >= 0.0f)
  {
    startPhi = vtkm::ATan2(relLightPos[1], relLightPos[0]);
  }
  else
  {
    startPhi = vtkm::ATan2(relLightPos[1], relLightPos[0]) + vtkm::Pi();
  }

  auto phi = startPhi;
  auto theta = startTheta;
  vtkm::Float32 lightSecs = 5;
  auto numFramesPerSecond = 30;
  auto lengthInFrames = lightSecs * numFramesPerSecond;
  auto deltaThetha = vtkm::TwoPif() / lengthInFrames;
  auto numLightFrames = static_cast<int>(vtkm::Round(vtkm::TwoPif() / deltaThetha));
  for (int frameIndex = 0; frameIndex < numLightFrames; ++frameIndex)
  {
    theta = startTheta + frameIndex * deltaThetha;
    vtkm::Vec3f_32 newRelLightPos;
    newRelLightPos[0] = startR * vtkm::Sin(theta) * vtkm::Cos(phi);
    newRelLightPos[1] = startR * vtkm::Sin(theta) * vtkm::Sin(phi);
    newRelLightPos[2] = startR * vtkm::Cos(theta);
    vtkm::Vec3f_32 newLightPos = internals::FromWikiAxis(newRelLightPos);
    scene->LightPosition = newLightPos;
    this->SetupScene();
    scene->Mapper->RenderCells(scene->DataSet.GetCellSet(),
                               scene->DataSet.GetCoordinateSystem(),
                               scene->DataSet.GetField(scene->FieldName),
                               scene->ColorTable,
                               scene->Camera,
                               scene->RangeMap->GetGlobalRange());
    if (mpi.Rank == 0)
    {
      this->SaveCanvas(frameIndex, "light_theta");
    }
    if ((frameIndex + 1) % numFramesPerSecond == 0)
    {
      Fmt::RawPrintln0("Finished light_theta {} frames for second {}",
                       (frameIndex + 1),
                       (frameIndex + 1) / numFramesPerSecond);
    }
  }
}

std::string Beams::GetSaveCanvasName(std::string frameName, std::string suffix)
{
  auto scene = this->Internals->Scene;

  std::string renderer = "direct";
  if (this->Internals->Renderer == RendererType::DirectVolume)
  {
    renderer = "direct";
  }
  else if (this->Internals->Renderer == RendererType::ShadowVolume)
  {
    renderer = "shadow";
  }
  else if (this->Internals->Renderer == RendererType::PhongVolume)
  {
    renderer = "phong";
  }

  std::stringstream ss;
  ss << scene->Id << "_" << renderer << ((suffix.length() > 0) ? ("_" + suffix) : "") << "_"
     << frameName << ".png";
  return ss.str();
}

void Beams::SaveCanvas(int frame, std::string suffix)
{
  auto scene = this->Internals->Scene;
  auto& mpi = this->Internals->Mpi;

  auto padZero = [](int num, int length = 3)
  {
    std::stringstream ss;
    ss << std::setw(length) << std::setfill('0') << num;
    return ss.str();
  };

  if (suffix.length() > 0)
  {
    suffix = "_" + suffix;
  }

  std::string renderer = "direct";
  if (this->Internals->Renderer == RendererType::DirectVolume)
  {
    renderer = "direct";
  }
  else if (this->Internals->Renderer == RendererType::ShadowVolume)
  {
    renderer = "shadow";
  }
  else if (this->Internals->Renderer == RendererType::PhongVolume)
  {
    renderer = "phong";
  }
  renderer = "_" + renderer;

  std::stringstream ss;
  ss << scene->Id << "_" << fmt::format("{:03}", mpi.Size) << renderer << suffix << "_"
     << padZero(frame) << ".png";
  scene->Canvas->SaveAs(ss.str());
  if (this->Internals->InteractiveMode == InteractiveMode::Headless ||
      this->Internals->InteractiveMode == InteractiveMode::SummerMovie ||
      this->Internals->InteractiveMode == InteractiveMode::SummerMovieFast)
  {
    Fmt::RawPrint0("Saved {}\n", ss.str());
  }
}

void Beams::SaveCanvas(std::string frameName, std::string suffix)
{
  auto scene = this->Internals->Scene;

  std::string renderer = "direct";
  if (this->Internals->Renderer == RendererType::DirectVolume)
  {
    renderer = "direct";
  }
  else if (this->Internals->Renderer == RendererType::ShadowVolume)
  {
    renderer = "shadow";
  }
  else if (this->Internals->Renderer == RendererType::PhongVolume)
  {
    renderer = "phong";
  }

  std::stringstream ss;
  ss << scene->Id << "_" << renderer << ((suffix.length() > 0) ? ("_" + suffix) : "") << "_"
     << frameName << ".png";
  scene->Canvas->SaveAs(ss.str());
  Fmt::RawPrint0("Saved {}\n", ss.str());
}
} // namespace beams
