//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//============================================================================

#include <vtkm/cont/DeviceAdapter.h>
#include <vtkm/cont/testing/MakeTestDataSet.h>
#include <vtkm/cont/testing/Testing.h>
#include <vtkm/io/VTKDataSetReader.h>
#include <vtkm/rendering/Actor.h>
#include <vtkm/rendering/Canvas.h>
#include <vtkm/rendering/CanvasRayTracer.h>
#include <vtkm/rendering/MapperTracker.h>
#include <vtkm/rendering/MapperVolume.h>
#include <vtkm/rendering/Scene.h>
#include <vtkm/rendering/View3D.h>
#include <vtkm/rendering/testing/RenderTest.h>
#include <vtkm/testing/Testing.h>

namespace
{


void RenderTests()
{
  using M = vtkm::rendering::MapperTracker;
  using C = vtkm::rendering::CanvasRayTracer;
  // using V3 = vtkm::rendering::View3D;

  const std::string fieldName = "E_magnitude";

  vtkm::cont::DataSet rectDS, unsDS;
  std::string rectfname =
    vtkm::cont::testing::Testing::DataPath("/research/matts_data/warpx_500.vtk");
  vtkm::io::VTKDataSetReader rectReader(rectfname);
  try
  {
    rectDS = rectReader.ReadDataSet();
  }
  catch (vtkm::io::ErrorIO& e)
  {
    std::string message("Error reading: ");
    message += rectfname;
    message += ", ";
    message += e.GetMessage();

    VTKM_TEST_FAIL(message.c_str());
  }

  vtkm::cont::ColorTable colorTable("cool to warm");
  colorTable.AddPointAlpha(0.0, 0.0f);
  colorTable.AddPointAlpha(0.01, 0.0f);
  colorTable.AddPointAlpha(0.1, 0.0f);
  colorTable.AddPointAlpha(0.2, 0.20f);
  colorTable.AddPointAlpha(0.3, 0.2f);
  colorTable.AddPointAlpha(0.4, 0.4f);
  colorTable.AddPointAlpha(0.5, 0.6f);
  colorTable.AddPointAlpha(0.6, 0.7f);
  colorTable.AddPointAlpha(0.7, 0.8f);
  colorTable.AddPointAlpha(0.8, 0.9f);
  colorTable.AddPointAlpha(0.9, 0.7f);
  colorTable.AddPointAlpha(1., 0.0f);

  C canvas(1024, 1024);
  vtkm::rendering::Color background(1.0f, 1.0f, 1.0f, 1.0f);
  vtkm::rendering::Color foreground(0.0f, 0.0f, 0.0f, 1.0f);
  canvas.SetBackgroundColor(background);
  canvas.SetForegroundColor(foreground);
  canvas.Clear();
  M mapper;
  mapper.SetCanvas(&canvas);

  vtkm::rendering::Camera camera;
  camera.ResetToBounds(rectDS.GetCoordinateSystem().GetBounds());
  camera.Azimuth(-65.0);
  camera.Elevation(-40.0);
  //camera.Zoom(0.4);
  std::cout << "BOUNDS " << rectDS.GetCoordinateSystem().GetBounds() << "\n";
  camera.Print();
  vtkm::Range range = rectDS.GetField(fieldName).GetRange().ReadPortal().Get(0);
  ;
  std::cout << "RANGE " << range << "\n";
  range.Min = 1000.f;
  range.Max = 1e9f;
  //range.Max = 1e10f;
  mapper.SetActiveColorTable(colorTable);
  mapper.SetSamples(10000);

  vtkm::Vec<vtkm::Float32, 3> lpos, lcolor;
  vtkm::Float32 lradius = 0.1f;
  lpos[0] = -9.2607e-05f;
  //lpos[1] = -3.71907e-05f;
  lpos[1] = -10.71907e-02f;
  lpos[2] = 9.13083e-03;
  //lpos[0] = -30000.f;
  //lpos[1] =  90200.f;
  //lpos[2] = -50000.f;
  //50200,-82806.7,-47458.4

  lcolor[0] = 2.;
  lcolor[1] = 2.f;
  lcolor[2] = 2.f;
  mapper.AddLight(lradius, lpos, lcolor);
  mapper.SetDensityScale(100000000.f);

  mapper.RenderCells(rectDS.GetCellSet(),
                     rectDS.GetCoordinateSystem(),
                     rectDS.GetField(fieldName),
                     colorTable,
                     camera,
                     range);
  canvas.SaveAs("warpx.png");

  // vtkm::rendering::testing::Render<M, C, V3>(rectDS, "hardyglobal", colorTable, "rect3D.png");
}

} //namespace

int UnitTestMapperVolume(int argc, char* argv[])
{
  return vtkm::cont::testing::Testing::Run(RenderTests, argc, argv);
}
