//=============================================================================
//
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//  Copyright 2016 National Technology & Engineering Solutions of Sandia, LLC (NTESS).
//  Copyright 2016 UT-Battelle, LLC.
//  Copyright 2016 Los Alamos National Security.
//
//  Under the terms of Contract DE-NA0003525 with NTESS,
//  the U.S. Government retains certain rights in this software.
//  Under the terms of Contract DE-AC52-06NA25396 with Los Alamos National
//  Laboratory (LANL), the U.S. Government retains certain rights in
//  this software.
//
//=============================================================================
#ifndef vtk_m_rendering_raytracing_ResidualTracker_h
#define vtk_m_rendering_raytracing_ResidualTracker_h

#include <vtkm/cont/DataSet.h>

#include <vtkm/rendering/raytracing/Lights.h>
#include <vtkm/rendering/raytracing/Ray.h>

namespace vtkm
{
namespace rendering
{
namespace raytracing
{

class ResidualTracker
{
public:
  using DefaultHandle = vtkm::cont::ArrayHandle<vtkm::FloatDefault>;
  using CartesianArrayHandle =
    vtkm::cont::ArrayHandleCartesianProduct<DefaultHandle, DefaultHandle, DefaultHandle>;

  VTKM_CONT
  ResidualTracker();

  VTKM_CONT
  void EnableCompositeBackground();

  VTKM_CONT
  void DisableCompositeBackground();

  VTKM_CONT
  void SetColorMap(const vtkm::cont::ArrayHandle<vtkm::Vec<vtkm::Float32, 4>>& colorMap);

  VTKM_CONT
  void SetData(const vtkm::cont::CoordinateSystem& coords,
               const vtkm::cont::Field& scalarField,
               const vtkm::cont::CellSetStructured<3>& cellset,
               const vtkm::Range& scalarRange);


  VTKM_CONT
  void Render(vtkm::rendering::raytracing::Ray<vtkm::Float32>& rays);

  VTKM_CONT
  void SetSamples(const vtkm::Int32& numSamples);

  VTKM_CONT
  void SetDensityScale(const vtkm::Float32& scale);

  VTKM_CONT
  void AddLight(const vtkm::Float32& radius,
                const vtkm::Vec<vtkm::Float32, 3>& position,
                const vtkm::Vec<vtkm::Float32, 3>& color);

  VTKM_CONT
  vtkm::Float32 GetMaxAlpha();

protected:
  template <typename Precision, typename Device>
  VTKM_CONT void RenderOnDevice(vtkm::rendering::raytracing::Ray<Precision>& rays, Device);
  template <typename Precision>
  struct RenderFunctor;

  bool IsSceneDirty;
  bool IsUniformDataSet;
  vtkm::Bounds SpatialExtent;
  // vtkm::cont::ArrayHandleVirtualCoordinates Coordinates;
  vtkm::cont::CoordinateSystem Coords;
  vtkm::cont::CellSetStructured<3> Cellset;
  const vtkm::cont::Field* ScalarField;
  vtkm::cont::ArrayHandle<vtkm::Vec<vtkm::Float32, 4>> ColorMap;
  vtkm::Int32 NumSamples;
  vtkm::Float32 DensityScale;
  vtkm::Range ScalarRange;
  Lights TheLights;
};
}
}
} //namespace vtkm::rendering::raytracing
#endif
