//=============================================================================
//
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//  Copyright 2016 National Technology & Engineering Solutions of Sandia, LLC (NTESS).
//  Copyright 2016 UT-Battelle, LLC.
//  Copyright 2016 Los Alamos National Security.
//
//  Under the terms of Contract DE-NA0003525 with NTESS,
//  the U.S. Government retains certain rights in this software.
//  Under the terms of Contract DE-AC52-06NA25396 with Los Alamos National
//  Laboratory (LANL), the U.S. Government retains certain rights in
//  this software.
//
//=============================================================================
#include <vtkm/rendering/raytracing/ResidualTracker.h>

#include <iostream>
#include <math.h>
#include <stdio.h>
#include <vtkm/cont/Algorithm.h>
#include <vtkm/cont/ArrayHandleCartesianProduct.h>
#include <vtkm/cont/ArrayHandleCounting.h>
#include <vtkm/cont/ArrayHandleUniformPointCoordinates.h>
#include <vtkm/cont/CellSetStructured.h>
#include <vtkm/cont/ColorTable.h>
#include <vtkm/cont/ErrorBadValue.h>
#include <vtkm/cont/Timer.h>
#include <vtkm/cont/TryExecute.h>
#include <vtkm/rendering/raytracing/Backgrounds.h>
#include <vtkm/rendering/raytracing/BoundingVolumeHierarchy.h>
#include <vtkm/rendering/raytracing/Camera.h>
#include <vtkm/rendering/raytracing/Logger.h>
#include <vtkm/rendering/raytracing/MeshOracleFactory.h>
#include <vtkm/rendering/raytracing/MinMaxVoxelGrid.h>
#include <vtkm/rendering/raytracing/Random.h>
#include <vtkm/rendering/raytracing/Ray.h>
#include <vtkm/rendering/raytracing/RayTracingTypeDefs.h>
#include <vtkm/rendering/raytracing/Sampling.h>
#include <vtkm/rendering/raytracing/ToneMapper.h>
#include <vtkm/worklet/DispatcherMapField.h>
#include <vtkm/worklet/DispatcherMapTopology.h>
#include <vtkm/worklet/WorkletMapField.h>
#include <vtkm/worklet/WorkletMapTopology.h>

namespace vtkm
{
namespace rendering
{
namespace raytracing
{
//static int debugRay = 211488;
//static int debugRay = -1;
// static int debugRay = 346286;

class Sampler : public vtkm::worklet::WorkletMapField
{
private:
  vtkm::Float32 MinScalar;
  vtkm::Float32 InverseDeltaScalar;
  vtkm::Float32 MaxDensity;

public:
  VTKM_CONT
  Sampler(const vtkm::Float32& minScalar,
          const vtkm::Float32& maxScalar,
          const vtkm::Float32& maxDensity)
    : MinScalar(minScalar)
    , MaxDensity(maxDensity)
  {
    if ((maxScalar - minScalar) != 0.f)
      InverseDeltaScalar = 1.f / (maxScalar - minScalar);
    else
      InverseDeltaScalar = minScalar;
  }
  using ControlSignature = void(FieldInOut,
                                FieldInOut,
                                FieldIn,
                                FieldInOut,
                                FieldInOut,
                                FieldInOut,
                                FieldInOut,
                                WholeArrayInOut,
                                WholeArrayInOut,
                                WholeArrayIn,
                                ExecObject meshOracle,
                                WholeArrayIn,
                                ExecObject lights,
                                ExecObject minmax);
  using ExecutionSignature =
    void(_1, _2, _3, _4, _5, _6, _7, _8, _9, _10, WorkIndex, _11, _12, _13, _14);


  template <typename ScalarPortalType,
            typename ColorBufferType,
            typename ColorMapType,
            typename LightsType,
            typename MinMaxType>
  VTKM_EXEC void operator()(vtkm::Vec<vtkm::Float32, 3>& rayDir,
                            vtkm::Vec<vtkm::Float32, 3>& rayOrigin,
                            const vtkm::Float32& minDistance,
                            vtkm::Float32& maxDistance,
                            vtkm::UInt8& status,
                            vtkm::Id& depth,
                            vtkm::Vec<vtkm::UInt32, 2>& rng,
                            ColorBufferType& colorBuffer,
                            ColorBufferType& throughputBuffer,
                            ScalarPortalType& scalars,
                            const vtkm::Id& pixelIndex,
                            MeshOracleWrapper& oracle,
                            const ColorMapType& colorMap,
                            const LightsType& lights,
                            const MinMaxType& minmax) const
  {
    if (status != RAY_ACTIVE)
    {
      return;
    }
    vtkm::Vec<vtkm::Float32, 4> color;
    BOUNDS_CHECK(colorBuffer, pixelIndex * 4 + 0);
    color[0] = colorBuffer.Get(pixelIndex * 4 + 0);
    BOUNDS_CHECK(colorBuffer, pixelIndex * 4 + 1);
    color[1] = colorBuffer.Get(pixelIndex * 4 + 1);
    BOUNDS_CHECK(colorBuffer, pixelIndex * 4 + 2);
    color[2] = colorBuffer.Get(pixelIndex * 4 + 2);
    BOUNDS_CHECK(colorBuffer, pixelIndex * 4 + 3);
    color[3] = colorBuffer.Get(pixelIndex * 4 + 3);

    vtkm::Vec<vtkm::Float32, 3> throughput;
    BOUNDS_CHECK(throughputBuffer, pixelIndex * 3 + 0);
    throughput[0] = static_cast<vtkm::Float32>(throughputBuffer.Get(pixelIndex * 3 + 0));
    BOUNDS_CHECK(throughputBuffer, pixelIndex * 3 + 1);
    throughput[1] = static_cast<vtkm::Float32>(throughputBuffer.Get(pixelIndex * 3 + 1));
    BOUNDS_CHECK(throughputBuffer, pixelIndex * 3 + 2);
    throughput[2] = static_cast<vtkm::Float32>(throughputBuffer.Get(pixelIndex * 3 + 2));
    constexpr Float32 epsilon = 1e-12f;
    //get the initial sample position;
    vtkm::Vec<vtkm::Float32, 3> sampleLocation;
    // find the distance to the first sample
    vtkm::Float32 distance = minDistance + epsilon;
    sampleLocation = rayOrigin + distance * rayDir;


    const vtkm::Id colorMapSize = colorMap.GetNumberOfValues() - 1;
    //std::cout<<"INxed "<<pixelIndex<<"\n";
    vtkm::Id cellId = -2;

    // Init the DDA traversal
    vtkm::Vec<vtkm::Float32, 3> deltaMax, delta;
    vtkm::Vec<vtkm::Int32, 3> voxel;

    vtkm::Float32 exitDistance =
      minmax.InitTraversal(sampleLocation, rayDir, deltaMax, delta, voxel);
    // delta max is relative to the sample location
    // so offset it by the current ray distance so its relative to the
    // ray origin
    deltaMax[0] += distance;
    deltaMax[1] += distance;
    deltaMax[2] += distance;
    exitDistance += distance;

    maxDistance = vtkm::Min(maxDistance, exitDistance);

    /* if (!minmax.IsValid(voxel))
      std::cout << pixelIndex << "\n";
    */
    vtkm::Float32 maxDensity = vtkm::Max(minmax.GetRange(voxel)[1], 0.0001f);
    vtkm::Float32 nextDistance = minmax.Exit(deltaMax);

    while (cellId != -1)
    {
      vtkm::Float32 step = -vtkm::Log(1.f - randomf(rng)) / maxDensity;
      distance += step;

      //vtkm::Float32 d = Min(nextDistance, distance); // dist clampled to voxel exit
      //TODO: get rid of constant and replate with value relative to bounds

      //if(pixelIndex == debugRay)
      //{
      //  std::cout<<" step "<<step<<" current distance "<<distance<<" next "<<nextDistance<<"\n";
      //}
      if (distance >= nextDistance)
      {
        // we step past the current voxel, so advance
        // to the next segment, update max density
        // and reset distance to the beginning
        minmax.Advance(deltaMax, delta, voxel, rayDir);

        if (!minmax.IsValid(voxel))
        {
          // ray exceeded max distance could be inside a voxel
          // or at the grid boundary
          status = RAY_COMPLETE;
          break;
        }


        distance = nextDistance;
        nextDistance = minmax.Exit(deltaMax);
        /*
        if (!minmax.IsValid(voxel))
          std::cout << pixelIndex << "!\n";
        */
        maxDensity = vtkm::Max(minmax.GetRange(voxel)[1], 0.0001f);

        //if(pixelIndex == debugRay)
        //{
        //  std::cout<<"advance "<<voxel<<" max density "<<maxDensity<<"\n";
        //  std::cout<<"dist "<<distance<<" next d "<<nextDistance<<" exit "<<exitDistance<<"\n";
        //  std::cout<<"delta max "<<deltaMax<<"\n";
        //}
        continue;
      }

      sampleLocation = rayOrigin + distance * rayDir;

      //if(pixelIndex == debugRay)
      //{
      //  std::cout<<"step "<<step<<" d "<<distance<<" next "<<nextDistance
      //           <<" actual voxel "<<minmax.GetVoxel(sampleLocation)
      //           <<" supposed voxel "<<voxel<<"\n";
      //}

      vtkm::Vec<vtkm::Float32, 3> pcoords;
      oracle.FindCell(sampleLocation, cellId, pcoords);

      if (cellId == -1)
      {
        //if(pixelIndex == debugRay)
        //{
        //  std::cout<<"Exited\n";
        //}
        status = RAY_EXITED_DOMAIN;
        break;
      }

      //maxDensity = vtkm::Max(vColor[3], 0.0001f);

      vtkm::Id cellIndices[8];
      vtkm::Vec<vtkm::Float32, 8> values;
      vtkm::Float32 scalar = 0.f;

      const vtkm::Int32 numIndices = oracle.GetCellIndices(cellIndices, cellId);
      for (vtkm::Int32 i = 0; i < numIndices; ++i)
      {
        values[i] = static_cast<vtkm::Float32>(scalars.Get(cellIndices[i]));
      }

      scalar = oracle.Interpolate(values, pcoords);
      scalar = (scalar - MinScalar) * InverseDeltaScalar;

      vtkm::Id colorIndex =
        static_cast<vtkm::Id>(scalar * static_cast<vtkm::Float32>(colorMapSize));
      constexpr vtkm::Id zero = 0;
      colorIndex = vtkm::Max(zero, vtkm::Min(colorMapSize, colorIndex));

      vtkm::Vec<vtkm::Float32, 4> sampleColor = colorMap.Get(colorIndex);
      // anything after the scatter is a transmittence estimator
      bool trans_only = depth == 0;

      //if(sampleColor[3] > maxDensity)
      //{
      //  std::cout<<"BAD"<<voxel<<" cell "<<cellId<<" sc "<<sampleColor[3]<<" max "<<maxDensity<<"\n";;
      //}
      if (!trans_only && randomf(rng) < ((sampleColor[3]) / maxDensity))
      {
        //if(pixelIndex == debugRay) std::cout<<"scatting event "<<sampleColor<<"\n";
        // reset ray
        if (randomf(rng) < .5f)
        {
          rayDir = UniformSphereSample(randomf(rng), randomf(rng));
          rayOrigin = sampleLocation;
          vtkm::Float32 ldist = lights.IntersectLights(rayOrigin, rayDir);
          maxDistance = ldist;
        }
        else
        {
          // TODO: comment this out and check register usage
          vtkm::Vec<vtkm::Float32, 3> lightSample;
          vtkm::Float32 ldist;
          lights.SampleLight(sampleLocation, rng, lightSample, ldist);

          rayDir = lightSample;
          maxDistance = ldist;
          rayOrigin = sampleLocation;
          //if(pixelIndex == debugRay)
          //{
          //  std::cout<<"["<<pixelIndex<<"] dir "<<rayDir<<" orig "<<rayOrigin<<" max "<<ldist<<"\n";
          //}
        }
        distance = 0.f;
        //vtkm::Float32 pdf;
        //pdf = 1.f / (4.f * vtkm::Pif());


        depth--;

        throughput[0] *= sampleColor[0];
        throughput[1] *= sampleColor[1];
        throughput[2] *= sampleColor[2];

        // max density hasn't changed
        // but the ray direction has so re-init traversal
        vtkm::Float32 newExit =
          minmax.InitTraversal(sampleLocation, rayDir, deltaMax, delta, voxel);

        maxDistance = vtkm::Min(maxDistance, newExit);
        nextDistance = minmax.Exit(deltaMax);
      }

      // emission
      vtkm::Float32 ef = 0.5f;
      if (sampleColor[3] > 0.40 && sampleColor[3] < .45)
      { // emission
        color[0] += throughput[0] * sampleColor[0] * ef;
        color[1] += throughput[1] * sampleColor[1] * ef;
        color[2] += throughput[2] * sampleColor[2] * ef;
      }


      // ratio tracking
      if (trans_only)
      {
        throughput[0] *= 1.f - sampleColor[3] / maxDensity;
        throughput[1] *= 1.f - sampleColor[3] / maxDensity;
        throughput[2] *= 1.f - sampleColor[3] / maxDensity;
        // russian roulette termination
        //vtkm::Float32 maxTr = vtkm::Max(throughput[0], vtkm::Max(throughput[1], throughput[2]));
        //maxTr = vtkm::Max(0.05f, 1.f - maxTr);
        //if(maxTr < 0.1f && randomf(rng) < maxTr)
        //{
        //  throughput[0] = 0.f;
        //  throughput[1] = 0.f;
        //  throughput[2] = 0.f;
        //  status = RAY_COMPLETE;
        //  break;
        //}
      }

    } // while sampling

    //color[0] = vtkm::Min(color[0], 1.f);
    //color[1] = vtkm::Min(color[1], 1.f);
    //color[2] = vtkm::Min(color[2], 1.f);
    //color[3] = vtkm::Min(color[3], 1.f);

    BOUNDS_CHECK(colorBuffer, pixelIndex * 4 + 0);
    colorBuffer.Set(pixelIndex * 4 + 0, color[0]);
    BOUNDS_CHECK(colorBuffer, pixelIndex * 4 + 1);
    colorBuffer.Set(pixelIndex * 4 + 1, color[1]);
    BOUNDS_CHECK(colorBuffer, pixelIndex * 4 + 2);
    colorBuffer.Set(pixelIndex * 4 + 2, color[2]);
    BOUNDS_CHECK(colorBuffer, pixelIndex * 4 + 3);
    colorBuffer.Set(pixelIndex * 4 + 3, color[3]);

    BOUNDS_CHECK(throughputBuffer, pixelIndex * 3 + 0);
    throughputBuffer.Set(pixelIndex * 3 + 0, throughput[0]);
    BOUNDS_CHECK(throughputBuffer, pixelIndex * 3 + 1);
    throughputBuffer.Set(pixelIndex * 3 + 1, throughput[1]);
    BOUNDS_CHECK(throughputBuffer, pixelIndex * 3 + 2);
    throughputBuffer.Set(pixelIndex * 3 + 2, throughput[2]);
  }
}; //Sampler

class CorrectColorMap : public vtkm::worklet::WorkletMapField
{
protected:
  vtkm::Float32 Factor;

public:
  VTKM_CONT
  CorrectColorMap(vtkm::Float32 factor)
    : Factor(factor)
  {
  }
  using ControlSignature = void(FieldIn, FieldOut);
  using ExecutionSignature = void(_1, _2);

  VTKM_EXEC
  void operator()(const vtkm::Vec<vtkm::Float32, 4>& in, vtkm::Vec<vtkm::Float32, 4>& out) const
  {
    out = in;
    out[3] *= Factor;
  }
}; //class

class CalcRayStart : public vtkm::worklet::WorkletMapField
{
  vtkm::Float32 Xmin;
  vtkm::Float32 Ymin;
  vtkm::Float32 Zmin;
  vtkm::Float32 Xmax;
  vtkm::Float32 Ymax;
  vtkm::Float32 Zmax;

public:
  VTKM_CONT
  CalcRayStart(const vtkm::Bounds boundingBox)
  {
    Xmin = static_cast<vtkm::Float32>(boundingBox.X.Min);
    Xmax = static_cast<vtkm::Float32>(boundingBox.X.Max);
    Ymin = static_cast<vtkm::Float32>(boundingBox.Y.Min);
    Ymax = static_cast<vtkm::Float32>(boundingBox.Y.Max);
    Zmin = static_cast<vtkm::Float32>(boundingBox.Z.Min);
    Zmax = static_cast<vtkm::Float32>(boundingBox.Z.Max);
  }

  VTKM_EXEC
  vtkm::Float32 rcp(vtkm::Float32 f) const { return 1.0f / f; }

  VTKM_EXEC
  vtkm::Float32 rcp_safe(vtkm::Float32 f) const { return rcp((fabs(f) < 1e-8f) ? 1e-8f : f); }

  using ControlSignature =
    void(FieldIn, FieldOut, FieldInOut, FieldInOut, FieldIn, FieldInOut, ExecObject lights);
  using ExecutionSignature = void(_1, _2, _3, _4, _5, _6, _7);
  template <typename Precision, typename LightObject>
  VTKM_EXEC void operator()(const vtkm::Vec<Precision, 3>& rayDir,
                            vtkm::Float32& minDistance,
                            vtkm::Float32& distance,
                            vtkm::Float32& maxDistance,
                            const vtkm::Vec<Precision, 3>& rayOrigin,
                            vtkm::UInt8& status,
                            const LightObject& lights) const
  {

    vtkm::Float32 dirx = static_cast<vtkm::Float32>(rayDir[0]);
    vtkm::Float32 diry = static_cast<vtkm::Float32>(rayDir[1]);
    vtkm::Float32 dirz = static_cast<vtkm::Float32>(rayDir[2]);
    vtkm::Float32 origx = static_cast<vtkm::Float32>(rayOrigin[0]);
    vtkm::Float32 origy = static_cast<vtkm::Float32>(rayOrigin[1]);
    vtkm::Float32 origz = static_cast<vtkm::Float32>(rayOrigin[2]);

    vtkm::Float32 invDirx = rcp_safe(dirx);
    vtkm::Float32 invDiry = rcp_safe(diry);
    vtkm::Float32 invDirz = rcp_safe(dirz);

    vtkm::Float32 odirx = origx * invDirx;
    vtkm::Float32 odiry = origy * invDiry;
    vtkm::Float32 odirz = origz * invDirz;

    vtkm::Float32 xmin = Xmin * invDirx - odirx;
    vtkm::Float32 ymin = Ymin * invDiry - odiry;
    vtkm::Float32 zmin = Zmin * invDirz - odirz;
    vtkm::Float32 xmax = Xmax * invDirx - odirx;
    vtkm::Float32 ymax = Ymax * invDiry - odiry;
    vtkm::Float32 zmax = Zmax * invDirz - odirz;


    minDistance = vtkm::Max(
      vtkm::Max(vtkm::Max(vtkm::Min(ymin, ymax), vtkm::Min(xmin, xmax)), vtkm::Min(zmin, zmax)),
      minDistance);
    vtkm::Float32 exitDistance =
      vtkm::Min(vtkm::Min(vtkm::Max(ymin, ymax), vtkm::Max(xmin, xmax)), vtkm::Max(zmin, zmax));
    maxDistance = vtkm::Min(maxDistance, exitDistance);
    if (maxDistance < minDistance)
    {
      status = RAY_EXITED_DOMAIN;
    }
    else
    {
      distance = minDistance;
      status = RAY_ACTIVE;
    }

    maxDistance = vtkm::Infinity32();
    ;
    vtkm::Float32 ldist = lights.IntersectLights(rayOrigin, rayDir);
    if (ldist < maxDistance)
    {
      maxDistance = ldist;
    }
  }
}; //class CalcRayStart

class FilterMissed : public vtkm::worklet::WorkletMapField
{
public:
  VTKM_CONT
  FilterMissed() {}

  using ControlSignature = void(FieldInOut);
  using ExecutionSignature = void(_1);

  VTKM_EXEC void operator()(vtkm::UInt8& status) const
  {
    if (status == RAY_EXITED_DOMAIN)
    {
      status = RAY_EXITED_MESH;
    }
  }
}; //class FilterMissed

ResidualTracker::ResidualTracker()
{
  IsSceneDirty = false;
  IsUniformDataSet = true;
  NumSamples = 100;
  DensityScale = .05f;
}

void ResidualTracker::SetColorMap(
  const vtkm::cont::ArrayHandle<vtkm::Vec<vtkm::Float32, 4>>& colorMap)
{
  ColorMap = colorMap;
}

void ResidualTracker::SetData(const vtkm::cont::CoordinateSystem& coords,
                              const vtkm::cont::Field& scalarField,
                              const vtkm::cont::CellSetStructured<3>& cellset,
                              const vtkm::Range& scalarRange)
{
  if (coords.GetData().IsType<CartesianArrayHandle>())
    IsUniformDataSet = false;
  IsSceneDirty = true;
  SpatialExtent = coords.GetBounds();
  // Coordinates = coords.GetData();
  Coords = coords;
  ScalarField = &scalarField;
  Cellset = cellset;
  ScalarRange = scalarRange;
}

template <typename Precision>
struct ResidualTracker::RenderFunctor
{
protected:
  vtkm::rendering::raytracing::ResidualTracker* Self;
  vtkm::rendering::raytracing::Ray<Precision>& Rays;

public:
  VTKM_CONT
  RenderFunctor(vtkm::rendering::raytracing::ResidualTracker* self,
                vtkm::rendering::raytracing::Ray<Precision>& rays)
    : Self(self)
    , Rays(rays)
  {
  }

  template <typename Device>
  VTKM_CONT bool operator()(Device)
  {
    VTKM_IS_DEVICE_ADAPTER_TAG(Device);

    this->Self->RenderOnDevice(this->Rays, Device());
    return true;
  }
};

void ResidualTracker::Render(vtkm::rendering::raytracing::Ray<vtkm::Float32>& rays)
{
  RenderFunctor<vtkm::Float32> functor(this, rays);
  vtkm::cont::TryExecute(functor);
}

vtkm::Float32 ResidualTracker::GetMaxAlpha()
{
  vtkm::Float32 maxAlpha = 0.f;
  vtkm::Id size = this->ColorMap.GetNumberOfValues();
  auto portal = this->ColorMap.ReadPortal();
  for (vtkm::Id i = 0; i < size; ++i)
  {
    maxAlpha = vtkm::Max(maxAlpha, portal.Get(i)[3]);
  }
  return maxAlpha;
}

//void
//ResidualTracker::Render(vtkm::rendering::raytracing::Ray<vtkm::Float64>& rays)
//{
//  RenderFunctor<vtkm::Float64> functor(this, rays);
//  vtkm::cont::TryExecute(functor);
//}

void ResidualTracker::AddLight(const vtkm::Float32& radius,
                               const vtkm::Vec<vtkm::Float32, 3>& position,
                               const vtkm::Vec<vtkm::Float32, 3>& color)
{
  TheLights.AddLight(radius, position, color);
}

template <typename Precision, typename Device>
void ResidualTracker::RenderOnDevice(vtkm::rendering::raytracing::Ray<Precision>& rays,
                                     Device device)
{
  vtkm::cont::Timer renderTimer{ device };
  renderTimer.Start();
  Logger* logger = Logger::GetInstance();
  logger->OpenLogEntry("volume_render_structured");
  logger->AddLogData("device", GetDeviceString(Device()));

  vtkm::Vec<vtkm::Float32, 3> extent;
  extent[0] = static_cast<vtkm::Float32>(this->SpatialExtent.X.Length());
  extent[1] = static_cast<vtkm::Float32>(this->SpatialExtent.Y.Length());
  extent[2] = static_cast<vtkm::Float32>(this->SpatialExtent.Z.Length());
  vtkm::Float32 mag = vtkm::Magnitude(extent);

  vtkm::Float32 alpha_point = 0.1f;

  vtkm::Float32 density = (vtkm::Log(DensityScale) / vtkm::Log(2.71828f)) / (-mag);
  density = density / alpha_point;
  std::cout << "mag " << mag << "\n";
  std::cout << "Correction " << density << "\n";
  vtkm::Float32 densityCorrection =
    (vtkm::Log(DensityScale) / vtkm::Log(2.71828f)) / (-mag * alpha_point);

  vtkm::cont::ArrayHandle<vtkm::Vec<vtkm::Float32, 4>> correctedColorMap;
  vtkm::worklet::DispatcherMapField<CorrectColorMap> correctDispatcher(densityCorrection);
  correctDispatcher.Invoke(ColorMap, correctedColorMap);

  std::cout << "$$$ " << densityCorrection << "\n";
  vtkm::Float32 maxDensity = this->GetMaxAlpha() * densityCorrection;
  std::cout << "$$$ Majorant " << maxDensity << "\n";
  vtkm::cont::Timer timer{ device };
  timer.Start();

  vtkm::worklet::DispatcherMapField<CalcRayStart> calcRayStartDispatcher(
    CalcRayStart(this->SpatialExtent));
  calcRayStartDispatcher.SetDevice(Device());
  calcRayStartDispatcher.Invoke(rays.Dir,
                                rays.MinDistance,
                                rays.Distance,
                                rays.MaxDistance,
                                rays.Origin,
                                rays.Status,
                                TheLights);

  // if the ray missed just make it miss the entire mesh (no mpi)
  vtkm::worklet::DispatcherMapField<FilterMissed> missedDispatcher;
  missedDispatcher.SetDevice(Device());
  missedDispatcher.Invoke(rays.Status);

  vtkm::Float64 time = timer.GetElapsedTime();
  logger->AddLogData("calc_ray_start", time);
  timer.Reset();
  timer.Start();

  // random numbers
  vtkm::cont::ArrayHandle<vtkm::Vec<vtkm::UInt32, 2>> rngs;
  rngs.Allocate(rays.NumRays);
  seedRng(rngs);

  bool isSupportedField =
    (ScalarField->GetAssociation() == vtkm::cont::Field::Association::POINTS ||
     ScalarField->GetAssociation() == vtkm::cont::Field::Association::CELL_SET);
  if (!isSupportedField)
    throw vtkm::cont::ErrorBadValue("Field not accociated with cell set or points");
  bool isAssocPoints = ScalarField->GetAssociation() == vtkm::cont::Field::Association::POINTS;

  MinMaxVoxelGrid minMaxGrid;
  minMaxGrid.Construct(Cellset, Coords, *ScalarField, correctedColorMap, ScalarRange);

  for (vtkm::Int32 i = 0; i < NumSamples; ++i)
  {
    Ray<Precision> raysCopy = rays.Copy();
    // init samples
    raysCopy.AddBuffer(3, "throughput");
    raysCopy.GetBuffer("throughput").InitConst(1.f);
    raysCopy.Buffers.at(0).InitConst(0.f);
    vtkm::cont::ArrayHandleConstant<vtkm::Id> depth(1, rays.NumRays);
    vtkm::cont::Algorithm::Copy(depth, raysCopy.Depth);

    if (IsUniformDataSet)
    {
      // nothing
    }
    else
    {
      CartesianArrayHandle vertices;
      // vertices = Coordinates.Cast<CartesianArrayHandle>();
      vertices = Coords.GetData().AsArrayHandle<CartesianArrayHandle>();
      MeshOracleContainer* oracle = MeshOracleFactory::Create(Cellset, Coords);
      if (isAssocPoints)
      {
        vtkm::worklet::DispatcherMapField<Sampler> samplerDispatcher(
          Sampler(vtkm::Float32(ScalarRange.Min), vtkm::Float32(ScalarRange.Max), maxDensity));
        samplerDispatcher.SetDevice(Device());
        samplerDispatcher.Invoke(raysCopy.Dir,
                                 raysCopy.Origin,
                                 raysCopy.MinDistance,
                                 raysCopy.MaxDistance,
                                 raysCopy.Status,
                                 raysCopy.Depth,
                                 rngs,
                                 raysCopy.Buffers.at(0).Buffer,
                                 raysCopy.GetBuffer("throughput").Buffer,
                                 vtkm::rendering::raytracing::GetScalarFieldArray(*ScalarField),
                                 oracle,
                                 correctedColorMap,
                                 TheLights,
                                 minMaxGrid);
      }
      else
      {
        // nothing for now
      }
    }

    vtkm::Vec<vtkm::Float32, 3> bgcolor;
    bgcolor[0] = 01.f;
    bgcolor[1] = 01.f;
    bgcolor[2] = 01.f;

    vtkm::worklet::DispatcherMapField<CheckLightContribution> lightsDispatcher(bgcolor);

    lightsDispatcher.SetDevice(Device());
    lightsDispatcher.Invoke(raysCopy.Origin,
                            raysCopy.Dir,
                            raysCopy.Depth,
                            raysCopy.Buffers.at(0).Buffer,
                            raysCopy.GetBuffer("throughput").Buffer,
                            TheLights);
    //bgcolor[0] = 0.1f;
    //bgcolor[1] = 0.1f;
    //bgcolor[2] = 0.1f;

    //Backgrounds::ConstantColor(raysCopy, bgcolor);


    //auto rportal = rays.Buffers[0].Buffer.GetPortalControl();
    //auto cportal = raysCopy.Buffers[0].Buffer.GetPortalControl();
    //vtkm::Id debugOffset = debugRay * 4;
    //std::cout<<"@@@@@@ RBuffer "<<rportal.Get(debugOffset + 0)<<" ";
    //std::cout<<rportal.Get(debugOffset + 1)<<" ";
    //std::cout<<rportal.Get(debugOffset + 2)<<" ";
    //std::cout<<rportal.Get(debugOffset + 3)<<"\n ";

    //std::cout<<"@@@@@@ CBuffer "<<cportal.Get(debugOffset + 0)<<" ";
    //std::cout<<cportal.Get(debugOffset + 1)<<" ";
    //std::cout<<cportal.Get(debugOffset + 2)<<" ";
    //std::cout<<cportal.Get(debugOffset + 3)<<"\n ";


    rays.Buffers[0].AddBuffer(raysCopy.Buffers[0]);
    std::cout << "sample " << i << "\n";
    //std::cout<<"@@@@@@ ADD RBuffer "<<rportal.Get(debugOffset + 0)<<" ";
    //std::cout<<rportal.Get(debugOffset + 1)<<" ";
    //std::cout<<rportal.Get(debugOffset + 2)<<" ";
    //std::cout<<rportal.Get(debugOffset + 3)<<"\n ";
  } // for samples
  rays.Buffers[0].ScalarMultiply(1.f / Precision(NumSamples));

  //vtkm::Id debugOffset = debugRay * 4;
  //auto rportal = rays.Buffers[0].Buffer.GetPortalControl();
  //rportal.Set(debugOffset + 0, 0.0f);
  //rportal.Set(debugOffset + 1, 1.0f);
  //rportal.Set(debugOffset + 2, 0.0f);

  //ToneMapper::ToneMapExp(rays, 0.4f);

  time = timer.GetElapsedTime();
  logger->AddLogData("sample", time);
  timer.Reset();

  time = renderTimer.GetElapsedTime();
  logger->CloseLogEntry(time);
} //Render

void ResidualTracker::SetSamples(const vtkm::Int32& numSamples)
{
  if (numSamples <= 0)
    throw vtkm::cont::ErrorBadValue("Samples must be positive.");
  NumSamples = numSamples;
}

void ResidualTracker::SetDensityScale(const vtkm::Float32& scale)
{
  if (scale <= 0.f)
    throw vtkm::cont::ErrorBadValue("Min throughput must be positive.");
  DensityScale = 1.f / scale;
}

}
}
} //namespace vtkm::rendering::raytracing
