//=============================================================================
//
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//  Copyright 2016 National Technology & Engineering Solutions of Sandia, LLC (NTESS).
//  Copyright 2016 UT-Battelle, LLC.
//  Copyright 2016 Los Alamos National Security.
//
//  Under the terms of Contract DE-NA0003525 with NTESS,
//  the U.S. Government retains certain rights in this software.
//  Under the terms of Contract DE-AC52-06NA25396 with Los Alamos National
//  Laboratory (LANL), the U.S. Government retains certain rights in
//  this software.
//
//=============================================================================
#ifndef vtk_m_rendering_raytracing_MinMaxVoxelGrid_h
#define vtk_m_rendering_raytracing_MinMaxVoxelGrid_h

#include <vtkm/cont/DataSet.h>

#include <vtkm/rendering/raytracing/Ray.h>

namespace vtkm
{
namespace rendering
{
namespace raytracing
{

template <typename Device>
class MinMaxExec
{
protected:
  using FloatHandle = vtkm::cont::ArrayHandle<vtkm::Float32>;
  using FloatPortal = typename FloatHandle::ReadPortalType;

  vtkm::Id3 Dims; // Cell Dims
  vtkm::Vec<vtkm::Float32, 3> Origin;
  vtkm::Vec<vtkm::Float32, 3> Spacing;
  vtkm::Vec<vtkm::Float32, 3> InvSpacing;
  FloatPortal MinPortal;
  FloatPortal MaxPortal;

public:
  MinMaxExec(){};
  MinMaxExec(vtkm::Id3 dims,
             vtkm::Vec<vtkm::Float32, 3> origin,
             vtkm::Vec<vtkm::Float32, 3> spacing,
             const FloatHandle& minValues,
             const FloatHandle& maxValues)
    : Dims(dims)
    , Origin(origin)
    , Spacing(spacing)
    , MinPortal(minValues.PrepareForInput(Device()))
    , MaxPortal(maxValues.PrepareForInput(Device()))
  {
    InvSpacing[0] = 1.f / spacing[0];
    InvSpacing[1] = 1.f / spacing[1];
    InvSpacing[2] = 1.f / spacing[2];
  };

  VTKM_EXEC
  vtkm::Vec<vtkm::Float32, 2> GetRange(const vtkm::Vec<vtkm::Float32, 3>& point) const
  {
    vtkm::Vec<vtkm::Float32, 3> temp = point;
    temp = temp - Origin;
    temp = temp * InvSpacing;
    vtkm::Vec<vtkm::Id, 3> logicalIndex = temp;
    // Bounds checking??
    vtkm::Id id = logicalIndex[0] + Dims[0] * (logicalIndex[1] + logicalIndex[2] * Dims[1]);
    vtkm::Vec<vtkm::Float32, 2> minmax;
    minmax[0] = MinPortal.Get(id);
    minmax[1] = MaxPortal.Get(id);
    return minmax;
  }

  VTKM_EXEC
  vtkm::Vec<vtkm::Id, 3> GetVoxel(const vtkm::Vec<vtkm::Float32, 3>& point) const
  {
    vtkm::Vec<vtkm::Float32, 3> temp = point;
    temp = temp - Origin;
    temp = temp * InvSpacing;
    vtkm::Vec<vtkm::Id, 3> logicalIndex = temp;
    return logicalIndex;
  }

  VTKM_EXEC
  inline bool IsInside(const vtkm::Vec<vtkm::Float32, 3>& point) const
  {
    bool inside = true;
    if (point[0] < Origin[0] || point[0] > Origin[0] + vtkm::Float32(Dims[0] + 1) * Spacing[0])
      inside = false;
    if (point[1] < Origin[1] || point[1] > Origin[1] + vtkm::Float32(Dims[1] + 1) * Spacing[1])
      inside = false;
    if (point[2] < Origin[2] || point[2] > Origin[2] + vtkm::Float32(Dims[2] + 1) * Spacing[2])
      inside = false;
    return inside;
  }

  VTKM_EXEC
  inline bool IsInside(const vtkm::Vec<vtkm::Int32, 3>& index) const
  {
    bool inside = true;
    vtkm::Int32 minIndex = vtkm::Min(index[0], vtkm::Min(index[1], index[2]));
    if (minIndex < 0)
      inside = false;
    if (index[0] >= Dims[0])
      inside = false;
    if (index[1] >= Dims[1])
      inside = false;
    if (index[2] >= Dims[2])
      inside = false;

    return inside;
  }
  // this assumes that the point is inside the grid. Since
  // the grid is conservative around the bounds of the data
  // set and we are only sampling inside the volume, this
  // should always be true
  // deltaMax = distance along the ray from the point to
  //            the next voxel boundary
  // delta    = step size
  // voxel    = current logical position inside the voxel
  //            grid
  VTKM_EXEC vtkm::Float32 InitTraversal(const vtkm::Vec<vtkm::Float32, 3>& point,
                                        const vtkm::Vec<vtkm::Float32, 3>& dir,
                                        vtkm::Vec<vtkm::Float32, 3>& deltaMax,
                                        vtkm::Vec<vtkm::Float32, 3>& delta,
                                        vtkm::Vec<vtkm::Int32, 3>& voxel) const
  {
    VTKM_ASSERT(IsInside(point));
    vtkm::Vec<vtkm::Float32, 3> temp = point;
    temp = temp - Origin;
    temp = temp * InvSpacing;
    voxel = temp;

    vtkm::Vec<vtkm::Float32, 3> step;
    step[0] = (dir[0] >= 0.f) ? 1.f : -1.f;
    step[1] = (dir[1] >= 0.f) ? 1.f : -1.f;
    step[2] = (dir[2] >= 0.f) ? 1.f : -1.f;

    vtkm::Vec<vtkm::Float32, 3> next_boundary;
    next_boundary[0] = (vtkm::Float32(voxel[0]) + step[0]) * Spacing[0];
    next_boundary[1] = (vtkm::Float32(voxel[1]) + step[1]) * Spacing[1];
    next_boundary[2] = (vtkm::Float32(voxel[2]) + step[2]) * Spacing[2];
    // correct next boundary for negative directions
    if (step[0] == -1.f)
      next_boundary[0] += Spacing[0];
    if (step[1] == -1.f)
      next_boundary[1] += Spacing[1];
    if (step[2] == -1.f)
      next_boundary[2] += Spacing[2];

    // distance to next voxel boundary
    deltaMax[0] =
      (dir[0] != 0.f) ? (next_boundary[0] - (point[0] - Origin[0])) / dir[0] : vtkm::Infinity32();
    deltaMax[1] =
      (dir[1] != 0.f) ? (next_boundary[1] - (point[1] - Origin[1])) / dir[1] : vtkm::Infinity32();
    deltaMax[2] =
      (dir[2] != 0.f) ? (next_boundary[2] - (point[2] - Origin[2])) / dir[2] : vtkm::Infinity32();

    // distance along ray to traverse x,y, and z of a voxel
    delta[0] = (dir[0] != 0) ? Spacing[0] / dir[0] * step[0] : vtkm::Infinity32();
    delta[1] = (dir[1] != 0) ? Spacing[1] / dir[1] * step[1] : vtkm::Infinity32();
    delta[2] = (dir[2] != 0) ? Spacing[2] / dir[2] * step[2] : vtkm::Infinity32();

    vtkm::Vec<vtkm::Float32, 3> exit_boundary;
    exit_boundary[0] = step[0] < 0.f ? 0.f : vtkm::Float32(Dims[0]) * Spacing[0];
    exit_boundary[1] = step[1] < 0.f ? 0.f : vtkm::Float32(Dims[1]) * Spacing[1];
    exit_boundary[2] = step[2] < 0.f ? 0.f : vtkm::Float32(Dims[2]) * Spacing[2];

    if (step[0] == -1.f)
      exit_boundary[0] += Spacing[0];
    if (step[1] == -1.f)
      exit_boundary[1] += Spacing[1];
    if (step[2] == -1.f)
      exit_boundary[2] += Spacing[2];

    vtkm::Vec<vtkm::Float32, 3> exit_dist;
    // distance to grid exit
    exit_dist[0] =
      (dir[0] != 0.f) ? (exit_boundary[0] - (point[0] - Origin[0])) / dir[0] : vtkm::Infinity32();
    exit_dist[1] =
      (dir[1] != 0.f) ? (exit_boundary[1] - (point[1] - Origin[1])) / dir[1] : vtkm::Infinity32();
    exit_dist[2] =
      (dir[2] != 0.f) ? (exit_boundary[2] - (point[2] - Origin[2])) / dir[2] : vtkm::Infinity32();
    return vtkm::Min(exit_dist[0], vtkm::Min(exit_dist[1], exit_dist[2]));
  }

  // returns the distance along the ray from initial query point
  // to exit the current voxel.
  VTKM_EXEC vtkm::Float32 Exit(vtkm::Vec<vtkm::Float32, 3>& deltaMax) const
  {
    return vtkm::Min(deltaMax[0], vtkm::Min(deltaMax[1], deltaMax[2]));
  }

  // advances to the next voxel along the ray
  VTKM_EXEC void Advance(vtkm::Vec<vtkm::Float32, 3>& deltaMax,
                         vtkm::Vec<vtkm::Float32, 3>& delta,
                         vtkm::Vec<vtkm::Int32, 3>& voxel,
                         const vtkm::Vec<vtkm::Float32, 3>& dir) const
  {
    vtkm::Int32 advanceDir = 0;
    for (vtkm::Int32 i = 1; i < 3; ++i)
    {
      if (deltaMax[i] < deltaMax[advanceDir])
      {
        advanceDir = i;
      }
    }
    deltaMax[advanceDir] += delta[advanceDir];
    voxel[advanceDir] += dir[advanceDir] < 0.f ? -1 : 1;

    /*
    VTKM_ASSERT(voxel[0] >= 0);
    VTKM_ASSERT(voxel[1] >= 0);
    VTKM_ASSERT(voxel[2] >= 0);

    VTKM_ASSERT(voxel[0] < Dims[0]);
    VTKM_ASSERT(voxel[1] < Dims[1]);
    VTKM_ASSERT(voxel[2] < Dims[2]);
    */
  }

  VTKM_EXEC
  vtkm::Vec<vtkm::Float32, 2> GetRange(const vtkm::Vec<Int32, 3>& voxel) const
  {
    vtkm::Id id = voxel[0] + Dims[0] * (voxel[1] + voxel[2] * Dims[1]);
    vtkm::Vec<vtkm::Float32, 2> minmax;
    minmax[0] = MinPortal.Get(id);
    minmax[1] = MaxPortal.Get(id);
    return minmax;
  }

  VTKM_EXEC
  bool IsValid(const vtkm::Vec<Int32, 3>& voxel) const
  {
    bool valid = true;
    if (voxel[0] < 0 || voxel[0] >= Dims[0] || voxel[1] < 0 || voxel[1] >= Dims[1] ||
        voxel[2] < 0 || voxel[2] >= Dims[2])
    {
      valid = false;
    }
    return valid;
  }
};

class MinMaxVoxelGrid : public vtkm::cont::ExecutionObjectBase
{
protected:
  vtkm::Id3 Dims;
  bool IsConstructed;

  vtkm::Vec<vtkm::Float32, 3> Origin;
  vtkm::Vec<vtkm::Float32, 3> Spacing;

  vtkm::cont::ArrayHandle<vtkm::Float32> MinField;
  vtkm::cont::ArrayHandle<vtkm::Float32> MaxField;

public:
  MinMaxVoxelGrid();

  void Construct(const vtkm::cont::DynamicCellSet& cellset,
                 const vtkm::cont::CoordinateSystem& coords,
                 const vtkm::cont::Field& field,
                 const vtkm::cont::ArrayHandle<vtkm::Vec<vtkm::Float32, 4>>& colorMap,
                 const vtkm::Range& scalarRange);

  void SetDims(vtkm::Id3 dims);
  vtkm::Id3 GetDims();

  template <typename Device>
  MinMaxExec<Device> PrepareForExecution(Device) const
  {
    return MinMaxExec<Device>(Dims, Origin, Spacing, MinField, MaxField);
  }
};

}
}
} //namespace vtkm::rendering::raytracing
#endif
