//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//  Copyright 2015 National Technology & Engineering Solutions of Sandia, LLC (NTESS).
//  Copyright 2015 UT-Battelle, LLC.
//  Copyright 2015 Los Alamos National Security.
//
//  Under the terms of Contract DE-NA0003525 with NTESS,
//  the U.S. Government retains certain rights in this software.
//
//  Under the terms of Contract DE-AC52-06NA25396 with Los Alamos National
//  Laboratory (LANL), the U.S. Government retains certain rights in
//  this software.
//============================================================================

#include <vtkm/rendering/raytracing/MeshOracleContainer.h>
#include <vtkm/rendering/raytracing/MeshOracleFactory.h>

namespace vtkm
{
namespace rendering
{
namespace raytracing
{

MeshOracleContainer* MeshOracleFactory::Create(const vtkm::cont::DynamicCellSet& cellset,
                                               const vtkm::cont::CoordinateSystem& coords)
{
  MeshOracleContainer* res = nullptr;

  using DefaultHandle = vtkm::cont::ArrayHandle<vtkm::FloatDefault>;
  using CartesianArrayHandle =
    vtkm::cont::ArrayHandleCartesianProduct<DefaultHandle, DefaultHandle, DefaultHandle>;

  using UniformHandle = vtkm::cont::ArrayHandleUniformPointCoordinates;
  enum MeshType
  {
    Unsupported = 0,
    Structured = 1,
    Unstructured = 2,
    UnstructuredSingle = 3
  };

  enum CoordType
  {
    Uniform = 0,
    Rectilinear = 1,
    Explicit = 2,
    Unknown = 3
  };

  MeshType meshType = Unsupported;

  // detect cell set
  if (cellset.IsSameType(vtkm::cont::CellSetStructured<3>()))
  {

    meshType = Structured;
    //coords.GetData().IsSameType(CartesianArrayHandle())
  }
  else if (cellset.IsSameType(vtkm::cont::CellSetSingleType<>()))
  {
    meshType = UnstructuredSingle;
  }
  else if (cellset.IsSameType(vtkm::cont::CellSetExplicit<>()))
  {
    meshType = Unstructured;
  }

  CoordType coordType = Unknown;
  // detect coordinate system
  if (coords.GetData().IsType<UniformHandle>())
  {
    coordType = Uniform;
  }
  else if (coords.GetData().IsType<CartesianArrayHandle>())
  {
    coordType = Rectilinear;
  }

  if (meshType == Structured && coordType == Rectilinear)
  {
    vtkm::cont::CellSetStructured<3> cells = cellset.Cast<vtkm::cont::CellSetStructured<3>>();
    CartesianArrayHandle cart = coords.GetData().Cast<CartesianArrayHandle>();
    res = new RectilinearContainer(cells, cart);
  }

  return res;
}

}
}
} //namespace vtkm::rendering::raytracing
