//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//  Copyright 2015 National Technology & Engineering Solutions of Sandia, LLC (NTESS).
//  Copyright 2015 UT-Battelle, LLC.
//  Copyright 2015 Los Alamos National Security.
//
//  Under the terms of Contract DE-NA0003525 with NTESS,
//  the U.S. Government retains certain rights in this software.
//
//  Under the terms of Contract DE-AC52-06NA25396 with Los Alamos National
//  Laboratory (LANL), the U.S. Government retains certain rights in
//  this software.
//============================================================================
#include <sstream>
#include <vtkm/CellShape.h>
#include <vtkm/cont/ErrorBadValue.h>
#include <vtkm/cont/DataSet.h>
#include <vtkm/cont/Timer.h>
#include <vtkm/cont/internal/DeviceAdapterListHelpers.h>
#include <vtkm/rendering/raytracing/MeshOracleBase.h>
#include <vtkm/rendering/raytracing/MeshOracleContainer.h>

namespace vtkm
{
namespace rendering
{
namespace raytracing
{

MeshOracleContainer::MeshOracleContainer(){};
MeshOracleContainer::~MeshOracleContainer(){};

MeshOracleWrapper
MeshOracleContainer::PrepareForExecution(const vtkm::cont::DeviceAdapterId deviceId)
{
  return MeshOracleWrapper(const_cast<MeshOracleBase*>(this->Construct(deviceId)));
}

VTKM_CONT
RectilinearContainer::RectilinearContainer(const vtkm::cont::CellSetStructured<3>& cellset,
                                           const CartesianArrayHandle& coords)
  : CellSet(cellset)
  , Coords(coords)
{
}

RectilinearContainer::~RectilinearContainer(){};

const MeshOracleBase* RectilinearContainer::Construct(const vtkm::cont::DeviceAdapterId deviceId)
{
  switch (deviceId.GetValue())
  {
#ifdef VTKM_ENABLE_OPENMP
    case VTKM_DEVICE_ADAPTER_OPENMP:
      using OMP = vtkm::cont::DeviceAdapterTagOpenMP;
      {
        RectilinearOracle<OMP> oracle(this->Coords, this->CellSet);
        Handle = make_OracleHandle(oracle);
      }
      return Handle.PrepareForExecution(OMP());
#endif
#ifdef VTKM_ENABLE_TBB
    case VTKM_DEVICE_ADAPTER_TBB:
      using TBB = vtkm::cont::DeviceAdapterTagTBB;
      {
        RectilinearOracle<TBB> oracle(this->Coords, this->CellSet);
        Handle = make_OracleHandle(oracle);
      }
      return Handle.PrepareForExecution(TBB());
#endif
#ifdef VTKM_ENABLE_CUDA
    case VTKM_DEVICE_ADAPTER_CUDA:
      using CUDA = vtkm::cont::DeviceAdapterTagCuda;
      {
        RectilinearOracle<CUDA> oracle(this->Coords, this->CellSet);
        Handle = make_OracleHandle(oracle);
      }
      return Handle.PrepareForExecution(CUDA());
#endif
    default:
      using SERIAL = vtkm::cont::DeviceAdapterTagSerial;
      {
        RectilinearOracle<SERIAL> oracle(this->Coords, this->CellSet);
        Handle = make_OracleHandle(oracle);
      }
      return Handle.PrepareForExecution(SERIAL());
  }
}

//VTKM_CONT
//UnstructuredSingleContainer::UnstructuredSingleContainer()
//{
//}
//
//VTKM_CONT
//UnstructuredSingleContainer::UnstructuredSingleContainer(
//  const vtkm::cont::CellSetSingleType<>& cellset,
//  const vtkm::cont::CoordinateSystem& coords,
//  IdHandle& faceConn,
//  Id4Handle& externalTriangles)
//  : FaceConnectivity(faceConn)
//  , Coords(coords)
//  , Cellset(cellset)
//{
//
//  this->ExternalTriangles = externalTriangles;
//
//  this->Intersector.SetUseWaterTight(true);
//
//  CellConnectivity =
//    Cellset.GetConnectivityArray(vtkm::TopologyElementTagPoint(), vtkm::TopologyElementTagCell());
//  vtkm::cont::ArrayHandleConstant<vtkm::UInt8> shapes =
//    Cellset.GetShapesArray(vtkm::TopologyElementTagPoint(), vtkm::TopologyElementTagCell());
//
//  ShapeId = shapes.GetPortalConstControl().Get(0);
//  CellTables tables;
//  NumIndices = tables.FaceLookUp(tables.CellTypeLookUp(ShapeId), 2);
//
//  if (NumIndices == 0)
//  {
//    std::stringstream message;
//    message << "Unstructured Mesh Connecitity Single type Error: unsupported cell type: ";
//    message << ShapeId;
//    throw vtkm::cont::ErrorBadValue(message.str());
//  }
//  vtkm::Id start = 0;
//  NumFaces = tables.FaceLookUp(tables.CellTypeLookUp(ShapeId), 1);
//  vtkm::Id numCells = CellConnectivity.GetPortalConstControl().GetNumberOfValues();
//  CellOffsets = vtkm::cont::make_ArrayHandleCounting<vtkm::Id>(start, NumIndices, numCells);
//
//  Logger* logger = Logger::GetInstance();
//  logger->OpenLogEntry("mesh_conn_construction");
//  vtkm::cont::Timer<cont::DeviceAdapterTagSerial> timer;
//
//  Intersector.SetData(Coords, ExternalTriangles);
//}
//
//const MeshOracleBase* UnstructuredSingleContainer::Construct(
//  const vtkm::cont::DeviceAdapterId deviceId)
//{
//  switch (deviceId.GetValue())
//  {
//#ifdef VTKM_ENABLE_OPENMP
//    case VTKM_DEVICE_ADAPTER_OPENMP:
//      using OMP = vtkm::cont::DeviceAdapterTagOpenMP;
//      {
//        MeshConnSingleType<OMP> conn(this->FaceConnectivity,
//                                     this->CellConnectivity,
//                                     this->CellOffsets,
//                                     this->ShapeId,
//                                     this->NumIndices,
//                                     this->NumFaces);
//        Handle = make_OracleHandle(conn);
//      }
//      return Handle.PrepareForExecution(OMP());
//#endif
//#ifdef VTKM_ENABLE_TBB
//    case VTKM_DEVICE_ADAPTER_TBB:
//      using TBB = vtkm::cont::DeviceAdapterTagTBB;
//      {
//        MeshConnSingleType<TBB> conn(this->FaceConnectivity,
//                                     this->CellConnectivity,
//                                     this->CellOffsets,
//                                     this->ShapeId,
//                                     this->NumIndices,
//                                     this->NumFaces);
//        Handle = make_OracleHandle(conn);
//      }
//      return Handle.PrepareForExecution(TBB());
//#endif
//#ifdef VTKM_ENABLE_CUDA
//    case VTKM_DEVICE_ADAPTER_CUDA:
//      using CUDA = vtkm::cont::DeviceAdapterTagCuda;
//      {
//        MeshConnSingleType<CUDA> conn(this->FaceConnectivity,
//                                      this->CellConnectivity,
//                                      this->CellOffsets,
//                                      this->ShapeId,
//                                      this->NumIndices,
//                                      this->NumFaces);
//        Handle = make_OracleHandle(conn);
//      }
//      return Handle.PrepareForExecution(CUDA());
//#endif
//    default:
//      using SERIAL = vtkm::cont::DeviceAdapterTagSerial;
//      {
//        MeshConnSingleType<SERIAL> conn(this->FaceConnectivity,
//                                        this->CellConnectivity,
//                                        this->CellOffsets,
//                                        this->ShapeId,
//                                        this->NumIndices,
//                                        this->NumFaces);
//        Handle = make_OracleHandle(conn);
//      }
//      return Handle.PrepareForExecution(SERIAL());
//  }
//}
//
//StructuredContainer::StructuredContainer(const vtkm::cont::CellSetStructured<3>& cellset,
//                                         const vtkm::cont::CoordinateSystem& coords,
//                                         Id4Handle& externalTriangles)
//  : Coords(coords)
//  , Cellset(cellset)
//{
//
//  ExternalTriangles = externalTriangles;
//  Intersector.SetUseWaterTight(true);
//
//  PointDims = Cellset.GetPointDimensions();
//  CellDims = Cellset.GetCellDimensions();
//
//  this->Intersector.SetData(Coords, ExternalTriangles);
//}
//
//const MeshOracleBase* StructuredContainer::Construct(
//  const vtkm::cont::DeviceAdapterId deviceId)
//{
//
//  MeshConnStructured conn(CellDims, PointDims);
//  Handle = make_OracleHandle(conn);
//
//  switch (deviceId.GetValue())
//  {
//#ifdef VTKM_ENABLE_OPENMP
//    case VTKM_DEVICE_ADAPTER_OPENMP:
//      return Handle.PrepareForExecution(vtkm::cont::DeviceAdapterTagOpenMP());
//#endif
//#ifdef VTKM_ENABLE_TBB
//    case VTKM_DEVICE_ADAPTER_TBB:
//      return Handle.PrepareForExecution(vtkm::cont::DeviceAdapterTagTBB());
//#endif
//#ifdef VTKM_ENABLE_CUDA
//    case VTKM_DEVICE_ADAPTER_CUDA:
//      return Handle.PrepareForExecution(vtkm::cont::DeviceAdapterTagCuda());
//#endif
//    default:
//      return Handle.PrepareForExecution(vtkm::cont::DeviceAdapterTagSerial());
//  }
//}
}
}
} //namespace vtkm::rendering::raytracing
