//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//  Copyright 2015 National Technology & Engineering Solutions of Sandia, LLC (NTESS).
//  Copyright 2015 UT-Battelle, LLC.
//  Copyright 2015 Los Alamos National Security.
//
//  Under the terms of Contract DE-NA0003525 with NTESS,
//  the U.S. Government retains certain rights in this software.
//
//  Under the terms of Contract DE-AC52-06NA25396 with Los Alamos National
//  Laboratory (LANL), the U.S. Government retains certain rights in
//  this software.
//============================================================================
#ifndef vtk_m_rendering_raytracing_Backgrounds_h
#define vtk_m_rendering_raytracing_Backgrounds_h

#include <vtkm/cont/ArrayHandleConstant.h>
#include <vtkm/rendering/raytracing/Ray.h>

#include <vtkm/worklet/DispatcherMapField.h>
#include <vtkm/worklet/WorkletMapField.h>

#include <vector>

namespace vtkm
{
namespace rendering
{
namespace raytracing
{
namespace detail
{
class ConstantColor : public vtkm::worklet::WorkletMapField
{
public:
  VTKM_CONT
  ConstantColor() {}
  using ControlSignature = void(FieldIn, WholeArrayInOut, WholeArrayInOut);
  using ExecutionSignature = void(_1, _2, _3, WorkIndex);

  template <typename BufferType>
  VTKM_EXEC inline void operator()(const vtkm::Vec<vtkm::Float32, 3>& bgColor,
                                   BufferType& frameBuffer,
                                   BufferType& throughputBuffer,
                                   const vtkm::Id& pixelIndex) const
  {
    vtkm::Vec<vtkm::Float32, 4> color;
    BOUNDS_CHECK(frameBuffer, pixelIndex * 4 + 0);
    color[0] = static_cast<vtkm::Float32>(frameBuffer.Get(pixelIndex * 4 + 0));
    BOUNDS_CHECK(frameBuffer, pixelIndex * 4 + 1);
    color[1] = static_cast<vtkm::Float32>(frameBuffer.Get(pixelIndex * 4 + 1));
    BOUNDS_CHECK(frameBuffer, pixelIndex * 4 + 2);
    color[2] = static_cast<vtkm::Float32>(frameBuffer.Get(pixelIndex * 4 + 2));
    BOUNDS_CHECK(frameBuffer, pixelIndex * 4 + 3);
    color[3] = static_cast<vtkm::Float32>(frameBuffer.Get(pixelIndex * 4 + 3));

    vtkm::Vec<vtkm::Float32, 3> throughput;
    BOUNDS_CHECK(throughputBuffer, pixelIndex * 3 + 0);
    throughput[0] = static_cast<vtkm::Float32>(throughputBuffer.Get(pixelIndex * 3 + 0));
    BOUNDS_CHECK(throughputBuffer, pixelIndex * 3 + 1);
    throughput[1] = static_cast<vtkm::Float32>(throughputBuffer.Get(pixelIndex * 3 + 1));
    BOUNDS_CHECK(throughputBuffer, pixelIndex * 3 + 2);
    throughput[2] = static_cast<vtkm::Float32>(throughputBuffer.Get(pixelIndex * 3 + 2));

    vtkm::Float32 inv_pdf = 4.f * vtkm::Pif();
    color[0] += throughput[0] * bgColor[0]; //* inv_pdf;
    color[1] += throughput[1] * bgColor[1]; //* inv_pdf;
    color[2] += throughput[2] * bgColor[2]; //* inv_pdf;
    color[3] = 1.f;

    BOUNDS_CHECK(frameBuffer, pixelIndex * 4 + 0);
    frameBuffer.Set(pixelIndex * 4 + 0, color[0]);
    BOUNDS_CHECK(frameBuffer, pixelIndex * 4 + 1);
    frameBuffer.Set(pixelIndex * 4 + 1, color[1]);
    BOUNDS_CHECK(frameBuffer, pixelIndex * 4 + 2);
    frameBuffer.Set(pixelIndex * 4 + 2, color[2]);
    BOUNDS_CHECK(frameBuffer, pixelIndex * 4 + 3);
    frameBuffer.Set(pixelIndex * 4 + 3, color[3]);
  }
}; //class ConstantColor
} // namespace detail

class Backgrounds
{
public:
  template <typename FloatType>
  static void ConstantColor(Ray<FloatType>& rays, const vtkm::Vec<vtkm::Float32, 3> color)
  {
    vtkm::cont::ArrayHandleConstant<vtkm::Vec<vtkm::Float32, 3>> colorHandle(color, rays.NumRays);

    vtkm::worklet::DispatcherMapField<detail::ConstantColor> dispatcher;
    dispatcher.Invoke(colorHandle, rays.Buffers.at(0).Buffer, rays.GetBuffer("throughput").Buffer);
  }
};
}
}
} //namespace vtkm::rendering::raytracing
#endif //vtk_m_rendering_raytracing_Ray_h
