//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//  Copyright 2016 National Technology & Engineering Solutions of Sandia, LLC (NTESS).
//  Copyright 2016 UT-Battelle, LLC.
//  Copyright 2016 Los Alamos National Security.
//
//  Under the terms of Contract DE-NA0003525 with NTESS,
//  the U.S. Government retains certain rights in this software.
//
//  Under the terms of Contract DE-AC52-06NA25396 with Los Alamos National
//  Laboratory (LANL), the U.S. Government retains certain rights in
//  this software.
//============================================================================

#include <vtkm/rendering/MapperTracker.h>

#include <vtkm/cont/Timer.h>
#include <vtkm/cont/TryExecute.h>

#include <vtkm/rendering/CanvasRayTracer.h>

#include <vtkm/rendering/raytracing/Camera.h>
#include <vtkm/rendering/raytracing/Logger.h>
#include <vtkm/rendering/raytracing/RayOperations.h>
#include <vtkm/rendering/raytracing/ResidualTracker.h>
#include <vtkm/rendering/raytracing/VolumeRendererStructured.h>

#include <sstream>

#define DEFAULT_SAMPLE_DISTANCE -1.f

namespace vtkm
{
namespace rendering
{

struct MapperTracker::InternalsType
{
  vtkm::rendering::CanvasRayTracer* Canvas;
  vtkm::rendering::raytracing::ResidualTracker Tracer;
  vtkm::Float32 SampleDistance;
  bool CompositeBackground;

  VTKM_CONT
  InternalsType()
    : Canvas(nullptr)
    , SampleDistance(DEFAULT_SAMPLE_DISTANCE)
    , CompositeBackground(true)
  {
  }
};

MapperTracker::MapperTracker()
  : Internals(new InternalsType)
{
}

MapperTracker::~MapperTracker() {}


void MapperTracker::AddLight(const vtkm::Float32& radius,
                             const vtkm::Vec<vtkm::Float32, 3>& position,
                             const vtkm::Vec<vtkm::Float32, 3>& color)
{
  this->Internals->Tracer.AddLight(radius, position, color);
}

void MapperTracker::SetCanvas(vtkm::rendering::Canvas* canvas)
{
  if (canvas != nullptr)
  {
    this->Internals->Canvas = dynamic_cast<CanvasRayTracer*>(canvas);

    if (this->Internals->Canvas == nullptr)
    {
      throw vtkm::cont::ErrorBadValue("Ray Tracer: bad canvas type. Must be CanvasRayTracer");
    }
  }
  else
  {
    this->Internals->Canvas = nullptr;
  }
}

vtkm::rendering::Canvas* MapperTracker::GetCanvas() const
{
  return this->Internals->Canvas;
}

void MapperTracker::RenderCells(const vtkm::cont::DynamicCellSet& cellset,
                                const vtkm::cont::CoordinateSystem& coords,
                                const vtkm::cont::Field& scalarField,
                                const vtkm::cont::ColorTable& vtkmNotUsed(colorTable),
                                const vtkm::rendering::Camera& camera,
                                const vtkm::Range& scalarRange)
{
  if (!cellset.IsSameType(vtkm::cont::CellSetStructured<3>()))
  {
    std::stringstream msg;
    std::string theType = typeid(cellset).name();
    msg << "Mapper volume: cell set type not currently supported\n";
    msg << "Type : " << theType << std::endl;
    throw vtkm::cont::ErrorBadValue(msg.str());
  }
  else
  {
    raytracing::Logger* logger = raytracing::Logger::GetInstance();
    logger->OpenLogEntry("mapper_volume");
    vtkm::cont::Timer tot_timer;
    vtkm::cont::Timer timer;
    tot_timer.Start();
    timer.Start();

    vtkm::rendering::raytracing::Camera rayCamera;
    vtkm::rendering::raytracing::Ray<vtkm::Float32> rays;

    std::cerr << "1\n";
    rayCamera.SetParameters(camera, *this->Internals->Canvas);
    std::cerr << "2\n";

    rayCamera.CreateRays(rays, coords.GetBounds());
    rays.Buffers.at(0).InitConst(0.f);
    raytracing::RayOperations::MapCanvasToRays(rays, camera, *this->Internals->Canvas);
    std::cerr << "3\n";

    this->Internals->Tracer.SetData(
      coords, scalarField, cellset.Cast<vtkm::cont::CellSetStructured<3>>(), scalarRange);
    std::cerr << "3.1\n";
    this->Internals->Tracer.SetColorMap(this->ColorMap);
    std::cerr << "4\n";

    this->Internals->Tracer.Render(rays);
    std::cerr << "\n5\n";

    timer.Reset();
    timer.Start();
    this->Internals->Canvas->WriteToCanvas(rays, rays.Buffers.at(0).Buffer, camera);
    std::cerr << "\n6\n";

    if (this->Internals->CompositeBackground)
    {
      this->Internals->Canvas->BlendBackground();
    }
    std::cerr << "7\n";
    vtkm::Float64 time = timer.GetElapsedTime();
    logger->AddLogData("write_to_canvas", time);
    time = tot_timer.GetElapsedTime();
    logger->CloseLogEntry(time);
  }
}

/*
void MapperTracker::StartScene()
{
  // Nothing needs to be done.
}

void MapperTracker::EndScene()
{
  // Nothing needs to be done.
}
*/

vtkm::rendering::Mapper* MapperTracker::NewCopy() const
{
  return new vtkm::rendering::MapperTracker(*this);
}

void MapperTracker::SetSamples(const vtkm::Int32 samples)
{
  this->Internals->Tracer.SetSamples(samples);
}

void MapperTracker::SetDensityScale(const vtkm::Float32 scale)
{
  this->Internals->Tracer.SetDensityScale(scale);
}

void MapperTracker::SetCompositeBackground(const bool compositeBackground)
{
  this->Internals->CompositeBackground = compositeBackground;
}
}
} // namespace vtkm::rendering
