/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkMergeFilter.cxx
  Language:  C++
  Date:      $Date$
  Version:   $Revision$


Copyright (c) 1993-1998 Ken Martin, Will Schroeder, Bill Lorensen.

This software is copyrighted by Ken Martin, Will Schroeder and Bill Lorensen.
The following terms apply to all files associated with the software unless
explicitly disclaimed in individual files. This copyright specifically does
not apply to the related textbook "The Visualization Toolkit" ISBN
013199837-4 published by Prentice Hall which is covered by its own copyright.

The authors hereby grant permission to use, copy, and distribute this
software and its documentation for any purpose, provided that existing
copyright notices are retained in all copies and that this notice is included
verbatim in any distributions. Additionally, the authors grant permission to
modify this software and its documentation for any purpose, provided that
such modifications are not distributed without the explicit consent of the
authors and that existing copyright notices are retained in all copies. Some
of the algorithms implemented by this software are patented, observe all
applicable patent law.

IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY FOR
DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY DERIVATIVES THEREOF,
EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE IS PROVIDED ON AN
"AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE
MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.


=========================================================================*/
#include "vtkMergeFilter.h"
#include "vtkPolyData.h"
#include "vtkStructuredGrid.h"
#include "vtkStructuredPoints.h"
#include "vtkUnstructuredGrid.h"
#include "vtkRectilinearGrid.h"

// Create object with no input or output.
vtkMergeFilter::vtkMergeFilter()
{
  this->Geometry = NULL;
  this->Scalars = NULL;
  this->Vectors = NULL;
  this->Normals = NULL;
  this->TCoords = NULL;
  this->Tensors = NULL;
  this->FieldData = NULL;

  this->Output = NULL;

  this->PolyData = vtkPolyData::New();
  this->PolyData->SetSource(this);
  
  this->StructuredPoints = vtkStructuredPoints::New();
  this->StructuredPoints->SetSource(this);
  
  this->StructuredGrid = vtkStructuredGrid::New();
  this->StructuredGrid->SetSource(this);
  
  this->UnstructuredGrid = vtkUnstructuredGrid::New();
  this->UnstructuredGrid->SetSource(this);
  
  this->RectilinearGrid = vtkRectilinearGrid::New();
  this->RectilinearGrid->SetSource(this);
}

vtkMergeFilter::~vtkMergeFilter()
{
  if (this->Geometry) {this->Geometry->UnRegister(this);}
  this->Geometry = NULL;

  this->PolyData->Delete();
  this->StructuredPoints->Delete();
  this->StructuredGrid->Delete();
  this->UnstructuredGrid->Delete();
  this->RectilinearGrid->Delete();
  // Output should only be one of the above. We set it to NULL
  // so that we don't free it twice
  this->Output = NULL;
  
  this->SetScalars((vtkDataSet *)NULL);
  this->SetVectors(NULL);
  this->SetNormals(NULL);
  this->SetTCoords(NULL);
  this->SetTensors(NULL);
  this->SetFieldData(NULL);  
}

void vtkMergeFilter::SetGeometry(vtkDataSet *input)
{
  if ( this->Geometry != input )
    {
    vtkDebugMacro(<<" setting Geometry to " << (void *)input);
    if (this->Geometry) {this->Geometry->UnRegister(this);}
    this->Geometry = input;
    if (this->Geometry) {this->Geometry->Register(this);}
    this->Modified();
    
    if ( input->GetDataSetType() == VTK_POLY_DATA )
      {
      this->Output = this->PolyData;
      }

    else if ( input->GetDataSetType() == VTK_STRUCTURED_POINTS )
      {
      this->Output = this->StructuredPoints;
      }

    else if ( input->GetDataSetType() == VTK_STRUCTURED_GRID )
      {
      this->Output = this->StructuredGrid;
      }

    else if ( input->GetDataSetType() == VTK_UNSTRUCTURED_GRID )
      {
      this->Output = this->UnstructuredGrid;
      }

    else if ( input->GetDataSetType() == VTK_RECTILINEAR_GRID )
      {
      this->Output = this->RectilinearGrid;
      }

    else
      {
      vtkErrorMacro(<<"Mismatch in data type");
      }

    this->Modified();
    }
}

void vtkMergeFilter::Update()
{
  unsigned long int mtime=0, dsMtime;

  // make sure geometry is defined
  if ( this->Geometry == NULL )
    {
    vtkErrorMacro(<< "No geometry input...can't execute!");
    return;
    }

  // prevent chasing our tail
  if (this->Updating)
    {
    return;
    }

  this->Updating = 1;
  this->Geometry->Update();
  mtime = this->Geometry->GetMTime();
  
  if ( this->Scalars ) 
    {
    this->Scalars->Update();
    dsMtime = this->Scalars->GetMTime();
    if ( dsMtime > mtime )
      {
      mtime = dsMtime;
      }
    }
  if ( this->Vectors )
    {
    this->Vectors->Update();
    dsMtime = this->Vectors->GetMTime();
    if ( dsMtime > mtime )
      {
      mtime = dsMtime;
      }
    }
  if ( this->Normals )
    {
    this->Normals->Update();
    dsMtime = this->Normals->GetMTime();
    if ( dsMtime > mtime )
      {
      mtime = dsMtime;
      }
    }
  if ( this->TCoords )
    {
    this->TCoords->Update();
    dsMtime = this->TCoords->GetMTime();
    if ( dsMtime > mtime )
      {
      mtime = dsMtime;
      }
    }
  if ( this->Tensors )
    {
    this->Tensors->Update();
    dsMtime = this->Tensors->GetMTime();
    if ( dsMtime > mtime )
      {
      mtime = dsMtime;
      }
    }
  if ( this->FieldData )
    {
    this->FieldData->Update();
    dsMtime = this->FieldData->GetMTime();
    if ( dsMtime > mtime )
      {
      mtime = dsMtime;
      }
    }
  this->Updating = 0;

  if ( mtime > this->ExecuteTime || this->GetMTime() > this->ExecuteTime )
    {
    if ( this->Geometry->GetDataReleased() )
      {
      this->Geometry->ForceUpdate();
      }
    if ( this->Scalars && this->Scalars->GetDataReleased() ) 
      {
      this->Scalars->ForceUpdate();
      }
    if ( this->Vectors && this->Vectors->GetDataReleased() ) 
      {
      this->Vectors->ForceUpdate();
      }
    if ( this->Normals && this->Normals->GetDataReleased() ) 
      {
      this->Normals->ForceUpdate();
      }
    if ( this->TCoords && this->TCoords->GetDataReleased() ) 
      {
      this->TCoords->ForceUpdate();
      }
    if ( this->Tensors && this->Tensors->GetDataReleased() ) 
      {
      this->Tensors->ForceUpdate();
      }
    if ( this->FieldData && this->FieldData->GetDataReleased() ) 
      {
      this->FieldData->ForceUpdate();
      }

    if ( this->StartMethod )
      {
      (*this->StartMethod)(this->StartMethodArg);
      }
    this->Output->Initialize(); //clear output
    // reset AbortExecute flag and Progress
    this->AbortExecute = 0;
    this->Progress = 0.0;
    this->Execute();
    this->ExecuteTime.Modified();
    if ( !this->AbortExecute )
      {
      this->UpdateProgress(1.0);
      }
    this->SetDataReleased(0);
    if ( this->EndMethod )
      {
      (*this->EndMethod)(this->EndMethodArg);
      }
    }
  
  if ( this->Geometry->ShouldIReleaseData() )
    {
    this->Geometry->ReleaseData();
    }

  if ( this->Scalars && this->Scalars->ShouldIReleaseData() ) 
    {
    this->Scalars->ReleaseData();
    }

  if ( this->Vectors && this->Vectors->ShouldIReleaseData() ) 
    {
    this->Vectors->ReleaseData();
    }

  if ( this->Normals && this->Normals->ShouldIReleaseData() ) 
    {
    this->Normals->ReleaseData();
    }

  if ( this->TCoords && this->TCoords->ShouldIReleaseData() ) 
    {
    this->TCoords->ReleaseData();
    }

  if ( this->Tensors && this->Tensors->ShouldIReleaseData() ) 
    {
    this->Tensors->ReleaseData();
    }

  if ( this->FieldData && this->FieldData->ShouldIReleaseData() ) 
    {
    this->FieldData->ReleaseData();
    }
}

// Merge it all together
void vtkMergeFilter::Execute()
{
  int numPts, numScalars=0, numVectors=0, numNormals=0, numTCoords=0;
  int numTensors=0, numTuples=0;
  int numCells, numCellScalars=0, numCellVectors=0, numCellNormals=0;
  int numCellTCoords=0, numCellTensors=0, numCellTuples=0;
  vtkPointData *pd;
  vtkScalars *scalars = NULL;
  vtkVectors *vectors = NULL;
  vtkNormals *normals = NULL;
  vtkTCoords *tcoords = NULL;
  vtkTensors *tensors = NULL;
  vtkFieldData *f = NULL;
  vtkCellData *cd;
  vtkScalars *cellScalars = NULL;
  vtkVectors *cellVectors = NULL;
  vtkNormals *cellNormals = NULL;
  vtkTCoords *cellTCoords = NULL;
  vtkTensors *cellTensors = NULL;
  vtkFieldData *cellf = NULL;
  vtkDataSet *output = (vtkDataSet *)this->Output;
  vtkPointData *outputPD = output->GetPointData();
  vtkCellData *outputCD = output->GetCellData();
  
  vtkDebugMacro(<<"Merging data!");

  // geometry needs to be copied
  output->CopyStructure(this->Geometry);
  if ( (numPts = this->Geometry->GetNumberOfPoints()) < 1 )
    {
    vtkWarningMacro(<<"Nothing to merge!");
    }
  numCells = this->Geometry->GetNumberOfCells();
  
  if ( this->Scalars ) 
    {
    pd = this->Scalars->GetPointData();
    scalars = pd->GetScalars();
    if ( scalars != NULL )
      {
      numScalars = scalars->GetNumberOfScalars();
      }
    cd = this->Scalars->GetCellData();
    cellScalars = cd->GetScalars();
    if ( cellScalars != NULL )
      {
      numCellScalars = cellScalars->GetNumberOfScalars();
      }
    }

  if ( this->Vectors ) 
    {
    pd = this->Vectors->GetPointData();
    vectors = pd->GetVectors();
    if ( vectors != NULL )
      {
      numVectors= vectors->GetNumberOfVectors();
      }
    cd = this->Vectors->GetCellData();
    cellVectors = cd->GetVectors();
    if ( cellVectors != NULL )
      {
      numCellVectors = cellVectors->GetNumberOfVectors();
      }
    }

  if ( this->Normals ) 
    {
    pd = this->Normals->GetPointData();
    normals = pd->GetNormals();
    if ( normals != NULL )
      {
      numNormals= normals->GetNumberOfNormals();
      }
    cd = this->Normals->GetCellData();
    cellNormals = cd->GetNormals();
    if ( cellNormals != NULL )
      {
      numCellNormals = cellNormals->GetNumberOfNormals();
      }
    }

  if ( this->TCoords ) 
    {
    pd = this->TCoords->GetPointData();
    tcoords = pd->GetTCoords();
    if ( tcoords != NULL )
      {
      numTCoords= tcoords->GetNumberOfTCoords();
      }
    cd = this->TCoords->GetCellData();
    cellTCoords = cd->GetTCoords();
    if ( cellTCoords != NULL )
      {
      numCellTCoords = cellTCoords->GetNumberOfTCoords();
      }
    }

  if ( this->Tensors ) 
    {
    pd = this->Tensors->GetPointData();
    tensors = pd->GetTensors();
    if ( tensors != NULL )
      {
      numTensors = tensors->GetNumberOfTensors();
      }
    cd = this->Tensors->GetCellData();
    cellTensors = cd->GetTensors();
    if ( cellTensors != NULL )
      {
      numCellTensors = cellTensors->GetNumberOfTensors();
      }
    }

  if ( this->FieldData ) 
    {
    pd = this->FieldData->GetPointData();
    f = pd->GetFieldData();
    if ( f != NULL )
      {
      numTuples = f->GetNumberOfTuples();
      }
    cd = this->FieldData->GetCellData();
    cellf = cd->GetFieldData();
    if ( cellf != NULL )
      {
      numCellTuples = cellf->GetNumberOfTuples();
      }
    }

  // merge data only if it is consistent
  if ( numPts == numScalars )
    {
    outputPD->SetScalars(scalars);
    }
  if ( numCells == numCellScalars )
    {
    outputCD->SetScalars(cellScalars);
    }

  if ( numPts == numVectors )
    {
    outputPD->SetVectors(vectors);
    }
  if ( numCells == numCellVectors )
    {
    outputCD->SetVectors(cellVectors);
    }
    
  if ( numPts == numNormals )
    {
    outputPD->SetNormals(normals);
    }
  if ( numCells == numCellNormals )
    {
    outputCD->SetNormals(cellNormals);
    }

  if ( numPts == numTCoords )
    {
    outputPD->SetTCoords(tcoords);
    }
  if ( numCells == numCellTCoords )
    {
    outputCD->SetTCoords(cellTCoords);
    }

  if ( numPts == numTensors )
    {
    outputPD->SetTensors(tensors);
    }
  if ( numCells == numCellTensors )
    {
    outputCD->SetTensors(cellTensors);
    }

  if ( numPts == numTuples )
    {
    outputPD->SetFieldData(f);
    }
  if ( numCells == numCellTuples )
    {
    outputCD->SetFieldData(cellf);
    }
}

// Get the output as vtkPolyData.
vtkPolyData *vtkMergeFilter::GetPolyDataOutput() 
{
  return this->PolyData;
}

// Get the output as vtkStructuredPoints.
vtkStructuredPoints *vtkMergeFilter::GetStructuredPointsOutput() 
{
  return this->StructuredPoints;
}

// Get the output as vtkStructuredGrid.
vtkStructuredGrid *vtkMergeFilter::GetStructuredGridOutput()
{
  return this->StructuredGrid;
}

// Get the output as vtkUnstructuredGrid.
vtkUnstructuredGrid *vtkMergeFilter::GetUnstructuredGridOutput()
{
  return this->UnstructuredGrid;
}

// Get the output as vtkRectilinearGrid. 
vtkRectilinearGrid *vtkMergeFilter::GetRectilinearGridOutput()
{
  return this->RectilinearGrid;
}

void vtkMergeFilter::PrintSelf(ostream& os, vtkIndent indent)
{
  vtkFilter::PrintSelf(os,indent);

  if ( this->Geometry )
    {
    os << indent << "Geometry: (" << this->Geometry << ")\n";
    os << indent << "Geometry type: " << this->Geometry->GetClassName() << "\n";
    }
  else
    {
    os << indent << "Geometry: (none)\n";
    }

  if ( this->Scalars )
    {
    os << indent << "Scalars: (" << this->Scalars << ")\n";
    }
  else
    {
    os << indent << "Scalars: (none)\n";
    }

  if ( this->Vectors )
    {
    os << indent << "Vectors: (" << this->Vectors << ")\n";
    }
  else
    {
    os << indent << "Vectors: (none)\n";
    }

  if ( this->Normals )
    {
    os << indent << "Normals: (" << this->Normals << ")\n";
    }
  else
    {
    os << indent << "Normals: (none)\n";
    }

  if ( this->TCoords )
    {
    os << indent << "TCoords: (" << this->TCoords << ")\n";
    }
  else
    {
    os << indent << "TCoords: (none)\n";
    }

  if ( this->Tensors )
    {
    os << indent << "Tensors: (" << this->Tensors << ")\n";
    }
  else
    {
    os << indent << "Tensors: (none)\n";
    }

  if ( this->FieldData )
    {
    os << indent << "Field Data: (" << this->FieldData << ")\n";
    }
  else
    {
    os << indent << "Field Data: (none)\n";
    }
}


void vtkMergeFilter::UnRegister(vtkObject *o)
{
  // detect the circular loop source <-> data
  // If we have two references and one of them is my data
  // and I am not being unregistered by my data, break the loop.
  if (this->ReferenceCount == 6 &&
      this->PolyData != o && this->StructuredGrid != o &&
      this->UnstructuredGrid != o && this->StructuredPoints != o &&
      this->RectilinearGrid != o &&
      this->PolyData->GetNetReferenceCount() == 1 &&
      this->StructuredGrid->GetNetReferenceCount() == 1 &&
      this->UnstructuredGrid->GetNetReferenceCount() == 1 &&
      this->StructuredPoints->GetNetReferenceCount() == 1 &&
      this->RectilinearGrid->GetNetReferenceCount() == 1)
    {
    this->PolyData->SetSource(NULL);
    this->StructuredGrid->SetSource(NULL);
    this->UnstructuredGrid->SetSource(NULL);
    this->StructuredPoints->SetSource(NULL);
    this->RectilinearGrid->SetSource(NULL);
    }
  if (this->ReferenceCount == 5 &&
      (this->PolyData == o || this->StructuredGrid == o ||
       this->UnstructuredGrid == o || this->RectilinearGrid == o ||
       this->StructuredPoints == o) &&
      (this->PolyData->GetNetReferenceCount() +
       this->StructuredPoints->GetNetReferenceCount() +
       this->RectilinearGrid->GetNetReferenceCount() +
       this->StructuredGrid->GetNetReferenceCount() +
       this->UnstructuredGrid->GetNetReferenceCount()) == 6)
    {
    this->PolyData->SetSource(NULL);
    this->StructuredGrid->SetSource(NULL);
    this->UnstructuredGrid->SetSource(NULL);
    this->StructuredPoints->SetSource(NULL);
    this->RectilinearGrid->SetSource(NULL);
    }
  
  this->vtkObject::UnRegister(o);
}

int vtkMergeFilter::InRegisterLoop(vtkObject *o)
{
  int num = 0;
  int cnum = 0;
  
  if (this->StructuredPoints->GetSource() == this)
    {
    num++;
    cnum += this->StructuredPoints->GetNetReferenceCount();
    }
  if (this->RectilinearGrid->GetSource() == this)
    {
    num++;
    cnum += this->RectilinearGrid->GetNetReferenceCount();
    }
  if (this->PolyData->GetSource() == this)
    {
    num++;
    cnum += this->PolyData->GetNetReferenceCount();
    }
  if (this->StructuredGrid->GetSource() == this)
    {
    num++;
    cnum += this->StructuredGrid->GetNetReferenceCount();
    }
  if (this->UnstructuredGrid->GetSource() == this)
    {
    num++;
    cnum += this->UnstructuredGrid->GetNetReferenceCount();
    }
  
  // if no one outside is using us
  // and our data objects are down to one net reference
  // and we are being asked by one of our data objects
  if (this->ReferenceCount == num &&
      cnum == (num + 1) &&
      (this->PolyData == o ||
       this->StructuredPoints == o ||
       this->RectilinearGrid == o ||
       this->StructuredGrid == o ||
       this->UnstructuredGrid == o))
    {
    return 1;
    }
  return 0;
}
