/*=========================================================================

  Program:   Visualization Toolkit
  Module:    VArray.hh
  Language:  C++
  Date:      $Date$
  Version:   $Revision$

This file is part of the Visualization Toolkit. No part of this file
or its contents may be copied, reproduced or altered in any way
without the express written consent of the authors.

Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen 1993, 1994 

=========================================================================*/
// .NAME vtkVoidArray - dynamic, self adjusting array of void* pointers
// .SECTION Description
// vtkVoidArray is an array of pointers to void. It provides methods
// for insertion and retrieval of these pointers values, and will 
// automatically resize itself to hold new data.

#ifndef __vtkVoidArray_h
#define __vtkVoidArray_h

#include "Object.hh"

class vtkVoidArray : public vtkObject 
{
public:
  vtkVoidArray():Array(NULL),Size(0),MaxId(-1),Extend(1000) {};
  int Allocate(const int sz, const int ext=1000);
  void Initialize();
  vtkVoidArray(const int sz, const int ext=1000);
  vtkVoidArray(const vtkVoidArray& fa);
  ~vtkVoidArray();
  virtual char *GetClassName() {return "vtkVoidArray";};
  void PrintSelf(ostream& os, vtkIndent indent);

  // access/insertion methods
  void* GetValue(const int id);
  vtkVoidArray &InsertValue(const int id, void* p);
  int InsertNextValue(void* v);
  void** GetPtr(const int id);
  void** WritePtr(const int id, const int number);

  // special operators
  vtkVoidArray &operator=(const vtkVoidArray& fa);
  void operator+=(const vtkVoidArray& fa);
  void operator+=(void* p);
  void* &operator[](const int i);

  // miscellaneous methods
  void Squeeze();
  int GetSize();
  int GetMaxId();
  void Reset();

private:
  void** Array;   // pointer to data
  int Size;       // allocated size of data
  int MaxId;     // maximum index inserted thus far
  int Extend;     // grow array by this point
  void** Resize(const int sz);  // function to resize data
};

// Description:
// Get the data at a particular index.
inline void* vtkVoidArray::GetValue(const int id) {return this->Array[id];};

// Description:
// Get the address of a particular data index.
inline void** vtkVoidArray::GetPtr(const int id) {return this->Array + id;};

// Description:
// Get the address of a particular data index. Make sure data is allocated
// for the number of items requested. Set MaxId according to the number of
// data values requested.
inline void** vtkVoidArray::WritePtr(const int id, const int number) 
{
  if ( (id + number) > this->Size ) this->Resize(id+number);
  this->MaxId = id + number - 1;
  return this->Array + id;
}

// Description:
// Insert data at a specified position in the array.
inline vtkVoidArray& vtkVoidArray::InsertValue(const int id, void* p)
{
  if ( id >= this->Size ) this->Resize(id);
  this->Array[id] = p;
  if ( id > this->MaxId ) this->MaxId = id;
  return *this;
}

// Description:
// Insert data at the end of the array. Return its location in the array.
inline int vtkVoidArray::InsertNextValue(void* p)
{
  this->InsertValue (++this->MaxId,p);
  return this->MaxId;
}
inline void vtkVoidArray::operator+=(void* p) 
{
  this->InsertNextValue(p);
}

// Description:
// Does insert or get (depending on location on lhs or rhs of statement). Does
// not do automatic resizing - user's responsibility to range check.
inline void* &vtkVoidArray::operator[](const int i)
{
  if (i > this->MaxId) this->MaxId = i; 
  return this->Array[i];
}

// Description:
// Resize object to just fit data requirement. Reclaims extra memory.
inline void vtkVoidArray::Squeeze() {this->Resize (this->MaxId+1);};

// Description:
// Get the allocated size of the object in terms of number of data items.
inline int vtkVoidArray::GetSize() {return this->Size;};

// Description:
// Returning the maximum index of data inserted so far.
inline int vtkVoidArray::GetMaxId() {return this->MaxId;};

// Description:
// Reuse the memory allocated by this object. Objects appears like
// no data has been previously inserted.
inline void vtkVoidArray::Reset() {this->MaxId = -1;};

#endif
