
/*****************************************************************************
*
* Copyright (c) 2000 - 2018, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-442911
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//  File: avtRemapFilter.C
// ************************************************************************* //

#include <avtRemapFilter.h>

#include <vector>

#include <vtkDataArray.h>
#include <vtkInformationVector.h>
#include <vtkInformation.h>
//#include <vtkUnstructuredGridAlgorithm.h>
#include <vtkUnstructuredGrid.h>
#include <vtkImplicitBoolean.h>
#include <vtkPlane.h>
#include <vtkCell.h>
#include <vtkAppendFilter.h>
#include <vtkCellData.h>

#include <verdict.h>

#include <avtExtents.h>
#include <avtDataAttributes.h>
#include <vtkVisItClipper.h>


// ****************************************************************************
//  Method: avtRemapFilter constructor
//
//  Programmer: rusu1 -- generated by xml2avt
//  Creation:   Tue Oct 23 14:58:25 PST 2018
//
// ****************************************************************************

avtRemapFilter::avtRemapFilter()
{
}


// ****************************************************************************
//  Method: avtRemapFilter destructor
//
//  Programmer: rusu1 -- generated by xml2avt
//  Creation:   Tue Oct 23 14:58:25 PST 2018
//
//  Modifications:
//
// ****************************************************************************

avtRemapFilter::~avtRemapFilter()
{
}


// ****************************************************************************
//  Method:  avtRemapFilter::Create
//
//  Programmer: rusu1 -- generated by xml2avt
//  Creation:   Tue Oct 23 14:58:25 PST 2018
//
// ****************************************************************************

avtFilter *
avtRemapFilter::Create()
{
    return new avtRemapFilter();
}


// ****************************************************************************
//  Method:      avtRemapFilter::SetAtts
//
//  Purpose:
//      Sets the state of the filter based on the attribute object.
//
//  Arguments:
//      a        The attributes to use.
//
//  Programmer: rusu1 -- generated by xml2avt
//  Creation:   Tue Oct 23 14:58:25 PST 2018
//
// ****************************************************************************

void
avtRemapFilter::SetAtts(const AttributeGroup *a)
{
    atts = *(const RemapAttributes*)a;
}


// ****************************************************************************
//  Method: avtRemapFilter::Equivalent
//
//  Purpose:
//      Returns true if creating a new avtRemapFilter with the given
//      parameters would result in an equivalent avtRemapFilter.
//
//  Programmer: rusu1 -- generated by xml2avt
//  Creation:   Tue Oct 23 14:58:25 PST 2018
//
// ****************************************************************************

bool
avtRemapFilter::Equivalent(const AttributeGroup *a)
{
    return (atts == *(RemapAttributes*)a);
}


// ****************************************************************************
//  Method: avtRemapFilter::ExecuteData
//
//  Purpose:
//      Sends the specified input and output through the Remap filter.
//
//  Arguments:
//      in_dr      The input data representation.
//
//  Returns:       The output data representation.
//
//  TODO:          (-) Refactor the clipping so that the planes of the
//                     rectilinear grid are only generated once.
//                 (-) How to determine intrinsic and extrinsic arrays?
//                 (-) How to handle avtOriginalCellNumbers?
//                 (-) How to handle ghost zones?
//                 (-) Make it work to support vector data.
//                 (-) Apply remapping to PointData as well.
//                 (-) Make it work for 3D
//                 (-) Make it run for multiple domains in parallel.
//
//  Programmer: rusu1 -- generated by xml2avt
//  Creation:   Tue Oct 23 14:58:25 PST 2018
//
// ****************************************************************************

avtDataRepresentation *
avtRemapFilter::ExecuteData(avtDataRepresentation *in_dr)
{

    // ----------------------------------------------------------- //
    // --- Convert the avtDataRepresentation into a vtkDataSet --- //
    // ----------------------------------------------------------- //
    vtkDataSet *in_ds = in_dr->GetDataVTK();
    int domain = in_dr->GetDomain();
    std::string label = in_dr->GetLabel();
    
    // If there are no cells, then return null. This check is okay for now because
    // we only support cell-centered operations for the time being. Soon, we will support
    // node-centered operations, and we will need to revise this.
    // TODO: revise this when we support node operations.
    double nCellsIn = in_ds->GetNumberOfCells();
    if (in_ds == NULL || in_ds->GetNumberOfPoints() == 0 || nCellsIn == 0)
    {
        return NULL;
    }
    
    
    
    // ------------------------------------- //
    // --- Generate the rectilinear grid --- //
    // ------------------------------------- //
    
    double bounds[6] = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    is3D = GetBounds(bounds);
std::cout << "is3D: " << is3D << std::endl;

    int width = atts.GetCellsX();
    int height = atts.GetCellsY();
    
    //
    // Setup whatever variables I can assuming the grid is 2D. Then, check if it
    // is actually 3D. If so, modify parameters and build rg. Otherwise, build
    // rg with 2D variable values.
    //
    int nCellsOut = width*height;
    double rCellVolume = (bounds[1] - bounds[0]) * (bounds[3] - bounds[2]) / (nCellsOut);
    vtkRectilinearGrid* rg;
    if (is3D)
    {
        int depth = atts.GetCellsZ();
        nCellsOut *= depth;
        rCellVolume *= (bounds[5] - bounds[4]);
        rg = CreateGrid(bounds, width, height, depth, 0, width, 0, height, 0, depth);
    }
    else
    {
        rg = CreateGrid(bounds, width, height, 0, width, 0, height);
    }

std::cout << "Just generated rectilinear grid. Here it is:" << std::endl;
rg->Print(std::cout);
    
    // --------------------------- //
    // --- Setup the variables --- //
    // --------------------------- //
    
    // Some datasets include avtOriginalCellNumbers, which I might want to include
    // in my remapped dataset, but I don't want to remap that array. So for now,
    // I will just remove them.
    // TODO: Figure out how to better handle avtOriginalCellNumbers
    in_ds->GetCellData()->RemoveArray("avtOriginalCellNumbers");
    
    //
    // If there are no variables, then just output the mesh.
    //
    int nVariables = in_ds->GetCellData()->GetNumberOfArrays();
    
    if (nVariables <= 0)
    {
        return new avtDataRepresentation(rg, 0, "", false);
    }
    
    vtkDataArray** vars;
    vars = new vtkDataArray*[nVariables];
    for (int vdx = 0, vdy = 0; vdx < nVariables; vdx++)
    {
        vars[vdx] = vtkDoubleArray::New();
        vars[vdx]->SetNumberOfComponents(1); // Can only handle scalar values now
        vars[vdx]->SetNumberOfTuples(nCellsOut);
        vars[vdx]->SetName(in_ds->GetCellData()->GetArray(vdx)->GetName());
        
        rg->GetCellData()->AddArray(vars[vdx]);
        rg->GetCellData()->SetScalars(vars[vdx]);
    }
    
    // Determine the variable type from the attributes
    RemapAttributes::VariableTypes varType =
        atts.GetVariableType();
    
//std::cout << "Added blank vars to rg:" << std::endl;
//PrintData(rg);
     
       
    // ------------------------------------------------------- //
    // --- Calculate volumes of each cell in Original Grid --- //
    // ------------------------------------------------------- //
    // add the original volume to the in_ds AFTER the variables have already
    // been added to rg so that we don't add an artificial variable.
    
    vtkDoubleArray* avtRemapOriginalVolume = CalculateCellVolumes(in_ds,
            "avtRemapOriginalVolume");
    in_ds->GetCellData()->AddArray(avtRemapOriginalVolume);
    //in_ds->GetCellData()->CopyFieldOn("avtRemapOriginalVolume"); // Deep copy
    //avtRemapOriginalVolume->Delete(); // Clean
    
std::cout << "In_ds after adding volumes" << std::endl;   
PrintData(in_ds);
//in_ds->Print(std::cout);
    
    
    // -------------------------------------------------------- //
    // --- Clip the input grid against the rectilinear grid --- //
    // -------------------------------------------------------- //
    
    // CURRENT CODE NEEDS REFACTOR.
    // We calculate all 4 planes each time, which costs xDim*yDim*zDim*6, but we can
    // precompute and setup a mapping between cells and planes to cost xDim+yDim+zDim+2
    // vtkUnstructuredGrid* allGrids = vtkUnstructuredGrid::New(); 
    //vtkAppendFilter* appender = vtkAppendFilter::New();

// --- DEBUG --- DEBUG -------- //
#include <float.h>
#include <cstdlib>
double DEBUG_maxDiff = DBL_MIN;
// --- DEBUG --- DEBUG -------- //
    
    // + ----- +
    // |   3   |
    // |0     1|
    // |   2   |
    // + ----- +
    
    for (vtkIdType rCell = 0; rCell < rg->GetNumberOfCells(); rCell++) {
        // Get the bounds from the cell.
        rg->GetCell(rCell)->GetBounds(bounds);
        
        // Loop over each plane.
        vtkVisItClipper* last = NULL; 
        std::vector<vtkVisItClipper*> clipperArray;      
        std::vector<vtkImplicitBoolean*> funcsArray;
        std::vector<vtkPlane*> planeArray;
        //for (int cdx = 0; cdx < 4; cdx++) { // future will be cdx < 6 for 3D
        for (int cdx = 0; cdx < 6; cdx++) { // future will be cdx < 6 for 3D
            double origin[3] = {0., 0., 0.}; // future change for 3D
            double normal[3] = {0., 0., 0.}; // future change for 3D
            if (cdx == 0) {
                origin[0] = bounds[cdx];
                normal[0] = -1.0;
            } else if (cdx == 1) {
                origin[0] = bounds[cdx];
                normal[0] = 1.0;
            } else if (cdx == 2) {
                origin[1] = bounds[cdx];
                normal[1] = -1.0;
            } else if (cdx == 3) {
                origin[1] = bounds[cdx];
                normal[1] = 1.0;
            } else if (cdx == 4) {
                origin[2] = bounds[cdx];
                normal[2] = -1.0;
            } else if (cdx == 5) {
                origin[2] = bounds[cdx];
                normal[2] = 1.0;
            }
            vtkPlane* plane = vtkPlane::New();
            plane->SetOrigin(origin);
            plane->SetNormal(normal);
            planeArray.push_back(plane);
        
            vtkImplicitBoolean* funcs = vtkImplicitBoolean::New();
            funcs->AddFunction(plane);
            funcsArray.push_back(funcs);
            
            vtkVisItClipper* clipper = vtkVisItClipper::New();
            clipper->SetInputData(in_ds);
            clipper->SetClipFunction(funcs);
            clipper->SetInsideOut(true);
            clipper->SetRemoveWholeCells(false);
            
            if (last != NULL) {
                clipper->SetInputConnection(last->GetOutputPort());
            }
            last = clipper;
            clipperArray.push_back(clipper);
        } // end clipping loop
        
        last->Update();
        
        // Another possible solution to the deletions below is:
        vtkUnstructuredGrid* ug = vtkUnstructuredGrid::New();
        ug->DeepCopy(last->GetOutput());
        
std::cout << "Just finished making the unstructured grid. Here it is:" << std::endl;
PrintData(ug);
//ug->Print(std::cout);
        
        
        
        // --- Calculate volume of subcells --- //
        // Now that we have the unstrucutred grid from the clipping, we can loop
        // over the subcells in that grid, calculate the volumes, and generate
        // A new volume for the rectilinear grid. We also need the ratio
        // between old and new volumes to weigh the variables (like mass, density,
        // pressure, etc.).
        
        vtkDoubleArray* subCellVolumes = CalculateCellVolumes(ug, "subCellVolume");
        
// --- DEBUG --- DEBUG --- DEBUG --- DEBUG --- DEBUG --- DEBUG --- DEBUG ---- //
        double DEBUG_rCellVolumeTEST = 0.0;
        for (vtkIdType j = 0; j < subCellVolumes->GetNumberOfTuples(); j++) {
            DEBUG_rCellVolumeTEST += subCellVolumes->GetComponent(j,0);
        }
        if (DEBUG_rCellVolumeTEST != rCellVolume)
        {
            double diff = std::abs(DEBUG_rCellVolumeTEST - rCellVolume);
            DEBUG_maxDiff = diff > DEBUG_maxDiff ? diff : DEBUG_maxDiff;
        }
// --- DEBUG --- DEBUG --- DEBUG --- DEBUG --- DEBUG --- DEBUG --- DEBUG ---- //
        
        // Get the volume of the original cells for this ugrid
        vtkDoubleArray* originalCellVolumes = vtkDoubleArray::SafeDownCast(
            ug->GetCellData()->GetArray("avtRemapOriginalVolume"));
        
        
        
        // --- Calculate variable updates --- //
        // Two types of variables: intrinsic (like density) and extrinsic (like
        // mass). To update the variables:
        //        (1) Intrinsic values must be made extrinsic within the volume of
        //            the sub cell, then totaled among the set of sub cells, then
        //            made intrinsic again within the volume of the new cell.
        //        (2) Extrinsic values must be made intrisic within the volume of
        //            original cell, then made extrinsic within the volume of the
        //            sub cell, then totaled among the set of sub cells.
        
        for (int vdx = 0; vdx < nVariables; vdx++)
        {
            double value = 0.0;
            vtkDataArray* myVariable = ug->GetCellData()->GetArray(vdx);
                // vtkDoubleArray::SafeDownCast(ug->GetCellData()->GetArray(vdx));
            if (varType == RemapAttributes::intrinsic) // like density
            {
                for (vtkIdType tuple = 0;
                     tuple < myVariable->GetNumberOfTuples(); tuple++)
                {
                    value += myVariable->GetComponent(tuple, 0) *
                        subCellVolumes->GetComponent(tuple, 0);
                }
                value /= rCellVolume;
                vars[vdx]->SetComponent(rCell, 0, value);
            }
            else if (varType == RemapAttributes::extrinsic) // like mass
            {
                for (vtkIdType tuple = 0;
                     tuple < myVariable->GetNumberOfTuples(); tuple++)
                {
                    value += myVariable->GetComponent(tuple, 0) / 
                        originalCellVolumes->GetComponent(tuple, 0) *
                        subCellVolumes->GetComponent(tuple, 0);
                }
                vars[vdx]->SetComponent(rCell, 0, value);
            }
            else
            {
                std::cout << "Should not be possible to get here... " << std::endl;
            }
        } // end vars loop
        
        
        
        // --- Calculate density-like features --- //
        // subCellDensities are the same as the original densities.
        // In order to calculate the avtRemapRectilinearDensities, I need to
        // normalize up to mass, sum the masses, and the normalize to density
        // using the new mass and new volume.
/*        vtkDoubleArray* subCellDensities =
            vtkDoubleArray::SafeDownCast(ug->GetCellData()->GetArray("d"));
        double newCellMass = 0.0;
        for (vtkIdType j = 0; j < subCellVolumes->GetNumberOfTuples(); j++) {
            newCellMass += subCellDensities->GetComponent(j,0) * 
                subCellVolumes->GetComponent(j,0);
        }

        double newCellDensity = 0.0;
        if (newCellVolume != 0.0)
            newCellDensity = newCellMass / newCellVolume;
        avtRemapRectilinearDensity->SetComponent(i, 0, newCellDensity);
*/
        
        // --- Clean up --- //
        subCellVolumes->Delete();
        ug->Delete();
        
        for (int cdx = 0; cdx < 4; cdx++) { // future will be 6 for 3D
            planeArray[cdx]->Delete();
            funcsArray[cdx]->Delete();
            clipperArray[cdx]->Delete();
        } // end clipper loop

    } // end rectilinear grid loop
    
std::cout << "DEBUG: max diff of cell volumes: " << DEBUG_maxDiff << std::endl;
    
        
std::cout << "Populated vars in rg:" << std::endl;
PrintData(rg);
    
    
    // ----------------------------------------------------- //
    // --- Attach the datasets onto the rectilinear grid --- //
    // ----------------------------------------------------- //
    
    //rg->GetCellData()->AddArray(avtRemapRectilinearVolume);
    //rg->GetCellData()->CopyFieldOn("avtRemapRectilinearVolume");
    
/*    avtRemapRectilinearDensity->SetName("d");
    rg->GetCellData()->AddArray(avtRemapRectilinearDensity);
    rg->GetCellData()->SetScalars(avtRemapRectilinearDensity);
    rg->GetCellData()->CopyFieldOn("d");
    avtRemapRectilinearDensity->Delete();
*/
    
/*    for (int vdx = 0; vdx < nVariables; vdx++)
    {
        rg->GetCellData()->AddArray(vars[vdx]);
        rg->GetCellData()->SetScalars(vars[vdx]);
    }
*/    
    avtDataRepresentation *out_dr =
            new avtDataRepresentation(rg, 0, "", false);
            
std::cout << "Created out_dr:" << std::endl;
PrintData(out_dr);
            
    // --- Clean up --- //
    avtRemapOriginalVolume->Delete();
    //avtRemapRectilinearVolume->Delete();
    // Delete rg?
    // Delete vars?
    // 
    
    return out_dr;
    
}

// ****************************************************************************
//  Method: avtResampleFilter::CalculateCellVolumes
//
//  Purpose:
//      Calculates the volumes of the cells in a vtkDataSet. Stores the volumes
//        in a vtkDoubleArray.
//
//  Arguments:
//      in_ds       Input vtkDataSet* that represents the input mesh.
//        name        The desired name for the array. If empty then vtk default.
//
//  Returns: av vtkDoubleArray* that stores the volumes of each cell.
//
//  Programmer: Edward Rusu
//  Creation:   Nov 9, 2018
//
// ****************************************************************************


inline void
Swap1(double &a, double &b)
{
    double tmp = a;
    a = b;
    b = tmp;
}

inline void
Swap3(double c[][3], int a, int b)
{
    Swap1(c[a][0], c[b][0]);
    Swap1(c[a][1], c[b][1]);
    Swap1(c[a][2], c[b][2]);
}

inline
void Copy3(double coords[][3], double a[], int i)
{
    a[0] = coords[i][0];
    a[1] = coords[i][1];
    a[2] = coords[i][2];
}

vtkDoubleArray*
avtRemapFilter::CalculateCellVolumes(vtkDataSet* in_ds, const char* name)
{
    vtkDoubleArray* volumeArray = vtkDoubleArray::New(); // return element
    
    // Set up necessary variables
    const int MAXPOINTS = 100; // Set maximum number of points per cell to 100
    double coordinates[MAXPOINTS][3]; // array of points for each cell in 3D
    
    // Get information from the mesh
    int nCells = in_ds->GetNumberOfCells();
    
    // Setup the volumeArray
    if (name != "")
    {
        volumeArray->SetName(name);
    }
    volumeArray->SetNumberOfComponents(1);
    volumeArray->SetNumberOfTuples(nCells);
    
    for (vtkIdType i = 0; i < nCells; i++)
    {
        double volume = 0.0;
        vtkCell* cell = in_ds->GetCell(i); // Get the cell
        vtkDataArray* pointData = cell->GetPoints()->GetData(); // Get the points
        for (vtkIdType j = 0; j < cell->GetNumberOfPoints(); j++)
        {
            coordinates[j][2] = 0; // Set to 0 in case 2D
            pointData->GetTuple(j, coordinates[j]); // Set the j-th entry in coordiantes to the tuple from pointData at j
        }
        
        int subdiv[3][4] = { {0,5,4,3}, {0,2,1,4}, {0,4,5,2} };
        // ^ Need this to be before switch otherwise ill-formed program.
        switch(cell->GetCellType()) // right now only support Triangle and Quad
        {
            case VTK_TRIANGLE:
std::cout << "VTK_TRIANGLE" << std::endl;
                volume = v_tri_area(3, coordinates);
                break;
            case VTK_QUAD:
std::cout << "VTK_QUAD" << std::endl;
                //volume = v_quad_area(3, coordinates);
                volume = v_quad_area(4, coordinates);
                // Based on V_QuadMetric, I believe this should be 4.
                // Should run tests and see if there is a change in the output
                break;
            case VTK_PIXEL:
std::cout << "VTK_PIXEL" << std::endl;
                Swap3(coordinates, 2, 3);
                volume = v_quad_area(4, coordinates);
                break;
            case VTK_VOXEL:
std::cout << "VTK_VOXEL" << std::endl;
                Swap3(coordinates, 2, 3);
                Swap3(coordinates, 6, 7);
                volume = v_hex_volume(8, coordinates);
                break;
            case VTK_HEXAHEDRON:
std::cout << "VTK_HEXAHEDRON" << std::endl;
                volume = v_hex_volume(8, coordinates);
                break;
            case VTK_TETRA:
std::cout << "VTK_TETRA" << std::endl;
                volume = v_tet_volume(4, coordinates);
                break;
            case VTK_WEDGE:
std::cout << "VTK_WEDGE" << std::endl;
                double tet_coordinates[4][3];
                volume = 0;
                for (int i = 0 ; i < 3 ; i++)
                {
                    for (int j = 0 ; j < 4 ; j++)
                        for (int k = 0 ; k < 3 ; k++)
                            tet_coordinates[j][k] = coordinates[subdiv[i][j]][k];
                    volume += v_tet_volume(4, tet_coordinates);
                }
                break;
            case VTK_PYRAMID:
std::cout << "VTK_PYRAMID" << std::endl;
                double one[4][3];
                double two[4][3];
                Copy3(coordinates,one[0], 0);
                Copy3(coordinates,one[1], 1);
                Copy3(coordinates,one[2], 2);
                Copy3(coordinates,one[3], 4);
                Copy3(coordinates,two[0], 0);
                Copy3(coordinates,two[1], 2);
                Copy3(coordinates,two[2], 3);
                Copy3(coordinates,two[3], 4);
                volume = v_tet_volume(4,one) + v_tet_volume(4, two);
                break;
            default:
                std::cout << "Cannot calculate volume for cell of type: "
                          << cell->GetCellType() << std::endl
                          << "Scalars won't be remapped." << std::endl;
                break;
        }
        
        volumeArray->SetComponent(i, 0, volume); // store the volume in our array
        
        // DEBUG CHECK
        //std::cout << "Volume for cell " << i << " of type " << cell->GetCellType()
        //          << ": " << volume << std::endl;
    }
    
    
    return volumeArray;
}

// ****************************************************************************
//  Method: avtResampleFilter::GetBounds
//
//  Purpose:
//      Obtains the bounds of the resampled volume.  This could come from
//      attributes, or the existing spatial extents of the input.
//
//  Arguments:
//      bounds       Output array.  Format is min/max X, then m/m Y, m/m Z.
//
//  Returns: whether or not these specify 3-dimensional bounds.
//
//  Programmer: Tom Fogal
//  Creation:   June 23, 2009
//
//  Modifications:
//
//    Hank Childs, Thu Aug 26 13:47:30 PDT 2010
//    Change extents names.
//
//    Hank Childs, Tue Nov 30 21:54:43 PST 2010
//    Remove const qualification.
//
// ****************************************************************************
bool avtRemapFilter::GetBounds(double bounds[6])
{
    if (!atts.GetUseExtents())
    {
        bounds[0] = atts.GetStartX();
        bounds[1] = atts.GetEndX();
        bounds[2] = atts.GetStartY();
        bounds[3] = atts.GetEndY();
        bounds[4] = atts.GetStartZ();
        bounds[5] = atts.GetEndZ();
    }
    else
    {
        const avtDataAttributes &datts = avtPluginDataTreeIterator::GetInput()->GetInfo().GetAttributes();
        avtExtents *exts = datts.GetDesiredSpatialExtents();
        if (exts->HasExtents())
        {
            exts->CopyTo(bounds);
        }
        else
        {
            GetSpatialExtents(bounds);
        }
    }
    if (fabs(bounds[4]) < 1e-100 && fabs(bounds[5]) < 1e-100)
    {
        return false;
    }
    else {
        return atts.GetIs3D();
    }
}

// ****************************************************************************
//  Method: CreateGrid
//
//  Purpose:
//      Creates a rectilinear grid that makes sense for the bounds of the
//      original dataset.
//
//  Arguments:
//      bounds    The bounds of the original dataset.
//      numX      The number of samples in X.
//      numY      The number of samples in Y.
//      minX      The minimum X index for this processor.
//      maxX      The maximum Y index for this processor.
//      minY      The minimum X index for this processor.
//      maxY      The maximum Y index for this processor.
//
//  Programmer:   Hank Childs
//  Creation:     March 26, 2001
//
//  Modifications:
//
//    Kathleen Bonnell, Mon Nov 19 15:31:36 PST 2001
//    Use vtkDataArray instead of vtkScalars to match VTK 4.0 API.
//
//    Hank Childs, Tue Feb  5 09:49:34 PST 2002
//    Use double for bounds.
//
//    Hank Childs, Fri Sep 30 10:50:24 PDT 2005
//    Add support for distributed resampling.
//
//    Hank Childs, Fri Jun  1 16:17:51 PDT 2007
//    Add support for cell-centered data.
//
//    Gunther H. Weber, Wed Aug  3 11:55:02 PDT 2011
//    Ensure that 2D data sets are 2D even for cell centered case
//
// ****************************************************************************

vtkRectilinearGrid *
avtRemapFilter::CreateGrid(const double *bounds, int numX, int numY, int numZ,
        int minX, int maxX, int minY, int maxY, int minZ, int maxZ)
{
    vtkDataArray *xc = NULL;
    vtkDataArray *yc = NULL;
    vtkDataArray *zc = NULL;

    double width  = bounds[1] - bounds[0];
    double height = bounds[3] - bounds[2];
    double depth  = bounds[5] - bounds[4];

    xc = GetCoordinates(bounds[0], width, numX+1, minX, maxX+1);
    yc = GetCoordinates(bounds[2], height, numY+1, minY, maxY+1);
    zc = GetCoordinates(bounds[4], depth, numZ+1, minZ, maxZ+1);

    vtkRectilinearGrid *rv = vtkRectilinearGrid::New();
    rv->SetDimensions(maxX-minX+1, maxY-minY+1, maxZ-minZ+1);
    rv->SetXCoordinates(xc);
    xc->Delete();
    rv->SetYCoordinates(yc);
    yc->Delete();
    rv->SetZCoordinates(zc);
    zc->Delete();

    return rv;
}

vtkRectilinearGrid *
avtRemapFilter::CreateGrid(const double *bounds,
        int numX, int numY, int minX, int maxX, int minY, int maxY)
{
    vtkDataArray *xc = NULL;
    vtkDataArray *yc = NULL;
    vtkDataArray *zc = NULL;

    double width  = bounds[1] - bounds[0];
    double height = bounds[3] - bounds[2];

    xc = GetCoordinates(bounds[0], width, numX+1, minX, maxX+1);
    yc = GetCoordinates(bounds[2], height, numY+1, minY, maxY+1);
    zc = GetCoordinates(bounds[4], 0, 1, 0, 1);

    vtkRectilinearGrid *rv = vtkRectilinearGrid::New();
    rv->SetDimensions(maxX-minX+1, maxY-minY+1, 1);
    rv->SetXCoordinates(xc);
    xc->Delete();
    rv->SetYCoordinates(yc);
    yc->Delete();
    rv->SetZCoordinates(zc);
    zc->Delete();

    return rv;
}


// ****************************************************************************
//  Function: GetCoordinates
//
//  Purpose:
//      Creates a coordinates array based on specifications.
//
//  Arguments:
//      start    The start of the coordinates array.
//      length   The length of the coordinates array.
//      numEls   The number of elements in the coordinates array.
//
//  Returns:     A vtkDataArray element for the coordinate.
//
//  Programmer:  Hank Childs
//  Creation:    March 26, 2001
//
//  Modifications:
//    Kathleen Bonnell, Mon Nov 19 15:31:36 PST 2001
//    Changes in VTK 4.0 API require use of vtkDataArray /vtkFloatArray
//    in place of vtkScalars. 
//
//    Hank Childs, Fri Sep 30 10:50:24 PDT 2005
//    Add support for distributed resampling.
//
// ****************************************************************************

vtkDataArray *
avtRemapFilter::GetCoordinates(double start, double length, int numEls, int myStart, int myStop)
{
    vtkDoubleArray *rv = vtkDoubleArray::New();

    //
    // Make sure we don't have any degenerate cases here.
    //
    if (length <= 0. || numEls <= 1 || myStart >= myStop)
    {
        rv->SetNumberOfValues(1);
        rv->SetValue(0, start);
        return rv;
    }

    int realNumEls = myStop - myStart;
    rv->SetNumberOfValues(realNumEls);
    double offset = length / (numEls-1);
    for (int i = myStart ; i < myStop ; i++)
    {
        rv->SetValue(i-myStart, start + i*offset);
    }

    return rv;
}











// **
// This function isn't real. Just using it to debug.
// **

void avtRemapFilter::PrintData(avtDataRepresentation* in_dr)
{
    PrintData(in_dr->GetDataVTK());
}

void avtRemapFilter::PrintData(vtkDataSet* in_ds)
{
    for (int i = 0; i < in_ds->GetCellData()->GetNumberOfArrays(); i++)
    {
        PrintData(in_ds->GetCellData()->GetArray(i));
        /*
        vtkDataArray* myArray = in_ds->GetCellData()->GetArray(i);
        std::cout << "Array name: " << myArray->GetName()
                  << std::endl;
        std::cout << "Values: ";
        for (int j = 0; j < myArray->GetNumberOfTuples(); j++)
        {
            std::cout << "{";
            for (int k = 0; k < myArray->GetNumberOfComponents(); k++)
            {
                std::cout << myArray->GetComponent(j,k);
                if (k != myArray->GetNumberOfComponents() - 1)
                {
                    std::cout << ", ";
                }
            }
            std::cout << "}";
            if (j != myArray->GetNumberOfTuples() - 1)
            {
                std::cout << "; ";
            }
        }
        std::cout << std::endl ;
        */
    }
    std::cout << std::endl;
}

void avtRemapFilter::PrintData(vtkDataArray* myArray)
{
    if (myArray->GetName()) {
        std::cout << "Array name: " << myArray->GetName()
                  << std::endl;
    }
    std::cout << "Values: ";
    for (int j = 0; j < myArray->GetNumberOfTuples(); j++)
    {
        std::cout << "{";
        for (int k = 0; k < myArray->GetNumberOfComponents(); k++)
        {
            std::cout << myArray->GetComponent(j,k);
            if (k != myArray->GetNumberOfComponents() - 1)
            {
                std::cout << ", ";
            }
        }
        std::cout << "}";
        if (j != myArray->GetNumberOfTuples() - 1)
        {
            std::cout << "; ";
        }
    }
    std::cout << std::endl ;
}






