/*=========================================================================

  Program:   ParaView
  Module:    vtkPVColorMap.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkPVColorMap.h"

#include "vtkCommand.h"
#include "vtkKWChangeColorButton.h"
#include "vtkKWCheckButton.h"
#include "vtkKWEntry.h"
#include "vtkKWFrame.h"
#include "vtkKWLabel.h"
#include "vtkKWFrameLabeled.h"
#include "vtkKWMenuButton.h"
#include "vtkKWOptionMenu.h"
#include "vtkKWPushButton.h"
#include "vtkKWScale.h"
#include "vtkPVTextPropertyEditor.h"
#include "vtkKWTkUtilities.h"
#include "vtkKWWidget.h"
#include "vtkLookupTable.h"
#include "vtkObjectFactory.h"
#include "vtkPVApplication.h"
#include "vtkPVArrayInformation.h"
#include "vtkPVDisplayGUI.h"
#include "vtkPVDataInformation.h"
#include "vtkPVDataSetAttributesInformation.h"
#include "vtkPVGenericRenderWindowInteractor.h"
#include "vtkPVRenderView.h"
#include "vtkPVSource.h"
#include "vtkPVSourceCollection.h"
#include "vtkPVWindow.h"
#include "vtkRenderer.h"
#include "vtkRendererCollection.h"
#include "vtkRenderWindow.h"
#include "vtkScalarBarActor.h"
#include "vtkScalarBarWidget.h"
#include "vtkPVProcessModule.h"
#include "vtkKWRange.h"
#include "vtkTextProperty.h"
#include "vtkPVTraceHelper.h"

#include "vtkTextProperty.h"
#include "vtkSMIntVectorProperty.h"
#include "vtkSMDoubleVectorProperty.h"
#include "vtkSMStringVectorProperty.h"
#include "vtkSMProxyProperty.h"
#include "vtkSMScalarBarWidgetProxy.h"
#include "vtkSMLookupTableProxy.h"
#include "vtkSMProxyManager.h"
#include "vtkSMProxy.h"
#include "vtkKWEvent.h"
#include "vtkMath.h"
#include "vtkSMRenderModuleProxy.h"

//----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPVColorMap);
vtkCxxRevisionMacro(vtkPVColorMap, "1.123");

/* 
 * This part was generated by ImageConvert from image:
 *    presets.png (zlib, base64)
 */
#define image_presets_width         16
#define image_presets_height        16
#define image_presets_pixel_size    4
#define image_presets_buffer_length 264

static unsigned char image_presets[] = 
  "eNrdkusNhSAMhZkHhtAlXAKHgCFkCZeAIXAIXeJcikiqQaN/L8kJr349LQGAQJG1FleB3T"
  "+JYkMIecvnNznuWJ7jqbay/jSIEULgzIvHOvjc4t+yR0//4P9VV3+SUgp932MYBoxaw6S7"
  "yTnM84zgA5ZlwbZtObblr6RMfJd5PSbeGDg3Zd4TH2Pl+f858d3ur/XB7/4+eETGt6SUrP"
  "WTP+WuvPfJn+pf63tz0Zm89G9N4qfSf/Kn/tfi3+K/vf+Z/wENH5fO";

//===========================================================================
//***************************************************************************
class vtkPVColorMapObserver : public vtkCommand
{
public:
  static vtkPVColorMapObserver* New() 
    {return new vtkPVColorMapObserver;};

  vtkPVColorMapObserver()
    {
    this->PVColorMap = 0;
    }

  virtual void Execute(vtkObject* wdg, unsigned long event,  
                       void* calldata)
    {
      if ( this->PVColorMap )
        {
        this->PVColorMap->ExecuteEvent(wdg, event, calldata);
        }
    }

  vtkPVColorMap* PVColorMap;
};

#define VTK_PV_COLOR_MAP_RED_HUE 0.0
#define VTK_PV_COLOR_MAP_RED_SATURATION 1.0
#define VTK_PV_COLOR_MAP_RED_VALUE 1.0
#define VTK_PV_COLOR_MAP_BLUE_HUE 0.6667
#define VTK_PV_COLOR_MAP_BLUE_SATURATION 1.0
#define VTK_PV_COLOR_MAP_BLUE_VALUE 1.0
#define VTK_PV_COLOR_MAP_BLACK_HUE 0.0
#define VTK_PV_COLOR_MAP_BLACK_SATURATION 0.0
#define VTK_PV_COLOR_MAP_BLACK_VALUE 0.0
#define VTK_PV_COLOR_MAP_WHITE_HUE 0.0
#define VTK_PV_COLOR_MAP_WHITE_SATURATION 0.0
#define VTK_PV_COLOR_MAP_WHITE_VALUE 1.0

#define VTK_USE_LAB_COLOR_MAP 1.1

//***************************************************************************
//===========================================================================
//vtkCxxSetObjectMacro(vtkPVColorMap,PVRenderView,vtkPVRenderView);
//----------------------------------------------------------------------------
// No register count because of reference loop.
void vtkPVColorMap::SetPVRenderView(vtkPVRenderView *rv)
{
  this->PVRenderView = rv;
}

//----------------------------------------------------------------------------
vtkPVColorMap::vtkPVColorMap()
{
  this->UseCount = 0;
  
  this->ScalarBarVisibility = 0;
  this->InternalScalarBarVisibility = 0;
  this->Initialized = 0;
  this->PVRenderView = NULL;
  this->ScalarBarObserver = NULL;

  // User interaface.
  this->ColorMapFrame = vtkKWFrameLabeled::New();
  this->ArrayNameLabel = vtkKWLabel::New();
  // Stuff for setting the range of the color map.
  this->ScalarRangeFrame = vtkKWFrame::New();
  this->ScalarRangeLockCheck = vtkKWCheckButton::New();
  this->ScalarRangeWidget = vtkKWRange::New();
  this->ScalarRangeWidget->ClampRangeOff();
  this->NumberOfColorsScale = vtkKWScale::New();  
  this->ColorEditorFrame = vtkKWFrame::New();
  this->StartColorButton = vtkKWChangeColorButton::New();
  this->Map = vtkKWLabel::New();
  this->EndColorButton = vtkKWChangeColorButton::New();

  this->ScalarRange[0] = this->WholeScalarRange[0] = VTK_LARGE_FLOAT;
  this->ScalarRange[1] = this->WholeScalarRange[1] = -VTK_LARGE_FLOAT;
  this->ScalarRangeLock = 0;

  this->VectorFrame = vtkKWFrameLabeled::New();
  this->VectorModeMenu = vtkKWOptionMenu::New() ;
  this->VectorComponentMenu = vtkKWOptionMenu::New();

  // Stuff for manipulating the scalar bar.
  this->ScalarBarFrame = vtkKWFrameLabeled::New();
  this->ScalarBarCheck = vtkKWCheckButton::New();
  this->ScalarBarTitleFrame = vtkKWFrame::New();
  this->ScalarBarTitleLabel = vtkKWLabel::New();
  this->ScalarBarTitleEntry = vtkKWEntry::New();
  this->ScalarBarVectorTitleEntry = vtkKWEntry::New();
  this->ScalarBarLabelFormatFrame = vtkKWFrame::New();
  this->ScalarBarLabelFormatLabel = vtkKWLabel::New();
  this->ScalarBarLabelFormatEntry = vtkKWEntry::New();

  this->BackButton = vtkKWPushButton::New();

  this->MapData = NULL;
  this->MapDataSize = 0;
  this->MapHeight = 25;
  this->MapWidth = 20;

  this->PresetsMenuButton = vtkKWMenuButton::New();

  this->VisitedFlag = 0;

  this->TitleTextPropertyWidget = vtkPVTextPropertyEditor::New();
  this->LabelTextPropertyWidget = vtkPVTextPropertyEditor::New();

  this->LabelTextProperty = vtkTextProperty::New();
  this->TitleTextProperty = vtkTextProperty::New();
  
  this->ScalarBarProxy = 0;
  this->ScalarBarProxyName = 0;

  this->LookupTableProxy = 0;
  this->LookupTableProxyName = 0;
  
  this->ScalarBarTitle = NULL;
  this->VectorMagnitudeTitle = new char[12];
  strcpy(this->VectorMagnitudeTitle, "Magnitude");
  this->VectorComponentTitles = NULL;
  this->ScalarBarVectorTitle = NULL;
  this->NumberOfVectorComponents = 0;
  this->VectorComponent = 0;
}

//----------------------------------------------------------------------------
vtkPVColorMap::~vtkPVColorMap()
{
  // Used to be in vtkPVActorComposite........

  this->SetPVRenderView(NULL);

  if (this->ScalarBarObserver)
    {
    this->ScalarBarObserver->Delete();
    this->ScalarBarObserver = NULL;
    }
    
  this->VectorFrame->Delete();
  this->VectorFrame = NULL;
  this->VectorModeMenu->Delete();
  this->VectorModeMenu = NULL;
  this->VectorComponentMenu->Delete();
  this->VectorComponentMenu = NULL;
  this->ScalarBarVectorTitleEntry->Delete();
  this->ScalarBarVectorTitleEntry = NULL;

  // User interaface.
  this->ColorMapFrame->Delete();
  this->ColorMapFrame = NULL;
  this->ArrayNameLabel->Delete();
  this->ArrayNameLabel = NULL;
  // Stuff for setting the range of the color map.
  this->ScalarRangeFrame->Delete();
  this->ScalarRangeFrame = NULL;
  this->ScalarRangeLockCheck->Delete();
  this->ScalarRangeLockCheck = NULL;
  this->ScalarRangeWidget->Delete();
  this->ScalarRangeWidget = NULL;
  this->NumberOfColorsScale->Delete();
  this->NumberOfColorsScale = NULL;

  this->ColorEditorFrame->Delete();
  this->ColorEditorFrame = NULL;
  this->StartColorButton->Delete();
  this->StartColorButton = NULL;
  this->Map->Delete();
  this->Map = NULL;
  this->EndColorButton->Delete();
  this->EndColorButton = NULL;
  
  this->ScalarBarFrame->Delete();
  this->ScalarBarFrame = NULL;
  this->ScalarBarCheck->Delete();
  this->ScalarBarCheck = NULL;
  this->ScalarBarTitleFrame->Delete();
  this->ScalarBarTitleFrame = NULL;
  this->ScalarBarTitleLabel->Delete();
  this->ScalarBarTitleLabel = NULL;
  this->ScalarBarTitleEntry->Delete();
  this->ScalarBarTitleEntry = NULL;
  this->ScalarBarLabelFormatFrame->Delete();
  this->ScalarBarLabelFormatFrame = NULL;
  this->ScalarBarLabelFormatLabel->Delete();
  this->ScalarBarLabelFormatLabel = NULL;
  this->ScalarBarLabelFormatEntry->Delete();
  this->ScalarBarLabelFormatEntry = NULL;

  this->BackButton->Delete();
  this->BackButton = NULL;

  if (this->MapData)
    {
    delete [] this->MapData;
    this->MapDataSize = 0;
    this->MapWidth = 0;
    this->MapHeight = 0;
    }

  if ( this->PresetsMenuButton )
    {
    this->PresetsMenuButton->Delete();
    }

  this->TitleTextPropertyWidget->Delete();
  this->TitleTextPropertyWidget = NULL;

  this->LabelTextPropertyWidget->Delete();
  this->LabelTextPropertyWidget = NULL;
  
  
  if (this->ScalarBarProxyName)
    {
    vtkSMObject::GetProxyManager()->UnRegisterProxy("displays", this->ScalarBarProxyName);
    }
  this->SetScalarBarProxyName(0);
  if (this->ScalarBarProxy)
    {
    vtkSMRenderModuleProxy* rm = this->GetPVApplication()->GetRenderModuleProxy();
    vtkSMProxyProperty* pp =  (rm)? vtkSMProxyProperty::SafeDownCast(
      rm->GetProperty("Displays")): 0;
    if (pp)
      {
      pp->RemoveProxy(this->ScalarBarProxy);
      rm->UpdateVTKObjects();
      }
    this->ScalarBarProxy->Delete();
    this->ScalarBarProxy = 0;
    }
 
  if (this->LookupTableProxyName)
    {
    vtkSMObject::GetProxyManager()->UnRegisterProxy("lookup_tables", 
      this->LookupTableProxyName);
    }
  this->SetLookupTableProxyName(0);
  
  if (this->LookupTableProxy)
    {
    this->LookupTableProxy->Delete();
    this->LookupTableProxy = 0;
    }
  
  if (this->ScalarBarTitle)
    {
    delete [] this->ScalarBarTitle;
    this->ScalarBarTitle = 0;
    }
  if (this->VectorMagnitudeTitle)
    {
    delete [] this->VectorMagnitudeTitle;
    this->VectorMagnitudeTitle = 0;
    }
  if (this->ScalarBarVectorTitle)
    {
    delete[] this->ScalarBarVectorTitle;
    this->ScalarBarVectorTitle = 0;
    }
  // This will delete the vector component titles
  this->SetNumberOfVectorComponents(0);

  this->LabelTextProperty->Delete();
  this->TitleTextProperty->Delete();
}


//----------------------------------------------------------------------------
void vtkPVColorMap::Create(vtkKWApplication *app)
{
  // Check if already created

  if (this->IsCreated())
    {
    vtkErrorMacro(<< this->GetClassName() << " already created");
    return;
    }

  // Call the superclass to create the whole widget

  this->Superclass::Create(app);
  
  vtkPVApplication* pvApp = vtkPVApplication::SafeDownCast(app);

  this->CreateParallelTclObjects(pvApp);

  // Now for the UI.

  const char *grid_settings = "-padx 1 -pady 2";

  // Color map

  this->ColorMapFrame->SetParent(this);
  this->ColorMapFrame->ShowHideFrameOn();
  this->ColorMapFrame->Create(app);
  this->ColorMapFrame->SetLabelText("Color Map");

  // Color map: parameter name

  this->ArrayNameLabel->SetParent(this->ColorMapFrame->GetFrame());
  this->ArrayNameLabel->Create(app);
  this->ArrayNameLabel->SetAnchorToWest();
  this->ArrayNameLabel->SetText("Parameter: ");

  // Color map: range
  this->ScalarRangeFrame->SetParent(this->ColorMapFrame->GetFrame());
  this->ScalarRangeFrame->Create(app);
  this->ScalarRangeLockCheck->SetParent(this->ScalarRangeFrame);
  this->ScalarRangeLockCheck->Create(app);
  this->ScalarRangeLockCheck->SetImageOption("PVUnlockedButton");
  this->ScalarRangeLockCheck->SetImageOption("PVLockedButton", "-selectimage");
  this->ScalarRangeLockCheck->SetReliefToFlat();
  this->ScalarRangeLockCheck->SetState(0);
  this->ScalarRangeLockCheck->SetIndicator(0);
  this->ScalarRangeLockCheck->SetBalloonHelpString(
    "Lock the range so that ParaView will not change it.");
  this->ScalarRangeLockCheck->SetCommand(this, "LockCheckCallback");
  this->Script("pack %s -side left -expand f",
               this->ScalarRangeLockCheck->GetWidgetName());

  this->ScalarRangeWidget->SetParent(this->ScalarRangeFrame);
  this->ScalarRangeWidget->Create(app);
  this->ScalarRangeWidget->SetWholeRange(
    -VTK_LARGE_FLOAT, VTK_LARGE_FLOAT);
  this->ScalarRangeWidget->ShowEntriesOn();
  this->ScalarRangeWidget->ShowLabelOff();
  this->ScalarRangeWidget->GetEntry1()->SetWidth(7);
  this->ScalarRangeWidget->GetEntry2()->SetWidth(7);
  this->ScalarRangeWidget->SetCommand(this, "ScalarRangeWidgetCallback");
  this->ScalarRangeWidget->SetEntry1PositionToLeft();
  this->ScalarRangeWidget->SetEntry2PositionToRight();
  this->ScalarRangeWidget->SetBalloonHelpString(
    "Set the minimum and maximum of the values of the color map");
  this->Script("pack %s -side left -fill x -expand t",
               this->ScalarRangeWidget->GetWidgetName());
               
  // Color map: gradient editor

  this->ColorEditorFrame->SetParent(this->ColorMapFrame->GetFrame());
  this->ColorEditorFrame->Create(app);

  this->StartColorButton->SetParent(this->ColorEditorFrame);
  this->StartColorButton->ShowLabelOff();
  this->StartColorButton->Create(app);
  this->StartColorButton->SetColor(1.0, 0.0, 0.0);
  this->StartColorButton->SetCommand(this, "StartColorButtonCallback");
  this->StartColorButton->SetBalloonHelpString("Select the minimum color.");

  this->Map->SetParent(this->ColorEditorFrame);
  this->Map->Create(app);
  this->Map->SetBorderWidth(0);
  this->Map->SetPadX(0);
  this->Map->SetPadY(0);

  this->Script("bind %s <Configure> {%s MapConfigureCallback %s}", 
               this->Map->GetWidgetName(), 
               this->GetTclName(), "%w %h");

  this->EndColorButton->SetParent(this->ColorEditorFrame);
  this->EndColorButton->ShowLabelOff();
  this->EndColorButton->Create(app);
  this->EndColorButton->SetColor(0.0, 0.0, 1.0);
  this->EndColorButton->SetCommand(this, "EndColorButtonCallback");
  this->EndColorButton->SetBalloonHelpString("Select the maximum color.");

  this->PresetsMenuButton->SetParent(this->ColorEditorFrame);
  this->PresetsMenuButton->Create(app);
  this->PresetsMenuButton->IndicatorOff();
  this->PresetsMenuButton->SetBalloonHelpString("Select a preset color map.");
  this->PresetsMenuButton->AddCommand(
    "Blue to Red", 
    this, "SetColorSchemeToBlueRed", "Set Color Scheme to Blue-Red");
  this->PresetsMenuButton->AddCommand(
    "Red to Blue", 
    this, "SetColorSchemeToRedBlue", "Set Color Scheme to Red-Blue");
  this->PresetsMenuButton->AddCommand(
    "Grayscale", 
    this, "SetColorSchemeToGrayscale", "Set Color Scheme to Grayscale");
  this->PresetsMenuButton->AddCommand(
    "CIELab Blue to Red", 
    this, "SetColorSchemeToLabBlueRed", "Set Color Scheme to Lab Blue To Red");
  
  this->PresetsMenuButton->SetImageOption(image_presets, 
                                          image_presets_width, 
                                          image_presets_height, 
                                          image_presets_pixel_size,
                                          image_presets_buffer_length);

  this->Script("grid %s %s %s %s -sticky news -padx 1 -pady 2",
               this->PresetsMenuButton->GetWidgetName(),
               this->StartColorButton->GetWidgetName(),
               this->Map->GetWidgetName(),
               this->EndColorButton->GetWidgetName());

  this->Script("grid columnconfigure %s 2 -weight 1",
               this->Map->GetParent()->GetWidgetName());
  this->Script("grid %s -ipadx 1",
               this->PresetsMenuButton->GetWidgetName());

  // Color map: resolution

  this->NumberOfColorsScale->SetParent(this->ColorMapFrame->GetFrame());
  this->NumberOfColorsScale->Create(app);
  this->NumberOfColorsScale->SetRange(2, 256);
  this->NumberOfColorsScale->SetValue(
    this->GetNumberOfColorsInternal());
  this->NumberOfColorsScale->DisplayLabel("Resolution:");
  this->NumberOfColorsScale->DisplayEntry();
  this->NumberOfColorsScale->DisplayEntryAndLabelOnTopOff();
  this->NumberOfColorsScale->SetEndCommand(this, 
                                           "NumberOfColorsScaleCallback");
  this->NumberOfColorsScale->SetEntryCommand(this, 
                                             "NumberOfColorsScaleCallback");
  this->NumberOfColorsScale->SetBalloonHelpString(
    "Select the discrete number of colors in the color map.");

  // Color map: pack

  this->Script("pack %s %s %s %s -side top -expand t -fill x -anchor nw",
               this->ArrayNameLabel->GetWidgetName(),
               this->ScalarRangeFrame->GetWidgetName(),
               this->ColorEditorFrame->GetWidgetName(),
               this->NumberOfColorsScale->GetWidgetName());


  if (this->NumberOfVectorComponents > 1)
    {
    this->VectorFrame->SetParent(this);
    this->VectorFrame->ShowHideFrameOn();
    this->VectorFrame->Create(app);
    this->VectorFrame->SetLabelText("Vector");

    this->VectorModeMenu->SetParent(this->VectorFrame->GetFrame());
    this->VectorModeMenu->Create(app);
    this->VectorModeMenu->AddEntryWithCommand("Magnitude", this, 
                                          "VectorModeMagnitudeCallback");
    this->VectorModeMenu->AddEntryWithCommand("Component", this, 
                                          "VectorModeComponentCallback");
    this->VectorModeMenu->SetValue("Magnitude");

    this->VectorComponentMenu->SetParent(this->VectorFrame->GetFrame());
    this->VectorComponentMenu->Create(app);
    this->UpdateVectorComponentMenu();
    this->Script("pack %s -side left -expand f -fill both -padx 2",
                 this->VectorModeMenu->GetWidgetName());
    this->Script("pack %s -side left -expand f -fill both -padx 2",
                 this->VectorComponentMenu->GetWidgetName());
    
    this->VectorModeMagnitudeCallback();
    }

  // Scalar bar frame

  this->ScalarBarFrame->SetParent(this);
  this->ScalarBarFrame->ShowHideFrameOn();
  this->ScalarBarFrame->Create(app);
  this->ScalarBarFrame->SetLabelText("Scalar Bar");

  // Scalar bar : Visibility

  this->ScalarBarCheck->SetParent(this->ScalarBarFrame->GetFrame());
  this->ScalarBarCheck->Create(app);
  this->ScalarBarCheck->SetText("Visibility");
  app->Script(
    "%s configure -command {%s ScalarBarCheckCallback}",
    this->ScalarBarCheck->GetWidgetName(),
    this->GetTclName());

  // Scalar bar : Title control

  this->ScalarBarTitleFrame->SetParent(this->ScalarBarFrame->GetFrame());
  this->ScalarBarTitleFrame->Create(app);

  this->ScalarBarTitleLabel->SetParent(this->ScalarBarTitleFrame);
  this->ScalarBarTitleLabel->SetText("Title:");
  this->ScalarBarTitleLabel->Create(app);
  this->ScalarBarTitleLabel->SetAnchorToWest();
  
  this->ScalarBarTitleEntry->SetParent(this->ScalarBarTitleFrame);
  this->ScalarBarTitleEntry->SetWidth(10);
  this->ScalarBarTitleEntry->Create(app);
  this->Script("bind %s <KeyPress-Return> {%s ScalarBarTitleEntryCallback}",
               this->ScalarBarTitleEntry->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ScalarBarTitleEntryCallback}",
               this->ScalarBarTitleEntry->GetWidgetName(),
               this->GetTclName()); 

  this->ScalarBarVectorTitleEntry->SetParent(this->ScalarBarTitleFrame);
  this->ScalarBarVectorTitleEntry->SetWidth(
    this->ScalarBarTitleEntry->GetWidth() / 2);
  this->ScalarBarVectorTitleEntry->Create(app);
  this->Script(
    "bind %s <KeyPress-Return> {%s ScalarBarVectorTitleEntryCallback}",
               this->ScalarBarVectorTitleEntry->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ScalarBarVectorTitleEntryCallback}",
               this->ScalarBarVectorTitleEntry->GetWidgetName(),
               this->GetTclName()); 

  this->TitleTextPropertyWidget->SetParent(this->ScalarBarTitleFrame);
  this->GetTitleTextPropertyInternal();
  this->TitleTextPropertyWidget->SetTextProperty(
    this->TitleTextProperty);
  this->TitleTextPropertyWidget->Create(app);
  this->TitleTextPropertyWidget->SetChangedCommand(
    this, "TitleTextPropertyWidgetCallback");
  this->TitleTextPropertyWidget->GetTraceHelper()->SetReferenceHelper(
    this->GetTraceHelper());
  this->TitleTextPropertyWidget->GetTraceHelper()->SetReferenceCommand(
    "GetTitleTextPropertyWidget");

  this->Script("grid %s -row 0 -column 0 -sticky nws %s",
               this->ScalarBarTitleLabel->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 0 -column 1 -sticky news %s",
               this->ScalarBarTitleEntry->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 1 -column 1 -columnspan 2 -sticky nws %s",
               this->TitleTextPropertyWidget->GetWidgetName(),
               grid_settings);

  if (this->NumberOfVectorComponents > 1)
    {
    this->Script("grid %s -row 0 -column 2 -sticky news %s",
                 this->ScalarBarVectorTitleEntry->GetWidgetName(),
                 grid_settings);
    this->Script("grid columnconfigure %s 2 -weight 3",
                 this->ScalarBarVectorTitleEntry->GetParent()->GetWidgetName());
    }

  // Scalar bar : Label control

  this->ScalarBarLabelFormatFrame->SetParent(this->ScalarBarFrame->GetFrame());
  this->ScalarBarLabelFormatFrame->Create(app);

  this->ScalarBarLabelFormatLabel->SetParent(this->ScalarBarLabelFormatFrame);
  this->ScalarBarLabelFormatLabel->SetText("Labels:");
  this->ScalarBarLabelFormatLabel->Create(app);
  this->ScalarBarLabelFormatLabel->SetAnchorToWest();
  
  this->ScalarBarLabelFormatEntry->SetParent(this->ScalarBarLabelFormatFrame);
  this->ScalarBarLabelFormatEntry->SetWidth(
    this->ScalarBarTitleEntry->GetWidth());
  this->ScalarBarLabelFormatEntry->Create(app);
  this->Script("bind %s <KeyPress-Return> {%s ScalarBarLabelFormatEntryCallback}",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ScalarBarLabelFormatEntryCallback}",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               this->GetTclName()); 
  this->ScalarBarLabelFormatEntry->SetBalloonHelpString(
    "Set the labels format (printf() style).");

  this->SetScalarBarLabelFormat(
    this->GetLabelFormatInternal());

  this->LabelTextPropertyWidget->SetParent(this->ScalarBarLabelFormatFrame);

  // this will update this->LabelTextProperty
  this->GetLabelTextPropertyInternal();
  this->LabelTextPropertyWidget->SetTextProperty(
    this->LabelTextProperty);
  this->LabelTextPropertyWidget->Create(app);
  this->LabelTextPropertyWidget->SetChangedCommand(this, 
    "LabelTextPropertyWidgetCallback");
  this->LabelTextPropertyWidget->GetTraceHelper()->SetReferenceHelper(
    this->GetTraceHelper());
  this->LabelTextPropertyWidget->GetTraceHelper()->SetReferenceCommand(
    "GetLabelTextPropertyWidget");

  this->Script("grid %s -row 0 -column 0 -sticky nws %s",
               this->ScalarBarLabelFormatLabel->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 0 -column 1 -sticky news %s",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 1 -column 1 -sticky nws %s",
               this->LabelTextPropertyWidget->GetWidgetName(),
               grid_settings);

  // Scalar bar : enable copy between text property widgets

  this->TitleTextPropertyWidget->ShowCopyOn();
  this->TitleTextPropertyWidget->GetCopyButton()->SetBalloonHelpString(
    "Copy the labels text properties to the title text properties.");
  ostrstream copy1;
  copy1 << "CopyValuesFrom " << this->LabelTextPropertyWidget->GetTclName() << ends;
  this->TitleTextPropertyWidget->GetCopyButton()->SetCommand(
    this->TitleTextPropertyWidget, copy1.str());
  copy1.rdbuf()->freeze(0);

  this->LabelTextPropertyWidget->ShowCopyOn();
  this->LabelTextPropertyWidget->GetCopyButton()->SetBalloonHelpString(
    "Copy the title text properties to the labels text properties.");
  ostrstream copy2;
  copy2 << "CopyValuesFrom " << this->TitleTextPropertyWidget->GetTclName() << ends;
  this->LabelTextPropertyWidget->GetCopyButton()->SetCommand(
    this->LabelTextPropertyWidget, copy2.str());
  copy2.rdbuf()->freeze(0);

  // Scalar bar: synchronize all those grids to have them aligned

  const char *widgets[2];this->PVRenderView->EventuallyRender();
  widgets[0] = this->ScalarBarTitleFrame->GetWidgetName();
  widgets[1] = this->ScalarBarLabelFormatFrame->GetWidgetName();

  int weights[2];
  weights[0] = 0;
  weights[1] = 4;

  float factors[2];
  factors[0] = 1.15;
  factors[1] = 0.0;

  vtkKWTkUtilities::SynchroniseGridsColumnMinimumSize(
    this->GetApplication()->GetMainInterp(), 2, widgets, factors, weights);

  // Scalar bar : pack/grid

  this->Script("pack %s -side top -fill y -anchor w",
               this->ScalarBarCheck->GetWidgetName());
  
  this->Script("pack %s %s -side top -expand t -fill both -anchor w",
               this->ScalarBarTitleFrame->GetWidgetName(),
               this->ScalarBarLabelFormatFrame->GetWidgetName());

  // Back button

  this->BackButton->SetParent(this);
  this->BackButton->Create(app);
  this->BackButton->SetText("Back");
  this->BackButton->SetCommand(this, "BackButtonCallback");

  // Pack

  this->Script("pack %s -side top -anchor n -fill x -padx 2 -pady 2", 
               this->ColorMapFrame->GetWidgetName());
  if (this->NumberOfVectorComponents > 1)
    {
    this->Script("pack %s -side top -anchor n -fill x -padx 2 -pady 2", 
                 this->VectorFrame->GetWidgetName());
    }
  this->Script("pack %s %s -side top -anchor n -fill x -padx 2 -pady 2", 
               this->ScalarBarFrame->GetWidgetName(),
               this->BackButton->GetWidgetName());

  this->SetColorSchemeToBlueRed();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::TitleTextPropertyWidgetCallback()
{
  double *color = this->TitleTextProperty->GetColor();
  this->SetTitleColor(color[0],color[1],color[2]);
  this->SetTitleOpacity(this->TitleTextProperty->GetOpacity());
  this->SetTitleFontFamily(this->TitleTextProperty->GetFontFamily());
  this->SetTitleBold(this->TitleTextProperty->GetBold());
  this->SetTitleItalic(this->TitleTextProperty->GetItalic());
  this->SetTitleShadow(this->TitleTextProperty->GetShadow());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::LabelTextPropertyWidgetCallback()
{
  double *color = this->LabelTextProperty->GetColor();
  this->SetLabelColor(color[0],color[1],color[2]);
  this->SetLabelOpacity(this->LabelTextProperty->GetOpacity());
  this->SetLabelFontFamily(this->LabelTextProperty->GetFontFamily());
  this->SetLabelBold(this->LabelTextProperty->GetBold());
  this->SetLabelItalic(this->LabelTextProperty->GetItalic());
  this->SetLabelShadow(this->LabelTextProperty->GetShadow());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::GetLabelTextPropertyInternal()
{
  vtkSMDoubleVectorProperty* dvp;
  vtkSMIntVectorProperty* ivp;
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("LabelTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty proxy.");
    return;
    }
 
  vtkSMProxy* textProperty = pp->GetProxy(0);

  dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    textProperty->GetProperty("Color"));
  if (dvp)
    {
    this->LabelTextProperty->SetColor(dvp->GetElements());
    }
  
  dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    textProperty->GetProperty("Opacity"));
  if (dvp)
    {
    this->LabelTextProperty->SetOpacity(dvp->GetElement(0));
    }
  
  ivp = vtkSMIntVectorProperty::SafeDownCast(
    textProperty->GetProperty("FontFamily"));
  if (ivp)
    {
    this->LabelTextProperty->SetFontFamily(ivp->GetElement(0));
    }

  ivp = vtkSMIntVectorProperty::SafeDownCast(
    textProperty->GetProperty("Bold"));
  if (ivp)
    {
    this->LabelTextProperty->SetBold(ivp->GetElement(0));
    }

  ivp = vtkSMIntVectorProperty::SafeDownCast(
    textProperty->GetProperty("Italic"));
  if (ivp)
    {
    this->LabelTextProperty->SetItalic(ivp->GetElement(0));
    }

  ivp = vtkSMIntVectorProperty::SafeDownCast(
    textProperty->GetProperty("Shadow"));
  if (ivp)
    {
    this->LabelTextProperty->SetShadow(ivp->GetElement(0));
    }
}


//----------------------------------------------------------------------------
void vtkPVColorMap::GetTitleTextPropertyInternal()
{
  vtkSMDoubleVectorProperty* dvp;
  vtkSMIntVectorProperty* ivp;

  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("TitleTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty proxy.");
    return;
    }
 
  vtkSMProxy* textProperty = pp->GetProxy(0);
  
  dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    textProperty->GetProperty("Color"));
  if (dvp)
    {
    this->TitleTextProperty->SetColor(dvp->GetElements());
    }
  
  dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    textProperty->GetProperty("Opacity"));
  if (dvp)
    {
    this->TitleTextProperty->SetOpacity(dvp->GetElement(0));
    }
  
  ivp = vtkSMIntVectorProperty::SafeDownCast(
    textProperty->GetProperty("FontFamily"));
  if (ivp)
    {
    this->TitleTextProperty->SetFontFamily(ivp->GetElement(0));
    }

  ivp = vtkSMIntVectorProperty::SafeDownCast(
    textProperty->GetProperty("Bold"));
  if (ivp)
    {
    this->TitleTextProperty->SetBold(ivp->GetElement(0));
    }

  ivp = vtkSMIntVectorProperty::SafeDownCast(
    textProperty->GetProperty("Italic"));
  if (ivp)
    {
    this->TitleTextProperty->SetItalic(ivp->GetElement(0));
    }

  ivp = vtkSMIntVectorProperty::SafeDownCast(
    textProperty->GetProperty("Shadow"));
  if (ivp)
    {
    this->TitleTextProperty->SetShadow(ivp->GetElement(0));
    }

}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateVectorComponentMenu()
{
  int idx;
  char numStr[5];

  if (this->NumberOfVectorComponents  == 1)
    {
    return;
    }

  this->VectorComponentMenu->DeleteAllEntries();
  if ( *(this->VectorComponentTitles[this->VectorComponent]) != '\0')
    {
    this->VectorComponentMenu->SetValue(
                         this->VectorComponentTitles[this->VectorComponent]);
    }
  else
    {
    sprintf(numStr, "%d", this->VectorComponent + 1);
    this->VectorComponentMenu->SetValue(numStr);
    } 

  for (idx = 0; idx < this->NumberOfVectorComponents; ++idx)
    {
    char command[64];
    sprintf(command, "VectorComponentCallback %d", idx);
    if (*(this->VectorComponentTitles[idx]) != '\0')
      {                                       
      this->VectorComponentMenu->AddEntryWithCommand(
                                           this->VectorComponentTitles[idx], 
                                           this, command);
      }
    else
      {
      sprintf(numStr, "%d", idx+1);
      this->VectorComponentMenu->AddEntryWithCommand(numStr, this, command);
      }
    }
} 

//----------------------------------------------------------------------------
void vtkPVColorMap::SetNumberOfVectorComponents(int  num)
{
  int idx;

  if (this->IsCreated() && num != 0)
    {
    vtkErrorMacro("You must set the number of vector components before "
      "you create this color map.");
    return;
    }

  if (num == this->NumberOfVectorComponents)
    {
    return;
    }

  // Get rid of old arrays.
  // Use for delete.  This number shold not be changed after creation.
  if (this->VectorComponentTitles)
    {
    for (idx = 0; idx < this->NumberOfVectorComponents; ++idx)
      {
      delete [] this->VectorComponentTitles[idx];
      this->VectorComponentTitles[idx] = NULL;
      }
    }

  delete[] this->VectorComponentTitles;
  this->VectorComponentTitles = NULL;

  this->NumberOfVectorComponents = num;

  // Set defaults for component titles.
  if (num > 0)
    {
    this->VectorComponentTitles = new char* [num];
    }
  for (idx = 0; idx < num; ++idx)
    {
    this->VectorComponentTitles[idx] = new char[4];  
    }
  if (num == 3)
    { // Use XYZ for default of three component vectors.
    strcpy(this->VectorComponentTitles[0], "X");
    strcpy(this->VectorComponentTitles[1], "Y");
    strcpy(this->VectorComponentTitles[2], "Z");
    }
  else
    {
    for (idx = 0; idx < num; ++idx)
      {
      sprintf(this->VectorComponentTitles[idx], "%d", idx+1);
      }
    }
  const char* arrayname = this->GetArrayName();
  if ( arrayname != NULL)
    {
    char *str2;
    str2 = new char [strlen(arrayname) + 128];
    sprintf(str2, "GetPVColorMap {%s} %d", arrayname, 
      this->NumberOfVectorComponents);
    this->GetTraceHelper()->SetReferenceCommand(str2);
    delete [] str2;
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::CreateParallelTclObjects(vtkPVApplication *pvApp)
{
  this->vtkKWObject::SetApplication(pvApp);
  
  vtkSMProxyManager* pxm = vtkSMObject::GetProxyManager();
  static int proxyNum = 0;
  // Create LookupTableProxy
  this->LookupTableProxy = vtkSMLookupTableProxy::SafeDownCast(
    pxm->NewProxy("lookup_tables","LookupTable"));
  if (!this->LookupTableProxy)
    {
    vtkErrorMacro("Failed to create LookupTableProxy");
    return;
    }
  this->LookupTableProxy->SetServers(
    vtkProcessModule::CLIENT | vtkProcessModule::RENDER_SERVER);
  ostrstream str1;
  str1 << "PVColorMap_LookupTable" << proxyNum << ends;
  this->SetLookupTableProxyName(str1.str());
  str1.rdbuf()->freeze(0);
  proxyNum++;
  pxm->RegisterProxy("lookup_tables", this->LookupTableProxyName , 
    this->LookupTableProxy);
  this->LookupTableProxy->CreateVTKObjects(1);
    
  // Create ScalarBarProxy
  this->ScalarBarProxy = vtkSMScalarBarWidgetProxy::SafeDownCast(
    pxm->NewProxy("displays","ScalarBarWidget"));
  if (!this->ScalarBarProxy)
    {
    vtkErrorMacro("Failed to create ScalarBarWidget proxy");
    return;
    }
  ostrstream str;
  str << "PVColorMap_ScalarBarWidget" << proxyNum << ends;
  this->SetScalarBarProxyName(str.str());
  str.rdbuf()->freeze(0);
  proxyNum++;
  pxm->RegisterProxy("displays",this->ScalarBarProxyName,this->ScalarBarProxy);
  this->ScalarBarProxy->SetServers(
    vtkProcessModule::CLIENT | vtkProcessModule::RENDER_SERVER);
  this->ScalarBarProxy->UpdateVTKObjects();
  this->InitializeObservers();

  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("LookupTable"));
  if (!pp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property LookupTable");
    return;
    }
  pp->RemoveAllProxies();
  pp->AddProxy(this->LookupTableProxy);
  this->ScalarBarProxy->UpdateVTKObjects();

  // Add to rendermodule.
  vtkSMRenderModuleProxy* rm = this->GetPVApplication()->GetRenderModuleProxy();
  pp =  vtkSMProxyProperty::SafeDownCast(
    rm->GetProperty("Displays"));
  if (!pp)
    {
    vtkErrorMacro("Failed to find property Displays on RenderModuleProxy.");
    }
  else
    {
    pp->AddProxy(this->ScalarBarProxy);
    rm->UpdateVTKObjects();
    }
}

//----------------------------------------------------------------------------
vtkSMProxy* vtkPVColorMap::GetProxyByName(const char* name)
{
  if (strcmp(name,"LookupTable") == 0)
    {
    return this->LookupTableProxy;
    }
  if (strcmp(name,"ScalarBarWidget") == 0)
    {
    return this->ScalarBarProxy;
    }
  vtkErrorMacro("Unknow proxy name : " << name);
  return NULL;
}
//----------------------------------------------------------------------------
void vtkPVColorMap::InitializeObservers()
{
  this->ScalarBarObserver = vtkPVColorMapObserver::New();
  this->ScalarBarObserver->PVColorMap = this;

  this->ScalarBarProxy->AddObserver(vtkCommand::InteractionEvent,
    this->ScalarBarObserver);
  this->ScalarBarProxy->AddObserver(vtkCommand::StartInteractionEvent, 
    this->ScalarBarObserver);
  this->ScalarBarProxy->AddObserver(vtkCommand::EndInteractionEvent, 
    this->ScalarBarObserver);
  this->ScalarBarProxy->AddObserver(vtkCommand::WidgetModifiedEvent,
    this->ScalarBarObserver);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::RenderView()
{
  if (this->PVRenderView)
    {
    this->PVRenderView->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::BackButtonCallback()
{
  if (this->PVRenderView == NULL)
    {
    return;
    }

  // This has a side effect of gathering and display information.
  this->PVRenderView->GetPVWindow()->GetCurrentPVSource()->GetPVOutput()->Update();
  this->PVRenderView->GetPVWindow()->ShowCurrentSourceProperties();
}


//----------------------------------------------------------------------------
void vtkPVColorMap::SetArrayName(const char* str)
{

  this->SetArrayNameInternal(str);
  if (str)
    {
    char *tmp;
    tmp = new char[strlen(str)+128];
    sprintf(tmp, "Parameter: %s", str);
    this->ArrayNameLabel->SetText(tmp);
    delete [] tmp;
    }

  if (str != NULL)
    {
    char *str2;
    str2 = new char [strlen(str) + 128];
    sprintf(str2, "GetPVColorMap {%s} %d", str, this->NumberOfVectorComponents);
    this->GetTraceHelper()->SetReferenceCommand(str2);
    delete [] str2;
    }
  this->Modified();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetArrayNameInternal(const char* str)
{
  vtkSMStringVectorProperty* svp = vtkSMStringVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("ArrayName"));
  if (!svp)
    {
    vtkErrorMacro("LookupTableProxy does not have property ArrayName");
    return;
    }
  svp->SetElement(0,str);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
int vtkPVColorMap::MatchArrayName(const char* str, int numberOfComponents)
{
  const char* arrayname = this->GetArrayName();
  if (str == NULL || arrayname == NULL)
    {
    return 0;
    }
  if (strcmp(str, arrayname) == 0 &&
    numberOfComponents == this->NumberOfVectorComponents)
    {
    return 1;
    }
  return 0;
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarBarTitleEntryCallback()
{
  this->SetScalarBarTitle(this->ScalarBarTitleEntry->GetValue());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarTitle(const char* name)
{
  if (this->ScalarBarTitle == NULL && name == NULL) 
    { 
    return;
    }

  if (this->ScalarBarTitle && name && (!strcmp(this->ScalarBarTitle, name))) 
    { 
    return;
    }

  if (this->ScalarBarTitle) 
    { 
    delete [] this->ScalarBarTitle; 
    this->ScalarBarTitle = NULL;
    }

  if (name)
    {
    this->ScalarBarTitle = new char[strlen(name) + 1];
    strcpy(this->ScalarBarTitle, name);
    } 
  this->ScalarBarTitleEntry->SetValue(this->ScalarBarTitle);
  this->UpdateScalarBarTitle();
  this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarBarTitle {%s}", 
    this->GetTclName(), name);
  this->RenderView();
  this->Modified();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleInternal(const char* title)
{
  vtkSMStringVectorProperty *svp = vtkSMStringVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("Title"));
  if (!svp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Title");
    return;
    }
  svp->SetElement(0,title);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarBarVectorTitleEntryCallback()
{
  this->SetScalarBarVectorTitle(this->ScalarBarVectorTitleEntry->GetValue());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarVectorTitle(const char* name)
{
  if (this->GetVectorModeInternal() == vtkLookupTable::MAGNITUDE)
    {
    if (this->VectorMagnitudeTitle == NULL && name == NULL) 
      { 
      return;
      }

    if (this->VectorMagnitudeTitle && name && 
      (!strcmp(this->VectorMagnitudeTitle, name)))
      {
      return;
      }

    if (this->VectorMagnitudeTitle)
      {
      delete [] this->VectorMagnitudeTitle;
      this->VectorMagnitudeTitle = NULL;
      }

    if (name)
      {
      this->VectorMagnitudeTitle = new char[strlen(name) + 1];
      strcpy(this->VectorMagnitudeTitle, name);
      }

    this->ScalarBarVectorTitleEntry->SetValue(name);

    this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarBarVectorTitle {%s}", 
      this->GetTclName(), name);
    }
  else
    {
    if (this->VectorComponentTitles == NULL)
      {
      return;
      }

    if (this->VectorComponentTitles[this->VectorComponent] == NULL && 
      name == NULL) 
      { 
      return;
      }

    if (this->VectorComponentTitles[this->VectorComponent] && 
      name && 
      (!strcmp(this->VectorComponentTitles[this->VectorComponent], name)))
      {
      return;
      }

    if (this->VectorComponentTitles[this->VectorComponent])
      {
      delete [] this->VectorComponentTitles[this->VectorComponent];
      this->VectorComponentTitles[this->VectorComponent] = NULL;
      }

    if (name)
      {
      this->VectorComponentTitles[this->VectorComponent] = 
        new char[strlen(name) + 1];
      strcpy(this->VectorComponentTitles[this->VectorComponent], name);
      }

    this->ScalarBarVectorTitleEntry->SetValue(name);

    this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarBarVectorTitle {%s}", 
      this->GetTclName(), name);

    this->UpdateVectorComponentMenu();

    }
    this->UpdateScalarBarTitle();
    this->Modified();
    this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarBarLabelFormatEntryCallback()
{
  this->SetScalarBarLabelFormat(this->ScalarBarLabelFormatEntry->GetValue());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarLabelFormat(const char* name)
{
  this->ScalarBarLabelFormatEntry->SetValue(name);
  this->SetLabelFormatInternal(name);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarBarLabelFormat {%s}", 
                      this->GetTclName(), name);
  this->RenderView();
  this->Modified();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelFormatInternal(const char* format)
{
  vtkSMStringVectorProperty *svp = vtkSMStringVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("LabelFormat"));
  if (!svp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property LabelFormat");
    return;
    }
  svp->SetElement(0,format);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::NumberOfColorsScaleCallback()
{
  int num = (int)(this->NumberOfColorsScale->GetValue());
  this->SetNumberOfColors(num);
}

//----------------------------------------------------------------------------
// Access for trace files.
void vtkPVColorMap::SetNumberOfColors(int num)
{
  this->NumberOfColorsScale->SetValue(num);

  this->SetNumberOfColorsInternal(num);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetNumberOfColors %d", this->GetTclName(),
    num);
  this->UpdateMap();
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetNumberOfColorsInternal(int num)
{
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("NumberOfTableValues"));
  if (!ivp)
    {
    vtkErrorMacro("LookupTableProxy does not have property NumberOfTableValues");
    return;
    }
  ivp->SetElement(0,num);
  this->LookupTableProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
int vtkPVColorMap::GetNumberOfColorsInternal()
{
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("NumberOfTableValues"));
  if (!ivp)
    {
    vtkErrorMacro("LookupTableProxy does not have property NumberOfTableValues");
    return 0;
    }
  return ivp->GetElement(0);
}
//----------------------------------------------------------------------------
void vtkPVColorMap::SetColorSchemeToRedBlue()
{
  double hr[2],sr[2],vr[2];
  hr[0] = VTK_PV_COLOR_MAP_RED_HUE;
  sr[0] = VTK_PV_COLOR_MAP_RED_SATURATION;
  vr[0] = VTK_PV_COLOR_MAP_RED_VALUE;
  
  hr[1] = VTK_PV_COLOR_MAP_BLUE_HUE;
  sr[1] = VTK_PV_COLOR_MAP_BLUE_SATURATION;
  vr[1] = VTK_PV_COLOR_MAP_BLUE_VALUE;

  this->StartColorButton->SetColor(1.0, 0.0, 0.0);
  this->EndColorButton->SetColor(0.0, 0.0, 1.0);

  /*this->SetHueRangeInternal(hr);
  this->SetSaturationRangeInternal(sr);
  this->SetValueRangeInternal(vr);*/
  this->SetHSVRangesInternal(hr,sr,vr);

  this->GetTraceHelper()->AddEntry("$kw(%s) SetColorSchemeToRedBlue", this->GetTclName());
  this->UpdateMap();
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetColorSchemeToBlueRed()
{
  double hr[2],sr[2],vr[2];
  hr[0] = VTK_PV_COLOR_MAP_BLUE_HUE;
  sr[0] = VTK_PV_COLOR_MAP_BLUE_SATURATION;
  vr[0] = VTK_PV_COLOR_MAP_BLUE_VALUE;
  hr[1] = VTK_PV_COLOR_MAP_RED_HUE;
  sr[1] = VTK_PV_COLOR_MAP_RED_SATURATION;
  vr[1] = VTK_PV_COLOR_MAP_RED_VALUE;

  this->StartColorButton->SetColor(0.0, 0.0, 1.0);
  this->EndColorButton->SetColor(1.0, 0.0, 0.0);

  /*this->SetHueRangeInternal(hr);
  this->SetSaturationRangeInternal(sr);
  this->SetValueRangeInternal(vr);*/
  this->SetHSVRangesInternal(hr,sr,vr);

  this->GetTraceHelper()->AddEntry("$kw(%s) SetColorSchemeToBlueRed", this->GetTclName());
  this->UpdateMap();
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetColorSchemeToGrayscale()
{
  double hr[2],sr[2],vr[2];
  hr[0] = VTK_PV_COLOR_MAP_BLACK_HUE;
  sr[0] = VTK_PV_COLOR_MAP_BLACK_SATURATION;
  vr[0] = VTK_PV_COLOR_MAP_BLACK_VALUE;
  hr[1] = VTK_PV_COLOR_MAP_WHITE_HUE;
  sr[1] = VTK_PV_COLOR_MAP_WHITE_SATURATION;
  vr[1] = VTK_PV_COLOR_MAP_WHITE_VALUE;

  this->StartColorButton->SetColor(0.0, 0.0, 0.0);
  this->EndColorButton->SetColor(1.0, 1.0, 1.0);

/*  this->SetHueRangeInternal(hr);
  this->SetSaturationRangeInternal(sr);
  this->SetValueRangeInternal(vr);*/
  this->SetHSVRangesInternal(hr,sr,vr);

  this->GetTraceHelper()->AddEntry("$kw(%s) SetColorSchemeToGrayscale", this->GetTclName());
  this->UpdateMap();
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetColorSchemeToLabBlueRed()
{
  double hr[2],sr[2],vr[2];
  double startHSV[3];
  double endHSV[3];
  //first calculate the gradient values
  //LabBlue[0]=83.91;
  //LabBlue[1]=-8.65;
  //LabBlue[2]=-12.18;
  hr[0]=57.93;
  sr[0]=-26.85;
  vr[0]=-30.21;
  startHSV[0] = hr[0];
  startHSV[1] = sr[0];
  startHSV[2] = vr[0];
  //LabRed[0]=83.91;
  //LabRed[1]=8.65;
  //LabRed[2]=12.18;
  
  hr[1]=57.93;
  sr[1]=26.85;
  vr[1]=30.21;
  endHSV[0] = hr[1];
  endHSV[1] = sr[1];
  endHSV[2] = vr[1];

  double rgb[3];
  double xyz[3];
  vtkMath::LabToXYZ(startHSV,xyz);
  vtkMath::XYZToRGB(xyz,rgb);
  this->StartColorButton->SetColor(rgb[0],rgb[1],rgb[2]);
  vtkMath::LabToXYZ(endHSV,xyz);
  vtkMath::XYZToRGB(xyz,rgb);
  this->EndColorButton->SetColor(rgb[0],rgb[1],rgb[2]);

/*  this->SetHueRangeInternal(hr);
  this->SetSaturationRangeInternal(sr);
  this->SetValueRangeInternal(vr);*/
  this->SetHSVRangesInternal(hr,sr,vr);

  this->GetTraceHelper()->AddEntry("$kw(%s) SetColorSchemeToLabBlueRed", this->GetTclName());
  this->UpdateMap();
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::StartColorButtonCallback(double r, double g, double b)
{
  double rgb[3];
  double hsv[3];

  // Convert RGB to HSV.
  rgb[0] = r;
  rgb[1] = g;
  rgb[2] = b;
  vtkMath::RGBToHSV(rgb, hsv);

  this->SetStartHSV(hsv[0], hsv[1], hsv[2]);
  
  // Lab color map uses hue > 1.1 as a flag.
  // We need to make sure the end color hue is not > 1.
  double hrange[2];
  this->GetHueRangeInternal(hrange);
  if (hrange[1] > 1.1)
    {
    double *tmp = this->EndColorButton->GetColor();
    vtkMath::RGBToHSV(tmp, hsv);
    this->SetEndHSV(hsv[0], hsv[1], hsv[2]);
    }  
}

//----------------------------------------------------------------------------
// Access for trace files.
void vtkPVColorMap::SetStartHSV(double h, double s, double v)
{
  double hr[2],sr[2],vr[2];
  double hsv[3];
  double rgb[3];

  this->GetHueRangeInternal(hr);
  this->GetSaturationRangeInternal(sr);
  this->GetValueRangeInternal(vr);

  if ( h == hr[0] && s == sr[0] && v == vr[0])
    {
    return;
    }

  // Change color button (should have no callback effect...)
  hsv[0] = h;  hsv[1] = s;  hsv[2] = v;
  if (hsv[0] > 1.1)
    { // Detect the Sandia hack for changing the interpolation.
    double xyz[3];
    vtkMath::LabToXYZ(hsv,xyz);
    vtkMath::XYZToRGB(xyz,rgb);
    this->StartColorButton->SetColor(rgb[0],rgb[1],rgb[2]);    
    }
  else
    {
    vtkMath::HSVToRGB(hsv, rgb);
    this->StartColorButton->SetColor(rgb);
    }
  hr[0] = h; sr[0] = s; vr[0] = v;
/*  this->SetHueRangeInternal(hr);
  this->SetSaturationRangeInternal(sr);
  this->SetValueRangeInternal(vr);*/
  this->SetHSVRangesInternal(hr,sr,vr);

  this->GetTraceHelper()->AddEntry("$kw(%s) SetStartHSV %g %g %g", 
                      this->GetTclName(), h, s, v);

  this->UpdateMap();
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetHSVRangesInternal(double hrange[2],
  double srange[2], double vrange[2])
{
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("HueRange"));
  if (!dvp)
    {
    vtkErrorMacro("LookupTableProxy does not have property HueRange");
    }
  dvp->SetElements(hrange);

  dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("SaturationRange"));
  if (!dvp)
    {
    vtkErrorMacro("LookupTableProxy does not have property SaturationRange");
    }
  dvp->SetElements(srange);

  dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("ValueRange"));
  if (!dvp)
    {
    vtkErrorMacro("LookupTableProxy does not have property ValueRange");
    }
  dvp->SetElements(vrange);

  this->LookupTableProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
/*void vtkPVColorMap::SetSaturationRangeInternal(double range[2])
{
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("SaturationRange"));
  if (!dvp)
    {
    vtkErrorMacro("ScalarBarProxy does not have propery SaturationRange");
    }
  dvp->SetElements(range);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetValueRangeInternal(double range[2])
{
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("ValueRange"));
  if (!dvp)
    {
    vtkErrorMacro("ScalarBarProxy does not have propery ValueRange");
    }
  dvp->SetElements(range);
  this->ScalarBarProxy->UpdateVTKObjects();
}
*/
//----------------------------------------------------------------------------
void vtkPVColorMap::EndColorButtonCallback(double r, double g, double b)
{
  double rgb[3];
  double hsv[3];

  // Convert RGB to HSV.
  rgb[0] = r;
  rgb[1] = g;
  rgb[2] = b;
  vtkMath::RGBToHSV(rgb, hsv);

  this->SetEndHSV(hsv[0], hsv[1], hsv[2]);

  // Lab color map uses hue > 1.1 as a flag.
  // We need to make sure the start color hue is not > 1.
  double hrange[2];
  this->GetHueRangeInternal(hrange);
  if (hrange[0] > 1.1)
    {
    double *tmp = this->StartColorButton->GetColor();
    vtkMath::RGBToHSV(tmp, hsv);
    this->SetStartHSV(hsv[0], hsv[1], hsv[2]);
    }
}


//----------------------------------------------------------------------------
// Access for trace files.
void vtkPVColorMap::SetEndHSV(double h, double s, double v)
{
  double hr[2],sr[2],vr[2];
  double hsv[3];
  double rgb[3];
  
  this->GetHueRangeInternal(hr);
  this->GetSaturationRangeInternal(sr);
  this->GetValueRangeInternal(vr);
  
  if ( hr[1] == h && sr[1] == s && vr[1] == v)
    {
    return;
    }

  // Change color button (should have no callback effect...)
  hsv[0] = h;  hsv[1] = s;  hsv[2] = v;
  if (hsv[0] > 1.1)
    { // Detect the Sandia hack for changing the interpolation.
    double xyz[3];
    vtkMath::LabToXYZ(hsv,xyz);
    vtkMath::XYZToRGB(xyz,rgb);
    this->EndColorButton->SetColor(rgb[0],rgb[1],rgb[2]);    
    }
  else
    {
    vtkMath::HSVToRGB(hsv, rgb);
    this->EndColorButton->SetColor(rgb);
    }
  hr[1] = h; sr[1] = s; vr[1] = v;
  this->SetHSVRangesInternal(hr,sr,vr);
  //this->SetHueRangeInternal(hr);
  //this->SetSaturationRangeInternal(sr);
  //this->SetValueRangeInternal(vr);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetEndHSV %g %g %g", 
                      this->GetTclName(), h, s, v);
  this->UpdateMap();
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarBarCheckCallback()
{
  this->SetScalarBarVisibility(this->ScalarBarCheck->GetState());
  if ( this->GetPVRenderView() )
    {
    this->GetPVRenderView()->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::LockCheckCallback()
{
  this->SetScalarRangeLock(this->ScalarRangeLockCheck->GetState());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarRangeLock(int val)
{
  if (this->ScalarRangeLock == val)
    {
    return;
    }
  this->ScalarRangeLock = val;
  this->ScalarRangeLockCheck->SetState(val);
  if ( ! val)
    {
    this->ResetScalarRange();
    }

  // Trace.
  this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarRangeLock %d", 
                      this->GetTclName(), val);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetVectorComponent(int component)
{
  this->SetVectorComponentInternal(component);
  this->UpdateVectorComponentMenu();
  this->UpdateScalarBarTitle();
  this->GetTraceHelper()->AddEntry("$kw(%s) SetVectorComponent %d", 
                      this->GetTclName(), component);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetVectorComponentInternal(int component)
{
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("VectorComponent"));
  if (!ivp)
    {
    vtkErrorMacro("LookupTableProxy does not have property VectorComponent");
    return;
    }
  ivp->SetElement(0,component);
  this->VectorComponent = component;
  this->LookupTableProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ResetScalarRange()
{
  this->ResetScalarRangeInternal();
  this->GetTraceHelper()->AddEntry("$kw(%s) ResetScalarRange", this->GetTclName());
}

//----------------------------------------------------------------------------
vtkPVApplication* vtkPVColorMap::GetPVApplication()
{
  if (this->GetApplication() == NULL)
    {
    return NULL;
    }
  
  if (this->GetApplication()->IsA("vtkPVApplication"))
    {  
    return (vtkPVApplication*)(this->GetApplication());
    }
  else
    {
    vtkErrorMacro("Bad typecast");
    return NULL;
    } 
}

//----------------------------------------------------------------------------
void vtkPVColorMap::IncrementUseCount()
{
  ++this->UseCount;
  this->UpdateInternalScalarBarVisibility();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::DecrementUseCount()
{
  --this->UseCount;
  this->UpdateInternalScalarBarVisibility();
}


//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarVisibility(int val)
{
  if (this->ScalarBarVisibility == val)
    {
    return;
    }
  this->ScalarBarVisibility = val;
  
  // Make sure the UI is up to date.
  if (val)
    {
    this->ScalarBarCheck->SetState(1);
    }
  else
    {
    this->ScalarBarCheck->SetState(0);
    }

  this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarBarVisibility %d", this->GetTclName(),
                      val);
  
  this->UpdateInternalScalarBarVisibility();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateInternalScalarBarVisibility()
{
  int visible = this->ScalarBarVisibility;

  if (this->UseCount == 0)
    {
    visible = 0;
    }

  if (this->InternalScalarBarVisibility == visible)
    {
    return;
    }
  this->InternalScalarBarVisibility = visible;


  if (!this->GetPVRenderView())
    {
    return;
    }
  
  this->SetVisibilityInternal(visible);
  this->Modified();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetVisibilityInternal(int visible)
{
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("Visibility"));
  if (!ivp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Visibility");
    return;
    }
  ivp->SetElement(0,visible);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SaveInBatchScript(ofstream *file)
{
  if (this->VisitedFlag)
    {
    return;
    }
  this->VisitedFlag = 1;
  this->LookupTableProxy->SaveInBatchScript(file);
  this->ScalarBarProxy->SaveInBatchScript(file);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::GetHueRangeInternal(double range[2])
{
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("HueRange"));
  if (!dvp || dvp->GetNumberOfElements() != 2)
    {
    vtkErrorMacro("LookupTableProxy does not have property HueRange");
    range[0] = range[1] = 0.0;
    return;
    }
  range[0] = dvp->GetElement(0);
  range[1] = dvp->GetElement(1);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::GetSaturationRangeInternal(double range[2])
{
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("SaturationRange"));
  if (!dvp || dvp->GetNumberOfElements() != 2)
    {
    vtkErrorMacro("LookupTableProxy does not have property SaturationRange");
    range[0] = range[1] = 0.0;
    return;
    }
  range[0] = dvp->GetElement(0);
  range[1] = dvp->GetElement(1);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::GetValueRangeInternal(double range[2])
{
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("ValueRange"));
  if (!dvp || dvp->GetNumberOfElements() != 2)
    {
    vtkErrorMacro("LookupTableProxy does not have property ValueRange");
    range[0] = range[1] = 0.0;
    return;
    }
  range[0] = dvp->GetElement(0);
  range[1] = dvp->GetElement(1);
}

//----------------------------------------------------------------------------
int vtkPVColorMap::GetVectorModeInternal()
{
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("VectorMode"));
  if (!ivp)
    {
    vtkErrorMacro("LookupTableProxy does not have property VectorMode");
    return 0;
    }
  return ivp->GetElement(0);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateScalarBarTitle()
{
  if (this->ScalarBarTitle == NULL || this->GetPVApplication() == NULL)
    {
    return;
    }
  
  if (this->GetVectorModeInternal() == vtkLookupTable::MAGNITUDE &&
      this->NumberOfVectorComponents > 1)
    {
    ostrstream ostr;
    ostr << this->ScalarBarTitle << " " << this->VectorMagnitudeTitle << ends;
    this->SetTitleInternal(ostr.str()); 
    ostr.rdbuf()->freeze(0);    
    this->ScalarBarVectorTitleEntry->SetValue(this->VectorMagnitudeTitle);
    }
  else if (this->NumberOfVectorComponents == 1)
    {
    this->SetTitleInternal(this->ScalarBarTitle);
    }
  else
    {
    ostrstream ostr;
    ostr << this->ScalarBarTitle << " " 
         << this->VectorComponentTitles[this->VectorComponent] << ends;
    this->SetTitleInternal(ostr.str());
    ostr.rdbuf()->freeze(0);    
    this->ScalarBarVectorTitleEntry->SetValue(
      this->VectorComponentTitles[this->VectorComponent]);
    }
  
}
//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarRangeWidgetCallback()
{
  double range[2];

  this->ScalarRangeWidget->GetRange(range);
  this->SetScalarRange(range[0], range[1]);
      
  if ( this->GetPVRenderView() )
    {
    this->GetPVRenderView()->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::MapConfigureCallback(int width, int height)
{
  this->UpdateMap(width, height);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetVectorModeInternal(int mode)
{
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("VectorMode"));
  if (!ivp)
    {
    vtkErrorMacro("LookupTableProxy does not have property VectorMode");
    return;
    }
  ivp->SetElement(0,mode);
  this->LookupTableProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::VectorModeMagnitudeCallback()
{
  this->GetTraceHelper()->AddEntry("$kw(%s) VectorModeMagnitudeCallback", 
                      this->GetTclName());
  this->SetVectorModeInternal(vtkLookupTable::MAGNITUDE);
  this->Script("pack forget %s",
               this->VectorComponentMenu->GetWidgetName());
  this->VectorModeMenu->SetValue("Magnitude");
  this->UpdateScalarBarTitle();
  this->ResetScalarRangeInternal();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::VectorModeComponentCallback()
{
  if (this->VectorComponentMenu->GetApplication() == NULL)
    { // Sanity check
    return;
    }

  this->GetTraceHelper()->AddEntry("$kw(%s) VectorModeComponentCallback", 
                      this->GetTclName());

  this->SetVectorModeInternal(vtkLookupTable::COMPONENT);
  this->Script("pack %s -side left -expand f -fill both -padx 2",
               this->VectorComponentMenu->GetWidgetName());
  this->VectorModeMenu->SetValue("Component");
  this->UpdateScalarBarTitle();
  this->ResetScalarRangeInternal();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::VectorComponentCallback(int component)
{
  this->SetVectorComponent(component);
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateMap()
{
  if (this->MapWidth && this->MapHeight)
    {
    this->UpdateMap(this->MapWidth,this->MapHeight);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateMap(int width, int height)
{
  int size;
  int i, j;
  double *range;
  double *wholeRange;
  double val, step;
  unsigned char *rgba;  
  unsigned char *ptr;  

  size = width*height;
  if (this->MapDataSize < size)
    {
    if (this->MapData)
      {
      delete [] this->MapData;
      }
    this->MapData = new unsigned char[size*3];
    this->MapDataSize = size;
    }
  this->MapWidth = width;
  this->MapHeight = height;

  vtkProcessModule* pm =vtkProcessModule::GetProcessModule();
  vtkLookupTable* lut = NULL;
  if (this->LookupTableProxy)
    {
    lut = vtkLookupTable::SafeDownCast(
      pm->GetObjectFromID(this->LookupTableProxy->GetID(0)));
    }
  if (!lut)
    {
    return;
    }

  range = this->ScalarRange;
  wholeRange = this->WholeScalarRange;

  //step = (range[1]-range[0])/(double)(width);
  step = (wholeRange[1]-wholeRange[0])/(double)(width);
  ptr = this->MapData;
  for (j = 0; j < height; ++j)
    {
    for (i = 0; i < width; ++i)
      {
      // This was the original.  Color map takes up the entire width.
      //val = range[0] + ((double)(i)*step);
      // Lets have x be constant whole range.
      // Color map shrinks as map range changes.
      val = wholeRange[0] + ((double)(i)*step);
      // Values out of range are background color.
      //if (val < range[0] || val > range[1])
      //  {
      //  ptr[0] = (unsigned char)(191);
      //  ptr[1] = (unsigned char)(191);
      //  ptr[2] = (unsigned char)(191);
      //  ptr += 3;
      //  }
      //else
      //  {
      
      // Colors clamp to min or max out of color map range.
      if (val < range[0])
        {
        val = range[0];
        }
      if (val > range[1])
        {
        val = range[1];
        }
        
      rgba = lut->MapValue(val);
      ptr[0] = rgba[0];
      ptr[1] = rgba[1];
      ptr[2] = rgba[2];
      ptr += 3;
      }
    }

  if (size > 0)
    {
    this->Map->SetImageOption(this->MapData, width, height, 3);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarPosition1(float x, float y)
{
  this->SetPosition1Internal(x,y);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarBarPosition1 %f %f", 
                      this->GetTclName(), x, y);
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetPosition1Internal(double x, double y)
{
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("Position"));
  if (!dvp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Position");
    return;
    }
  dvp->SetElement(0, x);
  dvp->SetElement(1, y);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::GetPosition1Internal(double pos[2])
{
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("Position"));
  if (!dvp || dvp->GetNumberOfElements() != 2 )
    {
    vtkErrorMacro("ScalarBarProxy does not have property Position"
      " or it does not have two elements");
    return;
    }
  pos[0] = dvp->GetElement(0);
  pos[1] = dvp->GetElement(1);
}
  

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarPosition2(float x, float y)
{
  this->SetPosition2Internal(x,y);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarBarPosition2 %f %f", 
                      this->GetTclName(), x, y);
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetPosition2Internal(double x, double y)
{
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("Position2"));
  if (!dvp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Position2");
    return;
    }
  dvp->SetElement(0, x);
  dvp->SetElement(1, y);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::GetPosition2Internal(double pos[2])
{
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("Position2"));
  if (!dvp && dvp->GetNumberOfElements() != 2 )
    {
    vtkErrorMacro("ScalarBarProxy does not have property Position2"
      " or it does not have two elements");
    return;
    }
  pos[0] = dvp->GetElement(0);
  pos[1] = dvp->GetElement(1);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarOrientation(int o)
{
  this->SetOrientationInternal(o);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarBarOrientation %d", 
                      this->GetTclName(), o);
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetOrientationInternal(int orientation)
{
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("Orientation"));
  if (!ivp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Orientation");
    return;
    }
  ivp->SetElement(0, orientation);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
int vtkPVColorMap::GetOrientationInternal()
{
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("Orientation"));
  if (!ivp && ivp->GetNumberOfElements() != 1 )
    {
    vtkErrorMacro("ScalarBarProxy does not have property Orientation"
      " or it does not have 1 element");
    return 0;
    }
  return ivp->GetElement(0); 
}
  
//----------------------------------------------------------------------------
const char* vtkPVColorMap::GetArrayName() 
{ 
  if (this->GetPVApplication() && this->ScalarBarProxy)
    {
    return this->GetArrayNameInternal();
    }
  return NULL;
}

//----------------------------------------------------------------------------
const char* vtkPVColorMap::GetArrayNameInternal()
{
  vtkSMStringVectorProperty* svp = vtkSMStringVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("ArrayName"));
  if (!svp || svp->GetNumberOfElements() != 1)
    {
    vtkErrorMacro("LookupTableProxy does not have property ArrayName");
    return NULL;
    }
  return svp->GetElement(0);
}

//----------------------------------------------------------------------------
int vtkPVColorMap::GetNumberOfVectorComponents()
{
  return this->NumberOfVectorComponents;
}
//----------------------------------------------------------------------------
const char* vtkPVColorMap::GetScalarBarLabelFormat()
{
  return this->GetLabelFormatInternal();
}

//----------------------------------------------------------------------------
const char* vtkPVColorMap::GetLabelFormatInternal()
{
  vtkSMStringVectorProperty* svp = vtkSMStringVectorProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("LabelFormat"));
  if (!svp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property LabelFormat");
    return "";
    }
  return svp->GetElement(0);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleColor(double r, double g, double b)
{
  this->SetTitleColorInternal(r,g,b);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetTitleColor %g %g %g", this->GetTclName(),
                      r, g, b);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleOpacity(double opacity)
{
  this->SetTitleOpacityInternal(opacity);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetTitleOpacity %g ", this->GetTclName(),
                      opacity);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleFontFamily(int font)
{
  this->SetTitleFontFamilyInternal(font);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetTitleFontFamily %d ", this->GetTclName(),
                      font);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleBold(int bold)
{
  this->SetTitleBoldInternal(bold);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetTitleBold %d ", this->GetTclName(),
                      bold);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleItalic(int italic)
{
  this->SetTitleItalicInternal(italic);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetTitleItalic %d ", this->GetTclName(),
                      italic);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleShadow(int shadow)
{
  this->SetTitleShadowInternal(shadow);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetTitleShadow %d ", this->GetTclName(),
                      shadow);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleColorInternal(double r, double g, double b)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("TitleTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    }
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("Color"));
  if (!dvp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Color");
    return;
    }
  dvp->SetElement(0,r);
  dvp->SetElement(1,g);
  dvp->SetElement(2,b);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleOpacityInternal(double opacity)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("TitleTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    } 
  
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("Opacity"));
  if (!dvp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property TitleFormatOpacity");
    return;
    }
  dvp->SetElement(0,opacity);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleFontFamilyInternal(int font)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("TitleTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    }
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("FontFamily"));
  if (!ivp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property FontFamily");
    return;
    }
  ivp->SetElement(0,font);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleBoldInternal(int bold)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("TitleTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    }

  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("Bold"));
  if (!ivp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Bold");
    return;
    }
  ivp->SetElement(0,bold);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleItalicInternal(int italic)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("TitleTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    }
  
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("Italic"));
  if (!ivp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Italic");
    return;
    }
  ivp->SetElement(0,italic);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetTitleShadowInternal(int shadow)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("TitleTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    }
  
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("Shadow"));
  if (!ivp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Shadow");
    return;
    }
  ivp->SetElement(0,shadow);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelColor(double r, double g, double b)
{
  this->SetLabelColorInternal(r,g,b);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetLabelColor %g %g %g", this->GetTclName(),
                      r, g, b);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelOpacity(double opacity)
{
  this->SetLabelOpacityInternal(opacity);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetLabelOpacity %g ", this->GetTclName(),
                      opacity);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelFontFamily(int font)
{
  this->SetLabelFontFamilyInternal(font);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetLabelFontFamily %d ", this->GetTclName(),
                      font);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelBold(int bold)
{
  this->SetLabelBoldInternal(bold);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetLabelBold %d ", this->GetTclName(),
                      bold);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelItalic(int italic)
{
  this->SetLabelItalicInternal(italic);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetLabelItalic %d ", this->GetTclName(),
                      italic);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelShadow(int shadow)
{
  this->SetLabelShadowInternal(shadow);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetLabelShadow %d ", this->GetTclName(),
                      shadow);
  this->Modified();
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelColorInternal(double r, double g, double b)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("LabelTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    }

  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("Color"));
  if (!dvp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Color");
    return;
    }

  dvp->SetElement(0,r);
  dvp->SetElement(1,g);
  dvp->SetElement(2,b);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelOpacityInternal(double opacity)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("LabelTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    }

  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("Opacity"));
  if (!dvp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Opacity");
    return;
    }
  dvp->SetElement(0,opacity);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelFontFamilyInternal(int font)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("LabelTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    }
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("FontFamily"));
  if (!ivp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property FontFamily");
    return;
    }
  ivp->SetElement(0,font);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelBoldInternal(int bold)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("LabelTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    }
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("Bold"));
  if (!ivp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Bold");
    return;
    }
  ivp->SetElement(0,bold);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelItalicInternal(int italic)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("LabelTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    }
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("Italic"));
  if (!ivp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Italic");
    return;
    }
  ivp->SetElement(0,italic);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetLabelShadowInternal(int shadow)
{
  vtkSMProxyProperty* pp = vtkSMProxyProperty::SafeDownCast(
    this->ScalarBarProxy->GetProperty("LabelTextProperty"));
  if (!pp || pp->GetNumberOfProxies() < 1)
    {
    vtkErrorMacro("Failed to find LabelTextProperty Proxy on ScalarBarProxy.");
    return;
    }
  vtkSMIntVectorProperty* ivp = vtkSMIntVectorProperty::SafeDownCast(
    pp->GetProxy(0)->GetProperty("Shadow"));
  if (!ivp)
    {
    vtkErrorMacro("ScalarBarProxy does not have property Shadow");
    return;
    }
  ivp->SetElement(0,shadow);
  this->ScalarBarProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ExecuteEvent(vtkObject* vtkNotUsed(wdg), 
                                 unsigned long event,  
                                 void* vtkNotUsed(calldata))
{
  switch ( event )
    {
    case vtkCommand::WidgetModifiedEvent:
      // the only things that have changed are the positions.
      // since we have no GUI for positions, we don't have any work to do here.
      //this->RenderView();
      // We don;t call EventuallyRender since that leads to 
      // non-smooth movement.
      this->GetPVApplication()->GetMainWindow()->GetInteractor()->Render();
      this->Modified();
      break;

    case vtkCommand::StartInteractionEvent:
      this->PVRenderView->GetPVWindow()->InteractiveRenderEnabledOn();
      this->RenderView();
      break;

    case vtkCommand::EndInteractionEvent:
      this->PVRenderView->GetPVWindow()->InteractiveRenderEnabledOff();
      this->RenderView();
      
      double pos1[2], pos2[2];
      this->GetPosition1Internal(pos1);
      this->GetPosition2Internal(pos2);
      
      this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarBarPosition1 %lf %lf", 
                          this->GetTclName(), pos1[0], pos1[1]);
      this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarBarPosition2 %lf %lf", 
                          this->GetTclName(), pos2[0], pos2[1]);
      this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarBarOrientation %d",
                          this->GetTclName(), this->GetOrientationInternal());
      break;
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SaveState(ofstream *file)
{

  *file << "set kw(" << this->GetTclName() << ") [$kw(" 
        << this->PVRenderView->GetPVWindow()->GetTclName() 
        << ") GetPVColorMap {" 
        << this->GetArrayNameInternal() << "} " 
        << this->NumberOfVectorComponents << "]\n";

  *file << "$kw(" << this->GetTclName() << ") SetScalarBarTitle {"
    << this->ScalarBarTitle << "}\n";
  
  *file << "$kw(" << this->GetTclName() << ") SetScalarBarVectorTitle {" 
    << this->VectorMagnitudeTitle << "}\n"; 
  
  *file << "$kw(" << this->GetTclName() << ") SetScalarBarLabelFormat {" 
    << this->GetLabelFormatInternal() << "}\n"; 
 
  double hr[2],sr[2],vr[2];
  this->GetHueRangeInternal(hr);
  this->GetSaturationRangeInternal(sr);
  this->GetValueRangeInternal(vr);

  *file << "$kw(" << this->GetTclName() << ") SetStartHSV " 
    << hr[0] << " " << sr[0] << " " << vr[0] << endl;
  *file << "$kw(" << this->GetTclName() << ") SetEndHSV " 
    << hr[1] << " " << sr[1] << " " << vr[1] << endl;

  *file << "$kw(" << this->GetTclName() << ") SetNumberOfColors " 
    << this->GetNumberOfColorsInternal() << endl;
  
  if (this->GetVectorModeInternal() == vtkLookupTable::MAGNITUDE)
    {
    *file << "$kw(" << this->GetTclName() << ") VectorModeMagnitudeCallback\n";
    }
  else if (this->GetVectorModeInternal() == vtkLookupTable::COMPONENT 
    && this->NumberOfVectorComponents > 1)
    {
    *file << "$kw(" << this->GetTclName() << ") VectorModeComponentCallback\n";
    *file << "$kw(" << this->GetTclName() << ") SetVectorComponent " 
      << this->VectorComponent << endl;
    }

  *file << "$kw(" << this->GetTclName() << ") SetScalarRange " 
    << this->ScalarRange[0] << " " << this->ScalarRange[1] << endl;
        
  *file << "$kw(" << this->GetTclName() << ") SetScalarRangeLock " 
        << this->ScalarRangeLock << "\n"; 

  *file << "$kw(" << this->GetTclName() << ") SetScalarBarVisibility " 
        << this->ScalarBarVisibility << endl;

  double pos1[2], pos2[2];
  this->GetPosition1Internal(pos1);
  this->GetPosition2Internal(pos2);

  *file << "$kw(" << this->GetTclName() << ") SetScalarBarPosition1 " 
    << pos1[0] << " " << pos1[1] << endl; 
  *file << "$kw(" << this->GetTclName() << ") SetScalarBarPosition2 " 
    << pos2[0] << " " << pos2[1] << endl;
  *file << "$kw(" << this->GetTclName() << ") SetScalarBarOrientation " 
    << this->GetOrientationInternal() << endl;
}



//----------------------------------------------------------------------------
void vtkPVColorMap::ResetScalarRangeInternal()
{
  double range[2];
  double tmp[2];
  vtkPVSource *pvs;
  vtkPVSourceCollection *sourceList;

  if (this->GetApplication() == NULL || this->PVRenderView == NULL)
    {
    vtkErrorMacro("Trying to reset scalar range without application and view.");
    return;
    }

  // Compute global scalar range ...
  range[0] = VTK_LARGE_FLOAT;
  range[1] = -VTK_LARGE_FLOAT;
  sourceList = this->PVRenderView->GetPVWindow()->GetSourceList("Sources");
  sourceList->InitTraversal();
  while ( (pvs = sourceList->GetNextPVSource()) )
    {    
    this->ComputeScalarRangeForSource(pvs, tmp);
    if (range[0] > tmp[0])
      {
      range[0] = tmp[0];
      }
    if (range[1] < tmp[1])
      {
      range[1] = tmp[1];
      }
    }
    
  // Get the computed range from the whole range.
  this->SetWholeScalarRange(range[0], range[1]);
  this->SetScalarRangeInternal(range[0], range[1]);

  if ( this->GetPVRenderView() )
    {
    this->GetPVRenderView()->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ResetScalarRangeInternal(vtkPVSource* pvs)
{
  if (this->GetApplication() == NULL || this->PVRenderView == NULL)
    {
    vtkErrorMacro("Trying to reset scalar range without application and view.");
    return;
    }
    
  double range[2];
  this->ComputeScalarRangeForSource(pvs, range);
  this->SetScalarRangeInternal(range[0], range[1]);

  if ( this->GetPVRenderView() )
    {
    this->GetPVRenderView()->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateForSource(vtkPVSource* pvs)
{
  double range[2];

  this->ComputeScalarRangeForSource(pvs, range);
  // Do we need to expand the whole range?
  if (range[0] > this->WholeScalarRange[0])
    {
    range[0] = this->WholeScalarRange[0];
    }
  if (range[1] < this->WholeScalarRange[1])
    {
    range[1] = this->WholeScalarRange[1];
    }
  
  // This will expand range if not locked.
  this->SetWholeScalarRange(range[0], range[1]);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ComputeScalarRangeForSource(vtkPVSource* pvs, 
                                                double* range)
{
  double tmp[2];
  int component = this->VectorComponent;
  if (this->GetVectorModeInternal() == vtkLookupTable::MAGNITUDE)
    {
    component = -1;
    }

  if (this->GetApplication() == NULL || this->PVRenderView == NULL)
    {
    vtkErrorMacro("Trying to reset scalar range without application and view.");
    return;
    }

  range[0] = VTK_DOUBLE_MAX;
  range[1] = -VTK_DOUBLE_MAX;

  const char* arrayname = this->GetArrayName();
  // For point data ...
  vtkPVArrayInformation *ai;
  ai = pvs->GetDataInformation()->GetPointDataInformation()->
    GetArrayInformation(arrayname);
  if (ai)
    {
    ai->GetComponentRange(component, tmp);
    if (tmp[0] < range[0])
      {
      range[0] = tmp[0];
      }
    if (tmp[1] > range[1])
      {
      range[1] = tmp[1];
      }
    }
  // For cell data ...
  ai = pvs->GetDataInformation()->GetCellDataInformation()->
    GetArrayInformation(arrayname);
  if (ai)
    {  
    ai->GetComponentRange(component, tmp);
    if (tmp[0] < range[0])
      {
      range[0] = tmp[0];
      }
    if (tmp[1] > range[1])
      {
      range[1] = tmp[1];
      }
    }
}



//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarRange(double min, double max)
{
  if (min == this->ScalarRange[0] && max == this->ScalarRange[1])
    {
    return;
    }
  this->SetScalarRangeInternal(min, max);
  this->GetTraceHelper()->AddEntry("$kw(%s) SetScalarRange %g %g", this->GetTclName(),
                      min, max);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarWidgetScalarRangeInternal(double min, double max)
{
  vtkSMDoubleVectorProperty* dvp = vtkSMDoubleVectorProperty::SafeDownCast(
    this->LookupTableProxy->GetProperty("ScalarRange"));
  if (!dvp)
    {
    vtkErrorMacro("LookupTableProxy does not have property ScalarRange");
    return;
    }
  dvp->SetElement(0,min);
  dvp->SetElement(1,max);
  this->LookupTableProxy->UpdateVTKObjects();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarRangeInternal(double min, double max)
{
  // Make sure new whole range does not cause problems. 
  if (max < min)
    {
    min = 0.0;
    max = 1.0;
    }
  if (min == max)
    {
    max = min + 0.0001;
    }

  // This will terminate any recursion.
  if (min == this->ScalarRange[0] && max == this->ScalarRange[1])
    {
    return;
    }
  this->ScalarRange[0] = min;
  this->ScalarRange[1] = max;
  this->SetScalarBarWidgetScalarRangeInternal(min,max);    
  this->ScalarRangeWidget->SetRange(min,max);    
  
  // Expand whole range if necessary.
  if (min < WholeScalarRange[0] || max > this->WholeScalarRange[1])
    {
    if (min > WholeScalarRange[0])
      {
      min = WholeScalarRange[0];
      }
    if (max < WholeScalarRange[1])
      {
      max = WholeScalarRange[1];
      }
    this->SetWholeScalarRange(min, max);
    }
    
    
  // User modified the scalar range.  Lets lock it 
  // so paraview will not over ride the users slection.
  // Also lock if user resets to range of data.
  if (this->ScalarRange[0] > this->WholeScalarRange[0] ||
      this->ScalarRange[1] < this->WholeScalarRange[1] )
    {
    this->ScalarRangeLockCheck->SetState(1);
    this->ScalarRangeLock = 1;
    }    
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetWholeScalarRange(double min, double max)
{
  // Make sure new whole range does not cause problems. 
  if (max < min)
    {
    min = 0.0;
    max = 1.0;
    }
  if (min == max)
    {
    max = min + 0.0001;
    }

  // This will terminate any recursion.
  if (min == this->WholeScalarRange[0] && max == this->WholeScalarRange[1])
    {
    return;
    }
    
  // Do not allow the whole range to shink smaller than a locked range.
  if (this->ScalarRangeLock)
    {
    if (min > this->ScalarRange[0])
      {
      min = this->ScalarRange[0];
      }
    if (max < this->ScalarRange[1])
      {
      max = this->ScalarRange[1];
      }
    }  
  this->WholeScalarRange[0] = min;
  this->WholeScalarRange[1] = max;
  this->ScalarRangeWidget->SetWholeRange(min,max);    
  
  // We might change the range also.
  double newRange[2];
  newRange[0] = this->ScalarRange[0];
  newRange[1] = this->ScalarRange[1];
  // Do not let the range outside of the whole range.
  if (newRange[0] < min)
    {
    newRange[0] = min;
    }
  if (newRange[1] > max)
    {
    newRange[1] = max;
    }
    
  // If the range is not locked by the user,
  // set the range to the whole rnage.
  if ( ! this->ScalarRangeLock)
    { // Expand the range with whole range.
    newRange[0] = min;
    newRange[1] = max;
    }
    
  this->SetScalarRangeInternal(newRange[0], newRange[1]);

  // Compute an appropriate resolution.
  double x = max-min;
  x = log10(x);
  // shift place value over 2 (100 divisions).
  int place = (int)(x) - 2;
  this->ScalarRangeWidget->SetResolution(pow(10.0, (double)(place)));
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateEnableState()
{
  this->Superclass::UpdateEnableState();

  this->PropagateEnableState(this->ColorMapFrame);
  this->PropagateEnableState(this->ArrayNameLabel);
  this->PropagateEnableState(this->NumberOfColorsScale);
  this->PropagateEnableState(this->ColorEditorFrame);
  this->PropagateEnableState(this->StartColorButton);
  this->PropagateEnableState(this->Map);
  this->PropagateEnableState(this->EndColorButton);
  this->PropagateEnableState(this->VectorFrame);
  this->PropagateEnableState(this->VectorModeMenu);
  this->PropagateEnableState(this->VectorComponentMenu);
  this->PropagateEnableState(this->ScalarBarVectorTitleEntry);
  this->PropagateEnableState(this->ScalarBarFrame);
  this->PropagateEnableState(this->ScalarBarCheck);
  this->PropagateEnableState(this->ScalarBarTitleFrame);
  this->PropagateEnableState(this->ScalarBarTitleLabel);
  this->PropagateEnableState(this->ScalarBarTitleEntry);
  this->PropagateEnableState(this->ScalarBarLabelFormatFrame);
  this->PropagateEnableState(this->ScalarBarLabelFormatLabel);
  this->PropagateEnableState(this->ScalarBarLabelFormatEntry);
  this->PropagateEnableState(this->TitleTextPropertyWidget);
  this->PropagateEnableState(this->LabelTextPropertyWidget);
  this->PropagateEnableState(this->PresetsMenuButton);
  this->PropagateEnableState(this->ScalarRangeFrame);
  this->PropagateEnableState(this->ScalarRangeWidget);
  this->PropagateEnableState(this->BackButton);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
  os << indent << "UseCount: " << this->UseCount << endl;

  os << indent << "ScalarBarProxy: " << this->ScalarBarProxy << endl;
  os << indent << "ScalarBarCheck: " << this->ScalarBarCheck << endl;
  os << indent << "TitleTextPropertyWidget: " << this->TitleTextPropertyWidget << endl;
  os << indent << "LabelTextPropertyWidget: " << this->LabelTextPropertyWidget << endl;
  
  os << indent << "ScalarBarVisibility: " << this->ScalarBarVisibility << endl;
  os << indent << "VisitedFlag: " << this->VisitedFlag << endl;
  os << indent << "ScalarBarCheck: " << this->ScalarBarCheck << endl;
  
  os << indent << "ScalarRange: " << this->ScalarRange[0] << ", " 
     << this->ScalarRange[1] << endl;
  os << indent << "WholeScalarRange: " << this->WholeScalarRange[0] << ", " 
     << this->WholeScalarRange[1] << endl;
  os << indent << "VectorComponent: " << this->VectorComponent << endl;
  os << indent << "ScalarBarTitle: " << ((this->ScalarBarTitle)?
    this->ScalarBarTitle : "NULL") << endl;

}

