/*=========================================================================

  Program:   ParaView
  Module:    vtkPVTrackballPan.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkPVTrackballPan.h"

#include "vtkCamera.h"
#include "vtkObjectFactory.h"
#include "vtkPVApplication.h"
#include "vtkPVDataInformation.h"
#include "vtkPVInteractorStyleCenterOfRotation.h"
#include "vtkPVSource.h"
#include "vtkPVWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkRenderWindow.h"
#include "vtkRenderer.h"
#include "vtkMath.h"

vtkCxxRevisionMacro(vtkPVTrackballPan, "1.8");
vtkStandardNewMacro(vtkPVTrackballPan);

//-------------------------------------------------------------------------
vtkPVTrackballPan::vtkPVTrackballPan()
{
}

//-------------------------------------------------------------------------
vtkPVTrackballPan::~vtkPVTrackballPan()
{
}

//-------------------------------------------------------------------------
void vtkPVTrackballPan::OnButtonDown(int, int, vtkRenderer *,
                                     vtkRenderWindowInteractor *)
{
}


//-------------------------------------------------------------------------
void vtkPVTrackballPan::OnButtonUp(int, int, vtkRenderer *,
                                    vtkRenderWindowInteractor *)
{
}

//-------------------------------------------------------------------------
void vtkPVTrackballPan::OnMouseMove(int x, int y, vtkRenderer *ren,
                                     vtkRenderWindowInteractor *rwi)
{
  if (ren == NULL)
    {
    return;
    }

  vtkCamera *camera = ren->GetActiveCamera();
  double pos[3], fp[3];
  camera->GetPosition(pos);
  camera->GetFocalPoint(fp);

  if (camera->GetParallelProjection())
    {
    camera->OrthogonalizeViewUp();
    double *up = camera->GetViewUp();
    double *vpn = camera->GetViewPlaneNormal();
    double right[3];
    double scale, tmp;
    camera->GetViewUp(up);
    camera->GetViewPlaneNormal(vpn);
    vtkMath::Cross(vpn, up, right);

    // These are different because y is flipped.
    int *size = ren->GetSize();
    float dx = (float)(x - rwi->GetLastEventPosition()[0]) / (float)(size[1]);
    float dy = (float)(rwi->GetLastEventPosition()[1] - y) / (float)(size[1]);

    scale = camera->GetParallelScale();
    dx *= scale * 2.0;
    dy *= scale * 2.0;

    tmp = (right[0]*dx + up[0]*dy);
    pos[0] += tmp;
    fp[0] += tmp; 
    tmp = (right[1]*dx + up[1]*dy); 
    pos[1] += tmp;
    fp[1] += tmp; 
    tmp = (right[2]*dx + up[2]*dy); 
    pos[2] += tmp;
    fp[2] += tmp; 
    camera->SetPosition(pos);
    camera->SetFocalPoint(fp);
    }
  else
    {
    double depth, worldPt[4], lastWorldPt[4];
    vtkPVApplication *app =
      static_cast<vtkPVApplication*>(this->GetApplication());
    vtkPVWindow *window = app->GetMainWindow();
    vtkPVSource *pvs = window->GetCurrentPVSource();
    if (pvs)
      {
      double bounds[6], center[3];
      int idx;
      pvs->GetDataInformation()->GetBounds(bounds);
      for (idx = 0; idx < 3; idx++)
        {
        center[idx] = (bounds[idx * 2] + bounds[idx * 2 + 1])/2.0;
        }
      ren->SetWorldPoint(center[0], center[1], center[2], 1.0);
      }
    else
      {
      float center[3];
      window->GetCenterOfRotationStyle()->GetCenter(center);
      ren->SetWorldPoint(center[0], center[1], center[2], 1.0);
      }
    
    ren->WorldToDisplay();
    depth = ren->GetDisplayPoint()[2];
    
    ren->SetDisplayPoint(x, y, depth);
    ren->DisplayToWorld();
    ren->GetWorldPoint(worldPt);
    if (worldPt[3])
      {
      worldPt[0] /= worldPt[3];
      worldPt[1] /= worldPt[3];
      worldPt[2] /= worldPt[3];
      worldPt[3] = 1.0;
      }
    
    ren->SetDisplayPoint(rwi->GetLastEventPosition()[0],
                         rwi->GetLastEventPosition()[1],
                         depth);
    ren->DisplayToWorld();
    ren->GetWorldPoint(lastWorldPt);
    if (lastWorldPt[3])
      {
      lastWorldPt[0] /= lastWorldPt[3];
      lastWorldPt[1] /= lastWorldPt[3];
      lastWorldPt[2] /= lastWorldPt[3];
      lastWorldPt[3] = 1.0;
      }

    pos[0] += lastWorldPt[0] - worldPt[0];
    pos[1] += lastWorldPt[1] - worldPt[1];
    pos[2] += lastWorldPt[2] - worldPt[2];

    fp[0] += lastWorldPt[0] - worldPt[0];
    fp[1] += lastWorldPt[1] - worldPt[1];
    fp[2] += lastWorldPt[2] - worldPt[2];

    camera->SetPosition(pos);
    camera->SetFocalPoint(fp);
    }
  ren->ResetCameraClippingRange();
  rwi->Render();
}

//-------------------------------------------------------------------------
void vtkPVTrackballPan::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}






