/*=========================================================================

  Program:   ParaView
  Module:    vtkPVColorMap.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkPVColorMap.h"

#include "vtkCommand.h"
#include "vtkKWChangeColorButton.h"
#include "vtkKWCheckButton.h"
#include "vtkKWEntry.h"
#include "vtkKWFrame.h"
#include "vtkKWLabel.h"
#include "vtkKWLabeledEntry.h"
#include "vtkKWLabeledFrame.h"
#include "vtkKWMenuButton.h"
#include "vtkKWOptionMenu.h"
#include "vtkKWPushButton.h"
#include "vtkKWScale.h"
#include "vtkKWTextProperty.h"
#include "vtkKWTkUtilities.h"
#include "vtkKWWidget.h"
#include "vtkLookupTable.h"
#include "vtkObjectFactory.h"
#include "vtkPVApplication.h"
#include "vtkPVArrayInformation.h"
#include "vtkPVDisplayGUI.h"
#include "vtkPVDataInformation.h"
#include "vtkPVDataSetAttributesInformation.h"
#include "vtkPVGenericRenderWindowInteractor.h"
#include "vtkPVRenderView.h"
#include "vtkPVSource.h"
#include "vtkPVSourceCollection.h"
#include "vtkPVWindow.h"
#include "vtkRenderer.h"
#include "vtkRendererCollection.h"
#include "vtkRenderWindow.h"
#include "vtkScalarBarActor.h"
#include "vtkScalarBarWidget.h"
#include "vtkPVProcessModule.h"
#include "vtkPVRenderModule.h"
#include "vtkKWRange.h"

#include "vtkRMScalarBarWidget.h"
#include "vtkKWEvent.h"
#include "vtkMath.h"
//----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPVColorMap);
vtkCxxRevisionMacro(vtkPVColorMap, "1.104.2.1");

int vtkPVColorMapCommand(ClientData cd, Tcl_Interp *interp,
                     int argc, char *argv[]);

/* 
 * This part was generated by ImageConvert from image:
 *    presets.png (zlib, base64)
 */
#define image_presets_width         16
#define image_presets_height        16
#define image_presets_pixel_size    4
#define image_presets_buffer_length 264

static unsigned char image_presets[] = 
  "eNrdkusNhSAMhZkHhtAlXAKHgCFkCZeAIXAIXeJcikiqQaN/L8kJr349LQGAQJG1FleB3T"
  "+JYkMIecvnNznuWJ7jqbay/jSIEULgzIvHOvjc4t+yR0//4P9VV3+SUgp932MYBoxaw6S7"
  "yTnM84zgA5ZlwbZtObblr6RMfJd5PSbeGDg3Zd4TH2Pl+f858d3ur/XB7/4+eETGt6SUrP"
  "WTP+WuvPfJn+pf63tz0Zm89G9N4qfSf/Kn/tfi3+K/vf+Z/wENH5fO";

//===========================================================================
//***************************************************************************
class vtkRMScalarBarWidgetObserver : public vtkCommand
{
public:
  static vtkRMScalarBarWidgetObserver *New() 
    {return new vtkRMScalarBarWidgetObserver;};

  vtkRMScalarBarWidgetObserver()
    {
    this->PVColorMap = 0;
    }

  virtual void Execute(vtkObject* wdg, unsigned long event,  
                       void* calldata)
    {
      if ( this->PVColorMap )
        {
        this->PVColorMap->ExecuteEvent(wdg, event, calldata);
        }
    }

  vtkPVColorMap* PVColorMap;
};

#define VTK_PV_COLOR_MAP_RED_HUE 0.0
#define VTK_PV_COLOR_MAP_RED_SATURATION 1.0
#define VTK_PV_COLOR_MAP_RED_VALUE 1.0
#define VTK_PV_COLOR_MAP_BLUE_HUE 0.6667
#define VTK_PV_COLOR_MAP_BLUE_SATURATION 1.0
#define VTK_PV_COLOR_MAP_BLUE_VALUE 1.0
#define VTK_PV_COLOR_MAP_BLACK_HUE 0.0
#define VTK_PV_COLOR_MAP_BLACK_SATURATION 0.0
#define VTK_PV_COLOR_MAP_BLACK_VALUE 0.0
#define VTK_PV_COLOR_MAP_WHITE_HUE 0.0
#define VTK_PV_COLOR_MAP_WHITE_SATURATION 0.0
#define VTK_PV_COLOR_MAP_WHITE_VALUE 1.0

#define VTK_USE_LAB_COLOR_MAP 1.1

//***************************************************************************
//===========================================================================
//vtkCxxSetObjectMacro(vtkPVColorMap,PVRenderView,vtkPVRenderView);
//----------------------------------------------------------------------------
// No register count because of reference loop.
void vtkPVColorMap::SetPVRenderView(vtkPVRenderView *rv)
{
  this->PVRenderView = rv;
}

//----------------------------------------------------------------------------
vtkPVColorMap::vtkPVColorMap()
{
  this->CommandFunction = vtkPVColorMapCommand;

  this->UseCount = 0;
  
  this->RMScalarBarWidget = vtkRMScalarBarWidget::New();

  this->ScalarBarVisibility = 0;
  this->InternalScalarBarVisibility = 0;
  this->Initialized = 0;
  this->PVRenderView = NULL;
  this->RMScalarBarObserver = NULL;

  // User interaface.
  this->ColorMapFrame = vtkKWLabeledFrame::New();
  this->ArrayNameLabel = vtkKWLabel::New();
  // Stuff for setting the range of the color map.
  this->ColorRangeFrame = vtkKWWidget::New();
  this->ColorRangeLabel = vtkKWLabel::New();
  this->ColorRangeWidget = vtkKWRange::New();
  this->ColorRangeWidget->ClampRangeOff();
  this->NumberOfColorsScale = vtkKWScale::New();  
  this->ColorEditorFrame = vtkKWWidget::New();
  this->StartColorButton = vtkKWChangeColorButton::New();
  this->Map = vtkKWLabel::New();
  this->EndColorButton = vtkKWChangeColorButton::New();

  this->VectorFrame = vtkKWLabeledFrame::New();
  this->VectorModeMenu = vtkKWOptionMenu::New() ;
  this->VectorComponentMenu = vtkKWOptionMenu::New();

  // Stuff for manipulating the scalar bar.
  this->ScalarBarFrame = vtkKWLabeledFrame::New();
  this->ScalarBarCheck = vtkKWCheckButton::New();
  this->ScalarBarTitleFrame = vtkKWWidget::New();
  this->ScalarBarTitleLabel = vtkKWLabel::New();
  this->ScalarBarTitleEntry = vtkKWEntry::New();
  this->ScalarBarVectorTitleEntry = vtkKWEntry::New();
  this->ScalarBarLabelFormatFrame = vtkKWWidget::New();
  this->ScalarBarLabelFormatLabel = vtkKWLabel::New();
  this->ScalarBarLabelFormatEntry = vtkKWEntry::New();

  this->BackButton = vtkKWPushButton::New();

  this->MapData = NULL;
  this->MapDataSize = 0;
  this->MapHeight = 25;
  this->MapWidth = 20;

  this->PresetsMenuButton = vtkKWMenuButton::New();

  this->VisitedFlag = 0;

  this->TitleTextPropertyWidget = vtkKWTextProperty::New();
  this->LabelTextPropertyWidget = vtkKWTextProperty::New();
}

//----------------------------------------------------------------------------
vtkPVColorMap::~vtkPVColorMap()
{
  // Used to be in vtkPVActorComposite........
  this->RMScalarBarWidget->Delete();

  this->SetPVRenderView(NULL);

  if (this->RMScalarBarObserver)
    {
    this->RMScalarBarObserver->Delete();
    this->RMScalarBarObserver = NULL;
    }
    
  this->VectorFrame->Delete();
  this->VectorFrame = NULL;
  this->VectorModeMenu->Delete();
  this->VectorModeMenu = NULL;
  this->VectorComponentMenu->Delete();
  this->VectorComponentMenu = NULL;
  this->ScalarBarVectorTitleEntry->Delete();
  this->ScalarBarVectorTitleEntry = NULL;

  // User interaface.
  this->ColorMapFrame->Delete();
  this->ColorMapFrame = NULL;
  this->ArrayNameLabel->Delete();
  this->ArrayNameLabel = NULL;
  // Stuff for setting the range of the color map.
  this->ColorRangeFrame->Delete();
  this->ColorRangeFrame = NULL;
  this->ColorRangeLabel->Delete();
  this->ColorRangeLabel = NULL;
  this->ColorRangeWidget->Delete();
  this->ColorRangeWidget = NULL;
  this->NumberOfColorsScale->Delete();
  this->NumberOfColorsScale = NULL;

  this->ColorEditorFrame->Delete();
  this->ColorEditorFrame = NULL;
  this->StartColorButton->Delete();
  this->StartColorButton = NULL;
  this->Map->Delete();
  this->Map = NULL;
  this->EndColorButton->Delete();
  this->EndColorButton = NULL;
  
  this->ScalarBarFrame->Delete();
  this->ScalarBarFrame = NULL;
  this->ScalarBarCheck->Delete();
  this->ScalarBarCheck = NULL;
  this->ScalarBarTitleFrame->Delete();
  this->ScalarBarTitleFrame = NULL;
  this->ScalarBarTitleLabel->Delete();
  this->ScalarBarTitleLabel = NULL;
  this->ScalarBarTitleEntry->Delete();
  this->ScalarBarTitleEntry = NULL;
  this->ScalarBarLabelFormatFrame->Delete();
  this->ScalarBarLabelFormatFrame = NULL;
  this->ScalarBarLabelFormatLabel->Delete();
  this->ScalarBarLabelFormatLabel = NULL;
  this->ScalarBarLabelFormatEntry->Delete();
  this->ScalarBarLabelFormatEntry = NULL;

  this->BackButton->Delete();
  this->BackButton = NULL;

  if (this->MapData)
    {
    delete [] this->MapData;
    this->MapDataSize = 0;
    this->MapWidth = 0;
    this->MapHeight = 0;
    }

  if ( this->PresetsMenuButton )
    {
    this->PresetsMenuButton->Delete();
    }

  this->TitleTextPropertyWidget->Delete();
  this->TitleTextPropertyWidget = NULL;

  this->LabelTextPropertyWidget->Delete();
  this->LabelTextPropertyWidget = NULL;
}


//----------------------------------------------------------------------------
void vtkPVColorMap::Create(vtkKWApplication *app)
{
  vtkPVApplication* pvApp = vtkPVApplication::SafeDownCast(app);

  // Call the superclass to create the widget and set the appropriate flags

  if (!this->vtkKWWidget::Create(pvApp, "frame", "-bd 0 -relief flat"))
    {
    vtkErrorMacro("Failed creating widget " << this->GetClassName());
    return;
    }
  
  this->CreateParallelTclObjects(pvApp);

  // Now for the UI.

  const char *grid_settings = "-padx 1 -pady 2";
  const char *label_settings = "-anchor w";

  // Color map

  this->ColorMapFrame->SetParent(this);
  this->ColorMapFrame->ShowHideFrameOn();
  this->ColorMapFrame->Create(app, 0);
  this->ColorMapFrame->SetLabel("Color Map");

  // Color map: parameter name

  this->ArrayNameLabel->SetParent(this->ColorMapFrame->GetFrame());
  this->ArrayNameLabel->Create(app, "-anchor w");
  this->ArrayNameLabel->SetLabel("Parameter: ");

  // Color map: range
  this->ColorRangeFrame->SetParent(this->ColorMapFrame->GetFrame());
  this->ColorRangeFrame->Create(app, "frame", "");
  this->ColorRangeLabel->SetParent(this->ColorRangeFrame);
  this->ColorRangeLabel->SetLabel("Range:");
  this->ColorRangeLabel->Create(app, "");
  this->Script("pack %s -side left -expand f",
               this->ColorRangeLabel->GetWidgetName());

  this->ColorRangeWidget->SetParent(this->ColorRangeFrame);
  this->ColorRangeWidget->Create(app, "");
  this->ColorRangeWidget->SetWholeRange(
    -VTK_LARGE_FLOAT, VTK_LARGE_FLOAT);
  this->ColorRangeWidget->ShowEntriesOn();
  this->ColorRangeWidget->ShowLabelOff();
  this->ColorRangeWidget->GetEntry1()->SetWidth(7);
  this->ColorRangeWidget->GetEntry2()->SetWidth(7);
  this->ColorRangeWidget->SetCommand(this, "ColorRangeWidgetCallback");
  this->ColorRangeWidget->SetEntriesPosition(
    vtkKWRange::POSITION_ALIGNED);
  this->ColorRangeWidget->SetBalloonHelpString(
    "Set the minimum and maximum of the values of the color map");
  this->Script("pack %s -side left -fill x -expand t",
               this->ColorRangeWidget->GetWidgetName());
               
  // Color map: gradient editor

  this->ColorEditorFrame->SetParent(this->ColorMapFrame->GetFrame());
  this->ColorEditorFrame->Create(app, "frame", "");

  this->StartColorButton->SetParent(this->ColorEditorFrame);
  this->StartColorButton->ShowLabelOff();
  this->StartColorButton->Create(app, "");
  this->StartColorButton->SetColor(1.0, 0.0, 0.0);
  this->StartColorButton->SetCommand(this, "StartColorButtonCallback");
  this->StartColorButton->SetBalloonHelpString("Select the minimum color.");

  this->Map->SetParent(this->ColorEditorFrame);
  this->Map->Create(app, 
                    "-relief flat -bd 0 -highlightthickness 0 -padx 0 -pady 0");
  this->Script("bind %s <Configure> {%s MapConfigureCallback %s}", 
               this->Map->GetWidgetName(), 
               this->GetTclName(), "%w %h");

  this->EndColorButton->SetParent(this->ColorEditorFrame);
  this->EndColorButton->ShowLabelOff();
  this->EndColorButton->Create(app, "");
  this->EndColorButton->SetColor(0.0, 0.0, 1.0);
  this->EndColorButton->SetCommand(this, "EndColorButtonCallback");
  this->EndColorButton->SetBalloonHelpString("Select the maximum color.");

  this->PresetsMenuButton->SetParent(this->ColorEditorFrame);
  this->PresetsMenuButton->Create(app, "-indicator 0");
  this->PresetsMenuButton->SetBalloonHelpString("Select a preset color map.");
  this->PresetsMenuButton->AddCommand(
    "Blue to Red", 
    this, "SetColorSchemeToBlueRed", "Set Color Scheme to Blue-Red");
  this->PresetsMenuButton->AddCommand(
    "Red to Blue", 
    this, "SetColorSchemeToRedBlue", "Set Color Scheme to Red-Blue");
  this->PresetsMenuButton->AddCommand(
    "Grayscale", 
    this, "SetColorSchemeToGrayscale", "Set Color Scheme to Grayscale");
  this->PresetsMenuButton->AddCommand(
    "CIELab Blue to Red", 
    this, "SetColorSchemeToLabBlueRed", "Set Color Scheme to Lab Blue To Red");
  
  this->PresetsMenuButton->SetImageOption(image_presets, 
                                          image_presets_width, 
                                          image_presets_height, 
                                          image_presets_pixel_size,
                                          image_presets_buffer_length);

  this->Script("grid %s %s %s %s -sticky news -padx 1 -pady 2",
               this->PresetsMenuButton->GetWidgetName(),
               this->StartColorButton->GetWidgetName(),
               this->Map->GetWidgetName(),
               this->EndColorButton->GetWidgetName());

  this->Script("grid columnconfigure %s 2 -weight 1",
               this->Map->GetParent()->GetWidgetName());
  this->Script("grid %s -ipadx 1",
               this->PresetsMenuButton->GetWidgetName());

  // Color map: resolution

  this->NumberOfColorsScale->SetParent(this->ColorMapFrame->GetFrame());
  this->NumberOfColorsScale->Create(app, "");
  this->NumberOfColorsScale->SetRange(2, 256);
  this->NumberOfColorsScale->SetValue(256);
  this->NumberOfColorsScale->DisplayLabel("Resolution:");
  this->NumberOfColorsScale->DisplayEntry();
  this->NumberOfColorsScale->DisplayEntryAndLabelOnTopOff();
  this->NumberOfColorsScale->SetEndCommand(this, 
                                           "NumberOfColorsScaleCallback");
  this->NumberOfColorsScale->SetEntryCommand(this, 
                                             "NumberOfColorsScaleCallback");
  this->NumberOfColorsScale->SetBalloonHelpString(
    "Select the discrete number of colors in the color map.");

  // Color map: pack

  this->Script("pack %s %s %s %s -side top -expand t -fill x -anchor nw",
               this->ArrayNameLabel->GetWidgetName(),
               this->ColorRangeFrame->GetWidgetName(),
               this->ColorEditorFrame->GetWidgetName(),
               this->NumberOfColorsScale->GetWidgetName());


  if (this->RMScalarBarWidget->GetNumberOfVectorComponents() > 1)
    {
    this->VectorFrame->SetParent(this);
    this->VectorFrame->ShowHideFrameOn();
    this->VectorFrame->Create(app, 0);
    this->VectorFrame->SetLabel("Vector");

    this->VectorModeMenu->SetParent(this->VectorFrame->GetFrame());
    this->VectorModeMenu->Create(app, "");
    this->VectorModeMenu->AddEntryWithCommand("Magnitude", this, 
                                          "VectorModeMagnitudeCallback");
    this->VectorModeMenu->AddEntryWithCommand("Component", this, 
                                          "VectorModeComponentCallback");
    this->VectorModeMenu->SetValue("Magnitude");

    this->VectorComponentMenu->SetParent(this->VectorFrame->GetFrame());
    this->VectorComponentMenu->Create(app, "");
    this->UpdateVectorComponentMenu();
    this->Script("pack %s -side left -expand f -fill both -padx 2",
                 this->VectorModeMenu->GetWidgetName());
    this->Script("pack %s -side left -expand f -fill both -padx 2",
                 this->VectorComponentMenu->GetWidgetName());
    
    this->VectorModeMagnitudeCallback();
    }

  // Scalar bar frame

  this->ScalarBarFrame->SetParent(this);
  this->ScalarBarFrame->ShowHideFrameOn();
  this->ScalarBarFrame->Create(app, 0);
  this->ScalarBarFrame->SetLabel("Scalar Bar");

  // Scalar bar : Visibility

  this->ScalarBarCheck->SetParent(this->ScalarBarFrame->GetFrame());
  this->ScalarBarCheck->Create(app, "-text Visibility");
  app->Script(
    "%s configure -command {%s ScalarBarCheckCallback}",
    this->ScalarBarCheck->GetWidgetName(),
    this->GetTclName());

  // Scalar bar : Title control

  this->ScalarBarTitleFrame->SetParent(this->ScalarBarFrame->GetFrame());
  this->ScalarBarTitleFrame->Create(app, "frame", "-bd 0");

  this->ScalarBarTitleLabel->SetParent(this->ScalarBarTitleFrame);
  this->ScalarBarTitleLabel->SetLabel("Title:");
  this->ScalarBarTitleLabel->Create(app, label_settings);
  
  this->ScalarBarTitleEntry->SetParent(this->ScalarBarTitleFrame);
  this->ScalarBarTitleEntry->SetWidth(10);
  this->ScalarBarTitleEntry->Create(app, "");
  this->Script("bind %s <KeyPress-Return> {%s ScalarBarTitleEntryCallback}",
               this->ScalarBarTitleEntry->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ScalarBarTitleEntryCallback}",
               this->ScalarBarTitleEntry->GetWidgetName(),
               this->GetTclName()); 

  this->ScalarBarVectorTitleEntry->SetParent(this->ScalarBarTitleFrame);
  this->ScalarBarVectorTitleEntry->SetWidth(
    this->ScalarBarTitleEntry->GetWidth() / 2);
  this->ScalarBarVectorTitleEntry->Create(app, "");
  this->Script(
    "bind %s <KeyPress-Return> {%s ScalarBarVectorTitleEntryCallback}",
               this->ScalarBarVectorTitleEntry->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ScalarBarVectorTitleEntryCallback}",
               this->ScalarBarVectorTitleEntry->GetWidgetName(),
               this->GetTclName()); 

  this->TitleTextPropertyWidget->SetParent(this->ScalarBarTitleFrame);
  this->TitleTextPropertyWidget->SetTextProperty(
    this->RMScalarBarWidget->GetScalarBar()->GetScalarBarActor()->GetTitleTextProperty());
  this->TitleTextPropertyWidget->SetActor2D(
    this->RMScalarBarWidget->GetScalarBar()->GetScalarBarActor());
  this->TitleTextPropertyWidget->Create(app);
  this->TitleTextPropertyWidget->SetChangedCommand(this, "RenderView");
  this->TitleTextPropertyWidget->SetTraceReferenceObject(this);
  this->TitleTextPropertyWidget->SetTraceReferenceCommand(
    "GetTitleTextPropertyWidget");

  this->Script("grid %s -row 0 -column 0 -sticky nws %s",
               this->ScalarBarTitleLabel->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 0 -column 1 -sticky news %s",
               this->ScalarBarTitleEntry->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 1 -column 1 -columnspan 2 -sticky nws %s",
               this->TitleTextPropertyWidget->GetWidgetName(),
               grid_settings);

  if (this->RMScalarBarWidget->GetNumberOfVectorComponents()> 1)
    {
    this->Script("grid %s -row 0 -column 2 -sticky news %s",
                 this->ScalarBarVectorTitleEntry->GetWidgetName(),
                 grid_settings);
    this->Script("grid columnconfigure %s 2 -weight 3",
                 this->ScalarBarVectorTitleEntry->GetParent()->GetWidgetName());
    }

  // Scalar bar : Label control

  this->ScalarBarLabelFormatFrame->SetParent(this->ScalarBarFrame->GetFrame());
  this->ScalarBarLabelFormatFrame->Create(app, "frame", "-bd 0");

  this->ScalarBarLabelFormatLabel->SetParent(this->ScalarBarLabelFormatFrame);
  this->ScalarBarLabelFormatLabel->SetLabel("Labels:");
  this->ScalarBarLabelFormatLabel->Create(app, label_settings);
  
  this->ScalarBarLabelFormatEntry->SetParent(this->ScalarBarLabelFormatFrame);
  this->ScalarBarLabelFormatEntry->SetWidth(
    this->ScalarBarTitleEntry->GetWidth());
  this->ScalarBarLabelFormatEntry->Create(app, "");
  this->Script("bind %s <KeyPress-Return> {%s ScalarBarLabelFormatEntryCallback}",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ScalarBarLabelFormatEntryCallback}",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               this->GetTclName()); 
  this->ScalarBarLabelFormatEntry->SetBalloonHelpString(
    "Set the labels format (printf() style).");

  //this->SetScalarBarLabelFormat(
  //  this->RMScalarBarWidget->GetScalarBar->GetScalarBarActor()->GetLabelFormat());

  this->LabelTextPropertyWidget->SetParent(this->ScalarBarLabelFormatFrame);
  this->LabelTextPropertyWidget->SetTextProperty(
    this->RMScalarBarWidget->GetScalarBar()->GetScalarBarActor()->GetLabelTextProperty());
  this->LabelTextPropertyWidget->SetActor2D(
    this->RMScalarBarWidget->GetScalarBar()->GetScalarBarActor());
  this->LabelTextPropertyWidget->Create(app);
  this->LabelTextPropertyWidget->SetChangedCommand(this, "RenderView");
  this->LabelTextPropertyWidget->SetTraceReferenceObject(this);
  this->LabelTextPropertyWidget->SetTraceReferenceCommand(
    "GetLabelTextPropertyWidget");

  this->Script("grid %s -row 0 -column 0 -sticky nws %s",
               this->ScalarBarLabelFormatLabel->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 0 -column 1 -sticky news %s",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 1 -column 1 -sticky nws %s",
               this->LabelTextPropertyWidget->GetWidgetName(),
               grid_settings);

  // Scalar bar : enable copy between text property widgets

  this->TitleTextPropertyWidget->ShowCopyOn();
  this->TitleTextPropertyWidget->GetCopyButton()->SetBalloonHelpString(
    "Copy the labels text properties to the title text properties.");
  ostrstream copy1;
  copy1 << "CopyValuesFrom " << this->LabelTextPropertyWidget->GetTclName() << ends;
  this->TitleTextPropertyWidget->GetCopyButton()->SetCommand(
    this->TitleTextPropertyWidget, copy1.str());
  copy1.rdbuf()->freeze(0);

  this->LabelTextPropertyWidget->ShowCopyOn();
  this->LabelTextPropertyWidget->GetCopyButton()->SetBalloonHelpString(
    "Copy the title text properties to the labels text properties.");
  ostrstream copy2;
  copy2 << "CopyValuesFrom " << this->TitleTextPropertyWidget->GetTclName() << ends;
  this->LabelTextPropertyWidget->GetCopyButton()->SetCommand(
    this->LabelTextPropertyWidget, copy2.str());
  copy2.rdbuf()->freeze(0);

  // Scalar bar: synchronize all those grids to have them aligned

  const char *widgets[2];
  widgets[0] = this->ScalarBarTitleFrame->GetWidgetName();
  widgets[1] = this->ScalarBarLabelFormatFrame->GetWidgetName();

  int weights[2];
  weights[0] = 0;
  weights[1] = 4;

  float factors[2];
  factors[0] = 1.15;
  factors[1] = 0.0;

  vtkKWTkUtilities::SynchroniseGridsColumnMinimumSize(
    this->GetApplication()->GetMainInterp(), 2, widgets, factors, weights);

  // Scalar bar : pack/grid

  this->Script("pack %s -side top -fill y -anchor w",
               this->ScalarBarCheck->GetWidgetName());
  
  this->Script("pack %s %s -side top -expand t -fill both -anchor w",
               this->ScalarBarTitleFrame->GetWidgetName(),
               this->ScalarBarLabelFormatFrame->GetWidgetName());

  // Back button

  this->BackButton->SetParent(this);
  this->BackButton->Create(app, "-text {Back}");
  this->BackButton->SetCommand(this, "BackButtonCallback");

  // Pack

  this->Script("pack %s -side top -anchor n -fill x -padx 2 -pady 2", 
               this->ColorMapFrame->GetWidgetName());
  if (this->RMScalarBarWidget->GetNumberOfVectorComponents() > 1)
    {
    this->Script("pack %s -side top -anchor n -fill x -padx 2 -pady 2", 
                 this->VectorFrame->GetWidgetName());
    }
  this->Script("pack %s %s -side top -anchor n -fill x -padx 2 -pady 2", 
               this->ScalarBarFrame->GetWidgetName(),
               this->BackButton->GetWidgetName());

  this->SetColorSchemeToBlueRed();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateVectorComponentMenu()
{
  int idx;
  char numStr[5];
  int numberOfVectorComponents;
  char **vectorComponentTitles;
  int vectorComponent;

  numberOfVectorComponents = this->RMScalarBarWidget->GetNumberOfVectorComponents();
  if (numberOfVectorComponents  == 1)
    {
    return;
    }

  vectorComponentTitles = this->RMScalarBarWidget->GetVectorComponentTitles();
  vectorComponent = this->RMScalarBarWidget->GetVectorComponent();

  this->VectorComponentMenu->ClearEntries();
  if ( *(vectorComponentTitles[vectorComponent]) != '\0')
    {
    this->VectorComponentMenu->SetValue(
                         vectorComponentTitles[vectorComponent]);
    }
  else
    {
    sprintf(numStr, "%d", vectorComponent + 1);
    this->VectorComponentMenu->SetValue(numStr);
    } 

  for (idx = 0; idx < numberOfVectorComponents; ++idx)
    {
    char command[64];
    sprintf(command, "VectorComponentCallback %d", idx);
    if (*(vectorComponentTitles[idx]) != '\0')
      {                                       
      this->VectorComponentMenu->AddEntryWithCommand(
                                           vectorComponentTitles[idx], 
                                           this, command);
      }
    else
      {
      sprintf(numStr, "%d", idx+1);
      this->VectorComponentMenu->AddEntryWithCommand(numStr, this, command);
      }
    }
} 

//----------------------------------------------------------------------------
void vtkPVColorMap::SetNumberOfVectorComponents(int  num)
{
  this->RMScalarBarWidget->SetNumberOfVectorComponents(num);
  const char *arrayname = this->RMScalarBarWidget->GetArrayName();
  if ( arrayname != NULL)
    {
    char *str2;
    str2 = new char [strlen(arrayname) + 128];
    sprintf(str2, "GetPVColorMap {%s} %d", arrayname, 
            this->RMScalarBarWidget->GetNumberOfVectorComponents());
    this->SetTraceReferenceCommand(str2);
    delete [] str2;
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::CreateParallelTclObjects(vtkPVApplication *pvApp)
{
  this->vtkKWObject::SetApplication(pvApp);
  vtkPVProcessModule* pm = pvApp->GetProcessModule();
  this->RMScalarBarWidget->Create(pm,pm->GetRenderModule()->GetRenderer2DID(),
    this->PVRenderView->GetPVWindow()->GetInteractorID());

  //this->ScalarBar = vtkScalarBarWidget::New();
  //this->ScalarBar->SetCurrentRenderer(pvApp->GetRenderModule()->GetRenderer2D());
  //
  //// Actor will be in server manager.  Widget will be in UI.
  //this->ScalarBar->SetScalarBarActor(
  //  vtkScalarBarActor::SafeDownCast(pm->GetObjectFromID(
  //  this->RMScalarBarWidget->GetScalarBarActorID())));
  //
  //this->ScalarBar->SetInteractor(
  //  this->PVRenderView->GetPVWindow()->GetInteractor());

  this->RMScalarBarObserver = vtkRMScalarBarWidgetObserver::New();
  this->RMScalarBarObserver->PVColorMap = this;

  this->RMScalarBarWidget->AddObserver(vtkCommand::InteractionEvent,
                                       this->RMScalarBarObserver);
  this->RMScalarBarWidget->AddObserver(vtkCommand::StartInteractionEvent, 
                                       this->RMScalarBarObserver);
  this->RMScalarBarWidget->AddObserver(vtkCommand::EndInteractionEvent, 
                                       this->RMScalarBarObserver);
  this->RMScalarBarWidget->AddObserver(vtkKWEvent::WidgetModifiedEvent,
                                       this->RMScalarBarObserver);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::RenderView()
{
  if (this->PVRenderView)
    {
    this->PVRenderView->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::BackButtonCallback()
{
  if (this->PVRenderView == NULL)
    {
    return;
    }

  // This has a side effect of gathering and display information.
  this->PVRenderView->GetPVWindow()->GetCurrentPVSource()->GetPVOutput()->Update();
  this->PVRenderView->GetPVWindow()->ShowCurrentSourceProperties();
}


//----------------------------------------------------------------------------
void vtkPVColorMap::SetArrayName(const char* str)
{
  this->RMScalarBarWidget->SetArrayName(str);
  str = this->RMScalarBarWidget->GetArrayName();

  if (str)
    {
    char *tmp;
    tmp = new char[strlen(str)+128];
    sprintf(tmp, "Parameter: %s", str);
    this->ArrayNameLabel->SetLabel(tmp);
    delete [] tmp;
    }

  if (str != NULL)
    {
    char *str2;
    str2 = new char [strlen(str) + 128];
    sprintf(str2, "GetPVColorMap {%s} %d", str, 
      this->RMScalarBarWidget->GetNumberOfVectorComponents());
    this->SetTraceReferenceCommand(str2);
    delete [] str2;
    }
  //this->Modified();
}

//----------------------------------------------------------------------------
int vtkPVColorMap::MatchArrayName(const char* str, int numberOfComponents)
{
  return this->RMScalarBarWidget->MatchArrayName(str,numberOfComponents);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarBarTitleEntryCallback()
{
  this->SetScalarBarTitle(this->ScalarBarTitleEntry->GetValue());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarTitle(const char* name)
{
  this->SetScalarBarTitleInternal(name);
  this->AddTraceEntry("$kw(%s) SetScalarBarTitle {%s}", 
                      this->GetTclName(), name);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarTitleInternal(const char* name)
{

  this->RMScalarBarWidget->SetScalarBarTitle(name);
}


//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarBarVectorTitleEntryCallback()
{
  this->SetScalarBarVectorTitle(this->ScalarBarVectorTitleEntry->GetValue());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarVectorTitle(const char* name)
{
  this->RMScalarBarWidget->SetScalarBarVectorTitle(name);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarBarLabelFormatEntryCallback()
{
  this->SetScalarBarLabelFormat(this->ScalarBarLabelFormatEntry->GetValue());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarLabelFormat(const char* name)
{
  this->RMScalarBarWidget->SetScalarBarLabelFormat(name);

  this->AddTraceEntry("$kw(%s) SetScalarBarLabelFormat {%s}", 
                      this->GetTclName(), name);

}

//----------------------------------------------------------------------------
void vtkPVColorMap::NumberOfColorsScaleCallback()
{
  int num = (int)(this->NumberOfColorsScale->GetValue());

  if (num == this->RMScalarBarWidget->GetNumberOfColors())
    {
    return;
    }

  this->AddTraceEntry("$kw(%s) SetNumberOfColors %d", this->GetTclName(),
                      num);
  this->RMScalarBarWidget->SetNumberOfColors(num);
}

//----------------------------------------------------------------------------
// Access for trace files.
void vtkPVColorMap::SetNumberOfColors(int num)
{
  this->NumberOfColorsScale->SetValue(num);
  this->NumberOfColorsScaleCallback();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetColorSchemeToRedBlue()
{
  double startHSV[3];
  double endHSV[3];
  startHSV[0] = VTK_PV_COLOR_MAP_RED_HUE;
  startHSV[1] = VTK_PV_COLOR_MAP_RED_SATURATION;
  startHSV[2] = VTK_PV_COLOR_MAP_RED_VALUE;
  endHSV[0] = VTK_PV_COLOR_MAP_BLUE_HUE;
  endHSV[1] = VTK_PV_COLOR_MAP_BLUE_SATURATION;
  endHSV[2] = VTK_PV_COLOR_MAP_BLUE_VALUE;

  this->StartColorButton->SetColor(1.0, 0.0, 0.0);
  this->EndColorButton->SetColor(0.0, 0.0, 1.0);

  this->RMScalarBarWidget->SetStartHSV(startHSV);
  this->RMScalarBarWidget->SetEndHSV(endHSV);
  this->AddTraceEntry("$kw(%s) SetColorSchemeToRedBlue", this->GetTclName());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetColorSchemeToBlueRed()
{
  double startHSV[3];
  double endHSV[3];
  startHSV[0] = VTK_PV_COLOR_MAP_BLUE_HUE;
  startHSV[1] = VTK_PV_COLOR_MAP_BLUE_SATURATION;
  startHSV[2] = VTK_PV_COLOR_MAP_BLUE_VALUE;
  endHSV[0] = VTK_PV_COLOR_MAP_RED_HUE;
  endHSV[1] = VTK_PV_COLOR_MAP_RED_SATURATION;
  endHSV[2] = VTK_PV_COLOR_MAP_RED_VALUE;

  this->StartColorButton->SetColor(0.0, 0.0, 1.0);
  this->EndColorButton->SetColor(1.0, 0.0, 0.0);

  this->RMScalarBarWidget->SetStartHSV(startHSV);
  this->RMScalarBarWidget->SetEndHSV(endHSV);
  this->AddTraceEntry("$kw(%s) SetColorSchemeToBlueRed", this->GetTclName());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetColorSchemeToGrayscale()
{
  double startHSV[3];
  double endHSV[3];
  startHSV[0] = VTK_PV_COLOR_MAP_BLACK_HUE;
  startHSV[1] = VTK_PV_COLOR_MAP_BLACK_SATURATION;
  startHSV[2] = VTK_PV_COLOR_MAP_BLACK_VALUE;
  endHSV[0] = VTK_PV_COLOR_MAP_WHITE_HUE;
  endHSV[1] = VTK_PV_COLOR_MAP_WHITE_SATURATION;
  endHSV[2] = VTK_PV_COLOR_MAP_WHITE_VALUE;

  this->StartColorButton->SetColor(0.0, 0.0, 0.0);
  this->EndColorButton->SetColor(1.0, 1.0, 1.0);

  this->RMScalarBarWidget->SetStartHSV(startHSV);
  this->RMScalarBarWidget->SetEndHSV(endHSV);
  this->AddTraceEntry("$kw(%s) SetColorSchemeToGrayscale", this->GetTclName());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetColorSchemeToLabBlueRed()
{
  double startHSV[3];
  double endHSV[3];

  //first calculate the gradient values
  //LabBlue[0]=83.91;
  //LabBlue[1]=-8.65;
  //LabBlue[2]=-12.18;
  startHSV[0]=57.93;
  startHSV[1]=-26.85;
  startHSV[2]=-30.21;

  //LabRed[0]=83.91;
  //LabRed[1]=8.65;
  //LabRed[2]=12.18;
  
  endHSV[0]=57.93;
  endHSV[1]=26.85;
  endHSV[2]=30.21;
  
  //this->StartColorButton->SetColor(0.0, 0.0, 1.0);
  //this->EndColorButton->SetColor(1.0, 0.0, 0.0);

  double rgb[3];
  double xyz[3];
  this->LabToXYZ(startHSV,xyz);
  this->XYZToRGB(xyz,rgb);
  this->StartColorButton->SetColor(rgb[0],rgb[1],rgb[2]);
  this->LabToXYZ(endHSV,xyz);
  this->XYZToRGB(xyz,rgb);
  this->EndColorButton->SetColor(rgb[0],rgb[1],rgb[2]);

  this->RMScalarBarWidget->SetStartHSV(startHSV);
  this->RMScalarBarWidget->SetEndHSV(endHSV);

  this->AddTraceEntry("$kw(%s) SetColorSchemeToBlueRed", this->GetTclName());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::StartColorButtonCallback(double r, double g, double b)
{
  double rgb[3];
  double hsv[3];

  // Convert RGB to HSV.
  rgb[0] = r;
  rgb[1] = g;
  rgb[2] = b;
  vtkMath::RGBToHSV(rgb, hsv);

  this->SetStartHSV(hsv[0], hsv[1], hsv[2]);
}


//----------------------------------------------------------------------------
// Access for trace files.
void vtkPVColorMap::SetStartHSV(double h, double s, double v)
{
  double hsv[3];
  double rgb[3];

  this->RMScalarBarWidget->GetStartHSV(hsv);
  if ( hsv[0] == h && 
       hsv[1] == s && hsv[2] == v)
    {
    return;
    }

  // Change color button (should have no callback effect...)
  hsv[0] = h;  hsv[1] = s;  hsv[2] = v;
  vtkMath::HSVToRGB(hsv, rgb);
  this->StartColorButton->SetColor(rgb);

  this->AddTraceEntry("$kw(%s) SetStartHSV %g %g %g", 
                      this->GetTclName(), h, s, v);

  this->RMScalarBarWidget->SetStartHSV(h,s,v);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::EndColorButtonCallback(double r, double g, double b)
{
  double rgb[3];
  double hsv[3];

  // Convert RGB to HSV.
  rgb[0] = r;
  rgb[1] = g;
  rgb[2] = b;
  vtkMath::RGBToHSV(rgb, hsv);

  this->SetEndHSV(hsv[0], hsv[1], hsv[2]);
}


//----------------------------------------------------------------------------
// Access for trace files.
void vtkPVColorMap::SetEndHSV(double h, double s, double v)
{
  double hsv[3];
  double rgb[3];

  this->RMScalarBarWidget->GetEndHSV(hsv);
  if ( hsv[0] == h && 
       hsv[1] == s && hsv[2] == v)
    {
    return;
    }

  // Change color button (should have no callback effect...)
  hsv[0] = h;  hsv[1] = s;  hsv[2] = v;
  vtkMath::HSVToRGB(hsv, rgb);
  this->EndColorButton->SetColor(rgb);

  this->AddTraceEntry("$kw(%s) SetEndHSV %g %g %g", 
                      this->GetTclName(), h, s, v);

  this->RMScalarBarWidget->SetEndHSV(h,s,v);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ScalarBarCheckCallback()
{
  this->SetScalarBarVisibility(this->ScalarBarCheck->GetState());
  if ( this->GetPVRenderView() )
    {
    this->GetPVRenderView()->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarRange(double min, double max)
{
  this->SetScalarRangeInternal(min, max);
  this->AddTraceEntry("$kw(%s) SetScalarRange %g %g", this->GetTclName(),
                      min, max);
}


//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarRangeInternal(double min, double max)
{
  this->RMScalarBarWidget->SetScalarRange(min,max);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetVectorComponent(int component)
{
  this->RMScalarBarWidget->SetVectorComponent(component);
  this->AddTraceEntry("$kw(%s) SetVectorComponent %d", 
                      this->GetTclName(), component);
}


//----------------------------------------------------------------------------
void vtkPVColorMap::ResetScalarRange()
{
  this->ResetScalarRangeInternal();
  this->AddTraceEntry("$kw(%s) ResetScalarRange", this->GetTclName());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::ResetScalarRangeInternal()
{
  double range[2];
  double tmp[2];
  vtkPVSourceCollection *sourceList;
  vtkPVSource *pvs;
  int component = this->RMScalarBarWidget->GetVectorComponent();
  if (this->RMScalarBarWidget->GetVectorMode() == vtkRMScalarBarWidget::MAGNITUDE)
    {
    component = -1;
    }

  if (this->GetApplication() == NULL || this->PVRenderView == NULL)
    {
    vtkErrorMacro("Trying to reset scalar range without application and view.");
    return;
    }

  range[0] = VTK_DOUBLE_MAX;
  range[1] = -VTK_DOUBLE_MAX;

  // Compute global scalar range ...
  sourceList = this->PVRenderView->GetPVWindow()->GetSourceList("Sources");
  sourceList->InitTraversal();
  while ( (pvs = sourceList->GetNextPVSource()) )
    {
    // For point data ...
    vtkPVArrayInformation *ai;
    ai = pvs->GetDataInformation()->GetPointDataInformation()->
      GetArrayInformation(this->RMScalarBarWidget->GetArrayName());
    if (ai)
      {
      ai->GetComponentRange(component, tmp);
      if (tmp[0] < range[0])
        {
        range[0] = tmp[0];
        }
      if (tmp[1] > range[1])
        {
        range[1] = tmp[1];
        }
      }
    // For cell data ...
    ai = pvs->GetDataInformation()->GetCellDataInformation()->
      GetArrayInformation(this->RMScalarBarWidget->GetArrayName());
    if (ai)
      {  
      ai->GetComponentRange(component, tmp);
      if (tmp[0] < range[0])
        {
        range[0] = tmp[0];
        }
      if (tmp[1] > range[1])
        {
        range[1] = tmp[1];
        }
      }
    }

  if (range[1] < range[0])
    {
    range[0] = 0.0;
    range[1] = 1.0;
    }
  if (range[0] == range[1])
    {
    range[1] = range[0] + 0.001;
    }

  this->SetScalarRangeInternal(range[0], range[1]);

  if ( this->GetPVRenderView() )
    {
    this->GetPVRenderView()->EventuallyRender();
    }
}


//----------------------------------------------------------------------------
void vtkPVColorMap::ResetScalarRangeInternal(vtkPVSource* pvs)
{
  double range[2];
  double tmp[2];

  int component = this->RMScalarBarWidget->GetVectorComponent();
  if (this->RMScalarBarWidget->GetVectorMode() == vtkRMScalarBarWidget::MAGNITUDE)
    {
    component = -1;
    }

  if (this->GetApplication() == NULL || this->PVRenderView == NULL)
    {
    vtkErrorMacro("Trying to reset scalar range without application and view.");
    return;
    }

  range[0] = VTK_DOUBLE_MAX;
  range[1] = -VTK_DOUBLE_MAX;

  // For point data ...
  vtkPVArrayInformation *ai;
  ai = pvs->GetDataInformation()->GetPointDataInformation()->
    GetArrayInformation(this->RMScalarBarWidget->GetArrayName());
  if (ai)
    {
    ai->GetComponentRange(component, tmp);
    if (tmp[0] < range[0])
      {
      range[0] = tmp[0];
      }
    if (tmp[1] > range[1])
      {
      range[1] = tmp[1];
      }
    }
  // For cell data ...
  ai = pvs->GetDataInformation()->GetCellDataInformation()->
    GetArrayInformation(this->RMScalarBarWidget->GetArrayName());
  if (ai)
    {  
    ai->GetComponentRange(component, tmp);
    if (tmp[0] < range[0])
      {
      range[0] = tmp[0];
      }
    if (tmp[1] > range[1])
      {
      range[1] = tmp[1];
      }
    }

  if (range[1] < range[0])
    {
    range[0] = 0.0;
    range[1] = 1.0;
    }
  if (range[0] == range[1])
    {
    range[1] = range[0] + 0.001;
    }

  this->SetScalarRangeInternal(range[0], range[1]);

  if ( this->GetPVRenderView() )
    {
    this->GetPVRenderView()->EventuallyRender();
    }
}





//----------------------------------------------------------------------------
vtkPVApplication* vtkPVColorMap::GetPVApplication()
{
  if (this->GetApplication() == NULL)
    {
    return NULL;
    }
  
  if (this->GetApplication()->IsA("vtkPVApplication"))
    {  
    return (vtkPVApplication*)(this->GetApplication());
    }
  else
    {
    vtkErrorMacro("Bad typecast");
    return NULL;
    } 
}

//----------------------------------------------------------------------------
void vtkPVColorMap::IncrementUseCount()
{
  ++this->UseCount;
  this->UpdateInternalScalarBarVisibility();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::DecrementUseCount()
{
  --this->UseCount;
  this->UpdateInternalScalarBarVisibility();
}


//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarVisibility(int val)
{
  if (this->ScalarBarVisibility == val)
    {
    return;
    }
  this->ScalarBarVisibility = val;
  
  // Make sure the UI is up to date.
  if (val)
    {
    this->ScalarBarCheck->SetState(1);
    }
  else
    {
    this->ScalarBarCheck->SetState(0);
    }

  this->AddTraceEntry("$kw(%s) SetScalarBarVisibility %d", this->GetTclName(),
                      val);
  
  this->UpdateInternalScalarBarVisibility();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateInternalScalarBarVisibility()
{
  int visible = this->ScalarBarVisibility;

  if (this->UseCount == 0)
    {
    visible = 0;
    }

  if (this->InternalScalarBarVisibility == visible)
    {
    return;
    }
  this->InternalScalarBarVisibility = visible;


  if (!this->GetPVRenderView())
    {
    return;
    }
  
  vtkPVProcessModule* pm = this->GetPVApplication()->GetProcessModule();
  vtkPVRenderModule* rm = pm->GetRenderModule();

  if (rm == NULL)
    {
    return;
    }
  this->RMScalarBarWidget->SetVisibility(visible);
  if (visible)
    {
    // This removes all renderers from the render window before enabling 
    // the widget. It then adds them back into the render window.
    // I assume the widget needs to know which renderer it uses.
    // It's the old poked renderer problem.
    this->GetPVRenderView()->Enable3DWidget(this->RMScalarBarWidget->GetScalarBar());
    }
  this->Modified();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SaveInBatchScript(ofstream *file)
{
  if (this->VisitedFlag)
    {
    return;
    }
  this->VisitedFlag = 1;

  vtkClientServerID lookupTableID = this->RMScalarBarWidget->GetLookupTableID();
  double startHSV[3], endHSV[3];
  this->RMScalarBarWidget->GetStartHSV(startHSV);
  this->RMScalarBarWidget->GetEndHSV(endHSV);

  *file << endl;
  *file << "set pvTemp" <<  lookupTableID 
        << " [$proxyManager NewProxy lookup_tables LookupTable]"
        << endl;
  *file << "  $proxyManager RegisterProxy lookup_tables pvTemp"
        << lookupTableID << " $pvTemp" << lookupTableID 
        << endl;
  *file << "  $pvTemp" << lookupTableID << " UnRegister {}" << endl;
  *file << "  [$pvTemp" << lookupTableID << " GetProperty "
        << "NumberOfTableValues] SetElements1 "
        << this->RMScalarBarWidget->GetNumberOfColors() << endl;
  *file << "  [$pvTemp" << lookupTableID << " GetProperty "
        << "HueRange] SetElements2 "
        << startHSV[0] << " " << endHSV[0] << endl;


        
  *file << "  [$pvTemp" << lookupTableID << " GetProperty "
        << "SaturationRange] SetElements2 "
        << startHSV[1] << " " << endHSV[1] << endl;
  *file << "  [$pvTemp" << lookupTableID << " GetProperty "
        << "ValueRange] SetElements2 "
        << startHSV[2] << " " << endHSV[2] << endl;
  *file << "  [$pvTemp" << lookupTableID << " GetProperty "
        << "ScalarRange] SetElements2 "
        << this->RMScalarBarWidget->GetScalarRange()[0] << " " << this->RMScalarBarWidget->GetScalarRange()[1] << endl;
  *file << "  [$pvTemp" << lookupTableID << " GetProperty "
        << "VectorComponent] SetElements1 "
        << this->RMScalarBarWidget->GetVectorComponent() << endl;
  *file << "  [$pvTemp" << lookupTableID << " GetProperty "
        << "VectorMode] SetElements1 "
        << this->RMScalarBarWidget->GetVectorMode() << endl;
  *file << "  $pvTemp" << lookupTableID << " UpdateVTKObjects"
        << endl;
  *file << "  $pvTemp" << lookupTableID << " Build"
        << endl;
  *file << endl;

  if (this->ScalarBarVisibility)
    {
    
    *file << endl;

    // First thing define the TextProperty (Title and Label )
    ostrstream pvTitle, pvLabel;
    pvTitle << "pvTitle" << this->RMScalarBarWidget->ScalarBarActorID << ends;
    this->TitleTextPropertyWidget->SaveInBatchScript( pvTitle.str(), file );

    pvLabel << "pvLabel" << this->RMScalarBarWidget->ScalarBarActorID << ends;
    this->LabelTextPropertyWidget->SaveInBatchScript( pvLabel.str(), file );

    *file << "set pvTemp" <<  this->RMScalarBarWidget->ScalarBarActorID
          << " [$proxyManager NewProxy rendering ScalarBarActor]"
          << endl;
    *file << "  $proxyManager RegisterProxy scalar_bar pvTemp"
          << this->RMScalarBarWidget->ScalarBarActorID << " $pvTemp" << this->RMScalarBarWidget->ScalarBarActorID
          << endl;
    *file << "  $pvTemp" << this->RMScalarBarWidget->ScalarBarActorID << " UnRegister {}" << endl;

    *file << "  [$pvTemp" << this->RMScalarBarWidget->ScalarBarActorID 
          << " GetProperty LookupTable] AddProxy $pvTemp"
          << this->RMScalarBarWidget->LookupTableID << endl;

    *file << "  [$pvTemp" << this->RMScalarBarWidget->ScalarBarActorID 
          << " GetProperty TitleTextProperty] AddProxy $" << pvTitle.str()
          << endl;

    *file << "  [$pvTemp" << this->RMScalarBarWidget->ScalarBarActorID 
          << " GetProperty LabelTextProperty] AddProxy $" << pvLabel.str()
          << endl;

    *file << "  [$pvTemp" << this->RMScalarBarWidget->ScalarBarActorID 
          << " GetProperty Orientation] SetElements1 "
          << this->RMScalarBarWidget->ScalarBar->GetScalarBarActor()->GetOrientation() << endl;

    *file << "  [$pvTemp" << this->RMScalarBarWidget->ScalarBarActorID 
          << " GetProperty Width] SetElements1 " 
          << this->RMScalarBarWidget->ScalarBar->GetScalarBarActor()->GetWidth() << endl;

    *file << "  [$pvTemp" << this->RMScalarBarWidget->ScalarBarActorID 
          << " GetProperty Height] SetElements1 " 
          << this->RMScalarBarWidget->ScalarBar->GetScalarBarActor()->GetHeight() << endl;

    const double *pos = 
     this->RMScalarBarWidget->ScalarBar->GetScalarBarActor()->GetPositionCoordinate()->GetValue();
    *file << "  [$pvTemp" << this->RMScalarBarWidget->ScalarBarActorID 
          << " GetProperty Position] SetElements2 " 
          << pos[0] << " " << pos[1] << endl;

    *file << "  [$pvTemp" << this->RMScalarBarWidget->ScalarBarActorID 
          << " GetProperty Title] SetElement 0 {" 
          << this->RMScalarBarWidget->ScalarBar->GetScalarBarActor()->GetTitle() << "}" << endl;

    *file << "  [$pvTemp" << this->RMScalarBarWidget->ScalarBarActorID 
          << " GetProperty LabelFormat] SetElement 0 {" 
          << this->RMScalarBarWidget->ScalarBar->GetScalarBarActor()->GetLabelFormat() << "}" 
          << endl;

    pvTitle.rdbuf()->freeze(0);
    pvLabel.rdbuf()->freeze(0);

    /*  ostrstream ttprop, tlprop;
    ttprop << "[$pvTemp" << this->ScalarBarActorID  << " GetTitleTextProperty]" << ends;
    this->TitleTextPropertyWidget->SaveInBatchScript(file, ttprop.str());
    ttprop.rdbuf()->freeze(0);

    tlprop << "[$pvTemp" << this->ScalarBarActorID  << " GetLabelTextProperty]" << ends;
    this->LabelTextPropertyWidget->SaveInBatchScript(file, tlprop.str());
    tlprop.rdbuf()->freeze(0);*/
    
    *file << "  [$Ren1 GetProperty Displayers] AddProxy $pvTemp" << this->RMScalarBarWidget->ScalarBarActorID << endl;

    *file << "  $pvTemp" << this->RMScalarBarWidget->ScalarBarActorID << " UpdateVTKObjects"
          << endl;
    *file << endl;
    }
}



//----------------------------------------------------------------------------
void vtkPVColorMap::ColorRangeWidgetCallback()
{
  double range[2];

  this->ColorRangeWidget->GetRange(range);
  
  // Avoid the bad range error
  if (range[1] <= range[0])
    {
    range[1] = range[0] + 0.00001;
    }

  this->SetScalarRange(range[0], range[1]);
  if ( this->GetPVRenderView() )
    {
    this->GetPVRenderView()->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::LabToXYZ(double Lab[3], double xyz[3])
{
  
  //LAB to XYZ
  double var_Y = ( Lab[0] + 16 ) / 116;
  double var_X = Lab[1] / 500 + var_Y;
  double var_Z = var_Y - Lab[2] / 200;
    
  if ( pow(var_Y,3) > 0.008856 ) var_Y = pow(var_Y,3);
  else var_Y = ( var_Y - 16 / 116 ) / 7.787;
                                                            
  if ( pow(var_X,3) > 0.008856 ) var_X = pow(var_X,3);
  else var_X = ( var_X - 16 / 116 ) / 7.787;

  if ( pow(var_Z,3) > 0.008856 ) var_Z = pow(var_Z,3);
  else var_Z = ( var_Z - 16 / 116 ) / 7.787;
  double ref_X =  95.047;
  double ref_Y = 100.000;
  double ref_Z = 108.883;
  xyz[0] = ref_X * var_X;     //ref_X =  95.047  Observer= 2 Illuminant= D65
  xyz[1] = ref_Y * var_Y;     //ref_Y = 100.000
  xyz[2] = ref_Z * var_Z;     //ref_Z = 108.883
}

//----------------------------------------------------------------------------
void vtkPVColorMap::XYZToRGB(double xyz[3], double rgb[3])
{
  
  //double ref_X =  95.047;        //Observer = 2 Illuminant = D65
  //double ref_Y = 100.000;
  //double ref_Z = 108.883;
 
  double var_X = xyz[0] / 100;        //X = From 0 to ref_X
  double var_Y = xyz[1] / 100;        //Y = From 0 to ref_Y
  double var_Z = xyz[2] / 100;        //Z = From 0 to ref_Y
 
  double var_R = var_X *  3.2406 + var_Y * -1.5372 + var_Z * -0.4986;
  double var_G = var_X * -0.9689 + var_Y *  1.8758 + var_Z *  0.0415;
  double var_B = var_X *  0.0557 + var_Y * -0.2040 + var_Z *  1.0570;
 
  if ( var_R > 0.0031308 ) var_R = 1.055 * ( pow(var_R, ( 1 / 2.4 )) ) - 0.055;
  else var_R = 12.92 * var_R;
  if ( var_G > 0.0031308 ) var_G = 1.055 * ( pow(var_G ,( 1 / 2.4 )) ) - 0.055;
  else  var_G = 12.92 * var_G;
  if ( var_B > 0.0031308 ) var_B = 1.055 * ( pow(var_B, ( 1 / 2.4 )) ) - 0.055;
  else var_B = 12.92 * var_B;
                                                                                                 
  rgb[0] = var_R;
  rgb[1] = var_G;
  rgb[2] = var_B;
  
  //clip colors. ideally we would do something different for colors
  //out of gamut, but not really sure what to do atm.
  if (rgb[0]<0) rgb[0]=0;
  if (rgb[1]<0) rgb[1]=0;
  if (rgb[2]<0) rgb[2]=0;
  if (rgb[0]>1) rgb[0]=1;
  if (rgb[1]>1) rgb[1]=1;
  if (rgb[2]>1) rgb[2]=1;

}

//----------------------------------------------------------------------------
void vtkPVColorMap::MapConfigureCallback(int width, int height)
{
  this->UpdateMap(width, height);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::VectorModeMagnitudeCallback()
{
  this->AddTraceEntry("$kw(%s) VectorModeMagnitudeCallback", 
                      this->GetTclName());
  this->RMScalarBarWidget->SetVectorModeToMagnitude();
  this->Script("pack forget %s",
               this->VectorComponentMenu->GetWidgetName());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::VectorModeComponentCallback()
{
  if (this->VectorComponentMenu->GetApplication() == NULL)
    { // Sanity check
    return;
    }

  this->AddTraceEntry("$kw(%s) VectorModeComponentCallback", 
                      this->GetTclName());

  this->RMScalarBarWidget->SetVectorModeToComponent();
  this->Script("pack %s -side left -expand f -fill both -padx 2",
               this->VectorComponentMenu->GetWidgetName());
}

//----------------------------------------------------------------------------
void vtkPVColorMap::VectorComponentCallback(int component)
{
  this->SetVectorComponent(component);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateMap(int width, int height)
{
  int size;
  int i, j;
  double *range;
  double val, step;
  unsigned char *rgba;  
  unsigned char *ptr;  

  size = width*height;
  if (this->MapDataSize < size)
    {
    if (this->MapData)
      {
      delete [] this->MapData;
      }
    this->MapData = new unsigned char[size*3];
    this->MapDataSize = size;
    }
  this->MapWidth = width;
  this->MapHeight = height;

  if (this->RMScalarBarWidget->GetLookupTable() == NULL)
    {
    return;
    }

  range = this->RMScalarBarWidget->GetLookupTable()->GetRange();
  step = (range[1]-range[0])/(double)(width);
  ptr = this->MapData;
  for (j = 0; j < height; ++j)
    {
    for (i = 0; i < width; ++i)
      {
      val = range[0] + ((double)(i)*step);
      rgba = this->RMScalarBarWidget->GetLookupTable()->MapValue(val);
      
      ptr[0] = rgba[0];
      ptr[1] = rgba[1];
      ptr[2] = rgba[2];
      ptr += 3;
      }
    }

  if (size > 0)
    {
    this->Map->SetImageOption(this->MapData, width, height, 3);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarPosition1(float x, float y)
{
  this->RMScalarBarWidget->SetScalarBarPosition1(x,y);
  this->AddTraceEntry("$kw(%s) SetScalarBarPosition1 %f %f", 
                      this->GetTclName(), x, y);
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarPosition2(float x, float y)
{
  this->RMScalarBarWidget->SetScalarBarPosition2(x,y);
  this->AddTraceEntry("$kw(%s) SetScalarBarPosition2 %f %f", 
                      this->GetTclName(), x, y);
  this->RenderView();
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SetScalarBarOrientation(int o)
{
  this->RMScalarBarWidget->SetScalarBarOrientation(o);
  this->AddTraceEntry("$kw(%s) SetScalarBarOrientation %d", 
                      this->GetTclName(), o);
  this->RenderView();
}

//----------------------------------------------------------------------------
const char* vtkPVColorMap::GetScalarBarTitle() 
{
  return this->RMScalarBarWidget->GetScalarBarTitle();
}
//----------------------------------------------------------------------------
const char* vtkPVColorMap::GetArrayName() 
{ 
  return this->RMScalarBarWidget->GetArrayName();
}
//----------------------------------------------------------------------------
int vtkPVColorMap::GetNumberOfVectorComponents()
{
  return this->RMScalarBarWidget->GetNumberOfVectorComponents();
}
//----------------------------------------------------------------------------
const char* vtkPVColorMap::GetScalarBarLabelFormat()
{
  return this->RMScalarBarWidget->GetScalarBarLabelFormat();
}
//----------------------------------------------------------------------------
double *vtkPVColorMap::GetScalarRange()
{
  return this->RMScalarBarWidget->GetScalarRange();
}
//----------------------------------------------------------------------------
int vtkPVColorMap::GetVectorComponent()
{
  return this->RMScalarBarWidget->GetVectorComponent();
}
//----------------------------------------------------------------------------
vtkClientServerID vtkPVColorMap::GetLookupTableID()
{
  return this->RMScalarBarWidget->GetLookupTableID();
}
//----------------------------------------------------------------------------
void vtkPVColorMap::ExecuteEvent(vtkObject* vtkNotUsed(wdg), 
                                 unsigned long event,  
                                 void* vtkNotUsed(calldata))
{
  switch ( event )
    {
    case vtkKWEvent::WidgetModifiedEvent:
      this->UpdateVectorComponentMenu();
      this->ScalarBarVectorTitleEntry->SetValue(
        this->RMScalarBarWidget->GetScalarBarVectorTitle());
      this->ScalarBarLabelFormatEntry->SetValue(
        this->RMScalarBarWidget->GetScalarBarLabelFormat());
      this->ScalarBarTitleEntry->SetValue(
        this->RMScalarBarWidget->GetScalarBarTitle());

      double range[2];
      this->RMScalarBarWidget->GetScalarRange(range);
      this->ColorRangeWidget->SetRange(range);      
      if (this->MapWidth > 0 && this->MapHeight > 0)
          {
          this->UpdateMap(this->MapWidth, this->MapHeight);
          }
      this->RenderView();
      this->Modified();
      break;

    case vtkCommand::StartInteractionEvent:
      this->PVRenderView->GetPVWindow()->InteractiveRenderEnabledOn();
      break;

    case vtkCommand::EndInteractionEvent:
      this->PVRenderView->GetPVWindow()->InteractiveRenderEnabledOff();
      this->RenderView();
      vtkScalarBarActor* sact = this->RMScalarBarWidget->GetScalarBar()->GetScalarBarActor();
      double *pos1 = sact->GetPositionCoordinate()->GetValue();
      double *pos2 = sact->GetPosition2Coordinate()->GetValue();
      this->AddTraceEntry("$kw(%s) SetScalarBarPosition1 %lf %lf", 
                          this->GetTclName(), pos1[0], pos1[1]);
      this->AddTraceEntry("$kw(%s) SetScalarBarPosition2 %lf %lf", 
                          this->GetTclName(), pos2[0], pos2[1]);
      this->AddTraceEntry("$kw(%s) SetScalarBarOrientation %d",
                          this->GetTclName(), sact->GetOrientation());

      break;
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::SaveState(ofstream *file)
{
  // Just manually initialize.
  *file << "set kw(" << this->GetTclName() << ") [$kw(" 
        << this->PVRenderView->GetPVWindow()->GetTclName() 
        << ") GetPVColorMap {" << this->RMScalarBarWidget->GetArrayName() << "} " 
        << this->RMScalarBarWidget->GetNumberOfVectorComponents() << "]\n";

  if (strcmp(this->RMScalarBarWidget->GetScalarBarTitle(), 
             this->RMScalarBarWidget->GetArrayName()) != 0)
    {
    *file << "$kw(" << this->GetTclName() << ") SetScalarBarTitle {" 
          << this->RMScalarBarWidget->GetScalarBarTitle() << "}\n"; 
    }

  if (strcmp(this->RMScalarBarWidget->GetVectorMagnitudeTitle(), "Magnitude") != 0)
    {
    *file << "$kw(" << this->GetTclName() << ") SetScalarBarVectorTitle {" 
          << this->RMScalarBarWidget->GetVectorMagnitudeTitle() << "}\n"; 
    }

  if (strcmp(this->RMScalarBarWidget->GetScalarBarLabelFormat(), "%-#6.3g") != 0)
    {
    *file << "$kw(" << this->GetTclName() << ") SetScalarBarLabelFormat {" 
          << this->RMScalarBarWidget->GetScalarBarLabelFormat()<< "}\n"; 
    }

  *file << "$kw(" << this->GetTclName() << ") SetScalarRange " 
        << this->RMScalarBarWidget->GetScalarRange()[0] << " " << 
        this->RMScalarBarWidget->GetScalarRange()[1] << endl;

  double startHSV[3];
  double endHSV[3];
  this->RMScalarBarWidget->GetStartHSV(startHSV);
  this->RMScalarBarWidget->GetEndHSV(endHSV);

  *file << "$kw(" << this->GetTclName() << ") SetStartHSV " 
        << startHSV[0] << " " << startHSV[1] << " " 
        << startHSV[2] << endl;
  *file << "$kw(" << this->GetTclName() << ") SetEndHSV " 
        << endHSV[0] << " " << endHSV[1] << " " 
        << endHSV[2] << endl;

  if (this->RMScalarBarWidget->GetNumberOfColors() != 256)
    {
    *file << "$kw(" << this->GetTclName() << ") SetNumberOfColors " 
          << this->RMScalarBarWidget->GetNumberOfColors() << endl;
    }
  if (this->RMScalarBarWidget->GetVectorMode() == vtkRMScalarBarWidget::MAGNITUDE)
    {
    *file << "$kw(" << this->GetTclName() << ") VectorModeMagnitudeCallback\n";
    }
  if (this->RMScalarBarWidget->GetVectorMode() == vtkRMScalarBarWidget::COMPONENT 
    && this->RMScalarBarWidget->GetNumberOfVectorComponents() > 1)
    {
    *file << "$kw(" << this->GetTclName() << ") VectorModeComponentCallback\n";
    if (this->RMScalarBarWidget->GetVectorComponent() != 0)
      {
      *file << "$kw(" << this->GetTclName() << ") SetVectorComponent " 
            << this->RMScalarBarWidget->GetVectorComponent() << endl;
      }
    }

  *file << "$kw(" << this->GetTclName() << ") SetScalarBarVisibility " 
        << this->ScalarBarVisibility << endl;

  vtkScalarBarActor* sact = this->RMScalarBarWidget->GetScalarBar()->GetScalarBarActor();
  double *pos1 = sact->GetPositionCoordinate()->GetValue();
  double *pos2 = sact->GetPosition2Coordinate()->GetValue();

  if (pos1[0] != 0.87 || pos1[1] != 0.25)
    {    
    *file << "$kw(" << this->GetTclName() << ") SetScalarBarPosition1 " 
          << pos1[0] << " " << pos1[1] << endl; 
    }
  if (pos2[0] != 0.13 || pos2[1] != 0.5)
    {
    *file << "$kw(" << this->GetTclName() << ") SetScalarBarPosition2 " 
          << pos2[0] << " " << pos2[1] << endl;
    }
  if (sact->GetOrientation() != 1)
    {
    *file << "$kw(" << this->GetTclName() << ") SetScalarBarOrientation " 
         << sact->GetOrientation() << endl;
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMap::Update()
{
  double range[2];
  double tmp[2];
  vtkPVSourceCollection *sourceList;
  vtkPVSource *pvs;
  int component = this->RMScalarBarWidget->GetVectorComponent();
  if (this->RMScalarBarWidget->GetVectorMode() == vtkRMScalarBarWidget::MAGNITUDE)
    {
    component = -1;
    }

  if (this->GetApplication() == NULL || this->PVRenderView == NULL)
    {
    vtkErrorMacro("Trying to reset scalar range without application and view.");
    return;
    }

  range[0] = VTK_DOUBLE_MAX;
  range[1] = -VTK_DOUBLE_MAX;

  // Compute global scalar range ...
  sourceList = this->PVRenderView->GetPVWindow()->GetSourceList("Sources");
  sourceList->InitTraversal();
  while ( (pvs = sourceList->GetNextPVSource()) )
    {
    // For point data ...
    vtkPVArrayInformation *ai;
    ai = pvs->GetDataInformation()->GetPointDataInformation()->
      GetArrayInformation(this->RMScalarBarWidget->GetArrayName());
    if (ai)
      {
      ai->GetComponentRange(component, tmp);
      if (tmp[0] < range[0])
        {
        range[0] = tmp[0];
        }
      if (tmp[1] > range[1])
        {
        range[1] = tmp[1];
        }
      }
    // For cell data ...
    ai = pvs->GetDataInformation()->GetCellDataInformation()->
      GetArrayInformation(this->RMScalarBarWidget->GetArrayName());
    if (ai)
      {  
      ai->GetComponentRange(component, tmp);
      if (tmp[0] < range[0])
        {
        range[0] = tmp[0];
        }
      if (tmp[1] > range[1])
        {
        range[1] = tmp[1];
        }
      }
    }

  if (range[1] < range[0])
    {
    range[0] = 0.0;
    range[1] = 1.0;
    }
  if (range[0] == range[1])
    {
    range[1] = range[0] + 0.001;
    }

  this->ColorRangeWidget->SetWholeRange(range[0], range[1]);

  // Compute an appropriate resolution.
  double x = range[1]-range[0];
  x = log10(x);
  // shift place value over 2 (100 divisions).
  int place = (int)(x) - 2;
  this->ColorRangeWidget->SetResolution(pow(10.0, (double)(place)));

  this->RMScalarBarWidget->GetScalarRange(tmp);
  this->ColorRangeWidget->SetRange(tmp[0], tmp[1]);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::UpdateEnableState()
{
  this->Superclass::UpdateEnableState();

  this->PropagateEnableState(this->ColorMapFrame);
  this->PropagateEnableState(this->ArrayNameLabel);
  this->PropagateEnableState(this->NumberOfColorsScale);
  this->PropagateEnableState(this->ColorEditorFrame);
  this->PropagateEnableState(this->StartColorButton);
  this->PropagateEnableState(this->Map);
  this->PropagateEnableState(this->EndColorButton);
  this->PropagateEnableState(this->VectorFrame);
  this->PropagateEnableState(this->VectorModeMenu);
  this->PropagateEnableState(this->VectorComponentMenu);
  this->PropagateEnableState(this->ScalarBarVectorTitleEntry);
  this->PropagateEnableState(this->ScalarBarFrame);
  this->PropagateEnableState(this->ScalarBarCheck);
  this->PropagateEnableState(this->ScalarBarTitleFrame);
  this->PropagateEnableState(this->ScalarBarTitleLabel);
  this->PropagateEnableState(this->ScalarBarTitleEntry);
  this->PropagateEnableState(this->ScalarBarLabelFormatFrame);
  this->PropagateEnableState(this->ScalarBarLabelFormatLabel);
  this->PropagateEnableState(this->ScalarBarLabelFormatEntry);
  this->PropagateEnableState(this->TitleTextPropertyWidget);
  this->PropagateEnableState(this->LabelTextPropertyWidget);
  this->PropagateEnableState(this->PresetsMenuButton);
  this->PropagateEnableState(this->ColorRangeFrame);
  this->PropagateEnableState(this->ColorRangeWidget);
  this->PropagateEnableState(this->BackButton);
}

//----------------------------------------------------------------------------
void vtkPVColorMap::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
  os << indent << "UseCount: " << this->UseCount << endl;

  os << indent << "RMScalarBarWidget: " << this->RMScalarBarWidget << endl;
  os << indent << "ScalarBarCheck: " << this->ScalarBarCheck << endl;
  os << indent << "TitleTextPropertyWidget: " << this->TitleTextPropertyWidget << endl;
  os << indent << "LabelTextPropertyWidget: " << this->LabelTextPropertyWidget << endl;
  
  os << indent << "ScalarBarVisibility: " << this->ScalarBarVisibility << endl;
  os << indent << "VisitedFlag: " << this->VisitedFlag << endl;
  os << indent << "ScalarBarCheck: " << this->ScalarBarCheck << endl;
}
