function(format_path out)
  set(path_segments ${ARGN})
  set(path_string "")
  foreach(segment IN LISTS path_segments)
    if(segment MATCHES "^[0-9]+$")
      string(APPEND path_string "[${segment}]")
    else()
      if(path_string STREQUAL "")
        string(APPEND path_string "${segment}")
      else()
        string(APPEND path_string ".${segment}")
      endif()
    endif()
  endforeach()
  if(path_string STREQUAL "")
    set(path_string "<root>")
  endif()
  set(${out} "${path_string}" PARENT_SCOPE)
endfunction()

macro(fail_at entity actual expected)
  format_path(formatted_path ${ARGN})
  set(RunCMake_TEST_FAILED "Attribute '${formatted_path}' ${entity} '${actual}' does not match expected ${entity} '${expected}'")
endmacro()

macro(fail_array_subset expected_index)
  format_path(formatted_path ${ARGN})
  set(RunCMake_TEST_FAILED "Attribute '${formatted_path}' array element (subset match) had no match for expected index ${expected_index}")
endmacro()

macro(bubble_error)
  set(RunCMake_TEST_FAILED "${RunCMake_TEST_FAILED}" PARENT_SCOPE)
endmacro()

function(json_matches out actual_node expected_node display_path)
  set(current_path ${display_path})

  string(JSON actual_type TYPE "${actual_node}")
  string(JSON expected_type TYPE "${expected_node}")
  if (NOT actual_type STREQUAL expected_type)
    fail_at("type" "${actual_type}" "${expected_type}" ${current_path})
    bubble_error()
    set(${out} FALSE PARENT_SCOPE)
    return()
  endif()

  if (expected_type STREQUAL "OBJECT")
    string(JSON expected_length LENGTH "${expected_node}")
    math(EXPR expected_last_index "${expected_length}-1")
    foreach(key_index RANGE ${expected_last_index})
      string(JSON key MEMBER "${expected_node}" "${key_index}")

      string(JSON probe ERROR_VARIABLE error GET "${actual_node}" "${key}")
      if (error STREQUAL "NOT_FOUND")
        fail_at("object member presence" "<missing>" "${key}" ${current_path} "${key}")
        bubble_error()
        set(${out} FALSE PARENT_SCOPE)
        return()
      endif()

      string(JSON actual_child_type TYPE "${actual_node}" "${key}")
      string(JSON expected_child_type TYPE "${expected_node}" "${key}")
      if (NOT actual_child_type STREQUAL expected_child_type)
        fail_at("type" "${actual_child_type}" "${expected_child_type}" ${current_path} "${key}")
        bubble_error()
        set(${out} FALSE PARENT_SCOPE)
        return()
      endif()

      if (expected_child_type STREQUAL "OBJECT" OR expected_child_type STREQUAL "ARRAY")
        string(JSON actual_child GET "${actual_node}" "${key}")
        string(JSON expected_child GET "${expected_node}" "${key}")
        set(next_path ${current_path})
        list(APPEND next_path "${key}")
        json_matches(is_ok "${actual_child}" "${expected_child}" "${next_path}")
        if (NOT is_ok)
          bubble_error()
          set(${out} FALSE PARENT_SCOPE)
          return()
        endif()
      else()
        string(JSON actual_value GET "${actual_node}" "${key}")
        string(JSON expected_value GET "${expected_node}" "${key}")
        if (NOT "${actual_value}" STREQUAL "${expected_value}")
          fail_at("value" "${actual_value}" "${expected_value}" ${current_path} "${key}")
          bubble_error()
          set(${out} FALSE PARENT_SCOPE)
          return()
        endif()
      endif()
    endforeach()

    set(${out} TRUE PARENT_SCOPE)
    return()

  elseif (expected_type STREQUAL "ARRAY")
    string(JSON actual_length LENGTH "${actual_node}")
    string(JSON expected_length LENGTH "${expected_node}")
    if (actual_length LESS expected_length)
      fail_at("array length (subset requirement)" "${actual_length}" ">= ${expected_length}" ${current_path})
      bubble_error()
      set(${out} FALSE PARENT_SCOPE)
      return()
    endif()

    math(EXPR expected_last_index "${expected_length}-1")
    math(EXPR actual_last_index "${actual_length}-1")
    foreach(expected_index RANGE ${expected_last_index})
      set(is_matched FALSE)
      set(best_error "")
      set(best_error_set FALSE)

      string(JSON expected_element_type TYPE "${expected_node}" "${expected_index}")

      if (expected_element_type STREQUAL "OBJECT" OR expected_element_type STREQUAL "ARRAY")
        string(JSON expected_element GET "${expected_node}" "${expected_index}")
        foreach(actual_index RANGE ${actual_last_index})
          string(JSON actual_element_type TYPE "${actual_node}" "${actual_index}")
          if (NOT actual_element_type STREQUAL expected_element_type)
            continue()
          endif()

          string(JSON actual_element GET "${actual_node}" "${actual_index}")
          set(next_path ${current_path})
          list(APPEND next_path "${actual_index}")

          json_matches(one_ok "${actual_element}" "${expected_element}" "${next_path}")
          if (one_ok)
            set(is_matched TRUE)
            break()
          else()
            if (NOT best_error_set)
              set(best_error "${RunCMake_TEST_FAILED}")
              set(best_error_set TRUE)
            endif()
            set(RunCMake_TEST_FAILED "")
          endif()
        endforeach()
      else()
        string(JSON expected_value GET "${expected_node}" "${expected_index}")
        foreach(actual_index RANGE ${actual_last_index})
          string(JSON actual_element_type TYPE "${actual_node}" "${actual_index}")
          if (NOT actual_element_type STREQUAL expected_element_type)
            continue()
          endif()
          string(JSON actual_value GET "${actual_node}" "${actual_index}")
          if ("${actual_value}" STREQUAL "${expected_value}")
            set(is_matched TRUE)
            break()
          endif()
        endforeach()
        if (NOT is_matched AND NOT best_error_set)
          set(best_error "")
          set(best_error_set TRUE)
        endif()
      endif()

      if (NOT is_matched)
        if (best_error_set AND NOT "${best_error}" STREQUAL "")
          set(RunCMake_TEST_FAILED "${best_error}")
        else()
          fail_array_subset("${expected_index}" ${current_path})
        endif()
        bubble_error()
        set(${out} FALSE PARENT_SCOPE)
        return()
      endif()
    endforeach()

    set(${out} TRUE PARENT_SCOPE)
    return()
  endif()
endfunction()

function(expect_value content expected)
  string(JSON actual ERROR_VARIABLE error GET "${content}" ${ARGN})
  if (error STREQUAL "NOT_FOUND")
    list(JOIN ARGN "." path_name)
    set(RunCMake_TEST_FAILED "Path '${path_name}' not found in JSON input" PARENT_SCOPE)
    return()
  endif()
  if (NOT "${actual}" STREQUAL "${expected}")
    fail_at("value" "${actual}" "${expected}" ${ARGN})
    bubble_error()
  endif()
endfunction()

function(expect_array content expected_length)
  string(JSON value_type ERROR_VARIABLE error TYPE "${content}" ${ARGN})
  if (error STREQUAL "NOT_FOUND")
    list(JOIN ARGN "." path_name)
    set(RunCMake_TEST_FAILED "Path '${path_name}' not found in JSON input" PARENT_SCOPE)
    return()
  endif()
  if (NOT value_type STREQUAL "ARRAY")
    fail_at("type" "${value_type}" "ARRAY" ${ARGN})
    bubble_error()
    return()
  endif()
  string(JSON actual_length LENGTH "${content}" ${ARGN})
  if (NOT actual_length EQUAL "${expected_length}")
    fail_at("length" "${actual_length}" "${expected_length}" ${ARGN})
    bubble_error()
  endif()
endfunction()

function(expect_object content expected_var)
  string(JSON actual_node ERROR_VARIABLE error GET "${content}" ${ARGN})
  if (error STREQUAL "NOT_FOUND")
    list(JOIN ARGN "." path_name)
    set(RunCMake_TEST_FAILED "Path '${path_name}' not found in JSON input" PARENT_SCOPE)
    return()
  endif()

  set(expected_text "${${expected_var}}")
  set(display_path ${ARGN})

  json_matches(is_ok "${actual_node}" "${expected_text}" "${display_path}")
  if (NOT is_ok)
    bubble_error()
    return()
  endif()
endfunction()
