# Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
# file Copyright.txt or https://cmake.org/licensing for details.


# determine the compiler to use for Ada programs
# NOTE, a generator may set CMAKE_Ada_COMPILER before
# loading this file to force a compiler.
# use environment variable ADA first if defined by user, next use
# the cmake variable CMAKE_GENERATOR_Ada which can be defined by a generator
# as a default compiler
# If the internal cmake variable _CMAKE_TOOLCHAIN_PREFIX is set, this is used
# as prefix for the tools (e.g. arm-elf-gnat, arm-elf-ar etc.)
#
# Sets the following variables:
#   CMAKE_Ada_COMPILER
#   CMAKE_Ada_BACKEND_COMPILER
#   CMAKE_Ada_COMPILER_AR
#   CMAKE_Ada_COMPILER_RANLIB
#
# If not already set before, it also sets
#   _CMAKE_TOOLCHAIN_PREFIX

include(${CMAKE_ROOT}/Modules/CMakeDetermineCompiler.cmake)

# Load system-specific compiler preferences for this language.
include(Platform/${CMAKE_SYSTEM_NAME}-Determine-Ada OPTIONAL)
include(Platform/${CMAKE_SYSTEM_NAME}-Ada OPTIONAL)
if(NOT CMAKE_Ada_COMPILER_NAMES)
  set(CMAKE_Ada_COMPILER_NAMES gnat)
endif()

if(CMAKE_GENERATOR MATCHES "^Ninja|Makefiles")
  if(NOT CMAKE_Ada_COMPILER)
    set(CMAKE_Ada_COMPILER_INIT NOTFOUND)

    # prefer the environment variable ADAC
    if(NOT $ENV{ADAC} STREQUAL "")
      separate_arguments(CMAKE_Ada_COMPILER_INIT UNIX_COMMAND PROGRAM $ENV{ADAC})
      list(POP_BACK CMAKE_Ada_COMPILER_INIT CMAKE_Ada_FLAGS_ENV_INIT)
      if(CMAKE_Ada_FLAGS_ENV_INIT)
        set(CMAKE_Ada_COMPILER_ARG1 "${CMAKE_Ada_FLAGS_ENV_INIT}" CACHE STRING "Arguments to Ada compiler")
      endif()
      if(NOT EXISTS ${CMAKE_Ada_COMPILER_INIT})
        message(FATAL_ERROR "Could not find compiler set in environment variable ADAC:\n$ENV{ADAC}.\n${CMAKE_Ada_COMPILER_INIT}")
      endif()
    endif()

    # next prefer the generator specified compiler
    if(CMAKE_GENERATOR_Ada)
      if(NOT CMAKE_Ada_COMPILER_INIT)
        set(CMAKE_Ada_COMPILER_INIT ${CMAKE_GENERATOR_Ada})
      endif()
    endif()

    # finally list compilers to try
    if(NOT CMAKE_Ada_COMPILER_INIT)
      set(CMAKE_Ada_COMPILER_LIST ${_CMAKE_TOOLCHAIN_PREFIX}gnat)
    endif()

    _cmake_find_compiler(Ada)
  else()
    _cmake_find_compiler_path(Ada)
  endif()
  mark_as_advanced(CMAKE_Ada_COMPILER)

  if(CMAKE_Ada_COMPILER MATCHES "gnat")
    # gnat tool is a front-end to the various GNAT tools
    # This tool is required for binding and linking executables
    # but for platform introspection, backend compiler (i.e. gcc) must be
    # defined as well.
    string(REGEX REPLACE "gnat((-[0-9]+(\\.[0-9]+)*)?(-[^.]+)?(.exe)?)$" "gcc\\1" _CMAKE_Ada_BACKEND_COMPILER "${CMAKE_Ada_COMPILER}")
    if (EXISTS "${_CMAKE_Ada_BACKEND_COMPILER}")
      set(CMAKE_Ada_BACKEND_COMPILER "${_CMAKE_Ada_BACKEND_COMPILER}" CACHE PATH "Ada backend compiler" FORCE)
    else()
      set(CMAKE_Ada_BACKEND_COMPILER "CMAKE_Ada_BACKEND_COMPILER-NOTFOUND" CACHE PATH "Ada backend compiler" FORCE)
    endif()
    unset(_CMAKE_BACKEND_COMPILER)
  endif()
  mark_as_advanced(CMAKE_Ada_BACKEND_COMPILER)

  # Each entry in this list is a set of extra flags to try
  # adding to the compile line to see if it helps produce
  # a valid identification file.
  set(CMAKE_Ada_COMPILER_ID_TEST_FLAGS_FIRST "make -v")
  set(CMAKE_Ada_COMPILER_ID_TEST_FLAGS)
else()
  message(FATAL_ERROR "Ada language not supported by \"${CMAKE_GENERATOR}\" generator")
endif()

# Build a small source file to identify the compiler.
if(NOT CMAKE_Ada_COMPILER_ID_RUN)
  set(CMAKE_Ada_COMPILER_ID_RUN 1)

  if (CMAKE_Ada_BACKEND_COMPILER MATCHES "gcc")
    # Ada compiler is based on GNU toolchain
    # Use it to get compiler and platform information
    include(${CMAKE_ROOT}/Modules/CMakeCompilerIdDetection.cmake)
    compiler_id_detection(CMAKE_Ada_COMPILER_ID_CONTENT C
      ID_STRING
      VERSION_STRINGS
      PLATFORM_DEFAULT_COMPILER)
    configure_file("${CMAKE_ROOT}/Modules/CompilerId/AdaCompiler.defs.in"
      "${CMAKE_PLATFORM_INFO_DIR}/CompilerIdAda/AdaCompiler.defs" @ONLY)
    execute_process(COMMAND "${CMAKE_Ada_BACKEND_COMPILER}" -x c -P -E "${CMAKE_PLATFORM_INFO_DIR}/CompilerIdAda/AdaCompiler.defs"
      OUTPUT_VARIABLE CMAKE_Ada_PLATFORM_ID_CONTENT
      RESULT_VARIABLE CMAKE_Ada_RESULT)
    if (CMAKE_Ada_RESULT)
      # Something goes wrong, ignore the data
      unset(CMAKE_Ada_PLATFORM_ID_CONTENT)
    endif()
    unset(CMAKE_Ada_RESULT)
  endif()

  # Try to identify the compiler.
  set(CMAKE_Ada_COMPILER_ID)
  set(CMAKE_Ada_PLATFORM_ID)
  include(${CMAKE_ROOT}/Modules/CMakeDetermineCompilerId.cmake)
  cmake_determine_compiler_id(Ada ADAFLAGS CompilerId/AdaCompiler.adb)

  unset(CMAKE_Ada_PLATFORM_ID_CONTENT)
endif()

_cmake_find_compiler_sysroot(Ada "${CMAKE_Ada_BACKEND_COMPILER}")

if(NOT _CMAKE_TOOLCHAIN_LOCATION)
  cmake_path(GET CMAKE_Ada_COMPILER PARENT_PATH _CMAKE_TOOLCHAIN_LOCATION)
endif ()

# if we have a Ada cross compiler, they have usually some prefix, like
# e.g. powerpc-linux-gnat, arm-elf-gnat or i586-mingw32msvc-gnat , optionally
# with a 3-component version number at the end (e.g. arm-eabi-gnat-4.5.2).
# The other tools of the toolchain usually have the same prefix

if (NOT _CMAKE_TOOLCHAIN_PREFIX)

  if("${CMAKE_Ada_COMPILER_ID}" MATCHES "GNU")
    cmake_path(GET CMAKE_Ada_COMPILER FILENAME COMPILER_BASENAME)
    if (COMPILER_BASENAME MATCHES "^(.+-)gnat(-[0-9]+(\\.[0-9]+)*)?(-[^.]+)?(\\.exe)?$")
      set(_CMAKE_TOOLCHAIN_PREFIX ${CMAKE_MATCH_1})
      set(_CMAKE_COMPILER_SUFFIX ${CMAKE_MATCH_4})
    endif ()
  endif()

endif ()

set(_CMAKE_PROCESSING_LANGUAGE "Ada")
include(CMakeFindBinUtils)
include(Compiler/${CMAKE_Ada_COMPILER_ID}-FindBinUtils OPTIONAL)
unset(_CMAKE_PROCESSING_LANGUAGE)

if(CMAKE_Ada_COMPILER_SYSROOT)
  string(CONCAT _SET_CMAKE_Ada_COMPILER_SYSROOT
    "set(CMAKE_Ada_COMPILER_SYSROOT \"${CMAKE_Ada_COMPILER_SYSROOT}\")\n"
    "set(CMAKE_COMPILER_SYSROOT \"${CMAKE_Ada_COMPILER_SYSROOT}\")")
else()
  set(_SET_CMAKE_Ada_COMPILER_SYSROOT "")
endif()

if(CMAKE_Ada_COMPILER_ARCHITECTURE_ID)
  set(_SET_CMAKE_Ada_COMPILER_ARCHITECTURE_ID
    "set(CMAKE_Ada_COMPILER_ARCHITECTURE_ID ${CMAKE_Ada_COMPILER_ARCHITECTURE_ID})")
else()
  set(_SET_CMAKE_Ada_COMPILER_ARCHITECTURE_ID "")
endif()

# configure all variables set in this file
configure_file(${CMAKE_ROOT}/Modules/CMakeAdaCompiler.cmake.in
  ${CMAKE_PLATFORM_INFO_DIR}/CMakeAdaCompiler.cmake
  @ONLY
  )

set(CMAKE_Ada_COMPILER_ENV_VAR "ADAC")
