# Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
# file Copyright.txt or https://cmake.org/licensing for details.


# This file sets the basic flags for the C++ language in CMake.
# It also loads the available platform file for the system-compiler
# if it exists.
# It also loads a system - compiler - processor (or target hardware)
# specific file, which is mainly useful for crosscompiling and embedded systems.

include(CMakeLanguageInformation)

# some compilers use different extensions (e.g. sdcc uses .rel)
# so set the extension here first so it can be overridden by the compiler specific file
if(UNIX)
  set(CMAKE_Ada_OUTPUT_EXTENSION .o)
else()
  set(CMAKE_Ada_OUTPUT_EXTENSION .obj)
endif()

set(_INCLUDED_FILE 0)

# Load compiler-specific information.
if(CMAKE_Ada_COMPILER_ID)
  include(Compiler/${CMAKE_Ada_COMPILER_ID}-Ada OPTIONAL)
endif()

set(CMAKE_BASE_NAME)
cmake_path(GET CMAKE_Ada_COMPILER STEM CMAKE_BASE_NAME)


# load the system- and compiler specific files
if(CMAKE_Ada_COMPILER_ID)
  include(Platform/${CMAKE_EFFECTIVE_SYSTEM_NAME}-${CMAKE_Ada_COMPILER_ID}-Ada OPTIONAL RESULT_VARIABLE _INCLUDED_FILE)
endif()
if (NOT _INCLUDED_FILE)
  include(Platform/${CMAKE_EFFECTIVE_SYSTEM_NAME}-${CMAKE_BASE_NAME} OPTIONAL
          RESULT_VARIABLE _INCLUDED_FILE)
endif ()

# load any compiler-wrapper specific information
if (CMAKE_Ada_COMPILER_WRAPPER)
  __cmake_include_compiler_wrapper(Ada)
endif ()

# We specify the compiler information in the system file for some
# platforms, but this language may not have been enabled when the file
# was first included.  Include it again to get the language info.
# Remove this when all compiler info is removed from system files.
if (NOT _INCLUDED_FILE)
  include(Platform/${CMAKE_SYSTEM_NAME} OPTIONAL)
endif ()

if(CMAKE_Ada_SIZEOF_DATA_PTR)
  foreach(f ${CMAKE_Ada_ABI_FILES})
    include(${f})
  endforeach()
  unset(CMAKE_Ada_ABI_FILES)
endif()

# This should be included before the _INIT variables are
# used to initialize the cache.  Since the rule variables
# have if blocks on them, users can still define them here.
# But, it should still be after the platform file so changes can
# be made to those values.

if(CMAKE_USER_MAKE_RULES_OVERRIDE)
  # Save the full path of the file so try_compile can use it.
  include(${CMAKE_USER_MAKE_RULES_OVERRIDE} RESULT_VARIABLE _override)
  set(CMAKE_USER_MAKE_RULES_OVERRIDE "${_override}")
endif()

if(CMAKE_USER_MAKE_RULES_OVERRIDE_Ada)
  # Save the full path of the file so try_compile can use it.
  include(${CMAKE_USER_MAKE_RULES_OVERRIDE_Ada} RESULT_VARIABLE _override)
  set(CMAKE_USER_MAKE_RULES_OVERRIDE_Ada "${_override}")
endif()


# Create a set of shared library variable specific to Ada
# For 90% of the systems, these are the same flags as the C versions
# so if these are not set just copy the flags from the c version
if(NOT CMAKE_SHARED_LIBRARY_CREATE_Ada_FLAGS)
  set(CMAKE_SHARED_LIBRARY_CREATE_Ada_FLAGS ${CMAKE_SHARED_LIBRARY_CREATE_C_FLAGS})
endif()

if(NOT CMAKE_Ada_COMPILE_OPTIONS_PIC)
  set(CMAKE_Ada_COMPILE_OPTIONS_PIC ${CMAKE_C_COMPILE_OPTIONS_PIC})
endif()

if(NOT CMAKE_Ada_COMPILE_OPTIONS_PIE)
  set(CMAKE_Ada_COMPILE_OPTIONS_PIE ${CMAKE_C_COMPILE_OPTIONS_PIE})
endif()
if(NOT CMAKE_Ada_LINK_OPTIONS_PIE)
  set(CMAKE_Ada_LINK_OPTIONS_PIE ${CMAKE_C_LINK_OPTIONS_PIE})
endif()
if(NOT CMAKE_Ada_LINK_OPTIONS_NO_PIE)
  set(CMAKE_Ada_LINK_OPTIONS_NO_PIE ${CMAKE_C_LINK_OPTIONS_NO_PIE})
endif()

if(NOT CMAKE_Ada_COMPILE_OPTIONS_DLL)
  set(CMAKE_Ada_COMPILE_OPTIONS_DLL ${CMAKE_C_COMPILE_OPTIONS_DLL})
endif()

if(NOT CMAKE_SHARED_LIBRARY_Ada_FLAGS)
  set(CMAKE_SHARED_LIBRARY_Ada_FLAGS ${CMAKE_SHARED_LIBRARY_C_FLAGS})
endif()

if(NOT DEFINED CMAKE_SHARED_LIBRARY_LINK_Ada_FLAGS)
  set(CMAKE_SHARED_LIBRARY_LINK_Ada_FLAGS ${CMAKE_SHARED_LIBRARY_LINK_C_FLAGS})
endif()

if(NOT CMAKE_SHARED_LIBRARY_RUNTIME_Ada_FLAG)
  set(CMAKE_SHARED_LIBRARY_RUNTIME_Ada_FLAG ${CMAKE_SHARED_LIBRARY_RUNTIME_C_FLAG})
endif()

if(NOT CMAKE_SHARED_LIBRARY_RUNTIME_Ada_FLAG_SEP)
  set(CMAKE_SHARED_LIBRARY_RUNTIME_Ada_FLAG_SEP ${CMAKE_SHARED_LIBRARY_RUNTIME_C_FLAG_SEP})
endif()

if(NOT CMAKE_SHARED_LIBRARY_RPATH_LINK_Ada_FLAG)
  set(CMAKE_SHARED_LIBRARY_RPATH_LINK_Ada_FLAG ${CMAKE_SHARED_LIBRARY_RPATH_LINK_C_FLAG})
endif()

if(NOT DEFINED CMAKE_EXE_EXPORTS_Ada_FLAG)
  set(CMAKE_EXE_EXPORTS_Ada_FLAG ${CMAKE_EXE_EXPORTS_C_FLAG})
endif()

if(NOT DEFINED CMAKE_SHARED_LIBRARY_SONAME_Ada_FLAG)
  set(CMAKE_SHARED_LIBRARY_SONAME_Ada_FLAG ${CMAKE_SHARED_LIBRARY_SONAME_C_FLAG})
endif()

if(NOT CMAKE_EXECUTABLE_RUNTIME_Ada_FLAG)
  set(CMAKE_EXECUTABLE_RUNTIME_Ada_FLAG ${CMAKE_SHARED_LIBRARY_RUNTIME_Ada_FLAG})
endif()

if(NOT CMAKE_EXECUTABLE_RUNTIME_Ada_FLAG_SEP)
  set(CMAKE_EXECUTABLE_RUNTIME_Ada_FLAG_SEP ${CMAKE_SHARED_LIBRARY_RUNTIME_Ada_FLAG_SEP})
endif()

if(NOT CMAKE_EXECUTABLE_RPATH_LINK_Ada_FLAG)
  set(CMAKE_EXECUTABLE_RPATH_LINK_Ada_FLAG ${CMAKE_SHARED_LIBRARY_RPATH_LINK_Ada_FLAG})
endif()

if(NOT DEFINED CMAKE_SHARED_LIBRARY_LINK_Ada_WITH_RUNTIME_PATH)
  set(CMAKE_SHARED_LIBRARY_LINK_Ada_WITH_RUNTIME_PATH ${CMAKE_SHARED_LIBRARY_LINK_C_WITH_RUNTIME_PATH})
endif()

if(NOT CMAKE_INCLUDE_FLAG_Ada)
  set(CMAKE_INCLUDE_FLAG_Ada ${CMAKE_INCLUDE_FLAG_C})
endif()

# for most systems a module is the same as a shared library
# so unless the variable CMAKE_MODULE_EXISTS is set just
# copy the values from the LIBRARY variables
if(NOT CMAKE_MODULE_EXISTS)
  set(CMAKE_SHARED_MODULE_Ada_FLAGS ${CMAKE_SHARED_LIBRARY_Ada_FLAGS})
  set(CMAKE_SHARED_MODULE_CREATE_Ada_FLAGS ${CMAKE_SHARED_LIBRARY_CREATE_Ada_FLAGS})
endif()

# repeat for modules
if(NOT CMAKE_SHARED_MODULE_CREATE_Ada_FLAGS)
  set(CMAKE_SHARED_MODULE_CREATE_Ada_FLAGS ${CMAKE_SHARED_MODULE_CREATE_C_FLAGS})
endif()

if(NOT CMAKE_SHARED_MODULE_Ada_FLAGS)
  set(CMAKE_SHARED_MODULE_Ada_FLAGS ${CMAKE_SHARED_MODULE_C_FLAGS})
endif()

# Initialize Ada link type selection flags from C versions.
foreach(type SHARED_LIBRARY SHARED_MODULE EXE)
  if(NOT CMAKE_${type}_LINK_STATIC_Ada_FLAGS)
    set(CMAKE_${type}_LINK_STATIC_Ada_FLAGS
      ${CMAKE_${type}_LINK_STATIC_C_FLAGS})
  endif()
  if(NOT CMAKE_${type}_LINK_DYNAMIC_Ada_FLAGS)
    set(CMAKE_${type}_LINK_DYNAMIC_Ada_FLAGS
      ${CMAKE_${type}_LINK_DYNAMIC_C_FLAGS})
  endif()
endforeach()

# add the flags to the cache based
# on the initial values computed in the platform/*.cmake files
# use _INIT variables so that this only happens the first time
# and you can set these flags in the cmake cache
set(CMAKE_Ada_FLAGS_INIT "$ENV{ADAFLAGS} ${CMAKE_Ada_FLAGS_INIT}")

cmake_initialize_per_config_variable(CMAKE_Ada_FLAGS "Flags used by the Ada compiler")

if(CMAKE_Ada_STANDARD_LIBRARIES_INIT)
  set(CMAKE_Ada_STANDARD_LIBRARIES "${CMAKE_Ada_STANDARD_LIBRARIES_INIT}"
    CACHE STRING "Libraries linked by default with all Ada applications.")
  mark_as_advanced(CMAKE_Ada_STANDARD_LIBRARIES)
endif()

if(NOT CMAKE_Ada_COMPILER_LAUNCHER AND DEFINED ENV{CMAKE_Ada_COMPILER_LAUNCHER})
  set(CMAKE_Ada_COMPILER_LAUNCHER "$ENV{CMAKE_Ada_COMPILER_LAUNCHER}"
    CACHE STRING "Compiler launcher for Ada.")
endif()

include(CMakeCommonLanguageInclude)

# now define the following rules:
# CMAKE_Ada_CREATE_SHARED_LIBRARY
# CMAKE_Ada_CREATE_SHARED_MODULE
# CMAKE_Ada_COMPILE_OBJECT
# CMAKE_Ada_LINK_EXECUTABLE

# variables supplied by the generator at use time
# <TARGET>
# <TARGET_BASE> the target without the suffix
# <OBJECTS>
# <OBJECT>
# <LINK_LIBRARIES>
# <FLAGS>
# <LINK_FLAGS>

# Ada compiler information
# <CMAKE_Ada_COMPILER>
# <CMAKE_SHARED_LIBRARY_CREATE_Ada_FLAGS>
# <CMAKE_Ada_SHARED_MODULE_CREATE_FLAGS>
# <CMAKE_Ada_LINK_FLAGS>

# Static library tools
# <CMAKE_AR>
# <CMAKE_RANLIB>


# create a shared C++ library
if(NOT CMAKE_Ada_CREATE_SHARED_LIBRARY)
  set(CMAKE_Ada_CREATE_SHARED_LIBRARY
      "<CMAKE_Ada_COMPILER> <CMAKE_SHARED_LIBRARY_Ada_FLAGS> <LANGUAGE_COMPILE_FLAGS> <LINK_FLAGS> <CMAKE_SHARED_LIBRARY_CREATE_Ada_FLAGS> <SONAME_FLAG><TARGET_SONAME> -o <TARGET> <OBJECTS> <LINK_LIBRARIES>")
endif()

# create a c++ shared module copy the shared library rule by default
if(NOT CMAKE_Ada_CREATE_SHARED_MODULE)
  set(CMAKE_Ada_CREATE_SHARED_MODULE ${CMAKE_Ada_CREATE_SHARED_LIBRARY})
endif()


# Create a static archive incrementally for large object file counts.
# If CMAKE_Ada_CREATE_STATIC_LIBRARY is set it will override these.
if(NOT DEFINED CMAKE_Ada_ARCHIVE_CREATE)
  set(CMAKE_Ada_ARCHIVE_CREATE "<CMAKE_AR> qc <TARGET> <LINK_FLAGS> <OBJECTS>")
endif()
if(NOT DEFINED CMAKE_Ada_ARCHIVE_APPEND)
  set(CMAKE_Ada_ARCHIVE_APPEND "<CMAKE_AR> q <TARGET> <LINK_FLAGS> <OBJECTS>")
endif()
if(NOT DEFINED CMAKE_Ada_ARCHIVE_FINISH)
  set(CMAKE_Ada_ARCHIVE_FINISH "<CMAKE_RANLIB> <TARGET>")
endif()

# compile an Ada file into an object file
if(NOT CMAKE_Ada_COMPILE_OBJECT)
  set(CMAKE_Ada_COMPILE_OBJECT
    "<CMAKE_Ada_COMPILER> <DEFINES> <INCLUDES> <FLAGS> -o <OBJECT> -c <SOURCE>")
endif()

if(NOT CMAKE_Ada_LINK_EXECUTABLE)
  set(CMAKE_Ada_LINK_EXECUTABLE
    "<CMAKE_Ada_COMPILER> <FLAGS> <CMAKE_Ada_LINK_FLAGS> <LINK_FLAGS> <OBJECTS> -o <TARGET> <LINK_LIBRARIES>")
endif()

mark_as_advanced(
CMAKE_VERBOSE_MAKEFILE
)

set(CMAKE_Ada_INFORMATION_LOADED 1)
