// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause

#include "vtkAnariTestInteractor.h"
#include "vtkObjectFactory.h"

#include "vtkAnariLightNode.h"
#include "vtkAnariPass.h"
#include "vtkAnariSceneGraph.h"
#include "vtkLight.h"
#include "vtkLightCollection.h"
#include "vtkOpenGLRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkRendererCollection.h"

#include <string>
#include <vector>

#include <iostream>

namespace
{
static std::vector<std::string> ActorNames;
}

//----------------------------------------------------------------------------
class vtkAnariTestLooper : public vtkCommand
{
  // for progressive rendering
public:
  vtkTypeMacro(vtkAnariTestLooper, vtkCommand);

  static vtkAnariTestLooper* New()
  {
    vtkAnariTestLooper* self = new vtkAnariTestLooper;
    self->RenderWindow = nullptr;
    self->ProgressiveCount = 0;
    return self;
  }

  void Execute(
    vtkObject* vtkNotUsed(caller), unsigned long eventId, void* vtkNotUsed(callData)) override
  {
    if (eventId == vtkCommand::TimerEvent)
    {
      if (this->RenderWindow)
      {
        this->RenderWindow->Render();
      }
    }
    else
    {
      this->ProgressiveCount = 0;
    }
  }
  vtkRenderWindow* RenderWindow;
  int ProgressiveCount;
};

//----------------------------------------------------------------------------
vtkStandardNewMacro(vtkAnariTestInteractor);

//----------------------------------------------------------------------------
vtkAnariTestInteractor::vtkAnariTestInteractor()
{
  this->SetPipelineControlPoints(nullptr, nullptr, nullptr);
  this->VisibleActor = -1;
  this->VisibleLight = -1;
  this->Looper = vtkAnariTestLooper::New();
}

//----------------------------------------------------------------------------
vtkAnariTestInteractor::~vtkAnariTestInteractor()
{
  this->Looper->Delete();
}

//----------------------------------------------------------------------------
void vtkAnariTestInteractor::SetPipelineControlPoints(
  vtkRenderer* g, vtkRenderPass* _O, vtkRenderPass* _G)
{
  this->GLRenderer = g;
  this->O = _O;
  this->G = _G;
}

//----------------------------------------------------------------------------
void vtkAnariTestInteractor::OnKeyPress()
{
  if (this->GLRenderer == nullptr)
  {
    return;
  }

  // Get the keypress
  vtkRenderWindowInteractor* rwi = this->Interactor;
  char* ckey = rwi->GetKeySym();
  std::string key = ckey != nullptr ? ckey : "";

  if (key == "c")
  {
    vtkRenderPass* current = this->GLRenderer->GetPass();

    if (current == this->G)
    {
      std::cerr << "ANARI rendering " << this->O << std::endl;
      this->GLRenderer->SetPass(this->O);
      this->GLRenderer->GetRenderWindow()->Render();
    }
    else if (current == this->O)
    {
      std::cerr << "GL rendering " << this->G << std::endl;
      this->GLRenderer->SetPass(this->G);
      this->GLRenderer->GetRenderWindow()->Render();
    }
  }

  if (key == "n")
  {
    vtkActorCollection* actors = this->GLRenderer->GetActors();

    this->VisibleActor++;
    std::cerr << "VISIBLE " << this->VisibleActor;
    if (this->VisibleActor == actors->GetNumberOfItems())
    {
      this->VisibleActor = -1;
    }
    for (int i = 0; i < actors->GetNumberOfItems(); i++)
    {
      if (this->VisibleActor == -1 || this->VisibleActor == i)
      {
        if (i < static_cast<int>(ActorNames.size()))
        {
          std::cerr << " : " << ActorNames[i] << " ";
        }
        vtkActor::SafeDownCast(actors->GetItemAsObject(i))->SetVisibility(1);
      }
      else
      {
        vtkActor::SafeDownCast(actors->GetItemAsObject(i))->SetVisibility(0);
      }
    }
    std::cerr << std::endl;
    this->GLRenderer->ResetCamera();
    this->GLRenderer->GetRenderWindow()->Render();
  }

  if (key == "l")
  {
    vtkLightCollection* lights = this->GLRenderer->GetLights();

    this->VisibleLight++;
    if (this->VisibleLight == lights->GetNumberOfItems())
    {
      this->VisibleLight = -1;
    }
    std::cerr << "LIGHT " << this->VisibleLight << "/" << lights->GetNumberOfItems() << std::endl;
    for (int i = 0; i < lights->GetNumberOfItems(); i++)
    {
      if (this->VisibleLight == -1 || this->VisibleLight == i)
      {
        vtkLight::SafeDownCast(lights->GetItemAsObject(i))->SwitchOn();
      }
      else
      {
        vtkLight::SafeDownCast(lights->GetItemAsObject(i))->SwitchOff();
      }
    }
    this->GLRenderer->GetRenderWindow()->Render();
  }

  if (key == "I" || key == "i")
  {
    vtkLightCollection* lights = this->GLRenderer->GetLights();

    for (int i = 0; i < lights->GetNumberOfItems(); i++)
    {
      vtkLight* light = vtkLight::SafeDownCast(lights->GetItemAsObject(i));
      double intens = vtkAnariLightNode::GetLightScale(light) * 1.5;
      vtkAnariLightNode::SetLightScale(intens, light);
      std::cerr << "intensity " << intens << std::endl;
    }

    this->GLRenderer->GetRenderWindow()->Render();
  }

  // Forward events
  vtkInteractorStyleTrackballCamera::OnKeyPress();
}

//------------------------------------------------------------------------------
void vtkAnariTestInteractor::AddName(const char* name)
{
  ActorNames.push_back(std::string(name));
}

//------------------------------------------------------------------------------
vtkCommand* vtkAnariTestInteractor::GetLooper(vtkRenderWindow* rw)
{
  rw->Render();
  vtkAnariTestLooper::SafeDownCast(this->Looper)->RenderWindow = rw;
  return this->Looper;
}
