/*=========================================================================

  Program:   ParaView
  Module:    vtkSMChartRepresentationProxy.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkSMChartRepresentationProxy.h"

#include "vtkChartRepresentation.h"
#include "vtkClientServerStream.h"
#include "vtkObjectFactory.h"
#include "vtkPVArrayInformation.h"
#include "vtkPVDataInformation.h"
#include "vtkPVExtractSelection.h"
#include "vtkPVXMLElement.h"
#include "vtkSMDomain.h"
#include "vtkSMProperty.h"
#include "vtkSMPropertyHelper.h"
#include "vtkSMSourceProxy.h"
#include "vtkSMTransferFunctionManager.h"

vtkStandardNewMacro(vtkSMChartRepresentationProxy);
//----------------------------------------------------------------------------
vtkSMChartRepresentationProxy::vtkSMChartRepresentationProxy() = default;

//----------------------------------------------------------------------------
vtkSMChartRepresentationProxy::~vtkSMChartRepresentationProxy() = default;

//----------------------------------------------------------------------------
vtkChartRepresentation* vtkSMChartRepresentationProxy::GetRepresentation()
{
  this->CreateVTKObjects();
  return vtkChartRepresentation::SafeDownCast(this->GetClientSideObject());
}

//----------------------------------------------------------------------------
void vtkSMChartRepresentationProxy::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//----------------------------------------------------------------------------
int vtkSMChartRepresentationProxy::ReadXMLAttributes(
  vtkSMSessionProxyManager* pm, vtkPVXMLElement* element)
{
  if (!this->Superclass::ReadXMLAttributes(pm, element))
  {
    return 0;
  }

  vtkSMProxy* optionsProxy = this->GetSubProxy("PlotOptions");
  if (optionsProxy)
  {
    const char* names[] = { "Input", "CompositeDataSetIndex", "AttributeType", nullptr };
    for (int cc = 0; names[cc] != nullptr; cc++)
    {
      vtkSMProperty* src = this->GetProperty(names[cc]);
      vtkSMProperty* dest = optionsProxy->GetProperty(names[cc]);
      if (src && dest)
      {
        this->LinkProperty(src, dest);
      }
    }
  }
  return 1;
}

//----------------------------------------------------------------------------
void vtkSMChartRepresentationProxy::SetPropertyModifiedFlag(const char* name, int flag)
{
  if (name && strcmp(name, "Input") == 0)
  {
    vtkSMPropertyHelper helper(this, name);
    for (unsigned int cc = 0; cc < helper.GetNumberOfElements(); cc++)
    {
      vtkSMSourceProxy* input = vtkSMSourceProxy::SafeDownCast(helper.GetAsProxy(cc));
      if (input)
      {
        input->CreateSelectionProxies();
        vtkSMSourceProxy* esProxy = input->GetSelectionOutput(helper.GetOutputPort(cc));
        if (!esProxy)
        {
          vtkErrorMacro("Input proxy does not support selection extraction.");
        }
        else
        {
          vtkSMProxy* selectionReprProxy = this->GetSubProxy("SelectionRepresentation");
          if (selectionReprProxy)
          {
            // We use these internal properties since we need to add consumer dependency
            // on this proxy so that MarkModified() is called correctly.

            // Based on the name of the Property, we either pass the id-based
            // selection generated by vtkPVExtractSelection or the original
            // input selection to the selection representation.
            vtkSMPropertyHelper(selectionReprProxy, "SelectionInput", /*quiet*/ true)
              .Set(esProxy, vtkPVExtractSelection::OUTPUT_PORT_SELECTION_IDS);

            vtkSMPropertyHelper(selectionReprProxy, "OriginalSelectionInput", /*quiet*/ true)
              .Set(esProxy, vtkPVExtractSelection::OUTPUT_PORT_SELECTION_ORIGINAL);
            selectionReprProxy->UpdateVTKObjects();
          }
        }
      }
    }
  }
  this->Superclass::SetPropertyModifiedFlag(name, flag);
}

//----------------------------------------------------------------------------
std::string vtkSMChartRepresentationProxy::GetDecoratedArrayName(const std::string& arrayname)
{
  std::ostringstream ss;
  ss << arrayname;

  if (vtkSMPropertyHelper(this, "UseSeparateColorMap", true).GetAsInt())
  {
    // Use global id for separate color map
    std::ostringstream ss1;
    ss1 << "Separate_" << this->GetGlobalIDAsString() << "_" << ss.str();
    return ss1.str();
  }
  return ss.str();
}

//----------------------------------------------------------------------------
vtkPVArrayInformation* vtkSMChartRepresentationProxy::GetArrayInformationForColorArray(
  bool checkRepresentedData)
{
  // now, determine a name for it if possible.
  vtkSMPropertyHelper inputHelper(this, "Input");
  vtkSMSourceProxy* input = vtkSMSourceProxy::SafeDownCast(inputHelper.GetAsProxy());
  vtkSMPropertyHelper colorArrayHelper(this, "ColorArrayName");
  unsigned int port = inputHelper.GetOutputPort();
  if (input)
  {
    vtkPVArrayInformation* arrayInfoFromData = nullptr;
    arrayInfoFromData = input->GetDataInformation(port)->GetArrayInformation(
      colorArrayHelper.GetInputArrayNameToProcess(), colorArrayHelper.GetInputArrayAssociation());
    if (arrayInfoFromData)
    {
      return arrayInfoFromData;
    }

    if (colorArrayHelper.GetInputArrayAssociation() == vtkDataObject::POINT_THEN_CELL)
    {
      // Try points...
      arrayInfoFromData = input->GetDataInformation(port)->GetArrayInformation(
        colorArrayHelper.GetInputArrayNameToProcess(), vtkDataObject::POINT);
      if (arrayInfoFromData)
      {
        return arrayInfoFromData;
      }

      // ... then cells
      arrayInfoFromData = input->GetDataInformation(port)->GetArrayInformation(
        colorArrayHelper.GetInputArrayNameToProcess(), vtkDataObject::CELL);
      if (arrayInfoFromData)
      {
        return arrayInfoFromData;
      }
    }
  }

  if (checkRepresentedData)
  {
    vtkPVArrayInformation* arrayInfo = this->GetRepresentedDataInformation()->GetArrayInformation(
      colorArrayHelper.GetInputArrayNameToProcess(), colorArrayHelper.GetInputArrayAssociation());
    if (arrayInfo)
    {
      return arrayInfo;
    }
  }

  return nullptr;
}

//----------------------------------------------------------------------------
bool vtkSMChartRepresentationProxy::SetScalarColoring(
  const char* arrayname, int attribute_type, int component)
{
  return this->SetScalarColoringInternal(arrayname, attribute_type, true, component);
}

//----------------------------------------------------------------------------
bool vtkSMChartRepresentationProxy::SetScalarColoringInternal(
  const char* arrayname, int attribute_type, bool useComponent, int component)
{
  if ((arrayname == nullptr || arrayname[0] == 0))
  {
    // scalar coloring already off. Nothing to do.
    return true;
  }

  vtkSMProperty* colorArray = this->GetProperty("ColorArrayName");
  if (!colorArray)
  {
    vtkWarningMacro("No 'ColorArrayName' property found.");
    return false;
  }

  vtkSMPropertyHelper colorArrayHelper(colorArray);
  colorArrayHelper.SetInputArrayToProcess(attribute_type, arrayname);

  if (arrayname == nullptr || arrayname[0] == '\0')
  {
    vtkSMPropertyHelper(this, "LookupTable", true).RemoveAllValues();
    vtkSMPropertyHelper(this, "ScalarOpacityFunction", true).RemoveAllValues();
    this->UpdateVTKObjects();
    return true;
  }

  // Now, setup transfer functions.
  bool haveComponent = useComponent;
  bool separate = (vtkSMPropertyHelper(this, "UseSeparateColorMap", true).GetAsInt() != 0);
  std::string decoratedArrayName = this->GetDecoratedArrayName(arrayname);
  vtkNew<vtkSMTransferFunctionManager> mgr;
  vtkSMProxy* lutProxy = nullptr;
  if (vtkSMProperty* lutProperty = this->GetProperty("LookupTable"))
  {
    lutProxy =
      mgr->GetColorTransferFunction(decoratedArrayName.c_str(), this->GetSessionProxyManager());
    if (useComponent)
    {
      if (component >= 0)
      {
        vtkSMPropertyHelper(lutProxy, "VectorMode").Set("Component");
        vtkSMPropertyHelper(lutProxy, "VectorComponent").Set(component);
        lutProxy->UpdateVTKObjects();
      }
      else
      {
        vtkSMPropertyHelper(lutProxy, "VectorMode").Set("Magnitude");
        lutProxy->UpdateVTKObjects();
      }
    }
    else
    {
      // No Component defined for coloring, in order to generate a valid trace
      // a component is needed, recover currently used component
      const char* vectorMode = vtkSMPropertyHelper(lutProxy, "VectorMode").GetAsString();
      haveComponent = true;
      if (strcmp(vectorMode, "Component") == 0)
      {
        component = vtkSMPropertyHelper(lutProxy, "VectorComponent").GetAsInt();
      }
      else // Magnitude
      {
        component = -1;
      }
    }

    vtkSMPropertyHelper(lutProperty).Set(lutProxy);

    // Get the array information for the color array to determine transfer function properties
    vtkPVArrayInformation* colorArrayInfo = this->GetArrayInformationForColorArray();
    if (colorArrayInfo)
    {
      if (colorArrayInfo->GetDataType() == VTK_STRING)
      {
        vtkSMPropertyHelper(lutProxy, "IndexedLookup", true).Set(1);
        lutProxy->UpdateVTKObjects();
      }
      if (haveComponent)
      {
        const char* componentName = colorArrayInfo->GetComponentName(component);
        if (strcmp(componentName, "") == 0)
        {
          haveComponent = false;
        }
      }
    }
  }

  if (vtkSMProperty* sofProperty = this->GetProperty("ScalarOpacityFunction"))
  {
    vtkSMProxy* sofProxy =
      mgr->GetOpacityTransferFunction(decoratedArrayName.c_str(), this->GetSessionProxyManager());
    vtkSMPropertyHelper(sofProperty).Set(sofProxy);
  }

  this->UpdateVTKObjects();
  return true;
}
