/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkSynchronizedTemplates3D.h
  Language:  C++
  Date:      $Date$
  Version:   $Revision$



Copyright (c) 1993-2001 Ken Martin, Will Schroeder, Bill Lorensen 
All rights reserved.

    THIS CLASS IS PATENT PENDING.

    Application of this software for commercial purposes requires 
    a license grant from Kitware. Contact:
        Ken Martin
        Kitware
        469 Clifton Corporate Parkway,
        Clifton Park, NY 12065
        Phone:1-518-371-3971 
    for more information.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Ken Martin, Will Schroeder, or Bill Lorensen nor the names
   of any contributors may be used to endorse or promote products derived
   from this software without specific prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/
// .NAME vtkSynchronizedTemplates3D - generate isosurface from structured points

// .SECTION Description
// vtkSynchronizedTemplates3D is a 3D implementation of the synchronized 
// template algorithm. Note that vtkContourFilter will automatically
// use this class when appropriate if vtk was built with patents.

// .SECTION Caveats
// This filter is specialized to 3D images (aka volumes).

// .SECTION See Also
// vtkContourFilter vtkSynchronizedTemplates2D

#ifndef __vtkSynchronizedTemplates3D_h
#define __vtkSynchronizedTemplates3D_h

#include "vtkPolyDataSource.h"
#include "vtkImageData.h"
#include "vtkContourValues.h"
#include "vtkMultiThreader.h"
#include "vtkKitwareContourFilter.h"

class VTK_EXPORT vtkSynchronizedTemplates3D : public vtkPolyDataSource
{
public:
  static vtkSynchronizedTemplates3D *New();

  vtkTypeMacro(vtkSynchronizedTemplates3D,vtkPolyDataSource);
  void PrintSelf(ostream& os, vtkIndent indent);
  
  // Description:
  // Set/Get the source for the scalar data to contour.
  void SetInput(vtkImageData *input);
  vtkImageData *GetInput();
  
  // Description:
  // Because we delegate to vtkContourValues
  unsigned long int GetMTime();

  // Description:
  // Set/Get the computation of normals. Normal computation is fairly
  // expensive in both time and storage. If the output data will be
  // processed by filters that modify topology or geometry, it may be
  // wise to turn Normals and Gradients off.
  vtkSetMacro(ComputeNormals,int);
  vtkGetMacro(ComputeNormals,int);
  vtkBooleanMacro(ComputeNormals,int);

  // Description:
  // Set/Get the computation of gradients. Gradient computation is
  // fairly expensive in both time and storage. Note that if
  // ComputeNormals is on, gradients will have to be calculated, but
  // will not be stored in the output dataset.  If the output data
  // will be processed by filters that modify topology or geometry, it
  // may be wise to turn Normals and Gradients off.
  vtkSetMacro(ComputeGradients,int);
  vtkGetMacro(ComputeGradients,int);
  vtkBooleanMacro(ComputeGradients,int);

  // Description:
  // Set/Get the computation of scalars.
  vtkSetMacro(ComputeScalars,int);
  vtkGetMacro(ComputeScalars,int);
  vtkBooleanMacro(ComputeScalars,int);

  // Description:
  // Set a particular contour value at contour number i. The index i ranges 
  // between 0<=i<NumberOfContours.
  void SetValue(int i, float value) {this->ContourValues->SetValue(i,value);}

  // Description:
  // Get the ith contour value.
  float GetValue(int i) {return this->ContourValues->GetValue(i);}

  // Description:
  // Get a pointer to an array of contour values. There will be
  // GetNumberOfContours() values in the list.
  float *GetValues() {return this->ContourValues->GetValues();}

  // Description:
  // Fill a supplied list with contour values. There will be
  // GetNumberOfContours() values in the list. Make sure you allocate
  // enough memory to hold the list.
  void GetValues(float *contourValues) {
    this->ContourValues->GetValues(contourValues);}

  // Description:
  // Set the number of contours to place into the list. You only really
  // need to use this method to reduce list size. The method SetValue()
  // will automatically increase list size as needed.
  void SetNumberOfContours(int number) {
    this->ContourValues->SetNumberOfContours(number);}

  // Description:
  // Get the number of contours in the list of contour values.
  int GetNumberOfContours() {
    return this->ContourValues->GetNumberOfContours();}

  // Description:
  // Generate numContours equally spaced contour values between specified
  // range. Contour values will include min/max range values.
  void GenerateValues(int numContours, float range[2]) {
    this->ContourValues->GenerateValues(numContours, range);}

  // Description:
  // Generate numContours equally spaced contour values between specified
  // range. Contour values will include min/max range values.
  void GenerateValues(int numContours, float rangeStart, float rangeEnd)
    {this->ContourValues->GenerateValues(numContours, rangeStart, rangeEnd);}

  // Description:
  // Needed by templated functions.
  int *GetExecuteExtent() {return this->ExecuteExtent;}
  void ThreadedExecute(vtkImageData *data, int *exExt, int threadId);

  // Description:
  // Get/Set the number of threads to create when rendering
  vtkSetClampMacro( NumberOfThreads, int, 1, VTK_MAX_THREADS );
  vtkGetMacro( NumberOfThreads, int );

  // Description:
  // Determines the chunk size fro streaming.  This filter will act like a
  // collector: ask for many input pieces, but generate one output.  Limit is
  // in KBytes
  void SetInputMemoryLimit(unsigned long limit);
  unsigned long GetInputMemoryLimit();  

protected:
  vtkSynchronizedTemplates3D();
  ~vtkSynchronizedTemplates3D();
  vtkSynchronizedTemplates3D(const vtkSynchronizedTemplates3D&) {};
  void operator=(const vtkSynchronizedTemplates3D&) {};

  int ComputeNormals;
  int ComputeGradients;
  int ComputeScalars;
  vtkContourValues *ContourValues;

  void Execute();
  void ExecuteInformation();

  void ComputeInputUpdateExtents(vtkDataObject *output);
  
  int ExecuteExtent[6];

  int NumberOfThreads;
  vtkMultiThreader *Threader;
  // temporary outputs
  vtkPolyData *Threads[VTK_MAX_THREADS];
  void InitializeOutput(int *ext,vtkPolyData *o);

private:
  //BTX
  friend VTK_EXPORT vtkKitwareContourFilter;
  //ETX
  
};







// template table.
//BTX

static int VTK_SYNCHONIZED_TEMPLATES_3D_TABLE_1[] = {
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,  592, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1312,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  585,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,  260,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,  948,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,  935,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,  250,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,  620,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,   16,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,  530,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1263,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,  988,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  288,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,  201,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,  874,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  106,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,  746,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1119,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
 404,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,  414,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1126,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,  736,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,   99,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,  786,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
 134,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  355,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1064, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1172,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,  448,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,   62,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,  687,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1211,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
 484,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
  44,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,  660, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,  822,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  161,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,  328,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1028,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,  441,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1162,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,  700,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,   72,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
 124,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,  773, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1080, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,  368,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1018, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,  315,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  174,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,  838,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,  647,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,   34,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,  494,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1218,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,  278,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,  975,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,  890, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,  214,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   9,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,  610, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1276,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  540,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1011,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,  305,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,  181,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,  848,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,  637,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,   27,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,  504,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1231,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  271,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,  965, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,  903,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  224,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    5,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,  603,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1292, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,  553,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1201, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,  471,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,   48,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,  667,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,  809,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
 151,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  335,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1038, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,  431,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1149,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,  710,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,   79,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,  117,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,  763,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1093,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,  378,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  110,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,  753,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1109,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
 391,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,  421,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1136,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,  723,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,   89,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,  796,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
 141,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,  345,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1051, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1185,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,  458,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,   55,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,  677,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    1,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,  596,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1299,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,  569,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,  264,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,  955,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,  919, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,  237,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,  627,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
  20,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,  517,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1247,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,  998,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,  295,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,  191,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,  861,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 2036,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1316,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2040,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0, 2404,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1641, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1648,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0, 2408,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1993, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2696,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0, 2080,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1344, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1592,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2361,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2460,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1688,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2594,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1837, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1470, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0, 2218,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2211,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1460,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1850,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0, 2598,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1776,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0, 2557,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2270,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1510,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1411,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2168,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2632,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1896,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1384, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0, 2132,    0,    0,    0, 
   0,    0,    0,    0,    0,    0, 2662,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1941,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1740,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2512,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0, 2315,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1546, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2184,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1424,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1886,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0, 2625,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0, 2564,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1792,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1497,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2254,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1562,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2322,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2499,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1724, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1954,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2666,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2116,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1371,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2374,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1602,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1675,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2444,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0, 2700,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 2000,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1334,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2067,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1572,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0, 2335,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2489,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1711,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1967, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2676,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0, 2103,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1361, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2384,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1615,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1665,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2431,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2707,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 2016,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1327,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2057,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1391,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2142,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2655,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1925,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1750,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0, 2525,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2299,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1533,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2191,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1434,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1873,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2615,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2574,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1805,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1487,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2241,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2587,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1821, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 1477, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0, 2228,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2201,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1447,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1860,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0, 2605,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1763,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0, 2541,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2283,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1520,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1401,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2155,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2642,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1909,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
2711,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0, 2029, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1320,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0, 2047,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2391,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1625,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0, 1655,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2418,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1977,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0, 2683,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0, 2090,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0, 1351,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0, 1579,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0, 2345,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0,    0, 2473,    0,    0,    0,    0,    0,    0,    0, 
   0,    0,    0,    0,    0,    0,    0,    0,    0,    0, 
   0, 1698 };

static int VTK_SYNCHONIZED_TEMPLATES_3D_TABLE_2[] = {
 -1,   0,   1,   2,  -1,   0,   4,   3,  -1,   3,   1,   2, 
  4,   3,   2,  -1,   3,   7,   5,  -1,   0,   1,   2,   3, 
  7,   5,  -1,   4,   7,   5,   0,   4,   5,  -1,   5,   1, 
  2,   5,   2,   7,   7,   2,   4,  -1,   1,   5,   6,  -1, 
  0,   5,   6,   2,   0,   6,  -1,   3,   0,   4,   5,   6, 
  1,  -1,   3,   5,   6,   3,   6,   4,   4,   6,   2,  -1, 
  1,   3,   7,   6,   1,   7,  -1,   0,   3,   7,   0,   7, 
  2,   2,   7,   6,  -1,   1,   0,   4,   1,   4,   6,   6, 
  4,   7,  -1,   4,   7,   2,   7,   6,   2,  -1,   8,   2, 
  9,  -1,   8,   0,   1,   9,   8,   1,  -1,   0,   4,   3, 
  2,   9,   8,  -1,   8,   4,   3,   8,   3,   9,   9,   3, 
  1,  -1,   3,   7,   5,   2,   9,   8,  -1,   1,   9,   8, 
  1,   8,   0,   3,   7,   5,  -1,   4,   7,   5,   4,   5, 
  0,   2,   9,   8,  -1,   5,   4,   7,   5,   9,   4,   5, 
  1,   9,   9,   8,   4,  -1,   2,   9,   8,   1,   5,   6, 
 -1,   6,   9,   8,   6,   8,   5,   5,   8,   0,  -1,   4, 
  3,   0,   2,   9,   8,   5,   6,   1,  -1,   8,   6,   9, 
  4,   6,   8,   4,   5,   6,   4,   3,   5,  -1,   1,   3, 
  7,   1,   7,   6,   9,   8,   2,  -1,   3,   7,   6,   3, 
  6,   8,   3,   8,   0,   9,   8,   6,  -1,   8,   2,   9, 
  4,   6,   0,   4,   7,   6,   6,   1,   0,  -1,   8,   6, 
  9,   8,   4,   6,   4,   7,   6,  -1,   4,   8,  10,  -1, 
  4,   8,  10,   0,   1,   2,  -1,   0,   8,  10,   3,   0, 
 10,  -1,   2,   8,  10,   2,  10,   1,   1,  10,   3,  -1, 
  3,   7,   5,   4,   8,  10,  -1,   1,   2,   0,   3,   7, 
  5,   8,  10,   4,  -1,  10,   7,   5,  10,   5,   8,   8, 
  5,   0,  -1,   5,  10,   7,   1,  10,   5,   1,   8,  10, 
  1,   2,   8,  -1,   4,   8,  10,   5,   6,   1,  -1,   0, 
  5,   6,   0,   6,   2,   8,  10,   4,  -1,   0,   8,  10, 
  0,  10,   3,   5,   6,   1,  -1,   5,  10,   3,   5,   2, 
 10,   5,   6,   2,   8,  10,   2,  -1,   7,   6,   1,   7, 
  1,   3,   4,   8,  10,  -1,   8,  10,   4,   0,   3,   2, 
  2,   3,   7,   2,   7,   6,  -1,  10,   0,   8,  10,   6, 
  0,  10,   7,   6,   6,   1,   0,  -1,  10,   2,   8,  10, 
  7,   2,   7,   6,   2,  -1,   4,   2,   9,  10,   4,   9, 
 -1,   4,   0,   1,   4,   1,  10,  10,   1,   9,  -1,   0, 
  2,   9,   0,   9,   3,   3,   9,  10,  -1,   3,   1,  10, 
  1,   9,  10,  -1,   4,   2,   9,   4,   9,  10,   7,   5, 
  3,  -1,   7,   5,   3,   4,   0,  10,  10,   0,   1,  10, 
  1,   9,  -1,   2,   5,   0,   2,  10,   5,   2,   9,  10, 
  7,   5,  10,  -1,   5,  10,   7,   5,   1,  10,   1,   9, 
 10,  -1,   9,  10,   4,   9,   4,   2,   1,   5,   6,  -1, 
  4,   9,  10,   4,   5,   9,   4,   0,   5,   5,   6,   9, 
 -1,   5,   6,   1,   0,   2,   3,   3,   2,   9,   3,   9, 
 10,  -1,   6,   3,   5,   6,   9,   3,   9,  10,   3,  -1, 
  4,   2,  10,   2,   9,  10,   7,   1,   3,   7,   6,   1, 
 -1,  10,   0,   9,  10,   4,   0,   9,   0,   6,   3,   7, 
  0,   6,   0,   7,  -1,   6,   0,   7,   6,   1,   0,   7, 
  0,  10,   2,   9,   0,  10,   0,   9,  -1,   6,  10,   7, 
  9,  10,   6,  -1,   7,  10,  11,  -1,   0,   1,   2,  10, 
 11,   7,  -1,   4,   3,   0,  10,  11,   7,  -1,   3,   1, 
  2,   3,   2,   4,  10,  11,   7,  -1,   3,  10,  11,   5, 
  3,  11,  -1,   3,  10,  11,   3,  11,   5,   1,   2,   0, 
 -1,   4,  10,  11,   4,  11,   0,   0,  11,   5,  -1,  10, 
  2,   4,  10,   5,   2,  10,  11,   5,   1,   2,   5,  -1, 
  5,   6,   1,   7,  10,  11,  -1,   6,   2,   0,   6,   0, 
  5,   7,  10,  11,  -1,   0,   4,   3,   5,   6,   1,  10, 
 11,   7,  -1,  10,  11,   7,   3,   5,   4,   4,   5,   6, 
  4,   6,   2,  -1,  11,   6,   1,  11,   1,  10,  10,   1, 
  3,  -1,   0,   6,   2,   0,  10,   6,   0,   3,  10,  10, 
 11,   6,  -1,   1,  11,   6,   0,  11,   1,   0,  10,  11, 
  0,   4,  10,  -1,  11,   4,  10,  11,   6,   4,   6,   2, 
  4,  -1,  10,  11,   7,   8,   2,   9,  -1,   8,   0,   1, 
  8,   1,   9,  11,   7,  10,  -1,   3,   0,   4,  10,  11, 
  7,   2,   9,   8,  -1,   7,  10,  11,   3,   9,   4,   3, 
  1,   9,   9,   8,   4,  -1,  11,   5,   3,  11,   3,  10, 
  8,   2,   9,  -1,   3,  10,   5,  10,  11,   5,   1,   8, 
  0,   1,   9,   8,  -1,   2,   9,   8,   4,  10,   0,   0, 
 10,  11,   0,  11,   5,  -1,   9,   4,   1,   9,   8,   4, 
  1,   4,   5,  10,  11,   4,   5,   4,  11,  -1,   1,   5, 
  6,   9,   8,   2,   7,  10,  11,  -1,  10,  11,   7,   8, 
  5,   9,   8,   0,   5,   5,   6,   9,  -1,   0,   4,   3, 
  8,   2,   9,   5,   6,   1,  10,  11,   7,  -1,   4,   3, 
  5,   4,   5,   6,   4,   6,   8,   9,   8,   6,  10,  11, 
  7,  -1,   2,   9,   8,   1,  10,   6,   1,   3,  10,  10, 
 11,   6,  -1,  10,   6,   3,  10,  11,   6,   3,   6,   0, 
  9,   8,   6,   0,   6,   8,  -1,   0,   4,  10,   0,  10, 
 11,   0,  11,   1,   6,   1,  11,   2,   9,   8,  -1,  11, 
  4,  10,  11,   6,   4,   8,   4,   9,   9,   4,   6,  -1, 
  7,   4,   8,  11,   7,   8,  -1,   8,  11,   7,   8,   7, 
  4,   0,   1,   2,  -1,   7,   3,   0,   7,   0,  11,  11, 
  0,   8,  -1,   2,   3,   1,   2,  11,   3,   2,   8,  11, 
 11,   7,   3,  -1,   3,   4,   8,   3,   8,   5,   5,   8, 
 11,  -1,   1,   2,   0,   3,   4,   5,   5,   4,   8,   5, 
  8,  11,  -1,   0,   8,   5,   8,  11,   5,  -1,   2,   5, 
  1,   2,   8,   5,   8,  11,   5,  -1,   7,   4,   8,   7, 
  8,  11,   6,   1,   5,  -1,   0,   5,   2,   5,   6,   2, 
  8,   7,   4,   8,  11,   7,  -1,   1,   5,   6,   0,  11, 
  3,   0,   8,  11,  11,   7,   3,  -1,  11,   3,   8,  11, 
  7,   3,   8,   3,   2,   5,   6,   3,   2,   3,   6,  -1, 
  4,   8,  11,   4,  11,   1,   4,   1,   3,   6,   1,  11, 
 -1,   2,   3,   6,   2,   0,   3,   6,   3,  11,   4,   8, 
  3,  11,   3,   8,  -1,   1,  11,   6,   1,   0,  11,   0, 
  8,  11,  -1,  11,   2,   8,   6,   2,  11,  -1,   9,  11, 
  7,   9,   7,   2,   2,   7,   4,  -1,   0,   1,   9,   0, 
  9,   7,   0,   7,   4,  11,   7,   9,  -1,   7,   9,  11, 
  3,   9,   7,   3,   2,   9,   3,   0,   2,  -1,   7,   9, 
 11,   7,   3,   9,   3,   1,   9,  -1,   3,  11,   5,   3, 
  2,  11,   3,   4,   2,   2,   9,  11,  -1,   5,   4,  11, 
  5,   3,   4,  11,   4,   9,   0,   1,   4,   9,   4,   1, 
 -1,   9,   0,   2,   9,  11,   0,  11,   5,   0,  -1,   9, 
  5,   1,  11,   5,   9,  -1,   5,   6,   1,   7,   2,  11, 
  7,   4,   2,   2,   9,  11,  -1,   5,   9,   0,   5,   6, 
  9,   0,   9,   4,  11,   7,   9,   4,   9,   7,  -1,   3, 
  0,   2,   3,   2,   9,   3,   9,   7,  11,   7,   9,   5, 
  6,   1,  -1,   6,   3,   5,   6,   9,   3,   7,   3,  11, 
 11,   3,   9,  -1,   2,  11,   4,   2,   9,  11,   4,  11, 
  3,   6,   1,  11,   3,  11,   1,  -1,   0,   3,   4,   6, 
  9,  11,  -1,   9,   0,   2,   9,  11,   0,   1,   0,   6, 
  6,   0,  11,  -1,   9,  11,   6,  -1,   9,   6,  11,  -1, 
  1,   2,   0,   6,  11,   9,  -1,   0,   4,   3,   6,  11, 
  9,  -1,   2,   4,   3,   2,   3,   1,   6,  11,   9,  -1, 
  7,   5,   3,  11,   9,   6,  -1,   3,   7,   5,   1,   2, 
  0,  11,   9,   6,  -1,   5,   0,   4,   5,   4,   7,  11, 
  9,   6,  -1,  11,   9,   6,   5,   1,   7,   7,   1,   2, 
  7,   2,   4,  -1,   9,   1,   5,  11,   9,   5,  -1,   9, 
  2,   0,   9,   0,  11,  11,   0,   5,  -1,   5,  11,   9, 
  5,   9,   1,   0,   4,   3,  -1,   3,   5,  11,   3,  11, 
  2,   3,   2,   4,   2,  11,   9,  -1,   7,  11,   9,   7, 
  9,   3,   3,   9,   1,  -1,   7,  11,   9,   3,   7,   9, 
  3,   9,   2,   3,   2,   0,  -1,   0,   9,   1,   0,   7, 
  9,   0,   4,   7,  11,   9,   7,  -1,   9,   7,  11,   9, 
  2,   7,   2,   4,   7,  -1,  11,   8,   2,   6,  11,   2, 
 -1,   1,   6,  11,   1,  11,   0,   0,  11,   8,  -1,   2, 
  6,  11,   2,  11,   8,   4,   3,   0,  -1,   4,  11,   8, 
  4,   1,  11,   4,   3,   1,   6,  11,   1,  -1,  11,   8, 
  2,  11,   2,   6,   5,   3,   7,  -1,   3,   7,   5,   1, 
  6,   0,   0,   6,  11,   0,  11,   8,  -1,   8,   2,   6, 
  8,   6,  11,   0,   4,   5,   5,   4,   7,  -1,   7,   1, 
  4,   7,   5,   1,   4,   1,   8,   6,  11,   1,   8,   1, 
 11,  -1,   2,   1,   5,   2,   5,   8,   8,   5,  11,  -1, 
  0,   5,   8,   8,   5,  11,  -1,   3,   0,   4,   5,   8, 
  1,   5,  11,   8,   8,   2,   1,  -1,   3,   8,   4,   3, 
  5,   8,   5,  11,   8,  -1,   2,   1,   3,   2,   3,  11, 
  2,  11,   8,  11,   3,   7,  -1,   7,   0,   3,   7,  11, 
  0,  11,   8,   0,  -1,   8,   1,  11,   8,   2,   1,  11, 
  1,   7,   0,   4,   1,   7,   1,   4,  -1,   7,   8,   4, 
 11,   8,   7,  -1,   8,  10,   4,   9,   6,  11,  -1,   0, 
  1,   2,   8,  10,   4,   6,  11,   9,  -1,  10,   3,   0, 
 10,   0,   8,   9,   6,  11,  -1,   6,  11,   9,   2,   8, 
  1,   1,   8,  10,   1,  10,   3,  -1,   4,   8,  10,   7, 
  5,   3,   9,   6,  11,  -1,  11,   9,   6,   3,   7,   5, 
  0,   1,   2,   8,  10,   4,  -1,   9,   6,  11,  10,   7, 
  8,   8,   7,   5,   8,   5,   0,  -1,   1,   2,   8,   1, 
  8,  10,   1,  10,   5,   7,   5,  10,   6,  11,   9,  -1, 
  9,   1,   5,   9,   5,  11,  10,   4,   8,  -1,   4,   8, 
 10,   0,  11,   2,   0,   5,  11,  11,   9,   2,  -1,   1, 
  5,  11,   1,  11,   9,   3,   0,  10,  10,   0,   8,  -1, 
 11,   2,   5,  11,   9,   2,   5,   2,   3,   8,  10,   2, 
  3,   2,  10,  -1,   4,   8,  10,   7,  11,   3,   3,  11, 
  9,   3,   9,   1,  -1,   3,   7,  11,   3,  11,   9,   3, 
  9,   0,   2,   0,   9,   4,   8,  10,  -1,   8,   7,   0, 
  8,  10,   7,   0,   7,   1,  11,   9,   7,   1,   7,   9, 
 -1,   9,   7,  11,   9,   2,   7,  10,   7,   8,   8,   7, 
  2,  -1,  11,  10,   4,  11,   4,   6,   6,   4,   2,  -1, 
  1,   6,  11,   0,   1,  11,   0,  11,  10,   0,  10,   4, 
 -1,   0,   2,   6,   0,   6,  10,   0,  10,   3,  10,   6, 
 11,  -1,  11,   1,   6,  11,  10,   1,  10,   3,   1,  -1, 
  3,   7,   5,   4,   6,  10,   4,   2,   6,   6,  11,  10, 
 -1,   0,   1,   6,   0,   6,  11,   0,  11,   4,  10,   4, 
 11,   3,   7,   5,  -1,   6,  10,   2,   6,  11,  10,   2, 
 10,   0,   7,   5,  10,   0,  10,   5,  -1,  11,   1,   6, 
 11,  10,   1,   5,   1,   7,   7,   1,  10,  -1,  10,   4, 
  2,  10,   2,   5,  10,   5,  11,   1,   5,   2,  -1,   4, 
 11,  10,   4,   0,  11,   0,   5,  11,  -1,   3,   2,  10, 
  3,   0,   2,  10,   2,  11,   1,   5,   2,  11,   2,   5, 
 -1,   3,  11,  10,   5,  11,   3,  -1,   3,  11,   1,   3, 
  7,  11,   1,  11,   2,  10,   4,  11,   2,  11,   4,  -1, 
  7,   0,   3,   7,  11,   0,   4,   0,  10,  10,   0,  11, 
 -1,   0,   2,   1,  10,   7,  11,  -1,   7,  11,  10,  -1, 
  6,   7,  10,   9,   6,  10,  -1,   6,   7,  10,   6,  10, 
  9,   2,   0,   1,  -1,  10,   9,   6,  10,   6,   7,   3, 
  0,   4,  -1,   7,  10,   9,   7,   9,   6,   4,   3,   2, 
  2,   3,   1,  -1,   6,   5,   3,   6,   3,   9,   9,   3, 
 10,  -1,   0,   1,   2,   3,   9,   5,   3,  10,   9,   9, 
  6,   5,  -1,   4,  10,   9,   4,   9,   5,   4,   5,   0, 
  5,   9,   6,  -1,   9,   5,  10,   9,   6,   5,  10,   5, 
  4,   1,   2,   5,   4,   5,   2,  -1,   5,   7,  10,   5, 
 10,   1,   1,  10,   9,  -1,   2,   0,   5,   2,   5,  10, 
  2,  10,   9,   7,  10,   5,  -1,   4,   3,   0,  10,   1, 
  7,  10,   9,   1,   1,   5,   7,  -1,   4,   5,   2,   4, 
  3,   5,   2,   5,   9,   7,  10,   5,   9,   5,  10,  -1, 
  3,  10,   1,   1,  10,   9,  -1,   0,   9,   2,   0,   3, 
  9,   3,  10,   9,  -1,   4,   1,   0,   4,  10,   1,  10, 
  9,   1,  -1,   4,   9,   2,  10,   9,   4,  -1,  10,   8, 
  2,  10,   2,   7,   7,   2,   6,  -1,  10,   8,   0,  10, 
  0,   6,  10,   6,   7,   6,   0,   1,  -1,   0,   4,   3, 
  2,   7,   8,   2,   6,   7,   7,  10,   8,  -1,   7,   8, 
  6,   7,  10,   8,   6,   8,   1,   4,   3,   8,   1,   8, 
  3,  -1,   5,   3,  10,   5,  10,   2,   5,   2,   6,   8, 
  2,  10,  -1,   0,   6,   8,   0,   1,   6,   8,   6,  10, 
  5,   3,   6,  10,   6,   3,  -1,   0,  10,   5,   0,   4, 
 10,   5,  10,   6,   8,   2,  10,   6,  10,   2,  -1,   4, 
 10,   8,   5,   1,   6,  -1,   5,   7,  10,   1,   5,  10, 
  1,  10,   8,   1,   8,   2,  -1,  10,   5,   7,  10,   8, 
  5,   8,   0,   5,  -1,   1,   5,   7,   1,   7,  10,   1, 
 10,   2,   8,   2,  10,   0,   4,   3,  -1,  10,   5,   7, 
 10,   8,   5,   3,   5,   4,   4,   5,   8,  -1,   2,  10, 
  8,   2,   1,  10,   1,   3,  10,  -1,   0,  10,   8,   3, 
 10,   0,  -1,   2,  10,   8,   2,   1,  10,   4,  10,   0, 
  0,  10,   1,  -1,   4,  10,   8,  -1,   8,   9,   6,   8, 
  6,   4,   4,   6,   7,  -1,   0,   1,   2,   8,   9,   4, 
  4,   9,   6,   4,   6,   7,  -1,   3,   6,   7,   3,   8, 
  6,   3,   0,   8,   9,   6,   8,  -1,   1,   8,   3,   1, 
  2,   8,   3,   8,   7,   9,   6,   8,   7,   8,   6,  -1, 
  8,   9,   6,   4,   8,   6,   4,   6,   5,   4,   5,   3, 
 -1,   4,   8,   9,   4,   9,   6,   4,   6,   3,   5,   3, 
  6,   0,   1,   2,  -1,   6,   8,   9,   6,   5,   8,   5, 
  0,   8,  -1,   6,   8,   9,   6,   5,   8,   2,   8,   1, 
  1,   8,   5,  -1,   5,   7,   4,   5,   4,   9,   5,   9, 
  1,   9,   4,   8,  -1,   4,   9,   7,   4,   8,   9,   7, 
  9,   5,   2,   0,   9,   5,   9,   0,  -1,   1,   7,   9, 
  1,   5,   7,   9,   7,   8,   3,   0,   7,   8,   7,   0, 
 -1,   3,   5,   7,   2,   8,   9,  -1,   8,   3,   4,   8, 
  9,   3,   9,   1,   3,  -1,   8,   3,   4,   8,   9,   3, 
  0,   3,   2,   2,   3,   9,  -1,   8,   1,   0,   9,   1, 
  8,  -1,   8,   9,   2,  -1,   4,   2,   7,   7,   2,   6, 
 -1,   1,   4,   0,   1,   6,   4,   6,   7,   4,  -1,   0, 
  7,   3,   0,   2,   7,   2,   6,   7,  -1,   1,   7,   3, 
  6,   7,   1,  -1,   3,   6,   5,   3,   4,   6,   4,   2, 
  6,  -1,   1,   4,   0,   1,   6,   4,   3,   4,   5,   5, 
  4,   6,  -1,   0,   6,   5,   2,   6,   0,  -1,   1,   6, 
  5,  -1,   5,   2,   1,   5,   7,   2,   7,   4,   2,  -1, 
  4,   5,   7,   0,   5,   4,  -1,   5,   2,   1,   5,   7, 
  2,   0,   2,   3,   3,   2,   7,  -1,   3,   5,   7,  -1, 
  3,   2,   1,   4,   2,   3,  -1,   0,   3,   4,  -1,   0, 
  2,   1,  -1 };

//ETX

#endif

