# DeviceAdapterAlgorithm::Schedule now accepts and passes arguments

Previously, if you wanted to schedule a functor to pass arguments, the
only way to do it was to put the arguments into the state of the
functor, which could be a pain. Now, in addition to the functor itself
and a scheduling range, Schedule also accepts an arbitrary number of
objects. These objects are passed to the functor when each instance is
executed.

So previously to call a functor with some data you would need to set it up
something like this.

``` cpp
struct MyFunctor
{
  ArgType1 Arg1;
  ArgType2 Arg2;
  
  VTKM_EXEC void operator=(vtkm::Id index) const
  {
    // Do something with this->Arg1 and this->Arg2.
  }
};

...

  MyFunctor functor;
  functor.Arg1 = arg1;
  functor.Arg2 = arg2;
  vtkm::cont::DeviceAdapterAlgorithm::Schedule(functor, numInstances);
```

Now you can have a stateless functor and pass in arguments.

``` cpp
struct MyFunctor
{
  VTKM_EXEC void operator=(vktm::Id index, ArgType1 arg1, ArgType2 arg2) const
  {
    // Do something with arg1 and arg2.
  }
}

...

  vtkm::cont::DeviceAdapterAlgorithm::Schedule(
    MyFunctor{}, numInstances, arg1, arg2);
```

You could even get fancy and use a lambda.

``` cpp
vtkm::cont::DeviceAdapterAlgorithm::Schedule(
  [](ArgType1 arg1, ArgType2 arg2) {
    // Do something with arg1 and arg2.
  },
  numInstances,
  arg1,
  arg2);
```

The `Schedule` method of `vtkm::cont::Algorithm` was changed similarly. An
additional benefit of `Algorithm::Schedule` is that any arguments that
inherit from `ExecutionObjectBase` will automatically have their
`PrepareForExecution` method called so you can schedule tasks that take
execution objects without jumping through hoops making sure the execution
object matches the device.

Previously, there was also a special form of creating "Task" objects
that held the state of a worklet and the Invocation it needed to
execute. Now that Schedule can pass parameters, the Invocation can
easily be passed using command line arguments, so none of that special
code is necessary and has been ripped out.
