//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//  Copyright 2014 National Technology & Engineering Solutions of Sandia, LLC (NTESS).
//  Copyright 2014 UT-Battelle, LLC.
//  Copyright 2014 Los Alamos National Security.
//
//  Under the terms of Contract DE-NA0003525 with NTESS,
//  the U.S. Government retains certain rights in this software.
//
//  Under the terms of Contract DE-AC52-06NA25396 with Los Alamos National
//  Laboratory (LANL), the U.S. Government retains certain rights in
//  this software.
//============================================================================
// OpenGL Graphics includes
//glew needs to go before glut
#include <GL/glew.h>
#include <vtkm/interop/internal/OpenGLHeaders.h>
#if defined(__APPLE__)
#include <GLUT/glut.h>
#else
#include <GL/glut.h>
#endif

#include <vtkm/interop/testing/TestingTransferFancyHandles.h>

#include <vtkm/internal/Configure.h>
#if (defined(VTKM_GCC) || defined(VTKM_CLANG))
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
#endif

#if defined(VTKM_GCC) && defined(VTKM_POSIX) && !defined(__APPLE__)
//
// 1. Some Linux distributions default linker implicitly enables the as-needed
// linking flag. This means that your shared library or executable will only
// link to libraries from which they use symbols. So if you explicitly link to
// pthread but don't use any symbols you won't have a 'DT_NEEDED' entry for
// pthread.
//
// 2. NVidia libGL (driver version 352 ) uses pthread but doesn't have
// a DT_NEEDED entry for the library. When you run ldd or readelf on the library
// you won't detect any reference to the pthread library. Aside this is odd
// since the mesa version does explicitly link to pthread. But if you run the
// following command:
//        "strings  /usr/lib/nvidia-352/libGL.so.1 | grep pthread | less"
// You will see the following:
// { pthread_create
//   pthread_self
//   pthread_equal
//   pthread_key_crea
//   ...
//   libpthread.so.0
//   libpthread.so
//   pthread_create
// }
//
// This is very strong evidence that this library is using pthread.
//
//
// 3. So what does this all mean?
//
// It means that on system that use the linking flag 'as-needed', are using
// the nvidia driver, and don't use pthread will generate binaries that crash
// on launch. The only way to work around this issue is to do either:
//
//
//  A: Specify 'no-as-needed' to the linker potentially causing over-linking
//  and a  slow down in link time.
//
//  B: Use a method from pthread, making the linker realize that pthread is
//  needed. Note we have to actually call the method so that a linker with
//  optimizations enabled doesn't remove the function and pthread requirement.
//
//
// So that is the explanation on why we have the following function which is
// used once, doesn't look to be useful and seems very crazy.
#include <iostream>
#include <pthread.h>
#define VTKM_NVIDIA_PTHREAD_WORKAROUND 1
static int vtkm_force_linking_to_pthread_to_fix_nvidia_libgl_bug()
{
  return static_cast<int>(pthread_self());
}
#endif

int UnitTestFancyTransferGLUT(int argc, char* argv[])
{
  //get glut to construct a context for us
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH);
  glutInitWindowSize(1024, 1024);
  glutCreateWindow("GLUT test");

  //get glew to bind all the opengl functions
  glewInit();

  if (!glewIsSupported("GL_VERSION_2_1"))
  {
    std::cerr << "OpenGL 2.1 not supported!" << std::endl;
    std::cerr << "GL_RENDERER: " << glGetString(GL_RENDERER) << std::endl;
    std::cerr << "GL_VERSION: " << glGetString(GL_VERSION) << std::endl;
    std::cerr << std::endl;
    std::cerr << "Version support:" << std::endl;
    std::cerr << "GL_VERSION_1_5: " << (int)glewIsSupported("GL_VERSION_1_5") << std::endl;
    std::cerr << "GL_VERSION_2_0: " << (int)glewIsSupported("GL_VERSION_2_0") << std::endl;
    std::cerr << "GL_VERSION_2_1: " << (int)glewIsSupported("GL_VERSION_2_1") << std::endl;
    std::cerr << "GL_VERSION_3_0: " << (int)glewIsSupported("GL_VERSION_3_0") << std::endl;
    std::cerr << "GL_VERSION_3_1: " << (int)glewIsSupported("GL_VERSION_3_1") << std::endl;
    std::cerr << "GL_VERSION_3_2: " << (int)glewIsSupported("GL_VERSION_3_2") << std::endl;
    std::cerr << "GL_VERSION_3_3: " << (int)glewIsSupported("GL_VERSION_3_3") << std::endl;
    std::cerr << "GL_VERSION_4_0: " << (int)glewIsSupported("GL_VERSION_4_0") << std::endl;
    std::cerr << "GL_VERSION_4_1: " << (int)glewIsSupported("GL_VERSION_4_1") << std::endl;
    std::cerr << "GL_VERSION_4_2: " << (int)glewIsSupported("GL_VERSION_4_2") << std::endl;
    std::cerr << "GL_VERSION_4_3: " << (int)glewIsSupported("GL_VERSION_4_3") << std::endl;
    std::cerr << "GL_VERSION_4_4: " << (int)glewIsSupported("GL_VERSION_4_4") << std::endl;
    std::cerr << "GL_VERSION_4_5: " << (int)glewIsSupported("GL_VERSION_4_5") << std::endl;
    return 1;
  }

#if defined(VTKM_NVIDIA_PTHREAD_WORKAROUND)
  std::cout << ::vtkm_force_linking_to_pthread_to_fix_nvidia_libgl_bug();
#endif

  return vtkm::interop::testing::TestingTransferFancyHandles::Run(argc, argv);
}
