// SPDX-FileCopyrightText: Copyright (c) Kitware Inc.
// SPDX-License-Identifier: BSD-3-Clause
/**
 * @class   vtkSMViewProxyInteractorHelper
 * @brief   helper class that make it easier to
 * hook vtkRenderWindowInteractor and vtkSMViewProxy.
 *
 * vtkSMViewProxyInteractorHelper is a helper class designed to make it easier
 * to hook up vtkRenderWindowInteractor to call methods on a vtkSMViewProxy (or
 * subclass). It's primarily designed to work with vtkSMRenderViewProxy (and
 * subclasses), but it should work with other types of views too.
 *
 * To use this helper, the view typically creates an instance for itself as register
 * itself (using vtkSMViewProxyInteractorHelper::SetViewProxy) and then calls
 * vtkSMViewProxyInteractorHelper::SetupInteractor(). This method will initialize
 * the interactor (potentially changing some ivars on the interactor to avoid
 * automatic rendering, using vtkRenderWindowInteractor::EnableRenderOff(), etc.)
 * and setup event observer to monitor interaction.
 *
 * vtkSMViewProxyInteractorHelper only using vtkSMViewProxy::StillRender() and
 * vtkSMViewProxy::InteractiveRender() APIs directly. However several properties can
 * be optionally present on the view proxy to dictate this class' behaviour. These
 * are as follows:
 *
 * \li \c NonInteractiveRenderDelay :- when present provides time in seconds to
 * delay the StillRender() call after user interaction has ended i.e.
 * vtkRenderWindowInteractor fires the vtkCommand::EndInteractionEvent. If
 * missing, or less than 0.01, the view will immediately render.
 *
 * \li \c WindowResizeNonInteractiveRenderDelay :- when present provides time in seconds to
 * delay the StillRender() call after the window has been resized, ie. the interactor
 * fires a vtkCommand::WindowResizeEvent. If missing or equals 0, the view will
 * immediately render.
 *
 * \li \c EnableRenderOnInteraction :- when present provides a flag whether the interactor
 * should trigger the render calls (either StillRender or InteractiveRender) as
 * a consequence of interaction. If missing, we treat EnableRender as ON.
 */

#ifndef vtkSMViewProxyInteractorHelper_h
#define vtkSMViewProxyInteractorHelper_h

#include "vtkObject.h"
#include "vtkRemotingViewsModule.h" //needed for exports
#include "vtkWeakPointer.h"         //needed for vtkWeakPointer

class vtkCommand;
class vtkRenderWindowInteractor;
class vtkSMViewProxy;

class VTKREMOTINGVIEWS_EXPORT vtkSMViewProxyInteractorHelper : public vtkObject
{
public:
  static vtkSMViewProxyInteractorHelper* New();
  vtkTypeMacro(vtkSMViewProxyInteractorHelper, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  ///@{
  /**
   * Set the view proxy.
   * This is a weak reference i.e. the view proxy's
   * reference count will be unchanged by this call.
   */
  void SetViewProxy(vtkSMViewProxy* proxy);
  vtkSMViewProxy* GetViewProxy();
  ///@}

  ///@{
  /**
   * Set the interactor to "help" the view with.
   * This is a weak reference i.e. the interactor's
   * reference count will be unchanged by this call.
   */
  void SetupInteractor(vtkRenderWindowInteractor* iren);
  vtkRenderWindowInteractor* GetInteractor();
  void CleanupInteractor() { this->SetupInteractor(nullptr); }
  ///@}

protected:
  vtkSMViewProxyInteractorHelper();
  ~vtkSMViewProxyInteractorHelper() override;

  ///@{
  /**
   * Handle event.
   */
  void Execute(vtkObject* caller, unsigned long event, void* calldata);
  void Render();
  void CleanupTimer();
  void Resize();
  ///@}

  /**
   * Forward to the render view proxy whether or not the window is being currently resized.
   * If the proxy is not a render view proxy, it does nothing.
   */
  void SetResizingWindow(bool resizingWindow);

  vtkCommand* Observer;
  vtkWeakPointer<vtkSMViewProxy> ViewProxy;
  vtkWeakPointer<vtkRenderWindowInteractor> Interactor;
  int DelayedRenderTimerId;
  bool Interacting;
  bool Interacted;

  int EndWindowResizeTimerId = -1;

private:
  vtkSMViewProxyInteractorHelper(const vtkSMViewProxyInteractorHelper&) = delete;
  void operator=(const vtkSMViewProxyInteractorHelper&) = delete;
};

#endif
