import argparse
import time

from animreader import AnimReader
from vtkhdfwriter import HDFWriter
import pathlib


class AnimToVTKHDF:
    """Main conversion class. Read Radioss Data files and append their content to a single VTKHDF file"""

    writer: HDFWriter
    data: list[AnimReader] = []

    def __init__(self, verbose: bool = False, static=True):
        self.verbose = verbose
        self.static = static

    def convert(self, inputf: list[str], outputf: str):
        start = time.time()

        self.writer = HDFWriter(outputf, len(inputf), self.verbose, self.static)
        for file in inputf:
            self.writer.write(self.read(file))
        self.writer.close()

        end = time.time()
        if self.verbose:
            print(f"Conversion took {round(end-start,2)}s.")

    def read(self, file: str):
        return AnimReader(file, self.verbose).read()


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        prog="AnimToVTKHDF",
        description="Convert Radioss Anim files to the VTKHDF format",
    )

    parser.add_argument(
        "-q",
        "--quiet",
        action="store_true",
        help="do not show the conversion operation details",
    )
    parser.add_argument(
        "-n",
        "--nostatic",
        action="store_true",
        help="do not write cells as static. Use when the connectivity does not change between time steps. If not set, only points and field values will be changed at each timestep.",
    )
    parser.add_argument(
        "infiles",
        nargs="*",
        type=pathlib.Path,
        help="input files to convert. All files must be part of a same time series, and have the same fields",
    )
    parser.add_argument("outfile", type=str, help="output file")

    args = parser.parse_args()
    converter = AnimToVTKHDF(verbose=not args.quiet, static=not args.nostatic)

    try:
        converter.convert(inputf=args.infiles, outputf=args.outfile)
    except ValueError as err:
        print(
            f"Could not write {args.infiles} as VTKHDF file {args.outfile}: {err.args}"
        )
