//=========================================================================
//
// Copyright 2018 Kitware, Inc.
// Author: Guilbert Pierre (spguilbert@gmail.com)
// Date: 03-27-2018
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//=========================================================================

// This slam algorithm is inspired by the LOAM algorithm:
// J. Zhang and S. Singh. LOAM: Lidar Odometry and Mapping in Real-time.
// Robotics: Science and Systems Conference (RSS). Berkeley, CA, July 2014.

// The algorithm is composed of three sequential steps:
//
// - Keypoints extraction: this step consists of extracting keypoints over
// the points clouds. To do that, the laser lines / scans are treated independently.
// The laser lines are projected onto the XY plane and are rescaled depending on
// their vertical angle. Then we compute their curvature and create two classes of
// keypoints. The edges keypoints which correspond to points with a high curvature
// and planar points which correspond to points with a low curvature.
//
// - Ego-Motion: this step consists of recovering the motion of the lidar
// sensor between two frames (two sweeps). The motion is modelized by a constant
// velocity and angular velocity between two frames (i.e null acceleration).
// Hence, we can parameterize the motion by a rotation and translation per sweep / frame
// and interpolate the transformation inside a frame using the timestamp of the points.
// Since the points clouds generated by a lidar are sparse we can't design a
// pairwise match between keypoints of two successive frames. Hence, we decided to use
// a closest-point matching between the keypoints of the current frame
// and the geometric features derived from the keypoints of the previous frame.
// The geometric features are lines or planes and are computed using the edges
// and planar keypoints of the previous frame. Once the matching is done, a keypoint
// of the current frame is matched with a plane / line (depending of the
// nature of the keypoint) from the previous frame. Then, we recover R and T by
// minimizing the function f(R, T) = sum(d(point, line)^2) + sum(d(point, plane)^2).
// Which can be writen f(R, T) = sum((R*X+T-P).t*A*(R*X+T-P)) where:
// - X is a keypoint of the current frame
// - P is a point of the corresponding line / plane
// - A = (n*n.t) with n being the normal of the plane
// - A = (I - n*n.t).t * (I - n*n.t) with n being a director vector of the line
// Since the function f(R, T) is a non-linear mean square error function
// we decided to use the Levenberg-Marquardt algorithm to recover its argmin.
//
// - Mapping: This step consists of refining the motion recovered in the Ego-Motion
// step and to add the new frame in the environment map. Thanks to the ego-motion
// recovered at the previous step it is now possible to estimate the new position of
// the sensor in the map. We use this estimation as an initial point (R0, T0) and we
// perform an optimization again using the keypoints of the current frame and the matched
// keypoints of the map (and not only the previous frame this time!). Once the position in the
// map has been refined from the first estimation it is then possible to update the map by
// adding the keypoints of the current frame into the map.
//
// In the following programs : "slam" and "slam.cxx" the lidar
// coordinate system {L} is a 3D coordinate system with its origin at the
// geometric center of the lidar. The world coordinate system {W} is a 3D
// coordinate system which coincides with {L} at the initial position. The
// points will be denoted by the ending letter L or W if they belong to
// the corresponding coordinate system.

// LOCAL
#include "Slam.h"
#include "CeresCostFunctions.h"
#include "PoseGraphOptimization.h"
// STD
#include <sstream>
#include <algorithm>
#include <cmath>
#include <chrono>
// EIGEN
#include <Eigen/Dense>
// CERES
#include <ceres/ceres.h>
// PCL
#include <pcl/common/transforms.h>

#define PRINT_VERBOSE(minVerbosityLevel, stream) if (this->Verbosity >= (minVerbosityLevel)) {std::cout << stream << std::endl;}
#define IF_VERBOSE(minVerbosityLevel, command) if (this->Verbosity >= (minVerbosityLevel)) { command; }

namespace
{
//-----------------------------------------------------------------------------
Eigen::Vector3d GetRPY(const Eigen::Matrix3d& rot)
{
  // Get euler angles according to ZYX convention.
  Eigen::Vector3d rpy;
  rpy.x() = std::atan2(rot(2, 1), rot(2, 2));
  rpy.y() = -std::asin(rot(2, 0));
  rpy.z() = std::atan2(rot(1, 0), rot(0, 0));
  return rpy;
}

//-----------------------------------------------------------------------------
Eigen::Isometry3d ArrayToIsometry(const Eigen::Matrix<double, 6, 1>& data)
{
  // Build translation part.
  Eigen::Translation3d trans(data(3), data(4), data(5));
  // Build rotation part from euler angles (ZYX convention).
  Eigen::Quaterniond rot(Eigen::AngleAxisd(data(2), Eigen::Vector3d::UnitZ()) *
                         Eigen::AngleAxisd(data(1), Eigen::Vector3d::UnitY()) *
                         Eigen::AngleAxisd(data(0), Eigen::Vector3d::UnitX()));
  return trans * rot;
}

//-----------------------------------------------------------------------------
std::array<double, 36> FlipAndConvertCovariance(const Eigen::Matrix<double, 6, 6>& covar)
{
  // Reshape covariance from DoF order (rX, rY, rZ, X, Y, Z) to (X, Y, Z, rX, rY, rZ)
  const double* c = covar.data();
  std::array<double, 36> cov = {c[21], c[22], c[23],   c[18], c[19], c[20],
                                c[27], c[28], c[29],   c[24], c[25], c[26],
                                c[33], c[34], c[35],   c[30], c[31], c[32],

                                c[ 3], c[ 4], c[ 5],   c[ 0], c[ 1], c[ 2],
                                c[ 9], c[10], c[11],   c[ 6], c[ 7], c[ 8],
                                c[15], c[16], c[17],   c[12], c[13], c[14]};
  return cov;
}

//-----------------------------------------------------------------------------
std::unordered_map<std::string, std::chrono::high_resolution_clock::time_point> startTimes;

void InitTime(const std::string& functionName)
{
  startTimes[functionName] = std::chrono::high_resolution_clock::now();
}

void StopTimeAndDisplay(const std::string& functionName)
{
  std::chrono::duration<double, std::milli> chrono_ms = std::chrono::high_resolution_clock::now() - startTimes[functionName];
  std::cout << "  -> " << functionName << " took : " << chrono_ms.count() << " ms" << std::endl;
}

//-----------------------------------------------------------------------------
Eigen::Vector3d Rad2Deg(const Eigen::Vector3d& val)
{
  return val / M_PI * 180;
}
}

//==============================================================================
//   Main SLAM use
//==============================================================================

//-----------------------------------------------------------------------------
Slam::Slam()
{
  this->Reset();
}

//-----------------------------------------------------------------------------
void Slam::Reset()
{
  this->EdgesPointsLocalMap = std::make_shared<RollingGrid>();
  this->PlanarPointsLocalMap = std::make_shared<RollingGrid>();
  this->BlobsPointsLocalMap = std::make_shared<RollingGrid>();
  this->SetVoxelGridResolution(10.);
  this->SetVoxelGridSize(50);
  this->SetVoxelGridLeafSizeEdges(0.45);
  this->SetVoxelGridLeafSizePlanes(0.6);
  this->SetVoxelGridLeafSizeBlobs(0.12);

  this->NbrFrameProcessed = 0;

  // n-DoF parameters
  this->Tworld = Eigen::Isometry3d::Identity();
  this->Trelative = Eigen::Isometry3d::Identity();
  this->MotionParametersEgoMotion = std::make_pair(Eigen::Isometry3d::Identity(), Eigen::Isometry3d::Identity());
  this->MotionParametersMapping = std::make_pair(Eigen::Isometry3d::Identity(), Eigen::Isometry3d::Identity());
}

//-----------------------------------------------------------------------------
Transform Slam::GetWorldTransform()
{
  return this->LogTrajectory.back();
}

//-----------------------------------------------------------------------------
std::array<double, 36> Slam::GetTransformCovariance()
{
  // Reshape covariance from DoF order (rX, rY, rZ, X, Y, Z) to (X, Y, Z, rX, rY, rZ)
  return FlipAndConvertCovariance(this->TworldCovariance);
}

//-----------------------------------------------------------------------------
std::vector<Transform> Slam::GetTrajectory()
{
  std::vector<Transform> slamPoses(this->LogTrajectory.begin(), this->LogTrajectory.end());
  return slamPoses;
}

//-----------------------------------------------------------------------------
std::vector<std::array<double, 36>> Slam::GetCovariances()
{
  std::vector<std::array<double, 36>> slamCovariances(this->LogCovariances.begin(), this->LogCovariances.end());
  return slamCovariances;
}

//-----------------------------------------------------------------------------
std::unordered_map<std::string, double> Slam::GetDebugInformation()
{
  std::unordered_map<std::string, double> map;
  map["EgoMotion: edges used"] = this->EgoMotionEdgesPointsUsed;
  map["EgoMotion: planes used"] = this->EgoMotionPlanesPointsUsed;
  map["Mapping: edges used"] = this->MappingEdgesPointsUsed;
  map["Mapping: planes used"] = this->MappingPlanesPointsUsed;
  map["Mapping: blobs used"] = this->MappingBlobsPointsUsed;
  map["Mapping: variance error"] = this->MappingVarianceError;
  return map;
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetEdgesMap()
{
  return this->EdgesPointsLocalMap->Get();
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetPlanarsMap()
{
  return this->PlanarPointsLocalMap->Get();
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetBlobsMap()
{
  return this->BlobsPointsLocalMap->Get();
}

//-----------------------------------------------------------------------------
void Slam::AddFrame(const PointCloud::Ptr& pc, const std::vector<size_t>& laserIdMapping)
{
  IF_VERBOSE(1, InitTime("SLAM frame processing"));

  if (pc->size() == 0)
  {
    std::cout << "Slam entry is an empty pointcloud" << std::endl;
    return;
  }

  PRINT_VERBOSE(2, std::endl << "#########################################################");
  PRINT_VERBOSE(1, "Processing frame " << this->NbrFrameProcessed);
  PRINT_VERBOSE(2, "#########################################################" << std::endl);

  // Compute the edges and planars keypoints
  IF_VERBOSE(3, InitTime("Keypoints extraction"));
  this->KeyPointsExtractor->ComputeKeyPoints(pc, laserIdMapping);
  this->CurrentEdgesPoints = this->KeyPointsExtractor->GetEdgePoints();
  this->CurrentPlanarsPoints = this->KeyPointsExtractor->GetPlanarPoints();
  this->CurrentBlobsPoints = this->KeyPointsExtractor->GetBlobPoints();
  // Set the max and min keypoints positions to reduce map searching radius and extracted keypoints
  this->SetFrameMinMaxKeypoints(this->KeyPointsExtractor->GetMinPoint(), this->KeyPointsExtractor->GetMaxPoint());
  PRINT_VERBOSE(2, "========== Keypoints extraction ==========" << std::endl <<
                   "Extracted features : " << this->CurrentEdgesPoints->size()   << " edges, "
                                           << this->CurrentPlanarsPoints->size() << " planes, "
                                           << this->CurrentBlobsPoints->size()   << " blobs.");
  IF_VERBOSE(3, StopTimeAndDisplay("Keypoints extraction"));

  // If the new frame is the first one we just add the
  // extracted keypoints into the map without running
  // odometry and mapping steps
  if (this->NbrFrameProcessed > 0)
  {
    // Perfom EgoMotion : compute Trelative from previous and current frame keypoints, and guess current Tworld
    IF_VERBOSE(3, InitTime("Ego-Motion"));
    this->ComputeEgoMotion();
    IF_VERBOSE(3, StopTimeAndDisplay("Ego-Motion"));

    // Transform the current keypoints to the
    // referential of the sensor at the end of
    // frame acquisition
    //IF_VERBOSE(3, InitTime("Undistortion"));
    //this->TransformCurrentKeypointsToEnd();
    //IF_VERBOSE(3, StopTimeAndDisplay("Undistortion"));

    // Perform Mapping : compute Tworld from map and current frame keypoints
    IF_VERBOSE(3, InitTime("Mapping"));
    this->Mapping();
    IF_VERBOSE(3, StopTimeAndDisplay("Mapping"));
  }

  // Update keypoints maps : add current keypoints to map using Tworld
  if (this->UpdateMap)
  {
    IF_VERBOSE(3, InitTime("Maps update"));
    this->UpdateMapsUsingTworld();
    IF_VERBOSE(3, StopTimeAndDisplay("Maps update"));
  }

  // Update the PreviousTworld data
  this->PreviousTworld = this->Tworld;  // CHECK unused ?

  // Current keypoints become previous ones
  this->PreviousEdgesPoints = this->CurrentEdgesPoints;
  this->PreviousPlanarsPoints = this->CurrentPlanarsPoints;
  this->PreviousBlobsPoints = this->CurrentBlobsPoints;  // CHECK unused ?
  this->NbrFrameProcessed++;

  // Log current frame processing results : pose, covariance and keypoints.
  this->LogCurrentFrameState(pc->header.stamp * 1e-6, pc->header.frame_id);

  // Motion and localization parameters estimation information display
  if (this->Verbosity >= 2)
  {
    std::cout << "========== SLAM results ==========" << std::endl;
    Eigen::Vector3d angles, trans;
    angles << Rad2Deg(GetRPY(this->Trelative.linear()));
    trans << this->Trelative.translation();
    std::cout << "Ego-Motion:   angles = [" << angles.transpose() << "] translation = [" << trans.transpose() << "]" << std::endl;
    angles << Rad2Deg(GetRPY(this->Tworld.linear()));
    trans << this->Tworld.translation();
    std::cout << "Localization: angles = [" << angles.transpose() << "] translation = [" << trans.transpose() << "]" << std::endl;
  }

  // Frame processing duration
  IF_VERBOSE(1, StopTimeAndDisplay("SLAM frame processing"));
}

//-----------------------------------------------------------------------------
void Slam::RunPoseGraphOptimization(const std::vector<Transform>& gpsPositions,
                                    const std::vector<std::array<double, 9>>& gpsCovariances,
                                    Eigen::Isometry3d& gpsToSensorOffset,
                                    const std::string& g2oFileName)
{
  IF_VERBOSE(1, InitTime("Pose graph optimization"));
  IF_VERBOSE(3, InitTime("PGO : optimization"));

  // Transform to modifiable vectors
  std::vector<Transform> slamPoses(this->LogTrajectory.begin(), this->LogTrajectory.end());
  std::vector<std::array<double, 36>> slamCovariances(this->LogCovariances.begin(), this->LogCovariances.end());

  if (this->LoggingTimeout == 0.)
  {
    std::cout << "[WARNING] SLAM logging is not enabled : covariances will be "
                 "arbitrarly set and maps will not be optimized during pose "
                 "graph optimization." << std::endl;

    // Set all poses covariances equal to twice the last one if we did not log it
    std::array<double, 36> fakeSlamCovariance = FlipAndConvertCovariance(this->TworldCovariance * 2);
    for (unsigned int i = 0; i < slamPoses.size(); i++)
      slamCovariances.emplace_back(fakeSlamCovariance);
  }

  // Init pose graph optimizer
  PoseGraphOptimization poseGraphOptimization;
  poseGraphOptimization.SetNbIteration(500);
  poseGraphOptimization.SetVerbose(this->Verbosity >= 2);
  poseGraphOptimization.SetSaveG2OFile(!g2oFileName.empty());
  poseGraphOptimization.SetG2OFileName(g2oFileName);
  poseGraphOptimization.SetGpsToSensorCalibration(gpsToSensorOffset);

  // Run pose graph optimization
  // TODO : templatize poseGraphOptimization to accept any STL container and avoid deque <-> vector copies
  std::vector<Transform> optimizedSlamPoses;
  if (!poseGraphOptimization.Process(slamPoses, gpsPositions,
                                     slamCovariances, gpsCovariances,
                                     optimizedSlamPoses))
  {
    std::cout << "[ERROR] Pose graph optimization failed." << std::endl;
    return;
  }

  IF_VERBOSE(3, StopTimeAndDisplay("PGO : optimization"));

  // Update GPS/LiDAR calibration
  gpsToSensorOffset = optimizedSlamPoses.front().GetIsometry();

  // Update SLAM trajectory and maps
  IF_VERBOSE(3, InitTime("PGO : Maps reset"));
  this->ClearMaps();
  IF_VERBOSE(3, StopTimeAndDisplay("PGO : Maps reset"));
  IF_VERBOSE(3, InitTime("PGO : frames keypoints aggregation"));
  PointCloud::Ptr aggregatedEdgesMap(new PointCloud());
  PointCloud::Ptr aggregatedPlanarsMap(new PointCloud());
  PointCloud::Ptr aggregatedBlobsMap(new PointCloud());
  for (unsigned int i = 0; i < optimizedSlamPoses.size(); i++)
  {
    // Update SLAM pose
    this->LogTrajectory[i].GetIsometry() = gpsToSensorOffset.inverse() * optimizedSlamPoses[i].GetIsometry();

    // Transform frame keypoints to world coordinates
    Eigen::Matrix4d currentTransform = this->LogTrajectory[i].GetMatrix();
    pcl::transformPointCloud(*this->LogEdgesPoints[i], *this->CurrentEdgesPoints, currentTransform);
    pcl::transformPointCloud(*this->LogPlanarsPoints[i], *this->CurrentPlanarsPoints, currentTransform);
    if (!this->FastSlam)
      pcl::transformPointCloud(*this->LogBlobsPoints[i], *this->CurrentBlobsPoints, currentTransform);

    // TODO: Deal with undistortion case (properly transform pointclouds before aggreagtion)

    // Aggregate new keypoints to maps
    *aggregatedEdgesMap += *this->CurrentEdgesPoints;
    *aggregatedPlanarsMap += *this->CurrentPlanarsPoints;
    if (!this->FastSlam)
      *aggregatedBlobsMap += *this->CurrentBlobsPoints;
  }

  IF_VERBOSE(3, StopTimeAndDisplay("PGO : frames keypoints aggregation"));
  IF_VERBOSE(3, InitTime("PGO : final SLAM map update"));

  // Set final pose
  Transform& finalPose = this->LogTrajectory.back();
  this->Tworld = finalPose.GetIsometry();

  // TODO : Deal with undistortion case (update motionParameters)

  // Update SLAM maps
  this->EdgesPointsLocalMap->Roll(this->Tworld.translation());
  this->EdgesPointsLocalMap->Add(aggregatedEdgesMap);
  this->PlanarPointsLocalMap->Roll(this->Tworld.translation());
  this->PlanarPointsLocalMap->Add(aggregatedPlanarsMap);
  if (!this->FastSlam)
  {
    this->BlobsPointsLocalMap->Roll(this->Tworld.translation());
    this->BlobsPointsLocalMap->Add(aggregatedBlobsMap);
  }

  // Processing duration
  IF_VERBOSE(3, StopTimeAndDisplay("PGO : final SLAM map update"));
  IF_VERBOSE(1, StopTimeAndDisplay("Pose graph optimization"));
}

//-----------------------------------------------------------------------------
void Slam::SetWorldTransformFromGuess(const Transform& poseGuess)
{
  // Reset previous frame keypoints because Ego-Motion is not valid since we imposed a discontinuity
  this->PreviousEdgesPoints->clear();
  this->PreviousPlanarsPoints->clear();

  // Set current pose
  this->Tworld = poseGuess.GetIsometry();
  // TODO update motionParameters
}

//==============================================================================
//   Main SLAM steps
//==============================================================================

//-----------------------------------------------------------------------------
void Slam::ComputeEgoMotion()
{
  // Initialize the IsKeypointUsed vectors
  this->EdgePointRejectionEgoMotion.clear(); this->EdgePointRejectionEgoMotion.resize(this->CurrentEdgesPoints->size());
  this->PlanarPointRejectionEgoMotion.clear(); this->PlanarPointRejectionEgoMotion.resize(this->CurrentPlanarsPoints->size());
  // Check that there is enough points to compute the EgoMotion
  if ((this->CurrentEdgesPoints->size() == 0 || this->PreviousEdgesPoints->size() == 0) &&
      (this->CurrentPlanarsPoints->size() == 0 || this->PreviousPlanarsPoints->size() == 0))
  {
    this->EgoMotionEdgesPointsUsed = 0;
    this->EgoMotionPlanesPointsUsed = 0;
    std::cout << "Not enough keypoints, EgoMotion skipped for this frame" << std::endl;
    return;
  }

  // reset the relative transform
  // TODO : keep last frame transform as an init for optimization ?
  this->Trelative = Eigen::Isometry3d::Identity();
  this->MotionParametersEgoMotion = std::make_pair(Eigen::Isometry3d::Identity(), Eigen::Isometry3d::Identity());

  // kd-tree to process fast nearest neighbor
  // among the keypoints of the previous pointcloud
  KDTreePCLAdaptor kdtreePreviousEdges(this->PreviousEdgesPoints);
  KDTreePCLAdaptor kdtreePreviousPlanes(this->PreviousPlanarsPoints);

  PRINT_VERBOSE(2, "========== Ego-Motion ==========" << std::endl <<
                   "previous edges: " << this->PreviousEdgesPoints->size() << ", current edges: " << this->CurrentEdgesPoints->size() << std::endl <<
                   "previous planes: " << this->PreviousPlanarsPoints->size() << ", current planes: " << this->CurrentPlanarsPoints->size());

  unsigned int usedEdges = 0;
  unsigned int usedPlanes = 0;
  Point currentPoint;

  unsigned int toReserve =   this->CurrentEdgesPoints->size()
                           + this->CurrentPlanarsPoints->size();
  this->Xvalues.reserve(toReserve);
  this->Avalues.reserve(toReserve);
  this->Pvalues.reserve(toReserve);
  this->TimeValues.reserve(toReserve);
  this->residualCoefficient.reserve(toReserve);

  IF_VERBOSE(3, InitTime("Ego-Motion : whole ICP-LM loop"));

  // ICP - Levenberg-Marquardt loop:
  // At each step of this loop an ICP matching is performed.
  // Once the keypoints are matched, we estimate the 6-DOF
  // parameters by minimizing a non-linear least square cost
  // function using a Levenberg-Marquardt algorithm
  for (unsigned int icpCount = 0; icpCount < this->EgoMotionICPMaxIter; ++icpCount)
  {
    IF_VERBOSE(3, InitTime("Ego-Motion : ICP"));

    // clear all keypoints matching data
    this->ResetDistanceParameters();

    // Init the undistortion interpolator
    if (this->Undistortion)
    {
      this->CreateWithinFrameTrajectory(this->WithinFrameTrajectory, WithinFrameTrajMode::EgoMotionTraj);
    }

    // loop over edges if there is enough previous edge keypoints
    if (this->PreviousEdgesPoints->size() > this->EgoMotionLineDistanceNbrNeighbors)
    {
      for (unsigned int edgeIndex = 0; edgeIndex < this->CurrentEdgesPoints->size(); ++edgeIndex)
      {
        // Find the closest correspondence edge line of the current edge point
        // Compute the parameters of the point - line distance
        // i.e A = (I - n*n.t)^2 with n being the director vector
        // and P a point of the line
        currentPoint = this->CurrentEdgesPoints->points[edgeIndex];
        int rejectionIndex = this->ComputeLineDistanceParameters(kdtreePreviousEdges, this->Trelative, currentPoint, MatchingMode::EgoMotion);
        this->EdgePointRejectionEgoMotion[edgeIndex] = rejectionIndex;
        this->MatchRejectionHistogramLine[rejectionIndex] += 1;
      }
    }

    // loop over planars if there is enough previous planar keypoints
    if (this->PreviousPlanarsPoints->size() > this->EgoMotionPlaneDistanceNbrNeighbors)
    {
      for (unsigned int planarIndex = 0; planarIndex < this->CurrentPlanarsPoints->size(); ++planarIndex)
      {
        // Find the closest correspondence plane of the current planar point
        // Compute the parameters of the point - plane distance
        // i.e A = n * n.t with n being a normal of the plane
        // and is a point of the plane
        currentPoint = this->CurrentPlanarsPoints->points[planarIndex];
        int rejectionIndex = this->ComputePlaneDistanceParameters(kdtreePreviousPlanes, this->Trelative, currentPoint, MatchingMode::EgoMotion);
        this->PlanarPointRejectionEgoMotion[planarIndex] = rejectionIndex;
        this->MatchRejectionHistogramPlane[rejectionIndex] += 1;
      }
    }

    usedEdges = this->MatchRejectionHistogramLine[6];
    usedPlanes = this->MatchRejectionHistogramPlane[6];
    // Skip this frame if there is too few geometric
    // keypoints matched
    if ((usedPlanes + usedEdges) < 20)
    {
      std::cout << "Too few geometric features, frame skipped" << std::endl;
      break;
    }

    IF_VERBOSE(3, StopTimeAndDisplay("Ego-Motion : ICP"));
    IF_VERBOSE(3, InitTime("Ego-Motion : LM optim"));

    double lossScale = this->EgoMotionInitLossScale + static_cast<double>(icpCount) * (this->EgoMotionFinalLossScale - this->EgoMotionInitLossScale) / (1.0 * this->EgoMotionICPMaxIter);

    // Convert to raw data
    // TODO : update Ceres cost function to take as arg the Isometry3d data
    Eigen::Matrix<double, 6, 1> TrelativeArray;
    TrelativeArray << GetRPY(this->Trelative.linear()), this->Trelative.translation();
    Eigen::Matrix<double, 12, 1> motionParametersEgoMotionArray;
    motionParametersEgoMotionArray << GetRPY(this->MotionParametersEgoMotion.first.linear()),
                                      this->MotionParametersEgoMotion.first.translation(),
                                      GetRPY(this->MotionParametersEgoMotion.second.linear()),
                                      this->MotionParametersEgoMotion.second.translation();

    // We want to estimate our 6-DOF parameters using a non
    // linear least square minimization. The non linear part
    // comes from the Euler Angle parametrization of the rotation
    // endomorphism of SO(3). To minimize it, we use CERES to perform
    // the Levenberg-Marquardt algorithm.
    ceres::Problem problem;
    for (unsigned int k = 0; k < Xvalues.size(); ++k)
    {
      if (this->Undistortion)
      {
        ceres::CostFunction* cost_function = new ceres::AutoDiffCostFunction<CostFunctions::MahalanobisDistanceInterpolatedMotionResidual, 1, 12>(
                                             new CostFunctions::MahalanobisDistanceInterpolatedMotionResidual(
                                                this->Avalues[k], this->Pvalues[k], this->Xvalues[k],
                                                this->TimeValues[k], this->residualCoefficient[k]));
        problem.AddResidualBlock(cost_function, new ceres::ScaledLoss(new ceres::ArctanLoss(lossScale), this->residualCoefficient[k],
                                                                      ceres::TAKE_OWNERSHIP), motionParametersEgoMotionArray.data());
      }
      else
      {
        ceres::CostFunction* cost_function = new ceres::AutoDiffCostFunction<CostFunctions::MahalanobisDistanceAffineIsometryResidual, 1, 6>(
                                             new CostFunctions::MahalanobisDistanceAffineIsometryResidual(this->Avalues[k], this->Pvalues[k],
                                                                                                          this->Xvalues[k], this->residualCoefficient[k]));
        problem.AddResidualBlock(cost_function, new ceres::ScaledLoss(new ceres::ArctanLoss(lossScale), this->residualCoefficient[k], ceres::TAKE_OWNERSHIP), TrelativeArray.data());
      }
    }

    ceres::Solver::Options options;
    options.max_num_iterations = this->EgoMotionLMMaxIter;
    options.linear_solver_type = ceres::DENSE_QR;  // TODO : try also DENSE_NORMAL_CHOLESKY or SPARSE_NORMAL_CHOLESKY
    options.minimizer_progress_to_stdout = false;

    ceres::Solver::Summary summary;
    ceres::Solve(options, &problem, &summary);
    PRINT_VERBOSE(4, summary.BriefReport());

    // Unpack Trelative and MotionParametersEgoMotion
    this->Trelative = ArrayToIsometry(TrelativeArray);
    this->MotionParametersEgoMotion.first = ArrayToIsometry(motionParametersEgoMotionArray.topRows(6));
    this->MotionParametersEgoMotion.second = ArrayToIsometry(motionParametersEgoMotionArray.bottomRows(6));

    IF_VERBOSE(3, StopTimeAndDisplay("Ego-Motion : LM optim"));

    // If no L-M iteration has been made since the
    // last ICP matching it means we reached a local
    // minimum for the ICP-LM algorithm
    if (summary.num_successful_steps == 1)
    {
      break;
    }
  }

  IF_VERBOSE(3, StopTimeAndDisplay("Ego-Motion : whole ICP-LM loop"));

  this->EgoMotionEdgesPointsUsed = usedEdges;
  this->EgoMotionPlanesPointsUsed  = usedPlanes;
  PRINT_VERBOSE(2, "Used keypoints : " << this->Xvalues.size() <<
                   " (" << usedEdges << " edges, " << usedPlanes << " planes).");

  // Integrate the relative motion to the world transformation
  this->UpdateTworldUsingTrelative();
}

//-----------------------------------------------------------------------------
void Slam::Mapping()
{
  // Check that there is enough key-points to compute the Mapping
  if (this->CurrentEdgesPoints->size() == 0 && this->CurrentPlanarsPoints->size() == 0)
  {
    this->MappingVarianceError = 10;
    this->MappingEdgesPointsUsed = 0;
    this->MappingPlanesPointsUsed = 0;
    this->MappingBlobsPointsUsed = 0;
    std::cout << "Not enough keypoints, Mapping skipped for this frame" << std::endl;
    return;
  }
  this->EdgePointRejectionMapping.clear(); this->EdgePointRejectionMapping.resize(this->CurrentEdgesPoints->size());
  this->PlanarPointRejectionMapping.clear(); this->PlanarPointRejectionMapping.resize(this->CurrentPlanarsPoints->size());

  // Update motion model parameters
  if (this->Undistortion)
  {
    // CHECK : what about call to UpdateTworldUsingTrelative() ?
    this->MotionParametersMapping.first = this->MotionParametersMapping.second;
  }

  // get keypoints from the map
  IF_VERBOSE(3, InitTime("Mapping : keypoints extraction"));
  PointCloud::Ptr subEdgesPointsLocalMap = this->EdgesPointsLocalMap->Get(this->Tworld.translation());
  PointCloud::Ptr subPlanarPointsLocalMap = this->PlanarPointsLocalMap->Get(this->Tworld.translation());

  // contruct kd-tree for fast closest points search
  KDTreePCLAdaptor kdtreeEdges(subEdgesPointsLocalMap);
  KDTreePCLAdaptor kdtreePlanes(subPlanarPointsLocalMap);
  pcl::KdTreeFLANN<Slam::Point>::Ptr kdtreeBlobs;

  PRINT_VERBOSE(2, "========== Mapping ==========" << std::endl <<
                   "Edges extracted from map: " << subEdgesPointsLocalMap->points.size() << ", "
                   "Planes extracted from map: " << subPlanarPointsLocalMap->points.size());

  if (!this->FastSlam)
  {
    PointCloud::Ptr subBlobPointsLocalMap = this->BlobsPointsLocalMap->Get(this->Tworld.translation());
    kdtreeBlobs.reset(new pcl::KdTreeFLANN<Slam::Point>());
    kdtreeBlobs->setInputCloud(subBlobPointsLocalMap);
    std::cout << "Blobs extracted from map: " << subBlobPointsLocalMap->points.size() << std::endl;
  }

  IF_VERBOSE(3, StopTimeAndDisplay("Mapping : keypoints extraction"));

  // Information about matches
  unsigned int usedEdges = 0;
  unsigned int usedPlanes = 0;
  unsigned int usedBlobs = 0;

  Point currentPoint;

  unsigned int toReserve =   this->CurrentEdgesPoints->size()
                           + this->CurrentPlanarsPoints->size()
                           + this->CurrentBlobsPoints->size();
  this->Xvalues.reserve(toReserve);
  this->Avalues.reserve(toReserve);
  this->Pvalues.reserve(toReserve);
  this->TimeValues.reserve(toReserve);
  this->residualCoefficient.reserve(toReserve);

  IF_VERBOSE(3, InitTime("Mapping : whole ICP-LM loop"));

  // ICP - Levenberg-Marquardt loop:
  // At each step of this loop an ICP matching is performed
  // Once the keypoints matched, we estimate the the 6-DOF
  // parameters by minimizing a non-linear least square cost
  // function using a Levenberg-Marquardt algorithm
  for (unsigned int icpCount = 0; icpCount < this->MappingICPMaxIter; ++icpCount)
  {
    IF_VERBOSE(3, InitTime("Mapping : ICP"));

    // clear all keypoints matching data
    this->ResetDistanceParameters();

    // Init the undistortion interpolator
    if (this->Undistortion)
    {
      this->CreateWithinFrameTrajectory(this->WithinFrameTrajectory, WithinFrameTrajMode::MappingTraj);
    }

    // loop over edges
    if (this->CurrentEdgesPoints->size() > 0 && subEdgesPointsLocalMap->points.size() > 10)
    {
      for (unsigned int edgeIndex = 0; edgeIndex < this->CurrentEdgesPoints->size(); ++edgeIndex)
      {
        // Find the closest correspondence edge line of the current edge point
        currentPoint = this->CurrentEdgesPoints->points[edgeIndex];
        int rejectionIndex = this->ComputeLineDistanceParameters(kdtreeEdges, this->Tworld, currentPoint, MatchingMode::Mapping);
        this->EdgePointRejectionMapping[edgeIndex] = rejectionIndex;
        this->MatchRejectionHistogramLine[rejectionIndex] += 1;
        usedEdges = this->Xvalues.size();
      }
    }
    // loop over surfaces
    if (this->CurrentPlanarsPoints->size() > 0 && subPlanarPointsLocalMap->size() > 10)
    {
      for (unsigned int planarIndex = 0; planarIndex < this->CurrentPlanarsPoints->size(); ++planarIndex)
      {
        // Find the closest correspondence plane of the current planar point
        currentPoint = this->CurrentPlanarsPoints->points[planarIndex];
        int rejectionIndex = this->ComputePlaneDistanceParameters(kdtreePlanes, this->Tworld, currentPoint, MatchingMode::Mapping);
        this->PlanarPointRejectionMapping[planarIndex] = rejectionIndex;
        this->MatchRejectionHistogramPlane[rejectionIndex] += 1;
        usedPlanes = this->Xvalues.size() - usedEdges;
      }
    }

    if (!this->FastSlam && this->NbrFrameProcessed > 10)
    {
      // loop over blobs
      for (unsigned int blobIndex = 0; blobIndex < this->CurrentBlobsPoints->size(); ++blobIndex)
      {
        // Find the closest correspondence plane of the current planar point
        currentPoint = this->CurrentBlobsPoints->points[blobIndex];
        this->ComputeBlobsDistanceParameters(kdtreeBlobs, this->Tworld, currentPoint, MatchingMode::Mapping);
        // CHECK do not update MatchRejectionHistogramBlob ?
        usedBlobs = this->Xvalues.size() - usedPlanes - usedEdges;
      }
    }

    // Skip this frame if there is too few geometric keypoints matched
    if ((usedPlanes + usedEdges + usedBlobs) < 20)
    {
      std::cout << "Too few geometric features, loop breaked "
                << "(" << usedPlanes << " planes, " << usedEdges << " edges, " << usedBlobs << " blobs)." << std::endl;
      break;
    }

    IF_VERBOSE(3, StopTimeAndDisplay("Mapping : ICP"));
    IF_VERBOSE(3, InitTime("Mapping : LM optim"));

    double lossScale = this->MappingInitLossScale + static_cast<double>(icpCount) * (this->MappingFinalLossScale - this->MappingInitLossScale) / (1.0 * this->MappingICPMaxIter);

    // Convert to raw data
    // TODO : update Ceres cost function to take as arg the Isometry3d data
    Eigen::Matrix<double, 6, 1> TworldArray;
    TworldArray << GetRPY(this->Tworld.linear()), this->Tworld.translation();
    Eigen::Matrix<double, 12, 1> motionParametersMappingArray;
    motionParametersMappingArray << GetRPY(this->MotionParametersMapping.first.linear()),
                                    this->MotionParametersMapping.first.translation(),
                                    GetRPY(this->MotionParametersMapping.second.linear()),
                                    this->MotionParametersMapping.second.translation();

    // We want to estimate our 6-DOF parameters using a non
    // linear least square minimization. The non linear part
    // comes from the Euler Angle parametrization of the rotation
    // endomorphism SO(3). To minimize it we use CERES to perform
    // the Levenberg-Marquardt algorithm.
    ceres::Problem problem;
    for (unsigned int k = 0; k < Xvalues.size(); ++k)
    {
      if (this->Undistortion)
      {
        ceres::CostFunction* cost_function = new ceres::AutoDiffCostFunction<CostFunctions::MahalanobisDistanceInterpolatedMotionResidual, 1, 12>(
                                                new CostFunctions::MahalanobisDistanceInterpolatedMotionResidual(
                                                  this->Avalues[k], this->Pvalues[k], this->Xvalues[k], 
                                                  this->TimeValues[k], this->residualCoefficient[k]));
        problem.AddResidualBlock(cost_function, 
                                 new ceres::ScaledLoss(new ceres::ArctanLoss(lossScale),
                                                       this->residualCoefficient[k],
                                                       ceres::TAKE_OWNERSHIP),
                                 motionParametersMappingArray.data());
      }
      else
      {
        ceres::CostFunction* cost_function = new ceres::AutoDiffCostFunction<CostFunctions::MahalanobisDistanceAffineIsometryResidual, 1, 6>(
                                                new CostFunctions::MahalanobisDistanceAffineIsometryResidual(
                                                  this->Avalues[k], this->Pvalues[k], this->Xvalues[k], this->residualCoefficient[k]));
        problem.AddResidualBlock(cost_function,
                                 new ceres::ScaledLoss(new ceres::ArctanLoss(lossScale),
                                                       this->residualCoefficient[k],
                                                       ceres::TAKE_OWNERSHIP),
                                 TworldArray.data());
      }
    }

    ceres::Solver::Options options;
    options.max_num_iterations = this->MappingLMMaxIter;
    options.linear_solver_type = ceres::DENSE_QR;  // TODO test other optimizer
    options.minimizer_progress_to_stdout = false;

    ceres::Solver::Summary summary;
    ceres::Solve(options, &problem, &summary);
    PRINT_VERBOSE(4, summary.BriefReport());

    // Unpack Tworld and MotionParametersMapping
    this->Tworld = ArrayToIsometry(TworldArray);
    this->MotionParametersMapping.first = ArrayToIsometry(motionParametersMappingArray.topRows(6));
    this->MotionParametersMapping.second = ArrayToIsometry(motionParametersMappingArray.bottomRows(6));

    IF_VERBOSE(3, StopTimeAndDisplay("Mapping : LM optim"));

    // If no L-M iteration has been made since the
    // last ICP matching it means we reached a local
    // minimum for the ICP-LM algorithm
    if (((summary.num_successful_steps == 1) ||
        (icpCount == (this->MappingICPMaxIter - 1))) &&
        !this->Undistortion)
    {
      // Now evaluate the quality of the parameters
      // estimated using an approximate computation
      // of the variance covariance matrix
      // Covariance computation options
      ceres::Covariance::Options covOptions;
      covOptions.apply_loss_function = true;
      covOptions.algorithm_type = ceres::CovarianceAlgorithmType::DENSE_SVD;

      // Computation of the variance-covariance matrix
      ceres::Covariance covariance(covOptions);
      std::vector<std::pair<const double*, const double*>> covariance_blocks;
      covariance_blocks.push_back(std::make_pair(TworldArray.data(), TworldArray.data()));
      covariance.Compute(covariance_blocks, &problem);
      double covarianceMat[6 * 6];
      covariance.GetCovarianceBlock(TworldArray.data(), TworldArray.data(), covarianceMat);
      for (int i = 0; i < 6; ++i)
        for (int j = 0; j < 6; ++j)
          this->TworldCovariance(i, j) = covarianceMat[i + 6 * j];
      break;
    }
  }

  IF_VERBOSE(3, StopTimeAndDisplay("Mapping : whole ICP-LM loop"));

  Eigen::SelfAdjointEigenSolver<Eigen::MatrixXd> eig(this->TworldCovariance);
  Eigen::MatrixXd D = eig.eigenvalues();

  this->MappingVarianceError = D(5);
  this->MappingEdgesPointsUsed = usedEdges;
  this->MappingPlanesPointsUsed = usedPlanes;
  this->MappingBlobsPointsUsed = usedBlobs;

  PRINT_VERBOSE(2, "Matches used: " << this->Xvalues.size() << " "
                   "(" << usedEdges << " edges, " << usedPlanes << " planes, " << usedBlobs << " blobs)." << std::endl <<
                   "Covariance eigen values: " << D.transpose() << std::endl <<
                   "Maximum variance eigen vector: " << eig.eigenvectors().col(5).transpose() << std::endl <<
                   "Maximum variance: " << D(5));

  if (this->Undistortion)
  {
    this->Tworld = this->MotionParametersMapping.second;
  }

  // CHECK order of next steps : UpdateMapsUsingTworld, UpdateCurrentKeypointsUsingTworld, CreateWithinFrameTrajectory

  // Transform the current keypoints in the sensor reference frame
  // corresponding to the end of the frame
  if (this->Undistortion)
  {
    this->UpdateCurrentKeypointsUsingTworld();
  }
  // Compute the undistortion interpolator before replacing previousTworld
  // this interpolator will be used to output the mapped current frame
  this->CreateWithinFrameTrajectory(this->WithinFrameTrajectory, WithinFrameTrajMode::MappingTraj);
}

//-----------------------------------------------------------------------------
void Slam::UpdateMapsUsingTworld()
{
  // Init the mapping interpolator
  if (this->Undistortion)
  {
    this->CreateWithinFrameTrajectory(this->WithinFrameTrajectory, WithinFrameTrajMode::MappingTraj);
  }

  // it would be nice to add the point from the frame directly to the map
  auto updateMap = [this] (std::shared_ptr<RollingGrid> map, PointCloud::Ptr frame)
  {
    PointCloud::Ptr temporaryMap(new PointCloud());
    temporaryMap->points.reserve(frame->size());
    for (size_t i = 0; i < frame->size(); ++i)
    {
      temporaryMap->push_back(frame->at(i));
      this->TransformToWorld(temporaryMap->at(i));
    }
    map->Roll(this->Tworld.translation());
    map->Add(temporaryMap);
  };

  updateMap(this->EdgesPointsLocalMap, this->CurrentEdgesPoints);
  updateMap(this->PlanarPointsLocalMap, this->CurrentPlanarsPoints);
  if (!this->FastSlam)
    updateMap(this->BlobsPointsLocalMap, this->CurrentBlobsPoints);
}

//-----------------------------------------------------------------------------
void Slam::UpdateTworldUsingTrelative()
{
  // Next estimation of Tworld using the odometry result.
  // The new pose of the sensor in the world referential is the previous one
  // composed with the relative motion estimated at the odometry step.
  if (this->Undistortion)
  {
    // This estimation will be used to undistort the frame and to initialize the
    // optimization.
    // CHECK : do not use MotionParametersMapping.first ?
    this->MotionParametersMapping.first = this->MotionParametersMapping.second * this->MotionParametersEgoMotion.first;
    this->MotionParametersMapping.second = this->MotionParametersMapping.second * this->MotionParametersEgoMotion.second;
  }
  else
  {
    // This estimation will be used to initialize the optimization.
    this->Tworld = this->Tworld * this->Trelative;
  }
}

//-----------------------------------------------------------------------------
void Slam::UpdateCurrentKeypointsUsingTworld()
{
  // Create the undistortion interpolator

  // Now transform the keypoints
  this->ExpressPointCloudInOtherReferencial(this->CurrentEdgesPoints);
  this->ExpressPointCloudInOtherReferencial(this->CurrentPlanarsPoints);
  this->ExpressPointCloudInOtherReferencial(this->CurrentBlobsPoints);
}

//-----------------------------------------------------------------------------
void Slam::LogCurrentFrameState(double time, const std::string& frameId)
{
  // If logging is enabled
  if (this->LoggingTimeout)
  {
    // Save current frame data to buffer
    this->LogTrajectory.emplace_back(this->Tworld, time, frameId);
    this->LogCovariances.emplace_back(FlipAndConvertCovariance(this->TworldCovariance));
    this->LogEdgesPoints.emplace_back(this->CurrentEdgesPoints);
    this->LogPlanarsPoints.emplace_back(this->CurrentPlanarsPoints);
    if (!this->FastSlam)
      this->LogBlobsPoints.emplace_back(this->CurrentBlobsPoints);

    // If a timeout is defined, forget too old data
    if (this->LoggingTimeout > 0)
    {
      // Forget all previous data older than LoggingTimeout
      while (time - this->LogTrajectory.front().time > this->LoggingTimeout)
      {
        this->LogTrajectory.pop_front();
        this->LogCovariances.pop_front();
        this->LogEdgesPoints.pop_front();
        this->LogPlanarsPoints.pop_front();
        if (!this->FastSlam)
          this->LogBlobsPoints.pop_front();
      }
    }
  }

  // If logging is disabled
  else
  {
    this->LogTrajectory.clear();
    this->LogTrajectory.emplace_back(this->Tworld, time, frameId);
  }
}

//==============================================================================
//   Features associations and optimization
//==============================================================================

//-----------------------------------------------------------------------------
void Slam::ResetDistanceParameters()
{
  this->Xvalues.resize(0);
  this->Avalues.resize(0);
  this->Pvalues.resize(0);
  this->TimeValues.resize(0);
  this->residualCoefficient.resize(0);
  this->MatchRejectionHistogramLine.clear();
  this->MatchRejectionHistogramLine.resize(this->NrejectionCauses, 0);
  this->MatchRejectionHistogramPlane.clear();
  this->MatchRejectionHistogramPlane.resize(this->NrejectionCauses, 0);
  this->MatchRejectionHistogramBlob.clear();
  this->MatchRejectionHistogramBlob.resize(this->NrejectionCauses, 0);
}

//-----------------------------------------------------------------------------
int Slam::ComputeLineDistanceParameters(KDTreePCLAdaptor& kdtreePreviousEdges, const Eigen::Isometry3d& transform,
                                        Point p, MatchingMode matchingMode)
{
  // number of neighbors edge points required to approximate
  // the corresponding egde line
  unsigned int requiredNearest;
  double eigenValuesRatio;
  std::vector<int> nearestIndex;
  std::vector<float> nearestDist;

  // maximum distance between keypoints
  // and their computed line
  double squaredMaxDist;

  // Transform the point using the current pose estimation
  Eigen::Vector3d P0(p.x, p.y, p.z);
  Eigen::Vector3d P, n;
  Eigen::Matrix3d A;
  // Time continious motion model to take
  // into account the rolling shutter distortion
  if (this->Undistortion)
  {
    this->ExpressPointInOtherReferencial(p);
  }
  // Rigid transform
  else
  {
    P = transform * P0;
    p.x = P(0); p.y = P(1); p.z = P(2);
  }

  if (matchingMode == MatchingMode::EgoMotion)
  {
    requiredNearest = this->EgoMotionLineDistanceNbrNeighbors;
    eigenValuesRatio = this->EgoMotionLineDistancefactor;
    squaredMaxDist = std::pow(this->EgoMotionMaxLineDistance, 2);
    GetEgoMotionLineSpecificNeighbor(nearestIndex, nearestDist, requiredNearest, kdtreePreviousEdges, p);
    if (nearestIndex.size() < this->EgoMotionMinimumLineNeighborRejection)
    {
      return 0;
    }
    requiredNearest = nearestIndex.size();
  }
  else if (matchingMode == MatchingMode::Mapping)
  {
    requiredNearest = this->MappingLineDistanceNbrNeighbors;
    eigenValuesRatio = this->MappingLineDistancefactor;
    squaredMaxDist = std::pow(this->MappingMaxLineDistance, 2);
    GetMappingLineSpecificNeigbbor(nearestIndex, nearestDist, this->MappingLineMaxDistInlier, requiredNearest, kdtreePreviousEdges, p);
    if (nearestIndex.size() < this->MappingMinimumLineNeighborRejection)
    {
      return 0;
    }
    requiredNearest = nearestIndex.size();
  }
  else
  {
    throw "ComputeLineDistanceParameters function got invalide step parameter";
  }

  // if the nearest edges are too far from the
  // current edge keypoint we skip this point.
  if (nearestDist[requiredNearest - 1] > this->MaxDistanceForICPMatching)
  {
    return 1;
  }

  // Compute PCA to determine best line approximation
  // of the requiredNearest nearest edges points extracted
  // Thans to the PCA we will check the shape of the neighborhood
  // and keep it if it is distributed along a line
  Eigen::MatrixXd data(requiredNearest, 3);
  for (unsigned int k = 0; k < requiredNearest; k++)
  {
    Point pt = kdtreePreviousEdges.getInputCloud()->points[nearestIndex[k]];
    data.row(k) << pt.x, pt.y, pt.z;
  }

  Eigen::Vector3d mean = data.colwise().mean();
  Eigen::MatrixXd centered = data.rowwise() - mean.transpose();
  Eigen::Matrix3d varianceCovariance = centered.transpose() * centered;
  Eigen::SelfAdjointEigenSolver<Eigen::Matrix3d> eig(varianceCovariance);

  // Eigen values
  Eigen::MatrixXd D = eig.eigenvalues();

  // if the first eigen value is significantly higher than
  // the second one, it means the sourrounding points are
  // distributed on a edge line
  if (D(2) > eigenValuesRatio * D(1))
  {
    // n is the director vector of the line
    n = eig.eigenvectors().col(2);
  }
  else
  {
    return 2;
  }

  // A = (I-n*n.t).t * (I-n*n.t) = (I - n*n.t)^2
  // since (I-n*n.t) is a symmetric matrix
  // Then it comes A (I-n*n.t)^2 = (I-n*n.t) since
  // A is the matrix of a projection endomorphism
  A = (this->I3 - n * n.transpose());

  // it would be the case if P1 = P2 For instance
  // if the sensor has some dual returns that hit the same point
  if (!std::isfinite(A(0, 0)))
  {
    return 3;
  }

  // Evaluate the distance from the fitted line distribution
  // of the neighborhood
  Eigen::Vector3d Xtemp;
  Point pt;
  double meanSquaredDist = 0;
  for (unsigned int k = 0; k < requiredNearest; ++k)
  {
    pt = kdtreePreviousEdges.getInputCloud()->points[nearestIndex[k]];
    Xtemp(0) = pt.x; Xtemp(1) = pt.y; Xtemp(2) = pt.z;
    double squaredDist = (Xtemp - mean).transpose() * A * (Xtemp - mean);
    if (squaredDist > squaredMaxDist)
    {
      return 4;
    }
    meanSquaredDist += squaredDist;
  }
  meanSquaredDist /= static_cast<double>(requiredNearest);
  double fitQualityCoeff = 1.0 - std::sqrt(std::abs(meanSquaredDist) / squaredMaxDist);

  // s represents the quality of the match
  double s = fitQualityCoeff;

  // store the distance parameters values
  this->Avalues.emplace_back(A);
  this->Pvalues.emplace_back(mean);
  this->Xvalues.emplace_back(P0);
  this->TimeValues.emplace_back(p.intensity);  // CHECK intensity ? Not time ?
  this->residualCoefficient.emplace_back(s);
  return 6;
}

//-----------------------------------------------------------------------------
int Slam::ComputePlaneDistanceParameters(KDTreePCLAdaptor& kdtreePreviousPlanes, const Eigen::Isometry3d& transform,
                                         Point p, MatchingMode matchingMode)
{
  // number of neighbors edge points required to approximate
  // the corresponding edge line
  unsigned int requiredNearest;
  double significantlyFactor1, significantlyFactor2;

  // maximum distance between keypoints
  // and their computed plane
  double squaredMaxDist;

  if (matchingMode == MatchingMode::EgoMotion)
  {
    significantlyFactor1 = this->EgoMotionPlaneDistancefactor1;
    significantlyFactor2 = this->EgoMotionPlaneDistancefactor2;
    requiredNearest = this->EgoMotionPlaneDistanceNbrNeighbors;
    squaredMaxDist = std::pow(this->EgoMotionMaxPlaneDistance, 2);
  }
  else if (matchingMode == MatchingMode::Mapping)
  {
    significantlyFactor1 = this->MappingPlaneDistancefactor1;
    significantlyFactor2 = this->MappingPlaneDistancefactor2;
    requiredNearest = this->MappingPlaneDistanceNbrNeighbors;
    squaredMaxDist = std::pow(this->MappingMaxPlaneDistance, 2);
  }
  else
  {
    throw "ComputeLineDistanceParameters function got invalide step parameter";
  }

  Eigen::Vector3d P, n;
  Eigen::Matrix3d A;

  // Transform the point using the current pose estimation
  Eigen::Vector3d P0(p.x, p.y, p.z);

  // Time continious motion model to take
  // into account the rolling shutter distortion
  if (this->Undistortion)
  {
    this->ExpressPointInOtherReferencial(p);
  }
  // Rigid transform
  else
  {
    P = transform * P0;
    p.x = P(0); p.y = P(1); p.z = P(2);
  }

  std::vector<int> nearestIndex(requiredNearest, -1);
  std::vector<double> nearestDist(requiredNearest, -1.0);
  kdtreePreviousPlanes.query(p, requiredNearest, nearestIndex.data(), nearestDist.data());

  // It means that there is not enought keypoints in the neighbohood
  if (nearestIndex[requiredNearest - 1] == -1)
  {
    return 0;
  }

  // if the nearest planars are too far from the
  // current planar keypoint we skip this point.
  if (nearestDist[requiredNearest - 1] > this->MaxDistanceForICPMatching)
  {
    return 1;
  }

  // Compute PCA to determine best line approximation
  // of the requiredNearest nearest edges points extracted
  // Thanks to the PCA we will check the shape of the neighborhood
  // and keep it if it is distributed along a line
  Eigen::MatrixXd data(requiredNearest,3);
  for (unsigned int k = 0; k < requiredNearest; k++)
  {
    Point pt = kdtreePreviousPlanes.getInputCloud()->points[nearestIndex[k]];
    data.row(k) << pt.x, pt.y, pt.z;
  }
  Eigen::Vector3d mean = data.colwise().mean();
  Eigen::MatrixXd centered = data.rowwise() - mean.transpose();
  Eigen::Matrix3d varianceCovariance = centered.transpose() * centered;
  Eigen::SelfAdjointEigenSolver<Eigen::Matrix3d> eig(varianceCovariance);

  // Eigenvalues
  Eigen::VectorXd D = eig.eigenvalues();

  // if the second eigen value is close to the highest one
  // and bigger than the smallest one it means that the points
  // are distributed among a plane
  if ( (significantlyFactor2 * D(1) > D(2)) && (D(1) > significantlyFactor1 * D(0)) )
  {
    n = eig.eigenvectors().col(0);
  }
  else
  {
    return 2;
  }

  A = n * n.transpose();

  // it would be the case if P1 = P2, P1 = P3
  // or P3 = P2. For instance if the sensor has
  // some dual returns that hit the same point
  if (!std::isfinite(A(0, 0)))
  {
    return 3;
  }

  Eigen::Vector3d Xtemp;
  Point pt;
  double meanSquaredDist = 0;
  for (unsigned int k = 0; k < requiredNearest; ++k)
  {
    pt = kdtreePreviousPlanes.getInputCloud()->points[nearestIndex[k]];
    Xtemp(0) = pt.x; Xtemp(1) = pt.y; Xtemp(2) = pt.z;
    double squaredDist = (Xtemp - mean).transpose() * A * (Xtemp - mean);
    if (squaredDist > squaredMaxDist)
    {
      return 4;
    }
    meanSquaredDist += squaredDist;
  }
  meanSquaredDist /= static_cast<double>(requiredNearest);
  double fitQualityCoeff = 1.0 - std::sqrt(std::abs(meanSquaredDist) / squaredMaxDist);

  // s represents the quality of the match
  double s = fitQualityCoeff;

  // store the distance parameters values
  this->Avalues.emplace_back(A);
  this->Pvalues.emplace_back(mean);
  this->Xvalues.emplace_back(P0);
  this->residualCoefficient.emplace_back(s);
  this->TimeValues.emplace_back(p.intensity);  // CHECK not time?
  return 6;
}

//-----------------------------------------------------------------------------
int Slam::ComputeBlobsDistanceParameters(pcl::KdTreeFLANN<Slam::Point>::Ptr kdtreePreviousBlobs, const Eigen::Isometry3d& transform,
                                         Point p, MatchingMode /*matchingMode*/)
{
  // number of neighbors blobs points required to approximate
  // the corresponding ellipsoide
  unsigned int requiredNearest = 25;

  // maximum distance between keypoints
  // and its neighbor
  double maxDist = this->MaxDistanceForICPMatching;
  float maxDiameterTol = std::pow(4.0, 2);

  // Usefull variables
  Eigen::Vector3d P0, P, n;
  Eigen::Matrix3d A;

  // Transform the point using the current pose estimation
  // TODO : undistortion case
  P << p.x, p.y, p.z;
  P0 = P;
  P = transform * P;
  p.x = P(0); p.y = P(1); p.z = P(2);

  std::vector<int> nearestIndex;
  std::vector<float> nearestDist;
  kdtreePreviousBlobs->nearestKSearch(p, requiredNearest, nearestIndex, nearestDist);

  // It means that there is not enought keypoints in the neighbohood
  if (nearestIndex.size() < requiredNearest)
  {
    return 0;
  }

  // if the nearest blobs is too far from the
  // current blob keypoint we skip this point.
  if (nearestDist[requiredNearest - 1] > maxDist)
  {
    return 1;
  }

  // check the diameter of the neighborhood
  // if the diameter is too big we don't want
  // to keep this blobs. We must do that since
  // the blobs fitted ellipsoide is assume to
  // encode the local neighborhood shape.
  float maxDiameter = 0;
  for (unsigned int i = 0; i < requiredNearest; ++i)
  {
    for (unsigned int j = 0; j < requiredNearest; ++j)
    {
      Point pt1 = kdtreePreviousBlobs->getInputCloud()->points[nearestIndex[i]];
      Point pt2 = kdtreePreviousBlobs->getInputCloud()->points[nearestIndex[j]];
      float neighborhoodDiameter = std::pow(pt1.x - pt2.x, 2) + std::pow(pt1.y - pt2.y, 2) + std::pow(pt1.z - pt2.z, 2);
      maxDiameter = std::max(maxDiameter, neighborhoodDiameter);
    }
  }
  if (maxDiameter > maxDiameterTol)
  {
    return 2;
  }

  // Compute PCA to determine best ellipsoide approximation
  // of the requiredNearest nearest blobs points extracted
  // Thanks to the PCA we will check the shape of the neighborhood
  // tune a distance function adapter to the distribution
  // (Mahalanobis distance)
  Eigen::MatrixXd data(requiredNearest, 3);

  for (unsigned int k = 0; k < requiredNearest; k++)
  {
    Point pt = kdtreePreviousBlobs->getInputCloud()->points[nearestIndex[k]];
    data.row(k) << pt.x, pt.y, pt.z;
  }

  Eigen::Vector3d mean = data.colwise().mean();
  Eigen::MatrixXd centered = data.rowwise() - mean.transpose();
  Eigen::Matrix3d varianceCovariance = centered.transpose() * centered;

  // Sigma is the inverse of the covariance
  // Matrix encoding the mahalanobis distance
  // check that the covariance matrix is inversible
  if (std::abs(varianceCovariance.determinant()) < 1e-6)
  {
    return 3;
  }
  Eigen::Matrix3d sigma = varianceCovariance.inverse();
  Eigen::SelfAdjointEigenSolver<Eigen::Matrix3d> eig(sigma);

  // rescale the variance covariance matrix to preserve the
  // shape of the mahalanobis distance but removing the
  // variance values scaling
  Eigen::MatrixXd D = eig.eigenvalues();
  Eigen::MatrixXd U = eig.eigenvectors();
  D = D / D(2);
  Eigen::Matrix3d diagD = Eigen::Matrix3d::Zero();
  diagD(0, 0) = D(0); diagD(1, 1) = D(1); diagD(2, 2) = D(2);
  A = U * diagD * U.transpose();

  if (!std::isfinite(A.determinant()))
  {
    return 4;
  }

  // Coefficient the distance
  // using the distance between the point
  // and its matching blob; The aim is to prevent
  // wrong matching to pull the point cloud in the
  // bad direction
  double s = 1.0;//1.0 - nearestDist[requiredNearest - 1] / maxDist;

  // store the distance parameters values
  this->Avalues.emplace_back(A);
  this->Pvalues.emplace_back(mean);
  this->Xvalues.emplace_back(P0);
  this->residualCoefficient.emplace_back(s);
  return 5;
}

//-----------------------------------------------------------------------------
void Slam::GetEgoMotionLineSpecificNeighbor(std::vector<int>& nearestValid, std::vector<float>& nearestValidDist,
                                            unsigned int nearestSearch, KDTreePCLAdaptor& kdtreePreviousEdges, const Point& p)
{
  // clear vector
  nearestValid.clear();
  nearestValid.resize(0);
  nearestValidDist.clear();
  nearestValidDist.resize(0);

  // get nearest neighbor of the query point
  std::vector<int> nearestIndex(nearestSearch, -1);
  std::vector<double> nearestDist(nearestSearch, -1.0);
  kdtreePreviousEdges.query(p, nearestSearch, nearestIndex.data(), nearestDist.data());

  // take the closest point
  std::vector<int> idAlreadyTook(this->KeyPointsExtractor->GetNLasers(), 0);
  Point closest = kdtreePreviousEdges.getInputCloud()->points[nearestIndex[0]];
  nearestValid.push_back(nearestIndex[0]);
  nearestValidDist.push_back(nearestDist[0]);

  // invalid all possible points that
  // are on the same scan line than the
  // closest one
  idAlreadyTook[(int)closest.laserId] = 1;

  // invalid all possible points from scan
  // lines that are too far from the closest one
  for (int k = 0; k < this->KeyPointsExtractor->GetNLasers(); ++k)
  {
    if (std::abs(int(closest.laserId) - k) > 4.0)  // TODO : add parameter to discard too far laser rings
    {
      idAlreadyTook[k] = 1;
    }
  }

  // Make a selection among the neighborhood
  // of the query point. We can only take one edge
  // per scan line
  int id;
  for (unsigned int k = 1; k < nearestIndex.size(); ++k)
  {
    id = kdtreePreviousEdges.getInputCloud()->points[nearestIndex[k]].laserId;
    if ( (idAlreadyTook[id] < 1) && (nearestDist[k] < this->MaxDistanceForICPMatching))
    {
      idAlreadyTook[id] = 1;
      nearestValid.push_back(nearestIndex[k]);
      nearestValidDist.push_back(nearestDist[k]);
    }
  }
}

//-----------------------------------------------------------------------------
void Slam::GetMappingLineSpecificNeigbbor(std::vector<int>& nearestValid, std::vector<float>& nearestValidDist, double maxDistInlier,
                                          unsigned int nearestSearch, KDTreePCLAdaptor& kdtreePreviousEdges, const Point& p)
{
  // reset vectors
  nearestValid.clear();
  nearestValid.resize(0);
  nearestValidDist.clear();
  nearestValidDist.resize(0);

  // to prevent square root when making camparisons
  maxDistInlier = std::pow(maxDistInlier, 2);

  // Take the neighborhood of the query point
  // get nearest neighbor of the query point
  std::vector<int> nearestIndex(nearestSearch, -1);
  std::vector<double> nearestDist(nearestSearch, -1.0);
  kdtreePreviousEdges.query(p, nearestSearch, nearestIndex.data(), nearestDist.data());

  // take the closest point
  std::vector<std::vector<int> > inliersList;
  Point closest = kdtreePreviousEdges.getInputCloud()->points[nearestIndex[0]];
  nearestValid.push_back(nearestIndex[0]);
  nearestValidDist.push_back(nearestDist[0]);

  Eigen::Vector3d P1, P2, dir, Pcdt;
  Eigen::Matrix3d D;
  P1 << closest.x, closest.y, closest.z;
  Point pclP2;
  Point inlierCandidate;

  // Loop over other neighbors of the neighborhood. For each of them
  // compute the line between closest point and current point and
  // compute the number of inlier that fit this line. Keep the line and its
  // inliers with the most inliers
  for (unsigned int ptIndex = 1; ptIndex < nearestIndex.size(); ++ptIndex)
  {
    std::vector<int> inlierIndex;
    pclP2 = kdtreePreviousEdges.getInputCloud()->points[nearestIndex[ptIndex]];
    P2 << pclP2.x, pclP2.y, pclP2.z;
    dir = (P2 - P1).normalized();
    D = this->I3 - dir * dir.transpose();
    D = D.transpose() * D;

    for (unsigned int candidateIndex = 1; candidateIndex < nearestIndex.size(); ++candidateIndex)
    {
      inlierCandidate = kdtreePreviousEdges.getInputCloud()->points[nearestIndex[candidateIndex]];
      Pcdt << inlierCandidate.x, inlierCandidate.y, inlierCandidate.z;
      if ( (Pcdt - P1).transpose() * D * (Pcdt - P1) < maxDistInlier)
      {
        inlierIndex.push_back(candidateIndex);
      }
    }
    inliersList.push_back(inlierIndex);
  }

  std::size_t maxInliers = 0;
  int indexMaxInliers = -1;
  for (unsigned int k = 0; k < inliersList.size(); ++k)
  {
    if (inliersList[k].size() > maxInliers)
    {
      maxInliers = inliersList[k].size();
      indexMaxInliers = k;
    }
  }

  // fill
  for (unsigned int k = 0; k < inliersList[indexMaxInliers].size(); ++k)
  {
    nearestValid.push_back(nearestIndex[inliersList[indexMaxInliers][k]]);
    nearestValidDist.push_back(nearestDist[inliersList[indexMaxInliers][k]]);
  }

  return;
}

//==============================================================================
//   Geometrical transformations
//==============================================================================

//-----------------------------------------------------------------------------
void Slam::TransformToWorld(Point& p)
{
  if (this->Undistortion)
  {
    this->ExpressPointInOtherReferencial(p);
  }
  else
  {
    Eigen::Vector3d P;
    P << p.x, p.y, p.z;

    P = this->Tworld * P;

    p.x = P(0);
    p.y = P(1);
    p.z = P(2);
  }
}

//-----------------------------------------------------------------------------
void Slam::CreateWithinFrameTrajectory(SampledSensorPath& path, WithinFrameTrajMode mode)
{
  const auto& motionParameters = (mode == WithinFrameTrajMode::EgoMotionTraj) ? this->MotionParametersEgoMotion : 
                                                                                this->MotionParametersMapping;

  path.Samples.resize(2);
  // Add orientation and position of the sensor at the beginning of the frame
  this->WithinFrameTrajectory.Samples[0].R = motionParameters.first.linear();
  this->WithinFrameTrajectory.Samples[0].T = motionParameters.first.translation();
  this->WithinFrameTrajectory.Samples[0].time = 0.0;
  // Add orientation and position of the sensor at the end of the frame
  this->WithinFrameTrajectory.Samples[1].R = motionParameters.second.linear();
  this->WithinFrameTrajectory.Samples[1].T = motionParameters.second.translation();
  this->WithinFrameTrajectory.Samples[1].time = 1.0;

  if (mode == WithinFrameTrajMode::UndistortionTraj)  // CHECK unreachable code (mode is never set to UndistortionTraj)
  {
    // Relative motion between t0 and t1
    Eigen::Isometry3d dH = motionParameters.second.inverse() * motionParameters.first;
    this->WithinFrameTrajectory.Samples[0].R = dH.linear();
    this->WithinFrameTrajectory.Samples[0].T = dH.translation();
    this->WithinFrameTrajectory.Samples[1].R = Eigen::Matrix3d::Identity();
    this->WithinFrameTrajectory.Samples[1].T = Eigen::Vector3d::Zero();
  }
}

//-----------------------------------------------------------------------------
void Slam::ExpressPointInOtherReferencial(Point& p)
{
  // interpolate the transform
  AffineIsometry iso = this->WithinFrameTrajectory(p.intensity);  // CHECK intensity ? not time ?
  Eigen::Vector3d X(p.x, p.y, p.z);
  Eigen::Vector3d Y = iso.R * X + iso.T;
  p.x = Y(0); p.y = Y(1); p.z = Y(2);
}

//-----------------------------------------------------------------------------
void Slam::ExpressPointCloudInOtherReferencial(PointCloud::Ptr& pointcloud)
{
  for (Point& point : *pointcloud)
  {
    ExpressPointInOtherReferencial(point);
  }
}

//==============================================================================
//   Rolling grids parameters setting
//==============================================================================

//-----------------------------------------------------------------------------
void Slam::ClearMaps()
{
  this->EdgesPointsLocalMap->Clear();
  this->PlanarPointsLocalMap->Clear();
  this->BlobsPointsLocalMap->Clear();
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridLeafSizeEdges(double size)
{
  this->EdgesPointsLocalMap->SetLeafSize(size);
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridLeafSizePlanes(double size)
{
  this->PlanarPointsLocalMap->SetLeafSize(size);
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridLeafSizeBlobs(double size)
{
  this->BlobsPointsLocalMap->SetLeafSize(size);
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridSize(int size)
{
  this->EdgesPointsLocalMap->SetGridSize(size);
  this->PlanarPointsLocalMap->SetGridSize(size);
  this->BlobsPointsLocalMap->SetGridSize(size);
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridResolution(double resolution)
{
  this->EdgesPointsLocalMap->SetVoxelResolution(resolution);
  this->PlanarPointsLocalMap->SetVoxelResolution(resolution);
  this->BlobsPointsLocalMap->SetVoxelResolution(resolution);
}

//-----------------------------------------------------------------------------
void Slam::SetFrameMinMaxKeypoints(const Eigen::Vector3d& minPoint, const Eigen::Vector3d& maxPoint)
{
  this->EdgesPointsLocalMap->SetMinMaxPoints(minPoint, maxPoint);
  this->PlanarPointsLocalMap->SetMinMaxPoints(minPoint, maxPoint);
  this->BlobsPointsLocalMap->SetMinMaxPoints(minPoint, maxPoint);
}