//==============================================================================
// Copyright 2018-2020 Kitware, Inc., Kitware SAS
// Author: Guilbert Pierre (Kitware SAS)
//         Cadart Nicolas (Kitware SAS)
// Creation date: 2018-03-27
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//==============================================================================

// This slam algorithm is inspired by the LOAM algorithm:
// J. Zhang and S. Singh. LOAM: Lidar Odometry and Mapping in Real-time.
// Robotics: Science and Systems Conference (RSS). Berkeley, CA, July 2014.

// The algorithm is composed of three sequential steps:
//
// - Keypoints extraction: this step consists of extracting keypoints over
// the points clouds. To do that, the laser lines / scans are treated independently.
// The laser lines are projected onto the XY plane and are rescaled depending on
// their vertical angle. Then we compute their curvature and create two classes of
// keypoints. The edges keypoints which correspond to points with a high curvature
// and planar points which correspond to points with a low curvature.
//
// - Ego-Motion: this step consists of recovering the motion of the lidar
// sensor between two frames (two sweeps). The motion is modelized by a constant
// velocity and angular velocity between two frames (i.e null acceleration).
// Hence, we can parameterize the motion by a rotation and translation per sweep / frame
// and interpolate the transformation inside a frame using the timestamp of the points.
// Since the points clouds generated by a lidar are sparse we can't design a
// pairwise match between keypoints of two successive frames. Hence, we decided to use
// a closest-point matching between the keypoints of the current frame
// and the geometric features derived from the keypoints of the previous frame.
// The geometric features are lines or planes and are computed using the edges
// and planar keypoints of the previous frame. Once the matching is done, a keypoint
// of the current frame is matched with a plane / line (depending of the
// nature of the keypoint) from the previous frame. Then, we recover R and T by
// minimizing the function f(R, T) = sum(d(point, line)^2) + sum(d(point, plane)^2).
// Which can be writen f(R, T) = sum((R*X+T-P).t*A*(R*X+T-P)) where:
// - X is a keypoint of the current frame
// - P is a point of the corresponding line / plane
// - A = (n*n.t) with n being the normal of the plane
// - A = (I - n*n.t).t * (I - n*n.t) with n being a director vector of the line
// Since the function f(R, T) is a non-linear mean square error function
// we decided to use the Levenberg-Marquardt algorithm to recover its argmin.
//
// - Localization: This step consists of refining the motion recovered in the Ego-Motion
// step and to add the new frame in the environment map. Thanks to the ego-motion
// recovered at the previous step it is now possible to estimate the new position of
// the sensor in the map. We use this estimation as an initial point (R0, T0) and we
// perform an optimization again using the keypoints of the current frame and the matched
// keypoints of the map (and not only the previous frame this time!). Once the position in the
// map has been refined from the first estimation it is then possible to update the map by
// adding the keypoints of the current frame into the map.
//
// In the following programs, three 3D coordinates system are used :
// - LIDAR {L} : attached to the geometric center of the LiDAR sensor. The
//   coordinates of the received pointclouds are expressed in this system.
//   LIDAR is rigidly linked (static transform) to BASE.
// - BASE  {B} : attached to the origin of the moving body (e.g. vehicle). We
//   are generally interested in tracking an other point of the moving body than
//   the LiDAR's (for example, we prefer to track the GPS antenna pose).
// - WORLD {W} : The world coordinate system {W} coincides with BASE at the
//   initial position. The output trajectory describes BASE origin in WORLD.

// LOCAL
#include "LidarSlam/Slam.h"
#include "LidarSlam/Utilities.h"
#ifdef USE_G2O
#include "LidarSlam/PoseGraphOptimization.h"
#endif  // USE_G2O
// CERES
#include <ceres/solver.h>
// PCL
#include <pcl/common/common.h>
#include <pcl/common/transforms.h>
#include <pcl/io/pcd_io.h>
// EIGEN
#include <Eigen/Dense>

#define PRINT_VERBOSE(minVerbosityLevel, stream) if (this->Verbosity >= (minVerbosityLevel)) {std::cout << stream << std::endl;}
#define IF_VERBOSE(minVerbosityLevel, command) if (this->Verbosity >= (minVerbosityLevel)) { command; }

namespace LidarSlam
{

namespace Utils
{
namespace
{
//-----------------------------------------------------------------------------
std::array<double, 36> Matrix6dToStdArray36(const Eigen::Matrix6d& covar)
{
  std::array<double, 36> cov;
  std::copy_n(covar.data(), 36, cov.begin());
  return cov;
}

//-----------------------------------------------------------------------------
//! Approximate pointcloud memory size
inline size_t PointCloudMemorySize(const Slam::PointCloud& cloud)
{
  return (sizeof(cloud) + (sizeof(Slam::PointCloud::PointType) * cloud.size()));
}

//-----------------------------------------------------------------------------
//! Approximate logged keypoints size
void LoggedKeypointsSize(const std::deque<PointCloudStorage<Slam::Point>>& log, size_t& totalMemory, size_t& totalPoints)
{
  totalMemory = 0;
  totalPoints = 0;
  for (auto const& storage: log)
  {
    totalPoints += storage.PointsSize();
    totalMemory += storage.MemorySize();
  }
}
} // end of anonymous namespace
} // end of Utils namespace

//==============================================================================
//   Main SLAM use
//==============================================================================

//-----------------------------------------------------------------------------
Slam::Slam()
{
  // Allocate a default Keypoint Extractor for device 0
  this->KeyPointsExtractors[0] = std::make_shared<SpinningSensorKeypointExtractor>();

  // Allocate maps
  for (auto k : KeypointTypes)
    this->LocalMaps[k] = std::make_shared<RollingGrid>();

  // Set default maps parameters
  this->SetVoxelGridResolution(10.);
  this->SetVoxelGridSize(50);
  this->SetVoxelGridLeafSize(EDGE, 0.30);
  this->SetVoxelGridLeafSize(PLANE, 0.60);
  this->SetVoxelGridLeafSize(BLOB, 0.30);

  // Reset SLAM internal state
  this->Reset();
}

//-----------------------------------------------------------------------------
void Slam::Reset(bool resetLog)
{
  // Reset keypoints maps
  this->ClearMaps();

  // n-DoF parameters
  this->Tworld = Eigen::Isometry3d::Identity();
  this->PreviousTworld = Eigen::Isometry3d::Identity();
  this->Trelative = Eigen::Isometry3d::Identity();
  this->WithinFrameMotion.SetTransforms(Eigen::Isometry3d::Identity(), Eigen::Isometry3d::Identity());

  // Reset pose uncertainty
  this->LocalizationUncertainty = LocalOptimizer::RegistrationError();

  // Reset point clouds
  this->CurrentFrames.clear();
  this->CurrentFrames.emplace_back(new PointCloud);
  for (auto k : KeypointTypes)
  {
    this->CurrentRawKeypoints[k].reset(new PointCloud);
    this->CurrentUndistortedKeypoints[k].reset(new PointCloud);
    this->CurrentWorldKeypoints[k].reset(new PointCloud);
  }

  // Reset keypoints matching results
  for (auto k : {EDGE, PLANE})
    this->EgoMotionMatchingResults[k] = KeypointsMatcher::MatchingResults();
  for (auto k : KeypointTypes)
    this->LocalizationMatchingResults[k] = KeypointsMatcher::MatchingResults();

  // Reset log history
  if (resetLog)
  {
    // Reset logged keypoints
    this->NbrFrameProcessed = 0;
    this->LogTrajectory.clear();
    this->LogKeypoints.clear();

    // Reset processing duration timers
    Utils::Timer::Reset();
  }
}

//-----------------------------------------------------------------------------
void Slam::SetNbThreads(int n)
{
  // Set number of threads for main processes
  this->NbThreads = n;
  // Set number of threads for keypoints extraction
  for (const auto& kv : this->KeyPointsExtractors) 
    kv.second->SetNbThreads(n); 
}

//-----------------------------------------------------------------------------
void Slam::AddFrames(const std::vector<PointCloud::Ptr>& frames)
{
  Utils::Timer::Init("SLAM frame processing");

  // Check that input frames are correct and can be processed
  if (!this->CheckFrames(frames))
    return;
  this->CurrentFrames = frames;

  PRINT_VERBOSE(2, "\n#########################################################");
  PRINT_VERBOSE(1, "Processing frame " << this->NbrFrameProcessed);
  PRINT_VERBOSE(2, "#########################################################\n");

  // Compute the edge and planar keypoints
  IF_VERBOSE(3, Utils::Timer::Init("Keypoints extraction"));
  this->ExtractKeypoints();
  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("Keypoints extraction"));

  // Estimate Trelative by extrapolating new pose with a constant velocity model
  // and/or registering current frame on previous one
  IF_VERBOSE(3, Utils::Timer::Init("Ego-Motion"));
  this->ComputeEgoMotion();
  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("Ego-Motion"));

  // Perform Localization : update Tworld from map and current frame keypoints
  // and optionally undistort keypoints clouds based on ego-motion
  IF_VERBOSE(3, Utils::Timer::Init("Localization"));
  this->Localization();
  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("Localization"));

  // Update keypoints maps : add current keypoints to map using Tworld
  if (this->UpdateMap)
  {
    IF_VERBOSE(3, Utils::Timer::Init("Maps update"));
    this->UpdateMapsUsingTworld();
    IF_VERBOSE(3, Utils::Timer::StopAndDisplay("Maps update"));
  }

  // Log current frame processing results : pose, covariance and keypoints.
  IF_VERBOSE(3, Utils::Timer::Init("Logging"));
  this->LogCurrentFrameState(Utils::PclStampToSec(this->CurrentFrames[0]->header.stamp), this->WorldFrameId);
  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("Logging"));

  // Motion and localization parameters estimation information display
  if (this->Verbosity >= 2)
  {
    SET_COUT_FIXED_PRECISION(3);
    std::cout << "========== SLAM results ==========\n";
    if (this->Undistortion)
    {
      Eigen::Isometry3d motion = this->WithinFrameMotion.GetTransformRange();
      std::cout << "Within frame motion:\n"
                   " translation = [" << motion.translation().transpose()                                        << "] m\n"
                   " rotation    = [" << Utils::Rad2Deg(Utils::RotationMatrixToRPY(motion.linear())).transpose() << "] °\n";
    }
    std::cout << "Ego-Motion:\n"
                 " translation = [" << this->Trelative.translation().transpose()                                        << "] m\n"
                 " rotation    = [" << Utils::Rad2Deg(Utils::RotationMatrixToRPY(this->Trelative.linear())).transpose() << "] °\n"
                 "Localization:\n"
                 " position    = [" << this->Tworld.translation().transpose()                                        << "] m\n"
                 " orientation = [" << Utils::Rad2Deg(Utils::RotationMatrixToRPY(this->Tworld.linear())).transpose() << "] °" << std::endl;
    RESET_COUT_FIXED_PRECISION;
  }

  if (this->Verbosity >= 5)
  {
    SET_COUT_FIXED_PRECISION(3);
    std::cout << "========== Memory usage ==========\n";
    for (auto k : KeypointTypes)
    {
      PointCloud::Ptr map = this->GetMap(k);
      std::cout << Utils::Capitalize(Utils::Plural(KeypointTypeNames.at(k))) << " map: "
                << map->size() << " points, " << Utils::PointCloudMemorySize(*map) * 1e-6 << " MB\n";
    }

    // Logged keypoints
    size_t memory, points;
    for (auto k : KeypointTypes)
    {
      Utils::LoggedKeypointsSize(this->LogKeypoints[k], memory, points);
      std::cout << Utils::Capitalize(Utils::Plural(KeypointTypeNames.at(k))) << " log  : "
                << this->LogKeypoints[k].size() << " frames, " << points << " points, " << memory * 1e-6 << " MB\n";
    }
    RESET_COUT_FIXED_PRECISION;
  }

  // Frame processing duration
  this->Latency = Utils::Timer::Stop("SLAM frame processing");
  this->NbrFrameProcessed++;
  IF_VERBOSE(1, Utils::Timer::StopAndDisplay("SLAM frame processing"));
}

//-----------------------------------------------------------------------------
void Slam::RunPoseGraphOptimization(const std::vector<Transform>& gpsPositions,
                                    const std::vector<std::array<double, 9>>& gpsCovariances,
                                    Eigen::Isometry3d& gpsToSensorOffset,
                                    const std::string& g2oFileName)
{
  #ifdef USE_G2O
  IF_VERBOSE(1, Utils::Timer::Init("Pose graph optimization"));
  IF_VERBOSE(3, Utils::Timer::Init("PGO : optimization"));

  // Transform to modifiable vectors
  std::vector<Transform> slamPoses(this->LogTrajectory.begin(), this->LogTrajectory.end());
  std::vector<std::array<double, 36>> slamCovariances(this->LogCovariances.begin(), this->LogCovariances.end());

  const unsigned int nbSlamPoses = slamPoses.size();

  if (this->LoggingTimeout == 0.)
  {
    PRINT_WARNING("SLAM logging is not enabled : covariances will be "
                  "arbitrarly set and maps will not be optimized during pose "
                  "graph optimization.");

    // Set all poses covariances equal to twice the last one if we did not log it
    std::array<double, 36> fakeSlamCovariance = Utils::Matrix6dToStdArray36(this->LocalizationUncertainty.Covariance * 2);
    for (unsigned int i = 0; i < nbSlamPoses; i++)
      slamCovariances.emplace_back(fakeSlamCovariance);
  }

  // Init pose graph optimizer
  PoseGraphOptimization poseGraphOptimization;
  poseGraphOptimization.SetNbIteration(500);
  poseGraphOptimization.SetVerbose(this->Verbosity >= 2);
  poseGraphOptimization.SetSaveG2OFile(!g2oFileName.empty());
  poseGraphOptimization.SetG2OFileName(g2oFileName);
  poseGraphOptimization.SetGpsToSensorCalibration(gpsToSensorOffset);

  // Run pose graph optimization
  // TODO : templatize poseGraphOptimization to accept any STL container and avoid deque <-> vector copies
  std::vector<Transform> optimizedSlamPoses;
  if (!poseGraphOptimization.Process(slamPoses, gpsPositions,
                                     slamCovariances, gpsCovariances,
                                     optimizedSlamPoses))
  {
    PRINT_ERROR("Pose graph optimization failed.");
    return;
  }

  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("PGO : optimization"));

  // Update GPS/LiDAR calibration
  gpsToSensorOffset = optimizedSlamPoses.front().GetIsometry();

  // Update SLAM trajectory and maps
  IF_VERBOSE(3, Utils::Timer::Init("PGO : SLAM reset"));
  this->Reset(false);
  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("PGO : SLAM reset"));
  IF_VERBOSE(3, Utils::Timer::Init("PGO : frames keypoints aggregation"));
  std::map<Keypoint, PointCloud> keypoints;
  std::map<Keypoint, PointCloud::Ptr> aggregatedKeypointsMap;
  for (auto k : KeypointTypes)
    aggregatedKeypointsMap[k].reset(new PointCloud);

  for (unsigned int i = 0; i < nbSlamPoses; i++)
  {
    // Update SLAM pose
    this->LogTrajectory[i].SetIsometry(gpsToSensorOffset.inverse() * optimizedSlamPoses[i].GetIsometry());

    // Transform frame keypoints to world coordinates
    std::map<Keypoint, PointCloud::Ptr> logKeypoints;
    for (auto k : KeypointTypes)
        logKeypoints[k] = this->UseKeypoints[k] ? this->LogKeypoints[k][i].GetCloud() : PointCloud::Ptr(new PointCloud);

    if (this->Undistortion && i >= 1)
    {
      // Init the undistortion interpolator
      LinearTransformInterpolator<double> interpolator;
      interpolator.SetTransforms(this->LogTrajectory[i - 1].GetIsometry(), this->LogTrajectory[i].GetIsometry());
      interpolator.SetTimes(this->LogTrajectory[i].time - this->LogTrajectory[i - 1].time, 0.);

      // Perform undistortion of keypoint clouds
      for (auto k : KeypointTypes)
      {
        keypoints[k].clear();
        keypoints[k].reserve(logKeypoints[k]->size());
        for (const Point& p : *logKeypoints[k])
          keypoints[k].push_back(Utils::TransformPoint(p, interpolator(p.time)));
      }
    }
    else
    {
      Eigen::Matrix4d currentTransform = this->LogTrajectory[i].GetMatrix();
      for (auto k : KeypointTypes)
        pcl::transformPointCloud(*logKeypoints[k],  keypoints[k],  currentTransform);
    }

    // Aggregate new keypoints to maps
    for (auto k : KeypointTypes)
      *aggregatedKeypointsMap[k] += keypoints[k];
  }

  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("PGO : frames keypoints aggregation"));
  IF_VERBOSE(3, Utils::Timer::Init("PGO : final SLAM map update"));

  // Set final pose
  this->Tworld         = this->LogTrajectory[nbSlamPoses - 1].GetIsometry();
  this->PreviousTworld = this->LogTrajectory[nbSlamPoses - 2].GetIsometry();

  // Update SLAM maps
  for (auto k : KeypointTypes)
  {
    // We do not do map.Add(aggregatedPoints) as this would try to roll the map
    // so that the entire aggregatedPoints can best fit into map. But if this
    // entire point cloud does not fit into map, the rolling grid will be
    // centered on the aggregatedPoints bounding box center.
    // Instead, we prefer to roll so that the last frame keypoints can fit,
    // ensuring that next frame will be matched efficiently to rolled map.
    if (this->UseKeypoints[k])
    {
      Eigen::Vector4f minPoint, maxPoint;
      pcl::getMinMax3D(keypoints[k], minPoint, maxPoint);
      this->LocalMaps[k]->Roll(minPoint.head<3>().cast<double>().array(), maxPoint.head<3>().cast<double>().array());
      this->LocalMaps[k]->Add(aggregatedKeypointsMap[k], false);
    }
  }

  // Processing duration
  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("PGO : final SLAM map update"));
  IF_VERBOSE(1, Utils::Timer::StopAndDisplay("Pose graph optimization"));
  #else
  #define UNUSED(var) (void)(var)
  UNUSED(gpsPositions); UNUSED(gpsCovariances); UNUSED(gpsToSensorOffset); UNUSED(g2oFileName);
  PRINT_ERROR("SLAM PoseGraphOptimization requires G2O, but it was not found.");
  #endif  // USE_G2O
}

//-----------------------------------------------------------------------------
void Slam::SetWorldTransformFromGuess(const Transform& poseGuess)
{
  // Set current pose
  this->Tworld = poseGuess.GetIsometry();

  // Ego-Motion estimation is not valid anymore since we imposed a discontinuity.
  // We reset previous pose so that previous ego-motion extrapolation results in Identity matrix.
  // We reset current frame keypoints so that ego-motion registration will be skipped for next frame.
  this->PreviousTworld = this->Tworld;
  for (auto k : KeypointTypes)
    this->CurrentRawKeypoints[k].reset(new PointCloud);
}

//-----------------------------------------------------------------------------
void Slam::SaveMapsToPCD(const std::string& filePrefix, PCDFormat pcdFormat) const
{
  IF_VERBOSE(3, Utils::Timer::Init("Keypoints maps saving to PCD"));

  // Save keypoints maps
  for (auto k : KeypointTypes)
  {
    if (this->UseKeypoints.at(k))
      savePointCloudToPCD(filePrefix + Utils::Plural(KeypointTypeNames.at(k)) + ".pcd",  *this->GetMap(k),  pcdFormat, true);
  }

  // TODO : save map origin (in which coordinates?) in title or VIEWPOINT field

  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("Keypoints maps saving to PCD"));
}

//-----------------------------------------------------------------------------
void Slam::LoadMapsFromPCD(const std::string& filePrefix, bool resetMaps)
{
  IF_VERBOSE(3, Utils::Timer::Init("Keypoints maps loading from PCD"));

  // In most of the cases, we would like to reset SLAM internal maps before
  // loading new maps to avoid conflicts.
  if (resetMaps)
    this->ClearMaps();

  for (auto k : KeypointTypes)
  {
    std::string path = filePrefix + Utils::Plural(KeypointTypeNames.at(k)) + ".pcd";
    PointCloud::Ptr keypoints(new PointCloud);
    if (pcl::io::loadPCDFile(path, *keypoints) == 0)
    {
      std::cout << "SLAM keypoints map successfully loaded from " << path << std::endl;
      this->LocalMaps[k]->Add(keypoints);
    }
  }
  // TODO : load/use map origin (in which coordinates?) in title or VIEWPOINT field
  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("Keypoints maps loading from PCD"));
}

//==============================================================================
//   SLAM results getters
//==============================================================================

//-----------------------------------------------------------------------------
Transform Slam::GetWorldTransform() const
{
  return this->LogTrajectory.empty() ? Transform::Identity() : this->LogTrajectory.back();
}

//-----------------------------------------------------------------------------
Transform Slam::GetLatencyCompensatedWorldTransform() const
{
  constexpr double MAX_EXTRAPOLATION_RATIO = 3.;

  // Get 2 last transforms
  unsigned int trajectorySize = this->LogTrajectory.size();
  if (trajectorySize == 0)
    return Transform::Identity();
  else if (trajectorySize == 1)
    return this->LogTrajectory.back();
  const Transform& previous = this->LogTrajectory[trajectorySize - 2];
  const Transform& current = this->LogTrajectory[trajectorySize - 1];
  const Eigen::Isometry3d& H0 = previous.GetIsometry();
  const Eigen::Isometry3d& H1 = current.GetIsometry();

  // Linearly compute normalized timestamp of Hpred.
  // We expect H0 and H1 to match with time 0 and 1.
  // If timestamps are not defined or too close, extrapolation is impossible.
  if (std::abs(current.time - previous.time) < 1e-6)
  {
    PRINT_WARNING("Unable to compute latency-compensated transform : timestamps undefined or too close.");
    return current;
  }
  // If requested extrapolation timestamp is too far from previous frames timestamps, extrapolation is impossible.
  if (std::abs(this->Latency / (current.time - previous.time)) > MAX_EXTRAPOLATION_RATIO)
  {
    PRINT_WARNING("Unable to compute latency-compensated transform : extrapolation time is too far.");
    return current;
  }

  // Extrapolate H0 and H1 to get expected Hpred at current time
  Eigen::Isometry3d Hpred = LinearInterpolation(H0, H1, current.time + this->Latency, previous.time, current.time);

  return Transform(Hpred, current.time, current.frameid);
}

//-----------------------------------------------------------------------------
std::array<double, 36> Slam::GetTransformCovariance() const
{
  return Utils::Matrix6dToStdArray36(this->LocalizationUncertainty.Covariance);
}

//-----------------------------------------------------------------------------
std::vector<Transform> Slam::GetTrajectory() const
{
  std::vector<Transform> slamPoses(this->LogTrajectory.begin(), this->LogTrajectory.end());
  return slamPoses;
}

//-----------------------------------------------------------------------------
std::vector<std::array<double, 36>> Slam::GetCovariances() const
{
  std::vector<std::array<double, 36>> slamCovariances(this->LogCovariances.begin(), this->LogCovariances.end());
  return slamCovariances;
}

//-----------------------------------------------------------------------------
std::unordered_map<std::string, double> Slam::GetDebugInformation() const
{
  std::unordered_map<std::string, double> map;
  for (auto k : {EDGE, PLANE})
  {
    std::string name = "EgoMotion: " + Utils::Plural(KeypointTypeNames.at(k)) + " used";
    map[name] = this->EgoMotionMatchingResults.at(k).NbMatches();
  }

  for (auto k : KeypointTypes)
  {
    std::string name = "Localization: " + Utils::Plural(KeypointTypeNames.at(k)) + " used";
    map[name] = this->LocalizationMatchingResults.at(k).NbMatches();
  }

  map["Localization: position error"]    = this->LocalizationUncertainty.PositionError;
  map["Localization: orientation error"] = this->LocalizationUncertainty.OrientationError;
  return map;
}

//-----------------------------------------------------------------------------
std::unordered_map<std::string, std::vector<double>> Slam::GetDebugArray() const
{
  auto toDoubleVector = [](auto const& scalars) { return std::vector<double>(scalars.begin(), scalars.end()); };

  std::unordered_map<std::string, std::vector<double>> map;
  for (auto k : {EDGE, PLANE})
  {
    std::string name = "EgoMotion: " + KeypointTypeNames.at(k) + " matches";
    map[name]  = toDoubleVector(this->EgoMotionMatchingResults.at(k).Rejections);
    name = "EgoMotion: " + KeypointTypeNames.at(k) + " weights";
    map[name]  = this->EgoMotionMatchingResults.at(k).Weights;
  }

  for (auto k : KeypointTypes)
  {
    std::string name = "Localization: " + KeypointTypeNames.at(k) + " matches";
    map[name]  = toDoubleVector(this->EgoMotionMatchingResults.at(k).Rejections);
    name = "Localization: " + KeypointTypeNames.at(k) + " weights";
    map[name]  = this->LocalizationMatchingResults.at(k).Weights;
  }

  return map;
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetOutputFrame()
{
  PointCloud::Ptr aggregatedOutput(new PointCloud);
  aggregatedOutput->header = Utils::BuildPclHeader(this->CurrentFrames[0]->header.stamp,
                                                   this->WorldFrameId,
                                                   this->NbrFrameProcessed);

  // Loop over frames of input
  for (unsigned int i = 0; i < this->CurrentFrames.size(); ++i)
  {
    PointCloud output;

    // Transform from LiDAR sensor to BASE coordinate system,
    // followed by rigid transform or undistortion
    Eigen::Isometry3d baseToLidar = this->GetBaseToLidarOffset(this->CurrentFrames[i]->front().device_id);
    if (this->Undistortion)
    {
      auto transformInterpolator = this->WithinFrameMotion;
      transformInterpolator.SetTransforms(this->Tworld * this->WithinFrameMotion.GetH0() * baseToLidar,
                                          this->Tworld * this->WithinFrameMotion.GetH1() * baseToLidar);
      output.reserve(this->CurrentFrames[i]->size());
      for (const Slam::Point& p : *this->CurrentFrames[i])
        output.push_back(Utils::TransformPoint(p, transformInterpolator(p.time)));
    }
    else
    {
      const Eigen::Isometry3d worldPose = this->Tworld * baseToLidar;
      pcl::transformPointCloud(*this->CurrentFrames[i], output, worldPose.matrix());
    }

    // Modify point-wise time offsets to match header.stamp
    double timeOffset = Utils::PclStampToSec(this->CurrentFrames[i]->header.stamp) - Utils::PclStampToSec(aggregatedOutput->header.stamp);
    for (auto& point: output)
      point.time += timeOffset;

    // Add registered (and undistorted) frame to aggregated output
    *aggregatedOutput += output;
  }

  return aggregatedOutput;
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetMap(Keypoint k) const
{
  PointCloud::Ptr map = this->LocalMaps.at(k)->Get();
  map->header = Utils::BuildPclHeader(this->CurrentFrames[0]->header.stamp,
                                      this->WorldFrameId,
                                      this->NbrFrameProcessed);
  return map;
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetKeypoints(Keypoint k, bool worldCoordinates) const
{
  return worldCoordinates ? this->CurrentWorldKeypoints.at(k) : this->CurrentUndistortedKeypoints.at(k);
}

//==============================================================================
//   Main SLAM steps
//==============================================================================

//-----------------------------------------------------------------------------
bool Slam::CheckFrames(const std::vector<PointCloud::Ptr>& frames)
{
  // Check input frames and return if they are all empty
  bool allFramesEmpty = true;
  for (unsigned int i = 0; i < frames.size(); ++i)
  {
    if (!frames[i] || !frames[i]->empty())
      allFramesEmpty = false;
    else
      PRINT_WARNING("SLAM input frame " << i << " is an empty pointcloud : frame ignored.");
  }
  if (allFramesEmpty)
  {
    PRINT_ERROR("SLAM input only contains empty pointclouds : exiting.");
    return false;
  }

  // Skip frames if it has the same timestamp as previous ones (will induce problems in extrapolation)
  if (frames[0]->header.stamp == this->CurrentFrames[0]->header.stamp)
  {
    PRINT_ERROR("SLAM frames have the same timestamp (" << frames[0]->header.stamp << ") as previous ones : frames ignored.");
    return false;
  }

  // Check frame dropping
  for (unsigned int i = 0; i < frames.size(); ++i)
  {
    unsigned int droppedFrames = frames[i]->header.seq - this->PreviousFramesSeq[i] - 1;
    if ((this->PreviousFramesSeq[i] > 0) && (droppedFrames > 0))
      PRINT_WARNING(droppedFrames << " frame(s)" << (frames.size() > 1 ? " from LiDAR device " + std::to_string(i) : "") << " were dropped by SLAM\n");
    this->PreviousFramesSeq[i] = frames[i]->header.seq;
  }

  return true;
}

//-----------------------------------------------------------------------------
void Slam::ExtractKeypoints()
{
  PRINT_VERBOSE(2, "========== Keypoints extraction ==========");

  // Current keypoints become previous ones
  this->PreviousRawKeypoints = this->CurrentRawKeypoints;

  // Reset current keypoints and adapt header
  pcl::PCLHeader header = Utils::BuildPclHeader(this->CurrentFrames[0]->header.stamp, this->BaseFrameId, this->NbrFrameProcessed);
  for (auto k : KeypointTypes)
  {
    this->CurrentRawKeypoints[k].reset(new PointCloud);
    this->CurrentRawKeypoints[k]->header = header;
  }

  // Extract keypoints from each input cloud
  for (const PointCloud::Ptr& frame: this->CurrentFrames)
  {
    // Get LiDAR device id
    int lidarDevice = frame->front().device_id;

    // Get keypoints extractor
    // Check if KE exists
    if (!this->KeyPointsExtractors.count(lidarDevice))
    {
      // If KE does not exist but we are only using a single KE, use default one
      if (this->KeyPointsExtractors.size() == 1)
      {
        PRINT_WARNING("Input frame comes from LiDAR device " << lidarDevice
                    << " but no keypoints extractor has been set for this device : using default extractor for device 0.");
        lidarDevice = 0;
      }
      // Otherwise ignore frame
      else
      {
        PRINT_ERROR("Input frame comes from LiDAR device " << lidarDevice
                    << " but no keypoints extractor has been set for this device : ignoring frame.");
        continue;
      }
    }
    KeypointExtractorPtr& ke = this->KeyPointsExtractors[lidarDevice];

    // Compute keypoints from this device frame
    ke->ComputeKeyPoints(frame);

    // Transform ke keypoints from LIDAR to BASE coordinates, 
    // aggregate them in CurrentRawKeypoints, and correct time offset
    Eigen::Isometry3d baseToLidar = this->GetBaseToLidarOffset(lidarDevice);
    for (auto k : KeypointTypes)
    {
      PointCloud::Ptr baseCloud;
      // If transform to apply is identity, avoid much work
      if (baseToLidar.isApprox(Eigen::Isometry3d::Identity()))
        baseCloud = ke->GetKeypoints(k);
      // If transform is set and non trivial, run transformation
      else
      {
        baseCloud.reset(new PointCloud);
        pcl::transformPointCloud(*ke->GetKeypoints(k), *baseCloud, baseToLidar.matrix());
      }

      // Add to current keypoints
      *this->CurrentRawKeypoints[k] += *baseCloud;

      // Modify point-wise time offsets to match header.stamp
      double timeOffset = Utils::PclStampToSec(ke->GetKeypoints(k)->header.stamp) - Utils::PclStampToSec(this->CurrentRawKeypoints[k]->header.stamp);
      for (unsigned int i = this->CurrentRawKeypoints[k]->size() - ke->GetKeypoints(k)->size(); i < this->CurrentRawKeypoints[k]->size(); i++)
        this->CurrentRawKeypoints[k]->at(i).time += timeOffset;
    }
  }

  if (this->Verbosity >= 2)
  {
    std::cout << "Extracted features : ";
    for (auto k : KeypointTypes)
      std::cout << this->CurrentRawKeypoints[k]->size() << " " << Utils::Plural(KeypointTypeNames.at(k)) << " ";
    std::cout << std::endl;
  }
}

//-----------------------------------------------------------------------------
void Slam::ComputeEgoMotion()
{
  PRINT_VERBOSE(2, "========== Ego-Motion ==========");

  // Reset ego-motion
  this->Trelative = Eigen::Isometry3d::Identity();

  // Linearly extrapolate previous motion to estimate new pose
  if (this->LogTrajectory.size() >= 2 &&
      (this->EgoMotion == EgoMotionMode::MOTION_EXTRAPOLATION ||
       this->EgoMotion == EgoMotionMode::MOTION_EXTRAPOLATION_AND_REGISTRATION))
  {
    // Estimate new Tworld with a constant velocity model
    const double t = Utils::PclStampToSec(this->CurrentFrames[0]->header.stamp);
    const double t1 = this->LogTrajectory[this->LogTrajectory.size() - 1].time;
    const double t0 = this->LogTrajectory[this->LogTrajectory.size() - 2].time;
    Eigen::Isometry3d nextTworldEstimation = LinearInterpolation(this->PreviousTworld, this->Tworld, t, t0, t1);
    this->Trelative = this->Tworld.inverse() * nextTworldEstimation;
  }

  // Refine Trelative estimation by registering current frame on previous one
  if (this->EgoMotion == EgoMotionMode::REGISTRATION ||
      this->EgoMotion == EgoMotionMode::MOTION_EXTRAPOLATION_AND_REGISTRATION)
  {
    // kd-tree to process fast nearest neighbor
    // among the keypoints of the previous pointcloud
    IF_VERBOSE(3, Utils::Timer::Init("EgoMotion : build KD tree"));
    std::map<Keypoint, KDTree> kdtreePrevious;
    // Kdtrees map initialization to parallelize their 
    // construction using OMP and avoid concurrency issues
    for (auto k : {EDGE, PLANE})
      kdtreePrevious[k] = KDTree();

    // The iteration is not directly on Keypoint types
    // because of openMP behaviour which needs int iteration on MSVC
    int nbKeypointTypes = static_cast<int>(KeypointTypes.size());
    #pragma omp parallel for num_threads(std::min(this->NbThreads, nbKeypointTypes))
    for (int i = 0; i < nbKeypointTypes; ++i)
    {
      Keypoint k = static_cast<Keypoint>(KeypointTypes[i]);
      if (kdtreePrevious.count(k))
        kdtreePrevious[k].Reset(this->PreviousRawKeypoints[k]);
    }

    if (this->Verbosity >= 2)
    {
      std::cout << "Keypoints extracted from previous frame : ";
      for (auto k : {EDGE, PLANE})
        std::cout << this->PreviousRawKeypoints[k]->size() << " " << Utils::Plural(KeypointTypeNames.at(k)) << " ";
      std::cout << std::endl;
    }

    IF_VERBOSE(3, Utils::Timer::StopAndDisplay("EgoMotion : build KD tree"));
    IF_VERBOSE(3, Utils::Timer::Init("Ego-Motion : whole ICP-LM loop"));

    // Reset ICP results
    unsigned int totalMatchedKeypoints = 0;

    // Init matching parameters
    KeypointsMatcher::Parameters matchingParams;
    matchingParams.NbThreads = this->NbThreads;
    matchingParams.SingleEdgePerRing = true;
    matchingParams.MaxDistanceForICPMatching = this->MaxDistanceForICPMatching;
    matchingParams.MinNbrMatchedKeypoints = this->MinNbrMatchedKeypoints;
    matchingParams.LineDistanceNbrNeighbors = this->EgoMotionLineDistanceNbrNeighbors;
    matchingParams.MinimumLineNeighborRejection = this->EgoMotionMinimumLineNeighborRejection;
    matchingParams.LineDistancefactor = this->EgoMotionLineDistancefactor;
    matchingParams.MaxLineDistance = this->EgoMotionMaxLineDistance;
    matchingParams.PlaneDistanceNbrNeighbors = this->EgoMotionPlaneDistanceNbrNeighbors;
    matchingParams.PlaneDistancefactor1 = this->EgoMotionPlaneDistancefactor1;
    matchingParams.PlaneDistancefactor2 = this->EgoMotionPlaneDistancefactor2;
    matchingParams.MaxPlaneDistance = this->EgoMotionMaxPlaneDistance;

    // ICP - Levenberg-Marquardt loop
    // At each step of this loop an ICP matching is performed. Once the keypoints
    // are matched, we estimate the the 6-DOF parameters by minimizing the
    // non-linear least square cost function using Levenberg-Marquardt algorithm.
    for (unsigned int icpIter = 0; icpIter < this->EgoMotionICPMaxIter; ++icpIter)
    {
      IF_VERBOSE(3, Utils::Timer::Init("  Ego-Motion : ICP"));

      // We want to estimate our 6-DOF parameters using a non linear least square
      // minimization. The non linear part comes from the parametrization of the
      // rotation endomorphism SO(3).
      // First, we need to build the point-to-line, point-to-plane and
      // point-to-blob ICP matches that will be optimized.
      // Then, we use CERES Levenberg-Marquardt optimization to minimize the problem.

      // Create a keypoints matcher
      // At each ICP iteration, the outliers removal is refined to be stricter
      double iterRatio = icpIter / static_cast<double>(this->EgoMotionICPMaxIter - 1);
      matchingParams.SaturationDistance = (1 - iterRatio) * this->EgoMotionInitSaturationDistance + iterRatio * this->EgoMotionFinalSaturationDistance;
      KeypointsMatcher matcher(matchingParams, this->Trelative);

      // Loop over keypoints to build the residuals
      for (auto k : {EDGE, PLANE})
        this->EgoMotionMatchingResults[k] = matcher.BuildMatchResiduals(this->CurrentRawKeypoints[k], kdtreePrevious[k], k);

      // Count matches and skip this frame
      // if there are too few geometric keypoints matched
      totalMatchedKeypoints = 0;
      for (auto k : {EDGE, PLANE})
        totalMatchedKeypoints += this->EgoMotionMatchingResults[k].NbMatches();

      if (totalMatchedKeypoints < this->MinNbrMatchedKeypoints)
      {
        PRINT_WARNING("Not enough keypoints, EgoMotion skipped for this frame.");
        break;
      }

      IF_VERBOSE(3, Utils::Timer::StopAndDisplay("  Ego-Motion : ICP"));
      IF_VERBOSE(3, Utils::Timer::Init("  Ego-Motion : LM optim"));

      // Init the optimizer with initial pose and parameters
      LocalOptimizer optimizer;
      optimizer.SetPosePrior(this->Trelative);
      optimizer.SetLMMaxIter(this->EgoMotionLMMaxIter);
      optimizer.SetNbThreads(this->NbThreads);

      // Add LiDAR ICP matches
      for (auto k : {EDGE, PLANE})
        optimizer.AddResiduals(this->EgoMotionMatchingResults[k].Residuals);

      // Run LM optimization
      ceres::Solver::Summary summary = optimizer.Solve();
      PRINT_VERBOSE(4, summary.BriefReport());

      // Get back optimized Trelative
      this->Trelative = optimizer.GetOptimizedPose();

      IF_VERBOSE(3, Utils::Timer::StopAndDisplay("  Ego-Motion : LM optim"));

      // If no L-M iteration has been made since the last ICP matching, it means
      // that we reached a local minimum for the ICP-LM algorithm.
      if (summary.num_successful_steps == 1)
      {
        break;
      }
    }

    IF_VERBOSE(3, Utils::Timer::StopAndDisplay("Ego-Motion : whole ICP-LM loop"));
    if (this->Verbosity >= 2)
    {
      std::cout << "Matched keypoints: " << totalMatchedKeypoints << " (";
      for (auto k : {EDGE, PLANE})
        std::cout << this->EgoMotionMatchingResults[k].NbMatches() << " " << Utils::Plural(KeypointTypeNames.at(k)) << " ";
      std::cout << ")" << std::endl;
    }
  }

  // Print EgoMotion results
  SET_COUT_FIXED_PRECISION(3);
  PRINT_VERBOSE(2, "Estimated Ego-Motion (motion since last frame):\n"
                   " translation = [" << this->Trelative.translation().transpose() << "] m\n"
                   " rotation    = [" << Utils::Rad2Deg(Utils::RotationMatrixToRPY(this->Trelative.linear())).transpose() << "] °");
  RESET_COUT_FIXED_PRECISION;
}

//-----------------------------------------------------------------------------
void Slam::Localization()
{
  PRINT_VERBOSE(2, "========== Localization ==========");

  // Integrate the relative motion to the world transformation
  this->PreviousTworld = this->Tworld;
  this->Tworld = this->PreviousTworld * this->Trelative;

  // Init undistorted keypoints clouds from raw points
  this->CurrentUndistortedKeypoints = this->CurrentRawKeypoints;

  // Init and run undistortion if required
  if (this->Undistortion)
  {
    IF_VERBOSE(3, Utils::Timer::Init("Localization : initial undistortion"));
    // Init the within frame motion interpolator time bounds
    this->InitUndistortion();
    // Undistort keypoints clouds
    this->RefineUndistortion();
    IF_VERBOSE(3, Utils::Timer::StopAndDisplay("Localization : initial undistortion"));
  }

  // Get keypoints from maps and build kd-trees for fast nearest neighbors search
  IF_VERBOSE(3, Utils::Timer::Init("Localization : keypoints extraction"));

  std::map<Keypoint, KDTree> kdtrees;
  // Initialization of std map elements to parallelize 
  // their construction with OMP avoiding concurrency issues
  for (auto k : KeypointTypes)
    kdtrees[k] = KDTree();

  // The iteration is not directly on Keypoint types
  // because of openMP behaviour which needs int iteration on MSVC
  int nbKeypointTypes = static_cast<int>(KeypointTypes.size());
  #pragma omp parallel for num_threads(std::min(this->NbThreads, nbKeypointTypes))
  for (int i = 0; i < nbKeypointTypes; ++i)
  {
    Keypoint k = static_cast<Keypoint>(KeypointTypes[i]);
    if (this->UseKeypoints[k])
    {
      // Estimate current keypoints bounding box
      PointCloud currWordKeypoints;
      pcl::transformPointCloud(*this->CurrentUndistortedKeypoints[k], currWordKeypoints, this->Tworld.matrix());
      Eigen::Vector4f minPoint, maxPoint;
      pcl::getMinMax3D(currWordKeypoints, minPoint, maxPoint);

      // Extract all points in maps lying in this bounding box
      PointCloud::Ptr localSubMap = this->LocalMaps[k]->Get(minPoint.head<3>().cast<double>().array(), maxPoint.head<3>().cast<double>().array());
      kdtrees[k].Reset(localSubMap);
    }
  }

  if (this->Verbosity >= 2)
  {
    std::cout << "Keypoints extracted from map : ";
    for (auto k : KeypointTypes)
      std::cout << kdtrees[k].GetInputCloud()->size() << " " << Utils::Plural(KeypointTypeNames.at(k)) << " ";
    std::cout << std::endl;
  }

  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("Localization : keypoints extraction"));
  IF_VERBOSE(3, Utils::Timer::Init("Localization : whole ICP-LM loop"));

  // Reset ICP results
  unsigned int totalMatchedKeypoints = 0;

  // Init matching parameters
  KeypointsMatcher::Parameters matchingParams;
  matchingParams.NbThreads = this->NbThreads;
  matchingParams.SingleEdgePerRing = false;
  matchingParams.MaxDistanceForICPMatching = this->MaxDistanceForICPMatching;
  matchingParams.MinNbrMatchedKeypoints = this->MinNbrMatchedKeypoints;
  matchingParams.LineDistanceNbrNeighbors = this->LocalizationLineDistanceNbrNeighbors;
  matchingParams.MinimumLineNeighborRejection = this->LocalizationMinimumLineNeighborRejection;
  matchingParams.LineDistancefactor = this->LocalizationLineDistancefactor;
  matchingParams.MaxLineDistance = this->LocalizationMaxLineDistance;
  matchingParams.PlaneDistanceNbrNeighbors = this->LocalizationPlaneDistanceNbrNeighbors;
  matchingParams.PlaneDistancefactor1 = this->LocalizationPlaneDistancefactor1;
  matchingParams.PlaneDistancefactor2 = this->LocalizationPlaneDistancefactor2;
  matchingParams.MaxPlaneDistance = this->LocalizationMaxPlaneDistance;
  matchingParams.BlobDistanceNbrNeighbors = this->LocalizationBlobDistanceNbrNeighbors;

  // ICP - Levenberg-Marquardt loop
  // At each step of this loop an ICP matching is performed. Once the keypoints
  // are matched, we estimate the the 6-DOF parameters by minimizing the
  // non-linear least square cost function using Levenberg-Marquardt algorithm.
  for (unsigned int icpIter = 0; icpIter < this->LocalizationICPMaxIter; ++icpIter)
  {
    IF_VERBOSE(3, Utils::Timer::Init("  Localization : ICP"));

    // We want to estimate our 6-DOF parameters using a non linear least square
    // minimization. The non linear part comes from the parametrization of the
    // rotation endomorphism SO(3).
    // First, we need to build the point-to-line, point-to-plane and
    // point-to-blob ICP matches that will be optimized.
    // Then, we use CERES Levenberg-Marquardt optimization to minimize problem.

    // Create a keypoints matcher
    // At each ICP iteration, the outliers removal is refined to be stricter
    double iterRatio = icpIter / static_cast<double>(this->LocalizationICPMaxIter - 1);
    matchingParams.SaturationDistance = (1 - iterRatio) * this->LocalizationInitSaturationDistance + iterRatio * this->LocalizationFinalSaturationDistance;
    KeypointsMatcher matcher(matchingParams, this->Tworld);

    // Loop over keypoints to build the point to line residuals
    for (auto k : KeypointTypes)
      this->LocalizationMatchingResults[k] = matcher.BuildMatchResiduals(this->CurrentUndistortedKeypoints[k], kdtrees[k], k);

    // Count matches and skip this frame
    // if there is too few geometric keypoints matched
    totalMatchedKeypoints = 0;
    for (auto k : KeypointTypes)
      totalMatchedKeypoints += this->LocalizationMatchingResults[k].NbMatches();

    if (totalMatchedKeypoints < this->MinNbrMatchedKeypoints)
    {
      // Reset state to previous one to avoid instability
      this->Trelative = Eigen::Isometry3d::Identity();
      this->Tworld = this->PreviousTworld;
      if (this->Undistortion)
        this->WithinFrameMotion.SetTransforms(Eigen::Isometry3d::Identity(), Eigen::Isometry3d::Identity());
      PRINT_ERROR("Not enough keypoints, Localization skipped for this frame.");
      break;
    }

    IF_VERBOSE(3, Utils::Timer::StopAndDisplay("  Localization : ICP"));
    IF_VERBOSE(3, Utils::Timer::Init("  Localization : LM optim"));

    // Init the optimizer with initial pose and parameters
    LocalOptimizer optimizer;
    optimizer.SetPosePrior(this->Tworld);
    optimizer.SetLMMaxIter(this->LocalizationLMMaxIter);
    optimizer.SetNbThreads(this->NbThreads);

    // Add LiDAR ICP matches
    for (auto k : KeypointTypes)
      optimizer.AddResiduals(this->LocalizationMatchingResults[k].Residuals);

    // *********************Add sensor constraints here*********************

    // Run LM optimization
    ceres::Solver::Summary summary = optimizer.Solve();
    PRINT_VERBOSE(4, summary.BriefReport());

    // Update Tworld and Trelative from optimization results
    this->Tworld = optimizer.GetOptimizedPose();
    this->Trelative = this->PreviousTworld.inverse() * this->Tworld;

    // Optionally refine undistortion
    if (this->Undistortion == UndistortionMode::REFINED)
      this->RefineUndistortion();

    IF_VERBOSE(3, Utils::Timer::StopAndDisplay("  Localization : LM optim"));

    // If no L-M iteration has been made since the last ICP matching, it means
    // that we reached a local minimum for the ICP-LM algorithm.
    // We evaluate the quality of the Tworld optimization using an approximate
    // computation of the variance covariance matrix.
    if ((summary.num_successful_steps == 1) || (icpIter == this->LocalizationICPMaxIter - 1))
    {
      this->LocalizationUncertainty = optimizer.EstimateRegistrationError();
      break;
    }
  }

  IF_VERBOSE(3, Utils::Timer::StopAndDisplay("Localization : whole ICP-LM loop"));

  // Optionally print localization optimization summary
  if (this->Verbosity >= 2)
  {
    SET_COUT_FIXED_PRECISION(3);
    std::cout << "Matched keypoints: " << totalMatchedKeypoints << " (";
    for (auto k : KeypointTypes)
      std::cout << this->LocalizationMatchingResults[k].NbMatches() << " " << Utils::Plural(KeypointTypeNames.at(k)) << " ";
    std::cout << ")"
              << "\nPosition uncertainty    = " << this->LocalizationUncertainty.PositionError    << " m"
              << " (along [" << this->LocalizationUncertainty.PositionErrorDirection.transpose()    << "])"
              << "\nOrientation uncertainty = " << this->LocalizationUncertainty.OrientationError << " °"
              << " (along [" << this->LocalizationUncertainty.OrientationErrorDirection.transpose() << "])"
              << std::endl;
    RESET_COUT_FIXED_PRECISION;
  }
}

//-----------------------------------------------------------------------------
void Slam::UpdateMapsUsingTworld()
{
  // The iteration is not directly on Keypoint types
  // because of openMP behaviour which needs int iteration on MSVC
  int nbKeypointTypes = static_cast<int>(KeypointTypes.size());
  #pragma omp parallel for num_threads(std::min(this->NbThreads, nbKeypointTypes))
  for (int i = 0; i < nbKeypointTypes; ++i)
  {
    Keypoint k = static_cast<Keypoint>(KeypointTypes[i]);
    this->CurrentWorldKeypoints[k]->clear();
    this->CurrentWorldKeypoints[k]->header = this->CurrentUndistortedKeypoints[k]->header;
    this->CurrentWorldKeypoints[k]->header.frame_id = this->WorldFrameId;
    if (this->UseKeypoints[k])
    {
      // Transform keypoints to WORLD coordinates
      this->CurrentWorldKeypoints[k]->points.reserve(this->CurrentUndistortedKeypoints[k]->size());
      for (const Point& p : *this->CurrentUndistortedKeypoints[k])
        this->CurrentWorldKeypoints[k]->push_back(Utils::TransformPoint(p, this->Tworld));
      // Add new keypoints to rolling grid
      this->LocalMaps[k]->Add(this->CurrentWorldKeypoints[k]);
    }
  }
}

//-----------------------------------------------------------------------------
void Slam::LogCurrentFrameState(double time, const std::string& frameId)
{
  // If logging is enabled
  if (this->LoggingTimeout)
  {
    // Save current frame data to buffer
    this->LogTrajectory.emplace_back(this->Tworld, time, frameId);
    this->LogCovariances.emplace_back(Utils::Matrix6dToStdArray36(this->LocalizationUncertainty.Covariance));
    for (auto k : KeypointTypes)
    {
      if (this->UseKeypoints[k])
        this->LogKeypoints[k].emplace_back(this->CurrentRawKeypoints[k], this->LoggingStorage);
    }

    // If a timeout is defined, forget too old data
    if (this->LoggingTimeout > 0)
    {
      // Forget all previous data older than LoggingTimeout, but keep at least 2 last transforms
      while (time - this->LogTrajectory.front().time > this->LoggingTimeout
             && this->LogTrajectory.size() > 2)
      {
        this->LogTrajectory.pop_front();
        this->LogCovariances.pop_front();
        for (auto k : KeypointTypes)
        {
          if (this->UseKeypoints[k])
            this->LogKeypoints[k].pop_front();
        }
      }
    }
  }

  // If logging is disabled, only keep last 2 transforms for latency compensation
  else
  {
    this->LogTrajectory.emplace_back(this->Tworld, time, frameId);
    while (this->LogTrajectory.size() > 2)
      this->LogTrajectory.pop_front();
  }
}

//==============================================================================
//   Undistortion helpers
//==============================================================================

//-----------------------------------------------------------------------------
Eigen::Isometry3d Slam::InterpolateScanPose(double time)
{
  if (this->LogTrajectory.empty())
    return this->Tworld;

  constexpr double MAX_EXTRAPOLATION_RATIO = 3.;
  const double prevPoseTime = this->LogTrajectory.back().time;
  const double currPoseTime = Utils::PclStampToSec(this->CurrentFrames[0]->header.stamp);

  if (std::abs(time / (currPoseTime - prevPoseTime)) > MAX_EXTRAPOLATION_RATIO)
  {
    PRINT_WARNING("Unable to interpolate scan pose from motion : extrapolation time is too far.");
    return this->Tworld;
  }

  return LinearInterpolation(this->PreviousTworld, this->Tworld, currPoseTime + time, prevPoseTime, currPoseTime);
}

//-----------------------------------------------------------------------------
void Slam::InitUndistortion()
{
  // Get 'time' field range
  double frameFirstTime = std::numeric_limits<double>::max();
  double frameLastTime  = std::numeric_limits<double>::lowest();
  for (auto k : KeypointTypes)
  {
    for (const Point& point: *this->CurrentUndistortedKeypoints[k])
    {
      frameFirstTime = std::min(frameFirstTime, point.time);
      frameLastTime  = std::max(frameLastTime, point.time);
    }
  }

  // Update interpolator timestamps and reset transforms
  this->WithinFrameMotion.SetTimes(frameFirstTime, frameLastTime);
  this->WithinFrameMotion.SetTransforms(Eigen::Isometry3d::Identity(), Eigen::Isometry3d::Identity());

  // Check time values
  if (this->WithinFrameMotion.GetTimeRange() < 1e-6)
  {
    // If frame duration is 0, it means that the time field is constant and cannot be used.
    // We reset timestamps to 0, to ensure no time offset will be used.
    PRINT_WARNING("'time' field is not properly set (constant value) and cannot be used for undistortion.");
    this->WithinFrameMotion.SetTimes(0., 0.);
  }
  else if (this->WithinFrameMotion.GetTimeRange() > 10.)
  {
    // If frame duration is bigger than 10 seconds, it is probably wrongly set
    PRINT_WARNING("'time' field looks not properly set (frame duration > 10 s) and can lead to faulty undistortion.");
  }
}

//-----------------------------------------------------------------------------
void Slam::RefineUndistortion()
{
  // Get previously applied undistortion
  Eigen::Isometry3d previousBaseBegin = this->WithinFrameMotion.GetH0();
  Eigen::Isometry3d previousBaseEnd = this->WithinFrameMotion.GetH1();

  // Extrapolate first and last poses to update within frame motion interpolator
  Eigen::Isometry3d worldToBaseBegin = this->InterpolateScanPose(this->WithinFrameMotion.GetTime0());
  Eigen::Isometry3d worldToBaseEnd = this->InterpolateScanPose(this->WithinFrameMotion.GetTime1());
  Eigen::Isometry3d baseToWorld = this->Tworld.inverse();
  Eigen::Isometry3d newBaseBegin = baseToWorld * worldToBaseBegin;
  Eigen::Isometry3d newBaseEnd = baseToWorld * worldToBaseEnd;
  this->WithinFrameMotion.SetTransforms(newBaseBegin, newBaseEnd);

  // Init the interpolator to use to remove previous undistortion and apply updated one
  auto transformInterpolator = this->WithinFrameMotion;
  transformInterpolator.SetTransforms(newBaseBegin * previousBaseBegin.inverse(),
                                      newBaseEnd   * previousBaseEnd.inverse());

  // Refine undistortion of keypoints clouds
  // The iteration is not directly on Keypoint types
  // because of openMP behaviour which needs int iteration on MSVC
  int nbKeypointTypes = static_cast<int>(KeypointTypes.size());
  #pragma omp parallel for num_threads(std::min(this->NbThreads, nbKeypointTypes))
  for (int i = 0; i < nbKeypointTypes; ++i)
  {
    Keypoint k = static_cast<Keypoint>(KeypointTypes[i]);
    for (Point& p : *this->CurrentUndistortedKeypoints[k])
      Utils::TransformPoint(p, transformInterpolator(p.time));
  }
}

//==============================================================================
//   Keypoints extraction parameters setting
//==============================================================================

//-----------------------------------------------------------------------------
std::map<uint8_t, Slam::KeypointExtractorPtr> Slam::GetKeyPointsExtractors() const
{
  return this->KeyPointsExtractors;
}
void Slam::SetKeyPointsExtractors(const std::map<uint8_t, KeypointExtractorPtr>& extractors)
{
  this->KeyPointsExtractors = extractors;
}

//-----------------------------------------------------------------------------
Slam::KeypointExtractorPtr Slam::GetKeyPointsExtractor(uint8_t deviceId) const
{
  return this->KeyPointsExtractors.count(deviceId) ? this->KeyPointsExtractors.at(deviceId) : KeypointExtractorPtr();
}
void Slam::SetKeyPointsExtractor(KeypointExtractorPtr extractor, uint8_t deviceId)
{
  this->KeyPointsExtractors[deviceId] = extractor;
}

//-----------------------------------------------------------------------------
Eigen::Isometry3d Slam::GetBaseToLidarOffset(uint8_t deviceId) const
{
  return this->BaseToLidarOffsets.count(deviceId) ? this->BaseToLidarOffsets.at(deviceId) : Eigen::UnalignedIsometry3d::Identity();
}
void Slam::SetBaseToLidarOffset(const Eigen::Isometry3d& transform, uint8_t deviceId)
{
  this->BaseToLidarOffsets[deviceId] = transform;
}

//==============================================================================
//   Rolling grids parameters setting
//==============================================================================

//-----------------------------------------------------------------------------
void Slam::ClearMaps()
{
  for (auto k : KeypointTypes)
    this->LocalMaps[k]->Reset();
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridLeafSize(Keypoint k, double size)
{
  this->LocalMaps[k]->SetLeafSize(size);
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridSize(int size)
{
  for (auto k : KeypointTypes)
    this->LocalMaps[k]->SetGridSize(size);
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridResolution(double resolution)
{
  for (auto k : KeypointTypes)
    this->LocalMaps[k]->SetVoxelResolution(resolution);
}

} // end of LidarSlam namespace