/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkDirect3DTexture.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkDirect3DShaderCache
 * @brief   manage Shader Programs within a context
 *
 * vtkDirect3DShaderCache manages shader program compilation and binding
*/

#ifndef vtkDirect3DShaderCache_h
#define vtkDirect3DShaderCache_h

#include "vtkRenderingDirect3DModule.h" // For export macro
#include "vtkObject.h"
#include "vtkDirect3DShader.h" // for vtkDirect3DShader::Type
#include <map> // for methods
class vtkDirect3DTransformFeedback;
class vtkDirect3DShaderProgram;
class vtkWindow;
class vtkDirect3DRenderWindow;

class VTKRENDERINGDIRECT3D_EXPORT vtkDirect3DShaderCache : public vtkObject
{
public:
  static vtkDirect3DShaderCache *New();
  vtkTypeMacro(vtkDirect3DShaderCache, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent) VTK_OVERRIDE;

  // make sure the specified shaders are compiled, linked, and bound
  virtual vtkDirect3DShaderProgram *ReadyShaderProgram(
    const char *vertexCode,
    const char *fragmentCode,
    const char *geometryCode);

  // make sure the specified shaders are compiled, linked, and bound
  // will increment the reference count on the shaders if it
  // needs to keep them around
  virtual vtkDirect3DShaderProgram *ReadyShaderProgram(
    std::map<vtkDirect3DShader::Type, vtkDirect3DShader *> shaders);

  // make sure the specified shaders are compiled, linked, and bound
  virtual vtkDirect3DShaderProgram *ReadyShaderProgram(
      vtkDirect3DShaderProgram *shader);

  /**
   * Release the current shader.  Basically go back to
   * having no shaders loaded.  This is useful for old
   * legacy code that relies on no shaders being loaded.
   */
  void ReleaseCurrentShader();

  /**
   * Free up any resources being used by the provided shader
   */
  virtual void ReleaseGraphicsResources(vtkWindow *win);

  /**
   * Get/Clear the last Shader bound, called by shaders as they release
   * their graphics resources
   */
  virtual void ClearLastShaderBound() { this->LastShaderBound = NULL; }
  vtkGetObjectMacro(LastShaderBound, vtkDirect3DShaderProgram);

  //@{
  /**
   * Get/Set the context. This does not increase the reference count of the
   * context to avoid reference loops.
   */
  void SetContext(vtkDirect3DRenderWindow *val) {
    this->Context = val; }
  vtkDirect3DRenderWindow* GetContext();
  //@}

protected:
  vtkDirect3DShaderCache();
  ~vtkDirect3DShaderCache();

  // perform System and Output replacments in place. Returns
  // the number of outputs
  virtual unsigned int ReplaceShaderValues(
    std::string &VSSource,
    std::string &FSSource,
    std::string &GSSource);

  virtual vtkDirect3DShaderProgram* GetShaderProgram(
    const char *vertexCode,
    const char *fragmentCode,
    const char *geometryCode);
  virtual vtkDirect3DShaderProgram* GetShaderProgram(
    std::map<vtkDirect3DShader::Type,vtkDirect3DShader *> shaders);
  virtual int BindShader(vtkDirect3DShaderProgram* shader);

  class Private;
  Private *Internal;
  vtkDirect3DShaderProgram *LastShaderBound;
  vtkDirect3DRenderWindow *Context;

private:
  vtkDirect3DShaderCache(const vtkDirect3DShaderCache&) VTK_DELETE_FUNCTION;
  void operator=(const vtkDirect3DShaderCache&) VTK_DELETE_FUNCTION;

};

#endif
