/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkDirect3DShaderCache.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkDirect3DShaderCache.h"

#include "vtkObjectFactory.h"

#include "vtkDirect3DConstantBufferObject.h"
#include "vtkDirect3DRenderWindow.h"
#include "vtkDirect3DShaderProgram.h"
#include "vtkDirect3DShader.h"

#include <cmath>
#include <sstream>



#include "vtksys/MD5.h"

class vtkDirect3DShaderCache::Private
{
public:
  vtksysMD5* md5;

  // map of hash to shader program structs
  std::map<std::string, vtkDirect3DShaderProgram *> ShaderPrograms;

  Private()
  {
  md5 = vtksysMD5_New();
  }

  ~Private()
  {
  vtksysMD5_Delete(this->md5);
  }

  //-----------------------------------------------------------------------------
  void ComputeMD5(const char* content,
                  const char* content2,
                  const char* content3,
                  std::string &hash)
  {
    unsigned char digest[16];
    char md5Hash[33];
    md5Hash[32] = '\0';

    vtksysMD5_Initialize(this->md5);
    if (content)
    {
      vtksysMD5_Append(this->md5,
        reinterpret_cast<const unsigned char *>(content),
        (int)strlen(content));
    }
    if (content2)
    {
      vtksysMD5_Append(this->md5,
        reinterpret_cast<const unsigned char *>(content2),
        (int)strlen(content2));
    }
    if (content3)
    {
      vtksysMD5_Append(this->md5,
        reinterpret_cast<const unsigned char *>(content3),
        (int)strlen(content3));
    }
    vtksysMD5_Finalize(this->md5, digest);
    vtksysMD5_DigestToHex(digest, md5Hash);

    hash = md5Hash;
  }


};

// ----------------------------------------------------------------------------
vtkStandardNewMacro(vtkDirect3DShaderCache);

// ----------------------------------------------------------------------------
vtkDirect3DShaderCache::vtkDirect3DShaderCache() : Internal(new Private)
{
  this->LastShaderBound  = NULL;
}

// ----------------------------------------------------------------------------
vtkDirect3DShaderCache::~vtkDirect3DShaderCache()
{
  typedef std::map<std::string,vtkDirect3DShaderProgram*>::const_iterator SMapIter;
  SMapIter iter = this->Internal->ShaderPrograms.begin();
  for ( ; iter != this->Internal->ShaderPrograms.end(); iter++)
  {
    iter->second->Delete();
  }

  delete this->Internal;
}

// perform System and Output replacments
unsigned int vtkDirect3DShaderCache::ReplaceShaderValues(
  std::string &VSSource,
  std::string &FSSource,
  std::string &GSSource)
{
  // first handle renaming any Fragment shader inputs
  // if we have a geometry shader. By deafult fragment shaders
  // assume their inputs come from a Vertex Shader. When we
  // have a Geometry shader we rename the frament shader inputs
  // to come from the geometry shader
  if (GSSource.size() > 0)
  {
    vtkDirect3DShaderProgram::Substitute(FSSource,"VSOut","GSOut");
  }

  return 0;
}

vtkDirect3DShaderProgram *vtkDirect3DShaderCache::ReadyShaderProgram(
    std::map<vtkDirect3DShader::Type,vtkDirect3DShader *> shaders)
{
  std::string VSSource = shaders[vtkDirect3DShader::Vertex]->GetSource();
  std::string FSSource = shaders[vtkDirect3DShader::Fragment]->GetSource();
  std::string GSSource = shaders[vtkDirect3DShader::Geometry]->GetSource();

  // fill in constant buffers
  vtkDirect3DShaderProgram::Substitute(VSSource,
    "//VTK::ConstantBuffer::Dec",
    shaders[vtkDirect3DShader::Vertex]->GetConstantBuffer()
      ->GetDeclaration().c_str());

  vtkDirect3DShaderProgram::Substitute(FSSource,
    "//VTK::ConstantBuffer::Dec",
    shaders[vtkDirect3DShader::Fragment]->GetConstantBuffer()
      ->GetDeclaration().c_str());

  unsigned int count =
    this->ReplaceShaderValues(VSSource,FSSource,GSSource);
  shaders[vtkDirect3DShader::Vertex]->SetSource(VSSource);
  shaders[vtkDirect3DShader::Fragment]->SetSource(FSSource);
  shaders[vtkDirect3DShader::Geometry]->SetSource(GSSource);

  vtkDirect3DShaderProgram *shader = this->GetShaderProgram(shaders);
  shader->SetNumberOfOutputs(count);

  return this->ReadyShaderProgram(shader);
}

// return NULL if there is an issue
vtkDirect3DShaderProgram *vtkDirect3DShaderCache::ReadyShaderProgram(
  const char *vertexCode, const char *fragmentCode, const char *geometryCode)
{
  // perform system wide shader replacements
  // desktops to not use percision statements
  std::string VSSource = vertexCode;
  std::string FSSource = fragmentCode;
  std::string GSSource = geometryCode;

  unsigned int count =
    this->ReplaceShaderValues(VSSource,FSSource,GSSource);
  vtkDirect3DShaderProgram *shader =
    this->GetShaderProgram(
      VSSource.c_str(), FSSource.c_str(), GSSource.c_str());
  shader->SetNumberOfOutputs(count);

  return this->ReadyShaderProgram(shader);
}

// return NULL if there is an issue
vtkDirect3DShaderProgram *vtkDirect3DShaderCache::ReadyShaderProgram(
    vtkDirect3DShaderProgram *shader)
{
  if (!shader)
  {
    return NULL;
  }

  // compile if needed
  if (!shader->GetCompiled() && !shader->CompileShader())
  {
    return NULL;
  }

  // bind if needed
  if (!this->BindShader(shader))
  {
    return NULL;
  }

  return shader;
}

vtkDirect3DShaderProgram *vtkDirect3DShaderCache::GetShaderProgram(
  std::map<vtkDirect3DShader::Type,vtkDirect3DShader *> shaders)
{
  // compute the MD5 and the check the map
  std::string result;
  this->Internal->ComputeMD5(
    shaders[vtkDirect3DShader::Vertex]->GetSource().c_str(),
    shaders[vtkDirect3DShader::Fragment]->GetSource().c_str(),
    shaders[vtkDirect3DShader::Geometry]->GetSource().c_str(), result);

  // does it already exist?
  typedef std::map<std::string,vtkDirect3DShaderProgram*>::const_iterator SMapIter;
  SMapIter found = this->Internal->ShaderPrograms.find(result);
  if (found == this->Internal->ShaderPrograms.end())
  {
    // create one
    vtkDirect3DShaderProgram *sps = vtkDirect3DShaderProgram::New();
    sps->SetVertexShader(shaders[vtkDirect3DShader::Vertex]);
    sps->SetFragmentShader(shaders[vtkDirect3DShader::Fragment]);
    sps->SetGeometryShader(shaders[vtkDirect3DShader::Geometry]);
    sps->SetMD5Hash(result); // needed?
    sps->SetContext(this->Context);
    this->Internal->ShaderPrograms.insert(std::make_pair(result, sps));
    return sps;
  }
  else
  {
    return found->second;
  }
}

vtkDirect3DShaderProgram *vtkDirect3DShaderCache::GetShaderProgram(
  const char *vertexCode,
  const char *fragmentCode,
  const char *geometryCode)
{
  // compute the MD5 and the check the map
  std::string result;
  this->Internal->ComputeMD5(vertexCode, fragmentCode, geometryCode, result);

  // does it already exist?
  typedef std::map<std::string,vtkDirect3DShaderProgram*>::const_iterator SMapIter;
  SMapIter found = this->Internal->ShaderPrograms.find(result);
  if (found == this->Internal->ShaderPrograms.end())
  {
    // create one
    vtkDirect3DShaderProgram *sps = vtkDirect3DShaderProgram::New();
    sps->GetVertexShader()->SetSource(vertexCode);
    sps->GetFragmentShader()->SetSource(fragmentCode);
    if (geometryCode != NULL)
    {
      sps->GetGeometryShader()->SetSource(geometryCode);
    }
    sps->SetMD5Hash(result); // needed?
    this->Internal->ShaderPrograms.insert(std::make_pair(result, sps));
    return sps;
  }
  else
  {
    return found->second;
  }
}

void vtkDirect3DShaderCache::ReleaseGraphicsResources(vtkWindow *win)
{
  // NOTE:
  // In the current implementation as of October 26th, if a shader
  // program is created by ShaderCache then it should make sure
  // that it releases the graphics resouces used by these programs.
  // It is not wisely for callers to do that since then they would
  // have to loop over all the programs were in use and invoke
  // release graphics resources individually.

  this->ReleaseCurrentShader();

  typedef std::map<std::string,vtkDirect3DShaderProgram*>::const_iterator SMapIter;
  SMapIter iter = this->Internal->ShaderPrograms.begin();
  for ( ; iter != this->Internal->ShaderPrograms.end(); iter++)
  {
    iter->second->ReleaseGraphicsResources(win);
  }
}

void vtkDirect3DShaderCache::ReleaseCurrentShader()
{
  // release prior shader
  if (this->LastShaderBound)
  {
    this->LastShaderBound->Release();
    this->LastShaderBound = NULL;
  }
}

int vtkDirect3DShaderCache::BindShader(vtkDirect3DShaderProgram* shader)
{
  if (this->LastShaderBound == shader)
  {
    return 1;
  }

  // release prior shader
  if (this->LastShaderBound)
  {
    this->LastShaderBound->Release();
  }
  shader->Bind();
  this->LastShaderBound = shader;
  return 1;
}


// ----------------------------------------------------------------------------
void vtkDirect3DShaderCache::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
}
