/*=========================================================================

  Program:   Visualization Toolkit

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkDirect3DShader
 * @brief   encapsulate a glsl shader
 *
 * vtkDirect3DShader represents a shader, vertex, fragment, geometry etc
*/

#ifndef vtkDirect3DShader_h
#define vtkDirect3DShader_h

#include "vtkRenderingDirect3DModule.h" // for export macro
#include "vtkObject.h"

#include <string> // For member variables.
#include <vector> // For member variables.

class vtkDirect3DConstantBufferObject;
class vtkDirect3DRenderWindow;

/**
 * @brief Vertex or Fragment shader, combined into a ShaderProgram.
 *
 * This class creates a Vertex, Fragment or Geometry shader, that can be attached to a
 * ShaderProgram in order to render geometry etc.
 */

class VTKRENDERINGDIRECT3D_EXPORT vtkDirect3DShader : public vtkObject
{
public:
  static vtkDirect3DShader *New();
  vtkTypeMacro(vtkDirect3DShader, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent);


  /** Available shader types. */
  enum Type {
    Vertex,    /**< Vertex shader */
    Fragment,  /**< Fragment shader */
    Geometry,  /**< Geometry shader */
    Unknown    /**< Unknown (default) */
  };

  /** Set the shader type. */
  void SetType(Type type);

  /** Get the shader type, typically Vertex or Fragment. */
  Type GetType() const { return this->ShaderType; }

  /** Set the shader source to the supplied string. */
  void SetSource(const std::string &source);

  /** Get the source for the shader. */
  std::string GetSource() const { return this->Source; }

  void SetContext(vtkDirect3DRenderWindow *);

  void BindConstantBuffer();

  vtkDirect3DConstantBufferObject *GetConstantBuffer() {
    return this->ConstantBuffer; }

protected:
  vtkDirect3DShader();
  ~vtkDirect3DShader();

  Type ShaderType;

  std::string Source;
  vtkDirect3DConstantBufferObject *ConstantBuffer;
  vtkDirect3DRenderWindow *Context;

private:
  vtkDirect3DShader(const vtkDirect3DShader&) VTK_DELETE_FUNCTION;
  void operator=(const vtkDirect3DShader&) VTK_DELETE_FUNCTION;
};


#endif
